// Copyright 2012 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "base/metrics/sample_map.h"

#include "base/check.h"
#include "base/numerics/safe_conversions.h"

namespace base {

typedef HistogramBase::Count Count;
typedef HistogramBase::Sample Sample;

namespace {

// An iterator for going through a SampleMap. The logic here is identical
// to that of the iterator for PersistentSampleMap but with different data
// structures. Changes here likely need to be duplicated there.
template <typename T, typename I>
class IteratorTemplateSM : public SampleCountIterator {
 public:
  explicit IteratorTemplateSM(T& sample_counts)
      : iter_(sample_counts.begin()), end_(sample_counts.end()) {
    SkipEmptyBuckets();
  }

  ~IteratorTemplateSM() override;

  // SampleCountIterator:
  bool Done() const override { return iter_ == end_; }
  void Next() override {
    DCHECK(!Done());
    ++iter_;
    SkipEmptyBuckets();
  }
  void Get(HistogramBase::Sample* min,
           int64_t* max,
           HistogramBase::Count* count) override;

 private:
  void SkipEmptyBuckets() {
    while (!Done() && iter_->second == 0) {
      ++iter_;
    }
  }

  I iter_;
  const I end_;
};

typedef std::map<HistogramBase::Sample, HistogramBase::Count> SampleToCountMapSM;
typedef IteratorTemplateSM<const SampleToCountMapSM,
                         SampleToCountMapSM::const_iterator>
    SampleMapIterator;

template <>
SampleMapIterator::~IteratorTemplateSM() = default;

// Get() for an iterator of a SampleMap.
template <>
void SampleMapIterator::Get(Sample* min, int64_t* max, Count* count) {
  DCHECK(!Done());
  *min = iter_->first;
  *max = strict_cast<int64_t>(iter_->first) + 1;
  // We do not have to do the following atomically -- if the caller needs thread
  // safety, they should use a lock. And since this is in local memory, if a
  // lock is used, we know the value would not be concurrently modified by a
  // different process (in contrast to PersistentSampleMap, where the value in
  // shared memory may be modified concurrently by a subprocess).
  *count = iter_->second;
}

typedef IteratorTemplateSM<SampleToCountMapSM, SampleToCountMapSM::iterator>
    ExtractingSampleMapIterator;

template <>
ExtractingSampleMapIterator::~IteratorTemplateSM() {
  // Ensure that the user has consumed all the samples in order to ensure no
  // samples are lost.
  DCHECK(Done());
}

// Get() for an extracting iterator of a SampleMap.
template <>
void ExtractingSampleMapIterator::Get(Sample* min, int64_t* max, Count* count) {
  DCHECK(!Done());
  *min = iter_->first;
  *max = strict_cast<int64_t>(iter_->first) + 1;
  // We do not have to do the following atomically -- if the caller needs thread
  // safety, they should use a lock. And since this is in local memory, if a
  // lock is used, we know the value would not be concurrently modified by a
  // different process (in contrast to PersistentSampleMap, where the value in
  // shared memory may be modified concurrently by a subprocess).
  *count = iter_->second;
  iter_->second = 0;
}

}  // namespace

SampleMap::SampleMap() : SampleMap(0) {}

SampleMap::SampleMap(uint64_t id)
    : HistogramSamples(id, std::make_unique<LocalMetadata>()) {}

SampleMap::~SampleMap() = default;

void SampleMap::Accumulate(Sample value, Count count) {
  // We do not have to do the following atomically -- if the caller needs
  // thread safety, they should use a lock. And since this is in local memory,
  // if a lock is used, we know the value would not be concurrently modified
  // by a different process (in contrast to PersistentSampleMap, where the
  // value in shared memory may be modified concurrently by a subprocess).
  sample_counts_[value] += count;
  IncreaseSumAndCount(strict_cast<int64_t>(count) * value, count);
}

Count SampleMap::GetCount(Sample value) const {
  auto it = sample_counts_.find(value);
  if (it == sample_counts_.end())
    return 0;
  return it->second;
}

Count SampleMap::TotalCount() const {
  Count count = 0;
  for (const auto& entry : sample_counts_) {
    count += entry.second;
  }
  return count;
}

std::unique_ptr<SampleCountIterator> SampleMap::Iterator() const {
  return std::make_unique<SampleMapIterator>(sample_counts_);
}

std::unique_ptr<SampleCountIterator> SampleMap::ExtractingIterator() {
  return std::make_unique<ExtractingSampleMapIterator>(sample_counts_);
}

bool SampleMap::AddSubtractImpl(SampleCountIterator* iter, Operator op) {
  Sample min;
  int64_t max;
  Count count;
  for (; !iter->Done(); iter->Next()) {
    iter->Get(&min, &max, &count);
    if (strict_cast<int64_t>(min) + 1 != max)
      return false;  // SparseHistogram only supports bucket with size 1.

    // We do not have to do the following atomically -- if the caller needs
    // thread safety, they should use a lock. And since this is in local memory,
    // if a lock is used, we know the value would not be concurrently modified
    // by a different process (in contrast to PersistentSampleMap, where the
    // value in shared memory may be modified concurrently by a subprocess).
    sample_counts_[min] += (op == HistogramSamples::ADD) ? count : -count;
  }
  return true;
}

}  // namespace base
