// %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
// %                                                               %
// %   IsoWood Include File version 0.2                            %
// %                                                               %
// %   IsoWood2.inc - macros for wood objects                      %
// %                                                               %
// %   for use with megapov 0.6                                    %
// %   written August 2000 - November 2000                         %
// %   by Christoph Hormann <chris_hormann@gmx.de>                 %
// %   newest version can be found at                              %
// %   http://www.schunter.etc.tu-bs.de/~chris/iso_wood.html       %
// %                                                               %
// %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
// %                                                               %
// %   Feel free to use this file for your povray work, you are    %
// %   also allowed to modify and redistribute it as long as you   %
// %   leave these comments intact and highlight any changes you   %
// %   make.                                                       %
// %                                                               %
// %   I encourage you to contact me if you have ideas for         %
// %   improvements                                                %
// %                                                               %
// %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


#version unofficial MegaPov 0.6;

//------------------------------------------------------------------------------------------

//--------------------------------------------
//===== This is the global quality parameter
//===== used by most of the macros:
//=====
//=====   0: plain color shape
//=====   1: textured shape
//=====   2: isosurface
//--------------------------------------------

#declare CH_Wood_Quality = 2;

//--------------------------------------------------------------------
//===== These are general parameters for all isosurfaces used.
//===== You can change them here or in another pov file if you like
//--------------------------------------------------------------------

#declare CH_Wood_ISOmethod = 2;
#declare CH_Wood_ISOaccuracy = 0.001;
#declare CH_Wood_ISOmaxgrad = 1.0;
#declare CH_Wood_ISOeval = true;


//==========================================================================================
//===== Warp macros ========================================================================
//==========================================================================================

#macro IW_Warp_01 (Seed, xSize, ySize, zSize)

  #local SZ=0.1;
  #local Density=0.14;

  warp {
    black_hole <xSize*0.5, 0, 0>, SZ
    falloff 1.5
    strength 2
    repeat <min(xSize, SZ/Density),min(ySize, SZ/Density), SZ/Density>
    turbulence <0.0, 0.35, 0.5>
    inverse
  }
#end //-------------------------------------------------------------------------------------

#macro IW_Warp_02 (Seed, xSize, ySize, zSize)

  #local SZ=0.1;
  #local Density=0.1;

  warp {
    black_hole <xSize*0.5, 0, 0>, SZ
    falloff 2.5
    strength 2
    repeat <min(xSize, SZ/Density),min(ySize, SZ/Density), SZ/Density>
    turbulence <0.02, 0.75, 0.75>
    inverse
  }

  #local SZ=0.15;
  #local Density=0.1;

  warp {
    black_hole <xSize*0.5, 0, 0>, SZ
    falloff 2.1
    strength 3.7
    repeat <min(xSize, SZ/Density),min(ySize, 0.7*SZ/Density), 0.8*SZ/Density>
    turbulence <0.0,0.4, 0.4>
    inverse
  }

#end //-------------------------------------------------------------------------------------


#macro IW_Warp_03 (Seed, xSize, ySize, zSize)

  warp {
    displace {
      bozo
      color_map { [0 rgb 0][0.5 rgb 0.01][1 rgb 0.08] }
      turbulence 0.2
      translate 3*x
      scale 0.3
      type 1
    }
  }
#end //-------------------------------------------------------------------------------------


#macro IW_Warp_04 (Seed, xSize, ySize, zSize)

  warp {
    displace {
      bozo
      color_map { [0 rgb 0][0.5 rgb 0.01][1 rgb 0.1] }
      turbulence 0.04
      scale 0.4
      type 1
    }
  }

  #local Cnt=0;
  #local rd=seed(Seed);

  #while (Cnt < 100)

    #local PsY=rand(rd)*ySize;
    #local PsZ=rand(rd)*zSize;

    #if (eval_pigment (
      bozo
      color_map { [0.6 rgb 0.0][0.9 rgb 0.7] }
      turbulence 0.04
      scale 0.4 , <xSize*0.5, PsY, PsZ>).red > 0.2)

    warp {
      black_hole <xSize*0.5, PsY, PsZ>, 0.1
      falloff 2.1
      strength 4.7
      inverse
    }

    #local Cnt=Cnt+1;

    #end

  #end

  #local Cnt=0;
  #local rd=seed(Seed+5);

  #while (Cnt < 100)

    #local PsY=rand(rd)*ySize;
    #local PsZ=rand(rd)*8;

    #if (eval_pigment (
      bozo
      color_map { [0.6 rgb 0.0][0.9 rgb 0.7] }
      turbulence 0.04
      scale 0.4 , <-xSize*0.5, PsY, PsZ>).red > 0.2)

    warp {
      black_hole <-xSize*0.5, PsY, PsZ>, 0.1
      falloff 2.1
      strength 4.7
      inverse
    }

    #local Cnt=Cnt+1;

    #end


  #end

#end //-------------------------------------------------------------------------------------


//------------------------------------------------------------------------------------------


//==========================================================================================
//===== Wood_Plank_01 - rounded box ========================================================
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------
#macro Wood_Plank_01 (Seed, xSize, ySize, zSize, Rotate, Translate, Round, Strength, Pigm, Fin, Norm)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*xSize;
#local Trsy = (rand(rd)-0.5)*Translate*ySize;
#local Trsz = (rand(rd)-0.5)*Translate*zSize;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local P_WoodA =
   pigment {
     Pigm
     color_map {
       [ 0.000  color rgb 0.0 ]
       [ 1.000  color rgb 1.0 ]
     }
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y
   }

#local T_WoodA = texture {
   pigment {
     Pigm
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y
   }
   normal { Norm }
   finish { Fin }
}


object {

#switch (CH_Wood_Quality)
  #case (1)
    box{ <-xSize*0.5,-ySize*0.5,-zSize*0.5>,
         < xSize*0.5, ySize*0.5, zSize*0.5>
      texture { T_WoodA }
    }
    #break
  #case (2)
    #local fnShape=function{ "rounded_box",<Round,xSize*0.5,ySize*0.5,zSize*0.5>}
    #local fnBumps=function{ pigment { P_WoodA }}

    isosurface{
      function {fnShape(x,y,z)+fnBumps(x,y,z)*Strength}
      contained_by{
        box{ <-xSize*0.5-0.1,-ySize*0.5-0.1,-zSize*0.5-0.1>,
             < xSize*0.5+0.1, ySize*0.5+0.1, zSize*0.5+0.1> }
      }

      method CH_Wood_ISOmethod
      accuracy CH_Wood_ISOaccuracy
      #if (CH_Wood_ISOeval)
         eval
      #else
         max_gradient CH_Wood_ISOmaxgrad
      #end

      texture {
        T_WoodA
      }
    }
    #break
#else  //0
  box{ <-xSize*0.5,-ySize*0.5,-zSize*0.5>,
       < xSize*0.5, ySize*0.5, zSize*0.5>
    texture { pigment { color rgb <1,0,0> } }
  }
#end

  translate <xSize*0.5, ySize*0.5, zSize*0.5>
}

#end  //------------------------------------------------------------------------------------


//==========================================================================================
//===== Wood_Plank_01W - rounded box with warp =============================================
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------
#macro Wood_Plank_01W (Seed, xSize, ySize, zSize, Rotate, Translate,
                      Round, Strength, Pigm, Fin, Norm, WarpNbr)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*xSize;
#local Trsy = (rand(rd)-0.5)*Translate*ySize;
#local Trsz = (rand(rd)-0.5)*Translate*zSize;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local P_WoodA =
   pigment {
     Pigm
     color_map {
       [ 0.000  color rgb 0.0 ]
       [ 1.000  color rgb 1.0 ]
     }
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y

     #switch (WarpNbr)
       #case (1)
         IW_Warp_01 (Seed, xSize, ySize, zSize)
         #break
       #case (2)
         IW_Warp_02 (Seed, xSize, ySize, zSize)
         #break
       #case (3)
         IW_Warp_03 (Seed, xSize, ySize, zSize)
         #break
       #case (4)
         IW_Warp_04 (Seed, xSize, ySize, zSize)
         #break
       #case (5)
         IW_Warp_05 (Seed, xSize, ySize, zSize)
         #break
       #case (6)
         IW_Warp_06 (Seed, xSize, ySize, zSize)
         #break
       #case (7)
         IW_Warp_07 (Seed, xSize, ySize, zSize)
         #break
       #case (8)
         IW_Warp_08 (Seed, xSize, ySize, zSize)
         #break
       #case (9)
         IW_Warp_09 (Seed, xSize, ySize, zSize)
     #end
   }

#local T_WoodA = texture {
   pigment {
     Pigm
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y

     #switch (WarpNbr)
       #case (1)
         IW_Warp_01 (Seed, xSize, ySize, zSize)
         #break
       #case (2)
         IW_Warp_02 (Seed, xSize, ySize, zSize)
         #break
       #case (3)
         IW_Warp_03 (Seed, xSize, ySize, zSize)
         #break
       #case (4)
         IW_Warp_04 (Seed, xSize, ySize, zSize)
         #break
       #case (5)
         IW_Warp_05 (Seed, xSize, ySize, zSize)
         #break
       #case (6)
         IW_Warp_06 (Seed, xSize, ySize, zSize)
         #break
       #case (7)
         IW_Warp_07 (Seed, xSize, ySize, zSize)
         #break
       #case (8)
         IW_Warp_08 (Seed, xSize, ySize, zSize)
         #break
       #case (9)
         IW_Warp_09 (Seed, xSize, ySize, zSize)
     #end
   }
   normal { Norm }
   finish { Fin }
}


object {

#switch (CH_Wood_Quality)
  #case (1)
    box{ <-xSize*0.5,-ySize*0.5,-zSize*0.5>,
         < xSize*0.5, ySize*0.5, zSize*0.5>
      texture { T_WoodA }
    }
    #break
  #case (2)
    #local fnShape=function{ "rounded_box",<Round,xSize*0.5,ySize*0.5,zSize*0.5>}
    #local fnBumps=function{ pigment { P_WoodA }}

    isosurface{
      function {fnShape(x,y,z)+fnBumps(x,y,z)*Strength}
      contained_by{
        box{ <-xSize*0.5-0.1,-ySize*0.5-0.1,-zSize*0.5-0.1>,
             < xSize*0.5+0.1, ySize*0.5+0.1, zSize*0.5+0.1> }
      }

      method CH_Wood_ISOmethod
      accuracy CH_Wood_ISOaccuracy
      #if (CH_Wood_ISOeval)
         eval
      #else
         max_gradient CH_Wood_ISOmaxgrad
      #end

      texture {
        T_WoodA
      }
    }
    #break
#else  //0
  box{ <-xSize*0.5,-ySize*0.5,-zSize*0.5>,
       < xSize*0.5, ySize*0.5, zSize*0.5>
    texture { pigment { color rgb <1,0,0> } }
  }
#end

  translate <xSize*0.5, ySize*0.5, zSize*0.5>
}

#end  //------------------------------------------------------------------------------------



//==========================================================================================
//===== Wood_Plank_01T - rounded box with separate texture =================================
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------
#macro Wood_Plank_01T (Seed, xSize, ySize, zSize, Rotate, Translate,
                       Round, Strength, Pigm, Text)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*xSize;
#local Trsy = (rand(rd)-0.5)*Translate*ySize;
#local Trsz = (rand(rd)-0.5)*Translate*zSize;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local P_WoodA =
   pigment {
     Pigm
     color_map {
       [ 0.000  color rgb 0.0 ]
       [ 1.000  color rgb 1.0 ]
     }
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y

   }

#local T_WoodA = texture {
   Text
   translate <Trsx, Trsy, Trsz>
   rotate Rotx*x
   rotate Roty*y

}


object {

#switch (CH_Wood_Quality)
  #case (1)
    box{ <-xSize*0.5,-ySize*0.5,-zSize*0.5>,
         < xSize*0.5, ySize*0.5, zSize*0.5>
      texture { T_WoodA }
    }
    #break
  #case (2)
    #local fnShape=function{ "rounded_box",<Round,xSize*0.5,ySize*0.5,zSize*0.5>}
    #local fnBumps=function{ pigment { P_WoodA }}

    isosurface{
      function {fnShape(x,y,z)+fnBumps(x,y,z)*Strength}
      contained_by{
        box{ <-xSize*0.5-0.1,-ySize*0.5-0.1,-zSize*0.5-0.1>,
             < xSize*0.5+0.1, ySize*0.5+0.1, zSize*0.5+0.1> }
      }

      method CH_Wood_ISOmethod
      accuracy CH_Wood_ISOaccuracy
      #if (CH_Wood_ISOeval)
         eval
      #else
         max_gradient CH_Wood_ISOmaxgrad
      #end

      texture {
        T_WoodA
      }
    }
    #break
#else  //0
  box{ <-xSize*0.5,-ySize*0.5,-zSize*0.5>,
       < xSize*0.5, ySize*0.5, zSize*0.5>
    texture { pigment { color rgb <1,0,0> } }
  }
#end

  translate <xSize*0.5, ySize*0.5, zSize*0.5>
}

#end  //------------------------------------------------------------------------------------


//==========================================================================================
//===== Wood_Plank_02 - rounded box cut at the front/back ==================================
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------

#macro Wood_Plank_02 (Seed, xSize, ySize, zSize, Rotate, Translate,
                      Round, Strength, Strength2, Pigm, Fin, Norm)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*xSize;
#local Trsy = (rand(rd)-0.5)*Translate*ySize;
#local Trsz = (rand(rd)-0.5)*Translate*zSize;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local P_WoodA =
   pigment {
     Pigm
     color_map {
       [ 0.000  color rgb 0.0 ]
       [ 1.000  color rgb 1.0 ]
     }
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y
   }

#local T_WoodA = texture {
   pigment {
     Pigm
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y

   }
   normal { Norm }
   finish { Fin }
}

object {

#switch (CH_Wood_Quality)
  #case (1)
    box{ <-xSize*0.5,-ySize*0.5,-zSize*0.5>,
         < xSize*0.5, ySize*0.5, zSize*0.5>
      texture { T_WoodA }
    }
    #break
  #case (2)
    #local fnShape =function{ "rounded_box",<Round,xSize*0.5,ySize*0.5,zSize*0.5+0.2>}
    #local fnShape2=function{ (abs(z)-zSize*0.5) }
    #local fnBumps =function{ pigment { P_WoodA }}


    isosurface{
      function {(fnShape(x,y,z)+fnBumps(x,y,z)*Strength) & (fnShape2(x,y,z)+fnBumps(x,y,z)*Strength2)}
      contained_by{
        box{ <-xSize*0.5-0.1,-ySize*0.5-0.1,-zSize*0.5-0.1>,
             < xSize*0.5+0.1, ySize*0.5+0.1, zSize*0.5+0.1> }
      }

      method CH_Wood_ISOmethod
      accuracy CH_Wood_ISOaccuracy
      #if (CH_Wood_ISOeval)
         eval
      #else
         max_gradient CH_Wood_ISOmaxgrad
      #end

      texture {
        T_WoodA
      }
    }
    #break
#else  //0
  box{ <-xSize*0.5,-ySize*0.5,-zSize*0.5>,
       < xSize*0.5, ySize*0.5, zSize*0.5>
    texture { pigment { color rgb <1,0,0> } }
  }
#end

  translate <xSize*0.5, ySize*0.5, zSize*0.5>
}


#end  //------------------------------------------------------------------------------------


//==========================================================================================
//===== Wood_Plank_02E - rounded box cut at the front/back increased roughness to ends =====
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------

#macro Wood_Plank_02E (Seed, xSize, ySize, zSize, Rotate, Translate,
                       Round, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*xSize;
#local Trsy = (rand(rd)-0.5)*Translate*ySize;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local P_WoodA =
   pigment {
     gradient z
     triangle_wave

     pigment_map {
       [0.2 Pigm color_map { [0 rgb 0.0][1 rgb Strength/Strength2 ] } scale 1/zSize ]
       [1.0 Pigm color_map { [0 rgb 0.0][1 rgb 1.0] } scale 1/zSize ]
     }

     scale zSize

     translate <Trsx, Trsy, 0>
     rotate Rotx*x
     rotate Roty*y

     #switch (WarpNbr)
       #case (1)
         IW_Warp_01 (Seed, xSize, ySize, zSize)
         #break
       #case (2)
         IW_Warp_02 (Seed, xSize, ySize, zSize)
         #break
       #case (3)
         IW_Warp_03 (Seed, xSize, ySize, zSize)
         #break
       #case (4)
         IW_Warp_04 (Seed, xSize, ySize, zSize)
         #break
       #case (5)
         IW_Warp_05 (Seed, xSize, ySize, zSize)
         #break
       #case (6)
         IW_Warp_06 (Seed, xSize, ySize, zSize)
         #break
       #case (7)
         IW_Warp_07 (Seed, xSize, ySize, zSize)
         #break
       #case (8)
         IW_Warp_08 (Seed, xSize, ySize, zSize)
         #break
       #case (9)
         IW_Warp_09 (Seed, xSize, ySize, zSize)
     #end

   }

#local T_WoodA = texture {
   pigment {
     Pigm
     translate <Trsx, Trsy, 0>
     rotate Rotx*x
     rotate Roty*y

     #switch (WarpNbr)
       #case (1)
         IW_Warp_01 (Seed, xSize, ySize, zSize)
         #break
       #case (2)
         IW_Warp_02 (Seed, xSize, ySize, zSize)
         #break
       #case (3)
         IW_Warp_03 (Seed, xSize, ySize, zSize)
         #break
       #case (4)
         IW_Warp_04 (Seed, xSize, ySize, zSize)
         #break
       #case (5)
         IW_Warp_05 (Seed, xSize, ySize, zSize)
         #break
       #case (6)
         IW_Warp_06 (Seed, xSize, ySize, zSize)
         #break
       #case (7)
         IW_Warp_07 (Seed, xSize, ySize, zSize)
         #break
       #case (8)
         IW_Warp_08 (Seed, xSize, ySize, zSize)
         #break
       #case (9)
         IW_Warp_09 (Seed, xSize, ySize, zSize)
     #end

   }
   normal { Norm }
   finish { Fin }
}

object {

#switch (CH_Wood_Quality)
  #case (1)
    box{ <-xSize*0.5,-ySize*0.5,-zSize*0.5>,
         < xSize*0.5, ySize*0.5, zSize*0.5>
      texture { T_WoodA }
    }
    #break
  #case (2)
    #local fnShape =function{ "rounded_box",<Round,xSize*0.5,ySize*0.5,zSize*0.5+0.2>}
    #local fnShape2=function{ (abs(z)-zSize*0.5) }
    #local fnBumps =function{ pigment { P_WoodA }}


    isosurface{
      function { (fnShape(x,y,z) & fnShape2(x,y,z))+fnBumps(x,y,z)*Strength2 }
      contained_by{
        box{ <-xSize*0.5-0.1,-ySize*0.5-0.1,-zSize*0.5-0.1>,
             < xSize*0.5+0.1, ySize*0.5+0.1, zSize*0.5+0.1> }
      }

      method CH_Wood_ISOmethod
      accuracy CH_Wood_ISOaccuracy
      #if (CH_Wood_ISOeval)
         eval
      #else
         max_gradient CH_Wood_ISOmaxgrad
      #end

      texture {
        T_WoodA
      }
    }
    #break
#else  //0
  box{ <-xSize*0.5,-ySize*0.5,-zSize*0.5>,
       < xSize*0.5, ySize*0.5, zSize*0.5>
    texture { pigment { color rgb <1,0,0> } }
  }
#end

  translate <xSize*0.5, ySize*0.5, zSize*0.5>
}


#end  //------------------------------------------------------------------------------------


//==========================================================================================
//===== Wood_Plank_02W - rounded box cut at the front/back with warp =======================
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------

#macro Wood_Plank_02W (Seed, xSize, ySize, zSize, Rotate, Translate,
                       Round, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*xSize;
#local Trsy = (rand(rd)-0.5)*Translate*ySize;
#local Trsz = (rand(rd)-0.5)*Translate*zSize;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local P_WoodA =
   pigment {
     Pigm
     color_map {
       [ 0.000  color rgb 0.0 ]
       [ 1.000  color rgb 1.0 ]
     }
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y

     #switch (WarpNbr)
       #case (1)
         IW_Warp_01 (Seed, xSize, ySize, zSize)
         #break
       #case (2)
         IW_Warp_02 (Seed, xSize, ySize, zSize)
         #break
       #case (3)
         IW_Warp_03 (Seed, xSize, ySize, zSize)
         #break
       #case (4)
         IW_Warp_04 (Seed, xSize, ySize, zSize)
         #break
       #case (5)
         IW_Warp_05 (Seed, xSize, ySize, zSize)
         #break
       #case (6)
         IW_Warp_06 (Seed, xSize, ySize, zSize)
         #break
       #case (7)
         IW_Warp_07 (Seed, xSize, ySize, zSize)
         #break
       #case (8)
         IW_Warp_08 (Seed, xSize, ySize, zSize)
         #break
       #case (9)
         IW_Warp_09 (Seed, xSize, ySize, zSize)
     #end
   }

#local T_WoodA = texture {
   pigment {
     Pigm

     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y

     #switch (WarpNbr)
       #case (1)
         IW_Warp_01 (Seed, xSize, ySize, zSize)
         #break
       #case (2)
         IW_Warp_02 (Seed, xSize, ySize, zSize)
         #break
       #case (3)
         IW_Warp_03 (Seed, xSize, ySize, zSize)
         #break
       #case (4)
         IW_Warp_04 (Seed, xSize, ySize, zSize)
         #break
       #case (5)
         IW_Warp_05 (Seed, xSize, ySize, zSize)
         #break
       #case (6)
         IW_Warp_06 (Seed, xSize, ySize, zSize)
         #break
       #case (7)
         IW_Warp_07 (Seed, xSize, ySize, zSize)
         #break
       #case (8)
         IW_Warp_08 (Seed, xSize, ySize, zSize)
         #break
       #case (9)
         IW_Warp_09 (Seed, xSize, ySize, zSize)
     #end

   }
   normal { Norm }
   finish { Fin }
}

object {

#switch (CH_Wood_Quality)
  #case (1)
    box{ <-xSize*0.5,-ySize*0.5,-zSize*0.5>,
         < xSize*0.5, ySize*0.5, zSize*0.5>
      texture { T_WoodA }
    }
    #break
  #case (2)
    #local fnShape =function{ "rounded_box",<Round,xSize*0.5,ySize*0.5,zSize*0.5+0.2>}
    #local fnShape2=function{ (abs(z)-zSize*0.5) }
    #local fnBumps =function{ pigment { P_WoodA }}


    isosurface{
      function {(fnShape(x,y,z)+fnBumps(x,y,z)*Strength) & (fnShape2(x,y,z)+fnBumps(x,y,z)*Strength2)}
      contained_by{
        box{ <-xSize*0.5-0.1,-ySize*0.5-0.1,-zSize*0.5-0.1>,
             < xSize*0.5+0.1, ySize*0.5+0.1, zSize*0.5+0.1> }
      }

      method CH_Wood_ISOmethod
      accuracy CH_Wood_ISOaccuracy
      #if (CH_Wood_ISOeval)
         eval
      #else
         max_gradient CH_Wood_ISOmaxgrad
      #end

      texture {
        T_WoodA
      }
    }
    #break
#else  //0
  box{ <-xSize*0.5,-ySize*0.5,-zSize*0.5>,
       < xSize*0.5, ySize*0.5, zSize*0.5>
    texture { pigment { color rgb <1,0,0> } }
  }
#end

  translate <xSize*0.5, ySize*0.5, zSize*0.5>
}


#end  //------------------------------------------------------------------------------------


//==========================================================================================
//===== Wood_Plank_02T - rounded box cut at the front/back with separate texture ===========
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------

#macro Wood_Plank_02T (Seed, xSize, ySize, zSize, Rotate, Translate,
                       Round, Strength, Strength2, Pigm, Text)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*xSize;
#local Trsy = (rand(rd)-0.5)*Translate*ySize;
#local Trsz = (rand(rd)-0.5)*Translate*zSize;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local P_WoodA =
   pigment {
     Pigm

     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y
   }

#local T_WoodA = texture {
   Text
   translate <Trsx, Trsy, Trsz>
   rotate Rotx*x
   rotate Roty*y
}

object {

#switch (CH_Wood_Quality)
  #case (1)
    box{ <-xSize*0.5,-ySize*0.5,-zSize*0.5>,
         < xSize*0.5, ySize*0.5, zSize*0.5>
      texture { T_WoodA }
    }
    #break
  #case (2)
    #local fnShape =function{ "rounded_box",<Round,xSize*0.5,ySize*0.5,zSize*0.5+0.2>}
    #local fnShape2=function{ (abs(z)-zSize*0.5) }
    #local fnBumps =function{ pigment { P_WoodA }}


    isosurface{
      function {(fnShape(x,y,z)+fnBumps(x,y,z)*Strength) & (fnShape2(x,y,z)+fnBumps(x,y,z)*Strength2)}
      contained_by{
        box{ <-xSize*0.5-0.1,-ySize*0.5-0.1,-zSize*0.5-0.1>,
             < xSize*0.5+0.1, ySize*0.5+0.1, zSize*0.5+0.1> }
      }

      method CH_Wood_ISOmethod
      accuracy CH_Wood_ISOaccuracy
      #if (CH_Wood_ISOeval)
         eval
      #else
         max_gradient CH_Wood_ISOmaxgrad
      #end

      texture {
        T_WoodA
      }
    }
    #break
#else  //0
  box{ <-xSize*0.5,-ySize*0.5,-zSize*0.5>,
       < xSize*0.5, ySize*0.5, zSize*0.5>
    texture { pigment { color rgb <1,0,0> } }
  }
#end

  translate <xSize*0.5, ySize*0.5, zSize*0.5>
}


#end  //------------------------------------------------------------------------------------


//==========================================================================================
//===== Wood_Plank_03 - chamfered box cut at the front/back ================================
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------

#macro Wood_Plank_03 (Seed, xSize, ySize, zSize, Rotate, Translate,
                      Round, Round2, Strength, Strength2, Pigm, Fin, Norm)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*xSize;
#local Trsy = (rand(rd)-0.5)*Translate*ySize;
#local Trsz = (rand(rd)-0.5)*Translate*zSize;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local P_WoodA =
   pigment {
     Pigm
     color_map {
       [ 0.000  color rgb 0.0 ]
       [ 1.000  color rgb 1.0 ]
     }
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y
   }

#local T_WoodA = texture {
   pigment {
     Pigm
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y
   }
   normal { Norm }
   finish { Fin }
}

object {

#switch (CH_Wood_Quality)
  #case (1)
    box{ <-xSize*0.5,-ySize*0.5,-zSize*0.5>,
         < xSize*0.5, ySize*0.5, zSize*0.5>
      texture { T_WoodA }
    }
    #break
  #case (2)
    #local fnShape =function{ (abs(x)-xSize*0.5) & (abs(y)-ySize*0.5)}
    #local fnShape2=function{ (abs(z)-zSize*0.5) }
    #local fnShape3=function{ (abs(x)+abs(y)-(xSize+ySize)*0.5)+Round }
    #local fnShape4=function{ ((abs(z)+abs(y)-(zSize+ySize)*0.5)+Round2) &
                              ((abs(z)+abs(x)-(zSize+xSize)*0.5)+Round2) }
    #local fnBumps =function{ pigment { P_WoodA }}


    isosurface{
      function {((fnShape (x,y,z)&fnShape3(x,y,z))+fnBumps(x,y,z)*Strength) &
                ((fnShape2(x,y,z)&fnShape4(x,y,z))+fnBumps(x,y,z)*Strength2)}
      contained_by{
        box{ <-xSize*0.5-0.1,-ySize*0.5-0.1,-zSize*0.5-0.1>,
             < xSize*0.5+0.1, ySize*0.5+0.1, zSize*0.5+0.1> }
      }

      method CH_Wood_ISOmethod
      accuracy CH_Wood_ISOaccuracy
      #if (CH_Wood_ISOeval)
         eval
      #else
         max_gradient CH_Wood_ISOmaxgrad
      #end

      texture {
        T_WoodA
      }
    }
    #break
#else  //0
  box{ <-xSize*0.5,-ySize*0.5,-zSize*0.5>,
       < xSize*0.5, ySize*0.5, zSize*0.5>
    texture { pigment { color rgb <1,0,0> } }
  }
#end

  translate <xSize*0.5, ySize*0.5, zSize*0.5>
}


#end  //------------------------------------------------------------------------------------



//==========================================================================================
//===== Wood_Plank_03E - chamfered box cut at the front/back increased roughness to ends ===
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------

#macro Wood_Plank_03E (Seed, xSize, ySize, zSize, Rotate, Translate,
                       Round, Round2, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*xSize;
#local Trsy = (rand(rd)-0.5)*Translate*ySize;
#local Trsz = (rand(rd)-0.5)*Translate*zSize;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local P_WoodA =
   pigment {
     gradient z
     triangle_wave

     pigment_map {
       [0.2 Pigm color_map { [0 rgb 0.0][1 rgb Strength/Strength2 ] } scale 1/zSize ]
       [1.0 Pigm color_map { [0 rgb 0.0][1 rgb 1.0] } scale 1/zSize ]
     }

     scale zSize

     translate <Trsx, Trsy, 0>
     rotate Rotx*x
     rotate Roty*y

     #switch (WarpNbr)
       #case (1)
         IW_Warp_01 (Seed, xSize, ySize, zSize)
         #break
       #case (2)
         IW_Warp_02 (Seed, xSize, ySize, zSize)
         #break
       #case (3)
         IW_Warp_03 (Seed, xSize, ySize, zSize)
         #break
       #case (4)
         IW_Warp_04 (Seed, xSize, ySize, zSize)
         #break
       #case (5)
         IW_Warp_05 (Seed, xSize, ySize, zSize)
         #break
       #case (6)
         IW_Warp_06 (Seed, xSize, ySize, zSize)
         #break
       #case (7)
         IW_Warp_07 (Seed, xSize, ySize, zSize)
         #break
       #case (8)
         IW_Warp_08 (Seed, xSize, ySize, zSize)
         #break
       #case (9)
         IW_Warp_09 (Seed, xSize, ySize, zSize)
     #end

   }

#local T_WoodA = texture {
   pigment {
     Pigm
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y

     #switch (WarpNbr)
       #case (1)
         IW_Warp_01 (Seed, xSize, ySize, zSize)
         #break
       #case (2)
         IW_Warp_02 (Seed, xSize, ySize, zSize)
         #break
       #case (3)
         IW_Warp_03 (Seed, xSize, ySize, zSize)
         #break
       #case (4)
         IW_Warp_04 (Seed, xSize, ySize, zSize)
         #break
       #case (5)
         IW_Warp_05 (Seed, xSize, ySize, zSize)
         #break
       #case (6)
         IW_Warp_06 (Seed, xSize, ySize, zSize)
         #break
       #case (7)
         IW_Warp_07 (Seed, xSize, ySize, zSize)
         #break
       #case (8)
         IW_Warp_08 (Seed, xSize, ySize, zSize)
         #break
       #case (9)
         IW_Warp_09 (Seed, xSize, ySize, zSize)
     #end

   }
   normal { Norm }
   finish { Fin }
}

object {

#switch (CH_Wood_Quality)
  #case (1)
    box{ <-xSize*0.5,-ySize*0.5,-zSize*0.5>,
         < xSize*0.5, ySize*0.5, zSize*0.5>
      texture { T_WoodA }
    }
    #break
  #case (2)
    #local fnShape =function{ (abs(x)-xSize*0.5) & (abs(y)-ySize*0.5)}
    #local fnShape2=function{ (abs(z)-zSize*0.5) }
    #local fnShape3=function{ (abs(x)+abs(y)-(xSize+ySize)*0.5)+Round }
    #local fnShape4=function{ ((abs(z)+abs(y)-(zSize+ySize)*0.5)+Round2) &
                              ((abs(z)+abs(x)-(zSize+xSize)*0.5)+Round2) }
    #local fnBumps =function{ pigment { P_WoodA }}


    isosurface{
      function {((fnShape (x,y,z)&fnShape3(x,y,z)&
                  fnShape2(x,y,z)&fnShape4(x,y,z))+fnBumps(x,y,z)*Strength2)}
      contained_by{
        box{ <-xSize*0.5-0.1,-ySize*0.5-0.1,-zSize*0.5-0.1>,
             < xSize*0.5+0.1, ySize*0.5+0.1, zSize*0.5+0.1> }
      }

      method CH_Wood_ISOmethod
      accuracy CH_Wood_ISOaccuracy
      #if (CH_Wood_ISOeval)
         eval
      #else
         max_gradient CH_Wood_ISOmaxgrad
      #end

      texture {
        T_WoodA
      }
    }
    #break
#else  //0
  box{ <-xSize*0.5,-ySize*0.5,-zSize*0.5>,
       < xSize*0.5, ySize*0.5, zSize*0.5>
    texture { pigment { color rgb <1,0,0> } }
  }
#end

  translate <xSize*0.5, ySize*0.5, zSize*0.5>
}


#end  //------------------------------------------------------------------------------------




//==========================================================================================
//===== Wood_Plank_03W - chamfered box cut at the front/back with warp =====================
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------

#macro Wood_Plank_03W (Seed, xSize, ySize, zSize, Rotate, Translate,
                       Round, Round2, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*xSize;
#local Trsy = (rand(rd)-0.5)*Translate*ySize;
#local Trsz = (rand(rd)-0.5)*Translate*zSize;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local P_WoodA =
   pigment {
     Pigm
     color_map {
       [ 0.000  color rgb 0.0 ]
       [ 1.000  color rgb 1.0 ]
     }
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y

     #switch (WarpNbr)
       #case (1)
         IW_Warp_01 (Seed, xSize, ySize, zSize)
         #break
       #case (2)
         IW_Warp_02 (Seed, xSize, ySize, zSize)
         #break
       #case (3)
         IW_Warp_03 (Seed, xSize, ySize, zSize)
         #break
       #case (4)
         IW_Warp_04 (Seed, xSize, ySize, zSize)
         #break
       #case (5)
         IW_Warp_05 (Seed, xSize, ySize, zSize)
         #break
       #case (6)
         IW_Warp_06 (Seed, xSize, ySize, zSize)
         #break
       #case (7)
         IW_Warp_07 (Seed, xSize, ySize, zSize)
         #break
       #case (8)
         IW_Warp_08 (Seed, xSize, ySize, zSize)
         #break
       #case (9)
         IW_Warp_09 (Seed, xSize, ySize, zSize)
     #end
   }

#local T_WoodA = texture {
   pigment {
     Pigm
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y

     #switch (WarpNbr)
       #case (1)
         IW_Warp_01 (Seed, xSize, ySize, zSize)
         #break
       #case (2)
         IW_Warp_02 (Seed, xSize, ySize, zSize)
         #break
       #case (3)
         IW_Warp_03 (Seed, xSize, ySize, zSize)
         #break
       #case (4)
         IW_Warp_04 (Seed, xSize, ySize, zSize)
         #break
       #case (5)
         IW_Warp_05 (Seed, xSize, ySize, zSize)
         #break
       #case (6)
         IW_Warp_06 (Seed, xSize, ySize, zSize)
         #break
       #case (7)
         IW_Warp_07 (Seed, xSize, ySize, zSize)
         #break
       #case (8)
         IW_Warp_08 (Seed, xSize, ySize, zSize)
         #break
       #case (9)
         IW_Warp_09 (Seed, xSize, ySize, zSize)
     #end
   }
   normal { Norm }
   finish { Fin }
}

object {

#switch (CH_Wood_Quality)
  #case (1)
    box{ <-xSize*0.5,-ySize*0.5,-zSize*0.5>,
         < xSize*0.5, ySize*0.5, zSize*0.5>
      texture { T_WoodA }
    }
    #break
  #case (2)
    #local fnShape =function{ (abs(x)-xSize*0.5) & (abs(y)-ySize*0.5)}
    #local fnShape2=function{ (abs(z)-zSize*0.5) }
    #local fnShape3=function{ (abs(x)+abs(y)-(xSize+ySize)*0.5)+Round }
    #local fnShape4=function{ ((abs(z)+abs(y)-(zSize+ySize)*0.5)+Round2) &
                              ((abs(z)+abs(x)-(zSize+xSize)*0.5)+Round2) }
    #local fnBumps =function{ pigment { P_WoodA }}


    isosurface{
      function {((fnShape (x,y,z)&fnShape3(x,y,z))+fnBumps(x,y,z)*Strength) &
                ((fnShape2(x,y,z)&fnShape4(x,y,z))+fnBumps(x,y,z)*Strength2)}
      contained_by{
        box{ <-xSize*0.5-0.1,-ySize*0.5-0.1,-zSize*0.5-0.1>,
             < xSize*0.5+0.1, ySize*0.5+0.1, zSize*0.5+0.1> }
      }

      method CH_Wood_ISOmethod
      accuracy CH_Wood_ISOaccuracy
      #if (CH_Wood_ISOeval)
         eval
      #else
         max_gradient CH_Wood_ISOmaxgrad
      #end

      texture {
        T_WoodA
      }
    }
    #break
#else  //0
  box{ <-xSize*0.5,-ySize*0.5,-zSize*0.5>,
       < xSize*0.5, ySize*0.5, zSize*0.5>
    texture { pigment { color rgb <1,0,0> } }
  }
#end

  translate <xSize*0.5, ySize*0.5, zSize*0.5>
}


#end  //------------------------------------------------------------------------------------



//==========================================================================================
//===== Wood_Plank_03T - chamfered box cut at the front/back with separate texture =========
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------

#macro Wood_Plank_03T (Seed, xSize, ySize, zSize, Rotate, Translate,
                       Round, Round2, Strength, Strength2, Pigm, Text)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*xSize;
#local Trsy = (rand(rd)-0.5)*Translate*ySize;
#local Trsz = (rand(rd)-0.5)*Translate*zSize;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local P_WoodA =
   pigment {
     Pigm
     color_map {
       [ 0.000  color rgb 0.0 ]
       [ 1.000  color rgb 1.0 ]
     }
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y
   }

#local T_WoodA = texture {
   Text
   translate <Trsx, Trsy, Trsz>
   rotate Rotx*x
   rotate Roty*y
}

object {

#switch (CH_Wood_Quality)
  #case (1)
    box{ <-xSize*0.5,-ySize*0.5,-zSize*0.5>,
         < xSize*0.5, ySize*0.5, zSize*0.5>
      texture { T_WoodA }
    }
    #break
  #case (2)
    #local fnShape =function{ (abs(x)-xSize*0.5) & (abs(y)-ySize*0.5)}
    #local fnShape2=function{ (abs(z)-zSize*0.5) }
    #local fnShape3=function{ (abs(x)+abs(y)-(xSize+ySize)*0.5)+Round }
    #local fnShape4=function{ ((abs(z)+abs(y)-(zSize+ySize)*0.5)+Round2) &
                              ((abs(z)+abs(x)-(zSize+xSize)*0.5)+Round2) }
    #local fnBumps =function{ pigment { P_WoodA }}


    isosurface{
      function {((fnShape (x,y,z)&fnShape3(x,y,z))+fnBumps(x,y,z)*Strength) &
                ((fnShape2(x,y,z)&fnShape4(x,y,z))+fnBumps(x,y,z)*Strength2)}
      contained_by{
        box{ <-xSize*0.5-0.1,-ySize*0.5-0.1,-zSize*0.5-0.1>,
             < xSize*0.5+0.1, ySize*0.5+0.1, zSize*0.5+0.1> }
      }

      method CH_Wood_ISOmethod
      accuracy CH_Wood_ISOaccuracy
      #if (CH_Wood_ISOeval)
         eval
      #else
         max_gradient CH_Wood_ISOmaxgrad
      #end

      texture {
        T_WoodA
      }
    }
    #break
#else  //0
  box{ <-xSize*0.5,-ySize*0.5,-zSize*0.5>,
       < xSize*0.5, ySize*0.5, zSize*0.5>
    texture { pigment { color rgb <1,0,0> } }
  }
#end

  translate <xSize*0.5, ySize*0.5, zSize*0.5>
}


#end  //------------------------------------------------------------------------------------



//==========================================================================================
//===== Wood_Cyl_01 - cylindrical wood =====================================================
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------

#macro Wood_Cyl_01 (Seed, Radius, Length, Rotate, Translate, Strength, Pigm, Fin, Norm)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*Radius;
#local Trsy = (rand(rd)-0.5)*Translate*Radius;
#local Trsz = (rand(rd)-0.5)*Translate*Radius;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local P_WoodA =
   pigment {
     Pigm
     color_map {
       [ 0.000  color rgb 0.0 ]
       [ 1.000  color rgb 1.0 ]
     }
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y
   }

#local T_WoodA = texture {
   pigment {
     Pigm
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y
   }
   normal { Norm }
   finish { Fin }
}


object {

#switch (CH_Wood_Quality)
  #case (1)
    cylinder{ <0,0,-Length*0.5>, < 0, 0, Length*0.5>, Radius
      texture { T_WoodA }
    }
    #break
  #case (2)

    #local fnShape =function{ y^2 + x^2 - Radius^2 }
    #local fnShape2=function{ (abs(z)-Length*0.5) }
    #local fnBumps =function{ pigment { P_WoodA }}

    isosurface{
      function {(fnShape(x,y,z) & fnShape2(x,y,z))+fnBumps(x,y,z)*Strength}

      method CH_Wood_ISOmethod
      accuracy CH_Wood_ISOaccuracy
      #if (CH_Wood_ISOeval)
         eval
      #else
         max_gradient CH_Wood_ISOmaxgrad
      #end

      contained_by{
         box{ < -Radius-0.1,-Radius-0.1, -Length*0.5-0.1>,
              <  Radius+0.1, Radius+0.1, Length*0.5+0.1> }
      }
      texture {
         T_WoodA
      }
    }
    #break
#else  //0
  cylinder{ <0,0,-Length*0.5>, < 0, 0, Length*0.5>, Radius
    texture { pigment { color rgb <1,0,0> } }
  }
#end

  translate <0, 0, Length*0.5>
}




#end  //------------------------------------------------------------------------------------


//==========================================================================================
//===== Wood_Cyl_01T - cylindrical wood with separate texture  =============================
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------

#macro Wood_Cyl_01T (Seed, Radius, Length, Rotate, Translate, Strength, Pigm, Text)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*Radius;
#local Trsy = (rand(rd)-0.5)*Translate*Radius;
#local Trsz = (rand(rd)-0.5)*Translate*Radius;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local P_WoodA =
   pigment {
     Pigm
     color_map {
       [ 0.000  color rgb 0.0 ]
       [ 1.000  color rgb 1.0 ]
     }
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y
   }

#local T_WoodA = texture {
   Text
   translate <Trsx, Trsy, Trsz>
   rotate Rotx*x
   rotate Roty*y
}


object {

#switch (CH_Wood_Quality)
  #case (1)
    cylinder{ <0,0,-Length*0.5>, < 0, 0, Length*0.5>, Radius
      texture { T_WoodA }
    }
    #break
  #case (2)

    #local fnShape =function{ y^2 + x^2 - Radius^2 }
    #local fnShape2=function{ (abs(z)-Length*0.5) }
    #local fnBumps =function{ pigment { P_WoodA }}

    isosurface{
      function {(fnShape(x,y,z) & fnShape2(x,y,z))+fnBumps(x,y,z)*Strength}

      method CH_Wood_ISOmethod
      accuracy CH_Wood_ISOaccuracy
      #if (CH_Wood_ISOeval)
         eval
      #else
         max_gradient CH_Wood_ISOmaxgrad
      #end

      contained_by{
         box{ < -Radius-0.1,-Radius-0.1, -Length*0.5-0.1>,
              <  Radius+0.1, Radius+0.1, Length*0.5+0.1> }
      }
      texture {
         T_WoodA
      }
    }
    #break
#else  //0
  cylinder{ <0,0,-Length*0.5>, < 0, 0, Length*0.5>, Radius
    texture { pigment { color rgb <1,0,0> } }
  }
#end

  translate <0, 0, Length*0.5>
}


#end  //------------------------------------------------------------------------------------


//==========================================================================================
//===== Wood_Cyl_02 - cylindrical wood with cut & chamfered ends ===========================
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------

#macro Wood_Cyl_02 (Seed, Radius, Length, Rotate, Translate, Round,
                    Strength, Strength2, Pigm, Fin, Norm)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*Radius;
#local Trsy = (rand(rd)-0.5)*Translate*Radius;
#local Trsz = (rand(rd)-0.5)*Translate*Radius;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local P_WoodA =
   pigment {
     Pigm
     color_map {
       [ 0.000  color rgb 0.0 ]
       [ 1.000  color rgb 1.0 ]
     }
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y
   }

#local T_WoodA = texture {
   pigment {
     Pigm
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y
   }
   normal { Norm }
   finish { Fin }
}

object {

#switch (CH_Wood_Quality)
  #case (1)
    cylinder{ <0,0,-Length*0.5>, < 0, 0, Length*0.5>, Radius
      texture { T_WoodA }
    }
    #break
  #case (2)

    #local fnShape =function{ y^2 + x^2 - Radius^2 }
    #local fnShape2=function{ (abs(z)-Length*0.5) }
    #local fnShape3=function{ sqrt( sqr(x)+sqr(y) )+(abs(z)-Length/2-(1-Round)*Radius)   }
    #local fnBumps =function{ pigment { P_WoodA }}


    isosurface{
      function {(fnShape(x,y,z)+fnBumps(x,y,z)*Strength) &
                ((fnShape2(x,y,z) & fnShape3(x,y,z))+fnBumps(x,y,z)*Strength2)}

      contained_by{
         box{ < -Radius-0.1,-Radius-0.1, -Length*0.5-0.1>,
              <  Radius+0.1, Radius+0.1, Length*0.5+0.1> }
      }

      method CH_Wood_ISOmethod
      accuracy CH_Wood_ISOaccuracy
      #if (CH_Wood_ISOeval)
         eval
      #else
         max_gradient CH_Wood_ISOmaxgrad
      #end

      texture {
         T_WoodA
      }
    }

    #break
#else  //0
  cylinder{ <0,0,-Length*0.5>, < 0, 0, Length*0.5>, Radius
    texture { pigment { color rgb <1,0,0> } }
  }
#end

  translate <0, 0, Length*0.5>
}


#end  //------------------------------------------------------------------------------------



//==========================================================================================
//===== Wood_Cyl_02T - cylindrical wood with cut & chamfered ends with separate texture ====
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------

#macro Wood_Cyl_02T (Seed, Radius, Length, Rotate, Translate, Round,
                     Strength, Strength2, Pigm, Text)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*Radius;
#local Trsy = (rand(rd)-0.5)*Translate*Radius;
#local Trsz = (rand(rd)-0.5)*Translate*Radius;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local P_WoodA =
   pigment {
     Pigm
     color_map {
       [ 0.000  color rgb 0.0 ]
       [ 1.000  color rgb 1.0 ]
     }
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y
   }

#local T_WoodA = texture {
   Text
   translate <Trsx, Trsy, Trsz>
   rotate Rotx*x
   rotate Roty*y
}

object {

#switch (CH_Wood_Quality)
  #case (1)
    cylinder{ <0,0,-Length*0.5>, < 0, 0, Length*0.5>, Radius
      texture { T_WoodA }
    }
    #break
  #case (2)

    #local fnShape =function{ y^2 + x^2 - Radius^2 }
    #local fnShape2=function{ (abs(z)-Length*0.5) }
    #local fnShape3=function{ sqrt( sqr(x)+sqr(y) )+(abs(z)-Length/2-(1-Round)*Radius)   }
    #local fnBumps =function{ pigment { P_WoodA }}


    isosurface{
      function {(fnShape(x,y,z)+fnBumps(x,y,z)*Strength) &
                ((fnShape2(x,y,z) & fnShape3(x,y,z))+fnBumps(x,y,z)*Strength2)}

      contained_by{
         box{ < -Radius-0.1,-Radius-0.1, -Length*0.5-0.1>,
              <  Radius+0.1, Radius+0.1, Length*0.5+0.1> }
      }

      method CH_Wood_ISOmethod
      accuracy CH_Wood_ISOaccuracy
      #if (CH_Wood_ISOeval)
         eval
      #else
         max_gradient CH_Wood_ISOmaxgrad
      #end

      texture {
         T_WoodA
      }
    }

    #break
#else  //0
  cylinder{ <0,0,-Length*0.5>, < 0, 0, Length*0.5>, Radius
    texture { pigment { color rgb <1,0,0> } }
  }
#end

  translate <0, 0, Length*0.5>
}


#end  //------------------------------------------------------------------------------------



//==========================================================================================
//===== Wood_Cyl2_01 - half cylindrical wood ===============================================
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------

#macro Wood_Cyl2_01 (Seed, Radius, Length, Rotate, Translate, Strength, Pigm, Fin, Norm)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*Radius;
#local Trsy = (rand(rd)-0.5)*Translate*Radius;
#local Trsz = (rand(rd)-0.5)*Translate*Radius;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local P_WoodA =
   pigment {
     Pigm
     color_map {
       [ 0.000  color rgb 0.0 ]
       [ 1.000  color rgb 1.0 ]
     }
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y
   }

#local T_WoodA = texture {
   pigment {
     Pigm
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y
   }
   normal { Norm }
   finish { Fin }
}

object {

#switch (CH_Wood_Quality)
  #case (1)
    intersection {
      cylinder{ <0,0,-Length*0.5>, < 0, 0, Length*0.5>, Radius }
      plane { y, 0 }
      texture { T_WoodA }
    }
    #break
  #case (2)

    #local fnShape =function{ y^2 + x^2 - Radius^2 }
    #local fnShape2=function{ (abs(z)-Length*0.5) }
    #local fnBumps =function{ pigment { P_WoodA }}
    #local fnHalf  =function{ -y }


    isosurface{
      function {(fnShape(x,y,z) & fnShape2(x,y,z) & fnHalf(x,y,z))+fnBumps(x,y,z)*Strength}

      method CH_Wood_ISOmethod
      accuracy CH_Wood_ISOaccuracy
      #if (CH_Wood_ISOeval)
         eval
      #else
         max_gradient CH_Wood_ISOmaxgrad
      #end

      contained_by{
         box{ < -Radius-0.1,-Radius-0.1, -Length*0.5-0.1>,
              <  Radius+0.1, Radius+0.1, Length*0.5+0.1> }
      }
      texture {
         T_WoodA
      }
    }

    #break
#else  //0
  intersection {
    cylinder{ <0,0,-Length*0.5>, < 0, 0, Length*0.5>, Radius }
    plane { y, 0 }
    texture { pigment { color rgb <1,0,0> } }
  }
#end

  translate <0, 0, Length*0.5>
}


#end  //------------------------------------------------------------------------------------


//==========================================================================================
//===== Wood_Cyl2_01T - half cylindrical wood with separate texture ========================
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------

#macro Wood_Cyl2_01T (Seed, Radius, Length, Rotate, Translate, Strength, Pigm, Text)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*Radius;
#local Trsy = (rand(rd)-0.5)*Translate*Radius;
#local Trsz = (rand(rd)-0.5)*Translate*Radius;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local P_WoodA =
   pigment {
     Pigm
     color_map {
       [ 0.000  color rgb 0.0 ]
       [ 1.000  color rgb 1.0 ]
     }
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y
   }

#local T_WoodA = texture {
   Text
   translate <Trsx, Trsy, Trsz>
   rotate Rotx*x
   rotate Roty*y
}

object {

#switch (CH_Wood_Quality)
  #case (1)
    intersection {
      cylinder{ <0,0,-Length*0.5>, < 0, 0, Length*0.5>, Radius }
      plane { y, 0 }
      texture { T_WoodA }
    }
    #break
  #case (2)

    #local fnShape =function{ y^2 + x^2 - Radius^2 }
    #local fnShape2=function{ (abs(z)-Length*0.5) }
    #local fnBumps =function{ pigment { P_WoodA }}
    #local fnHalf  =function{ -y }


    isosurface{
      function {(fnShape(x,y,z) & fnShape2(x,y,z) & fnHalf(x,y,z))+fnBumps(x,y,z)*Strength}

      method CH_Wood_ISOmethod
      accuracy CH_Wood_ISOaccuracy
      #if (CH_Wood_ISOeval)
         eval
      #else
         max_gradient CH_Wood_ISOmaxgrad
      #end

      contained_by{
         box{ < -Radius-0.1,-Radius-0.1, -Length*0.5-0.1>,
              <  Radius+0.1, Radius+0.1, Length*0.5+0.1> }
      }
      texture {
         T_WoodA
      }
    }

    #break
#else  //0
  intersection {
    cylinder{ <0,0,-Length*0.5>, < 0, 0, Length*0.5>, Radius }
    plane { y, 0 }
    texture { pigment { color rgb <1,0,0> } }
  }
#end

  translate <0, 0, Length*0.5>
}


#end  //------------------------------------------------------------------------------------


//=========================================================================================
//===== Wood_Cyl2_02 - half cylindrical wood with cut & chamfered ends ====================
//=========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------

#macro Wood_Cyl2_02 (Seed, Radius, Length, Rotate, Translate, Round,
                     Strength, Strength2, Pigm, Fin, Norm)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*Radius;
#local Trsy = (rand(rd)-0.5)*Translate*Radius;
#local Trsz = (rand(rd)-0.5)*Translate*Radius;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local P_WoodA =
   pigment {
     Pigm
     color_map {
       [ 0.000  color rgb 0.0 ]
       [ 1.000  color rgb 1.0 ]
     }
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y
   }

#local T_WoodA = texture {
   pigment {
     Pigm
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y
   }
   normal { Norm }
   finish { Fin }
}

object {

#switch (CH_Wood_Quality)
  #case (1)
    intersection {
      cylinder{ <0,0,-Length*0.5>, < 0, 0, Length*0.5>, Radius }
      plane { y, 0 }
      texture { T_WoodA }
    }
    #break
  #case (2)

    #local fnShape =function{ y^2 + x^2 - Radius^2 }
    #local fnShape2=function{ (abs(z)-Length*0.5) }
    #local fnShape3=function{ sqrt( sqr(x)+sqr(y) )+(abs(z)-Length/2-(1-Round)*Radius)   }
    #local fnBumps =function{ pigment { P_WoodA }}
    #local fnHalf  =function{ -y }


    isosurface{
      function {(fnShape(x,y,z)+fnBumps(x,y,z)*Strength) &
                ((fnShape2(x,y,z) & fnShape3(x,y,z) &
                  fnHalf(x, y, z))+fnBumps(x,y,z)*Strength2)}

      contained_by{
         box{ < -Radius-0.1,-Radius-0.1, -Length*0.5-0.1>,
              <  Radius+0.1, Radius+0.1, Length*0.5+0.1> }
      }

      method CH_Wood_ISOmethod
      accuracy CH_Wood_ISOaccuracy
      #if (CH_Wood_ISOeval)
         eval
      #else
         max_gradient CH_Wood_ISOmaxgrad
      #end

      texture {
         T_WoodA
      }
    }

    #break
#else  //0
  intersection {
    cylinder{ <0,0,-Length*0.5>, < 0, 0, Length*0.5>, Radius }
    plane { y, 0 }
    texture { pigment { color rgb <1,0,0> } }
  }
#end

  translate <0, 0, Length*0.5>
}


#end  //------------------------------------------------------------------------------------


//==========================================================================================
//===== Wood_Cyl2_02T - half cyl. wood with cut & chamfered ends with separate texture =====
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------

#macro Wood_Cyl2_02T (Seed, Radius, Length, Rotate, Translate, Round,
                      Strength, Strength2, Pigm, Text)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*Radius;
#local Trsy = (rand(rd)-0.5)*Translate*Radius;
#local Trsz = (rand(rd)-0.5)*Translate*Radius;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local P_WoodA =
   pigment {
     Pigm
     color_map {
       [ 0.000  color rgb 0.0 ]
       [ 1.000  color rgb 1.0 ]
     }
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y
   }

#local T_WoodA = texture {
   Text
   translate <Trsx, Trsy, Trsz>
   rotate Rotx*x
   rotate Roty*y
}

object {

#switch (CH_Wood_Quality)
  #case (1)
    intersection {
      cylinder{ <0,0,-Length*0.5>, < 0, 0, Length*0.5>, Radius }
      plane { y, 0 }
      texture { T_WoodA }
    }
    #break
  #case (2)

    #local fnShape =function{ y^2 + x^2 - Radius^2 }
    #local fnShape2=function{ (abs(z)-Length*0.5) }
    #local fnShape3=function{ sqrt( sqr(x)+sqr(y) )+(abs(z)-Length/2-(1-Round)*Radius)   }
    #local fnBumps =function{ pigment { P_WoodA }}
    #local fnHalf  =function{ -y }


    isosurface{
      function {(fnShape(x,y,z)+fnBumps(x,y,z)*Strength) &
                ((fnShape2(x,y,z) & fnShape3(x,y,z) &
                  fnHalf(x, y, z))+fnBumps(x,y,z)*Strength2)}

      contained_by{
         box{ < -Radius-0.1,-Radius-0.1, -Length*0.5-0.1>,
              <  Radius+0.1, Radius+0.1, Length*0.5+0.1> }
      }

      method CH_Wood_ISOmethod
      accuracy CH_Wood_ISOaccuracy
      #if (CH_Wood_ISOeval)
         eval
      #else
         max_gradient CH_Wood_ISOmaxgrad
      #end

      texture {
         T_WoodA
      }
    }

    #break
#else  //0
  intersection {
    cylinder{ <0,0,-Length*0.5>, < 0, 0, Length*0.5>, Radius }
    plane { y, 0 }
    texture { pigment { color rgb <1,0,0> } }
  }
#end

  translate <0, 0, Length*0.5>
}


#end  //------------------------------------------------------------------------------------



//==========================================================================================
//===== Wood_CylS_01 - cylindrical wood spike - 4 cuts =====================================
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------

#macro Wood_CylS_01 (Seed, Radius, Length, Rotate, Translate,
                     Strength, Strength2, Angle, Irreg, Pigm, Fin, Norm)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*Radius;
#local Trsy = (rand(rd)-0.5)*Translate*Radius;
#local Trsz = (rand(rd)-0.5)*Translate*Radius;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local P_WoodA =
   pigment {
     Pigm
     color_map {
       [ 0.000  color rgb 0.0 ]
       [ 1.000  color rgb 1.0 ]
     }
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y
   }

#local T_WoodA = texture {
   pigment {
     Pigm
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y
   }
   normal { Norm }
   finish { Fin }
}


#local fnShape =function{ y^2 + x^2 - Radius^2 }
#local fnShape2=function{ (abs(z)-Length*0.5) }

#local Ang=rand(rd)*2*pi;
#local RndA=sin(Ang);
#local RndB=cos(Ang);
#local RndC=rand(rd)*Irreg + Angle;
#local fnSpike1=function{ (RndA*x + RndB*y + RndC*(z-Length*0.5)) }
#local AngA=Ang+(pi/2) + (rand(rd)-0.5)*Irreg;
#local RndA=sin(AngA);
#local RndB=cos(AngA);
#local RndC=rand(rd)*Irreg + Angle;
#local fnSpike2=function{ (RndA*x + RndB*y + RndC*(z-Length*0.5)) }
#local AngA=Ang+(pi/2)*2 + (rand(rd)-0.5)*Irreg;
#local RndA=sin(AngA);
#local RndB=cos(AngA);
#local RndC=rand(rd)*Irreg + Angle;
#local fnSpike3=function{ (RndA*x + RndB*y + RndC*(z-Length*0.5)) }
#local AngA=Ang+(pi/2)*3 + (rand(rd)-0.5)*Irreg;
#local RndA=sin(AngA);
#local RndB=cos(AngA);
#local RndC=rand(rd)*Irreg + Angle;
#local fnSpike4=function{ (RndA*x + RndB*y + RndC*(z-Length*0.5)) }
#local AngA=Ang+(pi/2)*4 + (rand(rd)-0.5)*Irreg;


#local fnSpike=function{ fnSpike1(x, y, z) & fnSpike2(x, y, z) & fnSpike3(x, y, z) & fnSpike4(x, y, z) }

#local fnBumps =function{ pigment { P_WoodA }}


isosurface{
  function {(( (fnShape(x,y,z) & fnShape2(x,y,z))+fnBumps(x,y,z)*Strength) &
             (fnSpike(x,y,z)+fnBumps(x,y,z)*Strength2)) }

  method CH_Wood_ISOmethod
  accuracy CH_Wood_ISOaccuracy
  #if (CH_Wood_ISOeval)
     eval
  #else
     max_gradient CH_Wood_ISOmaxgrad
  #end

  contained_by{
     box{ < -Radius-0.1,-Radius-0.1, -Length*0.5-0.1>,
          <  Radius+0.1, Radius+0.1, Length*0.5+0.1> }
  }
  texture {
     T_WoodA
  }
  translate <0, 0, Length*0.5>
}


#end  //------------------------------------------------------------------------------------



//==========================================================================================
//===== Wood_CylS_02 - cylindrical wood spike - 8 cuts =====================================
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------

#macro Wood_CylS_02 (Seed, Radius, Length, Rotate, Translate,
                     Strength, Strength2, Angle, Irreg, Pigm, Fin, Norm)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*Radius;
#local Trsy = (rand(rd)-0.5)*Translate*Radius;
#local Trsz = (rand(rd)-0.5)*Translate*Radius;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local P_WoodA =
   pigment {
     Pigm
     color_map {
       [ 0.000  color rgb 0.0 ]
       [ 1.000  color rgb 1.0 ]
     }
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y
   }

#local T_WoodA = texture {
   pigment {
     Pigm
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y
   }
   normal { Norm }
   finish { Fin }
}


#local fnShape =function{ y^2 + x^2 - Radius^2 }
#local fnShape2=function{ (abs(z)-Length*0.5) }

#local Ang=rand(rd)*2*pi;
#local RndA=sin(Ang);
#local RndB=cos(Ang);
#local RndC=rand(rd)*Irreg + Angle;
#local fnSpike1=function{ (RndA*x + RndB*y + RndC*(z-Length*0.5)) }
#local AngA=Ang+(pi/4) + (rand(rd)-0.5)*Irreg;
#local RndA=sin(AngA);
#local RndB=cos(AngA);
#local RndC=rand(rd)*Irreg + Angle;
#local fnSpike2=function{ (RndA*x + RndB*y + RndC*(z-Length*0.5)) }
#local AngA=Ang+(pi/4)*2 + (rand(rd)-0.5)*Irreg;
#local RndA=sin(AngA);
#local RndB=cos(AngA);
#local RndC=rand(rd)*Irreg + Angle;
#local fnSpike3=function{ (RndA*x + RndB*y + RndC*(z-Length*0.5)) }
#local AngA=Ang+(pi/4)*3 + (rand(rd)-0.5)*Irreg;
#local RndA=sin(AngA);
#local RndB=cos(AngA);
#local RndC=rand(rd)*Irreg + Angle;
#local fnSpike4=function{ (RndA*x + RndB*y + RndC*(z-Length*0.5)) }
#local AngA=Ang+(pi/4)*4 + (rand(rd)-0.5)*Irreg;
#local RndA=sin(AngA);
#local RndB=cos(AngA);
#local RndC=rand(rd)*Irreg + Angle;
#local fnSpike5=function{ (RndA*x + RndB*y + RndC*(z-Length*0.5)) }
#local AngA=Ang+(pi/4)*5 + (rand(rd)-0.5)*Irreg;
#local RndA=sin(AngA);
#local RndB=cos(AngA);
#local RndC=rand(rd)*Irreg + Angle;
#local fnSpike6=function{ (RndA*x + RndB*y + RndC*(z-Length*0.5)) }
#local AngA=Ang+(pi/4)*6 + (rand(rd)-0.5)*Irreg;
#local RndA=sin(AngA);
#local RndB=cos(AngA);
#local RndC=rand(rd)*Irreg + Angle;
#local fnSpike7=function{ (RndA*x + RndB*y + RndC*(z-Length*0.5)) }
#local AngA=Ang+(pi/4)*7 + (rand(rd)-0.5)*Irreg;
#local RndA=sin(AngA);
#local RndB=cos(AngA);
#local RndC=rand(rd)*Irreg + Angle;
#local fnSpike8=function{ (RndA*x + RndB*y + RndC*(z-Length*0.5)) }

#local fnSpike=function{ fnSpike1(x, y, z) & fnSpike2(x, y, z) & fnSpike3(x, y, z) & fnSpike4(x, y, z) &
                         fnSpike5(x, y, z) & fnSpike6(x, y, z) & fnSpike7(x, y, z) & fnSpike8(x, y, z) }

#local fnBumps =function{ pigment { P_WoodA }}


isosurface{
  function {(( (fnShape(x,y,z) & fnShape2(x,y,z))+fnBumps(x,y,z)*Strength) &
             (fnSpike(x,y,z)+fnBumps(x,y,z)*Strength2)) }

  method CH_Wood_ISOmethod
  accuracy CH_Wood_ISOaccuracy
  #if (CH_Wood_ISOeval)
     eval
  #else
     max_gradient CH_Wood_ISOmaxgrad
  #end

  contained_by{
     box{ < -Radius-0.1,-Radius-0.1, -Length*0.5-0.1>,
          <  Radius+0.1, Radius+0.1, Length*0.5+0.1> }
  }
  texture {
     T_WoodA
  }
  translate <0, 0, Length*0.5>
}


#end  //------------------------------------------------------------------------------------



//==========================================================================================
//==========================================================================================

#macro Wood_Box_01 (Seed, Start, End, Rotate, Translate, Round, Strength, Pigm, Fin, Norm)

  #local xSize = abs(End.x - Start.x);
  #local ySize = abs(End.y - Start.y);
  #local zSize = abs(End.z - Start.z);

  object {
    Wood_Plank_01 (Seed, xSize, ySize, zSize, Rotate, Translate, Round, Strength, Pigm, Fin, Norm)
    translate < min(End.x, Start.x), min(End.y, Start.y)*0.5, min(End.z, Start.z)*0.5 >
  }
#end  //------------------------------------------------------------------------------------

#macro Wood_Box_02 (Seed, Start, End, Rotate, Translate,
                      Round, Strength, Strength2, Pigm, Fin, Norm)
  #local xSize = abs(End.x - Start.x);
  #local ySize = abs(End.y - Start.y);
  #local zSize = abs(End.z - Start.z);

  object {
    Wood_Plank_02 (Seed, xSize, ySize, zSize, Rotate, Translate,
                      Round, Strength, Strength2, Pigm, Fin, Norm)
    translate < min(End.x, Start.x), min(End.y, Start.y)*0.5, min(End.z, Start.z)*0.5 >
  }
#end  //------------------------------------------------------------------------------------

#macro Wood_Box_03 (Seed, xSize, ySize, zSize, Rotate, Translate,
                      Round, Round2, Strength, Strength2, Pigm, Fin, Norm)
  #local xSize = abs(End.x - Start.x);
  #local ySize = abs(End.y - Start.y);
  #local zSize = abs(End.z - Start.z);

  object {
    Wood_BoxPlank_03 (Seed, xSize, ySize, zSize, Rotate, Translate,
                      Round, Round2, Strength, Strength2, Pigm, Fin, Norm)
    translate < min(End.x, Start.x), min(End.y, Start.y)*0.5, min(End.z, Start.z)*0.5 >
  }
#end  //------------------------------------------------------------------------------------


//==========================================================================================
//==========================================================================================


#macro Wood_Box_02E (Seed, Start, End, Rotate, Translate,
                      Round, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)
  #local xSize = abs(End.x - Start.x);
  #local ySize = abs(End.y - Start.y);
  #local zSize = abs(End.z - Start.z);

  object {
    Wood_Plank_02E (Seed, xSize, ySize, zSize, Rotate, Translate,
                      Round, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)
    translate < min(End.x, Start.x), min(End.y, Start.y)*0.5, min(End.z, Start.z)*0.5 >
  }
#end  //------------------------------------------------------------------------------------

#macro Wood_Box_03E (Seed, xSize, ySize, zSize, Rotate, Translate,
                      Round, Round2, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)
  #local xSize = abs(End.x - Start.x);
  #local ySize = abs(End.y - Start.y);
  #local zSize = abs(End.z - Start.z);

  object {
    Wood_BoxPlank_03E (Seed, xSize, ySize, zSize, Rotate, Translate,
                      Round, Round2, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)
    translate < min(End.x, Start.x), min(End.y, Start.y)*0.5, min(End.z, Start.z)*0.5 >
  }
#end  //------------------------------------------------------------------------------------


//==========================================================================================
//==========================================================================================

#macro Wood_Box_02W (Seed, Start, End, Rotate, Translate,
                      Round, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)
  #local xSize = abs(End.x - Start.x);
  #local ySize = abs(End.y - Start.y);
  #local zSize = abs(End.z - Start.z);

  object {
    Wood_Plank_02W (Seed, xSize, ySize, zSize, Rotate, Translate,
                      Round, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)
    translate < min(End.x, Start.x), min(End.y, Start.y)*0.5, min(End.z, Start.z)*0.5 >
  }
#end  //------------------------------------------------------------------------------------

#macro Wood_Box_03W (Seed, xSize, ySize, zSize, Rotate, Translate,
                      Round, Round2, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)
  #local xSize = abs(End.x - Start.x);
  #local ySize = abs(End.y - Start.y);
  #local zSize = abs(End.z - Start.z);

  object {
    Wood_BoxPlank_03W (Seed, xSize, ySize, zSize, Rotate, Translate,
                      Round, Round2, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)
    translate < min(End.x, Start.x), min(End.y, Start.y)*0.5, min(End.z, Start.z)*0.5 >
  }
#end  //------------------------------------------------------------------------------------


//==========================================================================================
//==========================================================================================


#macro Wood_Box_01T (Seed, Start, End, Rotate, Translate, Round, Strength, Pigm, Text)

  #local xSize = abs(End.x - Start.x);
  #local ySize = abs(End.y - Start.y);
  #local zSize = abs(End.z - Start.z);

  object {
    Wood_Plank_01T (Seed, xSize, ySize, zSize, Rotate, Translate, Round, Strength, Pigm, Text)
    translate < min(End.x, Start.x), min(End.y, Start.y)*0.5, min(End.z, Start.z)*0.5 >
  }
#end  //------------------------------------------------------------------------------------

#macro Wood_Box_02T (Seed, Start, End, Rotate, Translate,
                     Round, Strength, Strength2, Pigm, Fin, Norm)
  #local xSize = abs(End.x - Start.x);
  #local ySize = abs(End.y - Start.y);
  #local zSize = abs(End.z - Start.z);

  object {
    Wood_Plank_02T (Seed, xSize, ySize, zSize, Rotate, Translate,
                    Round, Strength, Strength2, Pigm, Text)
    translate < min(End.x, Start.x), min(End.y, Start.y)*0.5, min(End.z, Start.z)*0.5 >
  }
#end  //------------------------------------------------------------------------------------

#macro Wood_Box_03T (Seed, xSize, ySize, zSize, Rotate, Translate,
                     Round, Round2, Strength, Strength2, Pigm, Text)
  #local xSize = abs(End.x - Start.x);
  #local ySize = abs(End.y - Start.y);
  #local zSize = abs(End.z - Start.z);

  object {
    Wood_BoxPlank_03T (Seed, xSize, ySize, zSize, Rotate, Translate,
                       Round, Round2, Strength, Strength2, Pigm, Text)
    translate < min(End.x, Start.x), min(End.y, Start.y)*0.5, min(End.z, Start.z)*0.5 >
  }
#end  //------------------------------------------------------------------------------------

//==========================================================================================
//==========================================================================================


//==========================================================================================
//===== Wood_Plank_02N - rounded box cut at the front/back (distorted) =====================
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------

#macro Wood_Plank_02N (Seed, xSize, ySize, zSize, Rotate, Translate,
                      Round, Strength, Strength2, Pigm, Fin, Norm, C_Map, WarpNbr)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*xSize;
#local Trsy = (rand(rd)-0.5)*Translate*ySize;
#local Trsz = (rand(rd)-0.5)*Translate*zSize;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local P_WoodA =
   pigment {
     Pigm
     color_map {
       [ 0.000  color rgb 0.0 ]
       [ 1.000  color rgb 1.0 ]
     }
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y

     #switch (WarpNbr)
       #case (1)
         IW_Warp_01 (Seed, xSize, ySize, zSize)
         #break
       #case (2)
         IW_Warp_02 (Seed, xSize, ySize, zSize)
         #break
       #case (3)
         IW_Warp_03 (Seed, xSize, ySize, zSize)
         #break
       #case (4)
         IW_Warp_04 (Seed, xSize, ySize, zSize)
         #break
       #case (5)
         IW_Warp_05 (Seed, xSize, ySize, zSize)
         #break
       #case (6)
         IW_Warp_06 (Seed, xSize, ySize, zSize)
         #break
       #case (7)
         IW_Warp_07 (Seed, xSize, ySize, zSize)
         #break
       #case (8)
         IW_Warp_08 (Seed, xSize, ySize, zSize)
         #break
       #case (9)
         IW_Warp_09 (Seed, xSize, ySize, zSize)
     #end

   }

#ifndef (CH_Wood_Fn_Dx)
  #declare CH_Wood_Fn_Dx=function{ noise3d(0, 0, z*2)*0.2 - 0.1}
#end
#ifndef (CH_Wood_Fn_Dy)
  #declare CH_Wood_Fn_Dy=function{ noise3d(0, 0, -z*2)*0.2 - 0.1 }
#end


#local fnShape =function{ "rounded_box",<Round,xSize*0.5,ySize*0.5,zSize*0.5+0.2>}
#local fnShape2=function{ (abs(z)-zSize*0.5) }
#local fnBumps =function{ pigment { P_WoodA }}

#local fnShapeT =function{ fnShape(x+CH_Wood_Fn_Dx(0, 0, z),y+CH_Wood_Fn_Dy(0, 0, z),z) }
#local fnShape2T=function{ fnShape2(x+CH_Wood_Fn_Dx(0, 0, z),y+CH_Wood_Fn_Dy(0, 0, z),z) }
#local fnBumpsT =function{ fnBumps(x+CH_Wood_Fn_Dx(0, 0, z),y+CH_Wood_Fn_Dy(0, 0, z),z) }

#local T_WoodA = texture {
   pigment {
     function { fnBumpsT }
     color_map { C_Map }
   }
   normal { Norm }
   finish { Fin }
}

isosurface{
  function {(fnShapeT(x,y,z)+fnBumpsT(x,y,z)*Strength) & (fnShape2T(x,y,z)+fnBumpsT(x,y,z)*Strength2)}
  contained_by{
     box{ <-xSize*0.5-1.1,-ySize*0.5-1.1,-zSize*0.5-1.1>,
          < xSize*0.5+1.1, ySize*0.5+1.1, zSize*0.5+1.1> }
  }

  method CH_Wood_ISOmethod
  accuracy CH_Wood_ISOaccuracy
  #if (CH_Wood_ISOeval)
     eval
  #else
     max_gradient CH_Wood_ISOmaxgrad
  #end

  texture {
    T_WoodA
  }
  translate <xSize*0.5, ySize*0.5, zSize*0.5>
}


#end  //------------------------------------------------------------------------------------


//==========================================================================================
//===== Wood_Plank_03N - chamfered box cut at the front/back (distorted) ===================
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------

#macro Wood_Plank_03N (Seed, xSize, ySize, zSize, Rotate, Translate,
                      Round, Round2, Strength, Strength2, Pigm, Fin, Norm, C_Map, WarpNbr)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*xSize;
#local Trsy = (rand(rd)-0.5)*Translate*ySize;
#local Trsz = (rand(rd)-0.5)*Translate*zSize;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local P_WoodA =
   pigment {
     Pigm
     color_map {
       [ 0.000  color rgb 0.0 ]
       [ 1.000  color rgb 1.0 ]
     }
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y

     #switch (WarpNbr)
       #case (1)
         IW_Warp_01 (Seed, xSize, ySize, zSize)
         #break
       #case (2)
         IW_Warp_02 (Seed, xSize, ySize, zSize)
         #break
       #case (3)
         IW_Warp_03 (Seed, xSize, ySize, zSize)
         #break
       #case (4)
         IW_Warp_04 (Seed, xSize, ySize, zSize)
         #break
       #case (5)
         IW_Warp_05 (Seed, xSize, ySize, zSize)
         #break
       #case (6)
         IW_Warp_06 (Seed, xSize, ySize, zSize)
         #break
       #case (7)
         IW_Warp_07 (Seed, xSize, ySize, zSize)
         #break
       #case (8)
         IW_Warp_08 (Seed, xSize, ySize, zSize)
         #break
       #case (9)
         IW_Warp_09 (Seed, xSize, ySize, zSize)
     #end

   }

#ifndef (CH_Wood_Fn_Dx)
  #declare CH_Wood_Fn_Dx=function{ noise3d(0, 0, z*2)*0.2 - 0.1}
#end
#ifndef (CH_Wood_Fn_Dy)
  #declare CH_Wood_Fn_Dy=function{ noise3d(0, 0, -z*2)*0.2 - 0.1 }
#end

#local fnShape =function{ (abs(x)-xSize*0.5) & (abs(y)-ySize*0.5)}
#local fnShape2=function{ (abs(z)-zSize*0.5) }
#local fnShape3=function{ (abs(x)+abs(y)-(xSize+ySize)*0.5)+Round }
#local fnShape4=function{ ((abs(z)+abs(y)-(zSize+ySize)*0.5)+Round2) &
                              ((abs(z)+abs(x)-(zSize+xSize)*0.5)+Round2) }
#local fnBumps =function{ pigment { P_WoodA }}

#local fnShapeT =function{ fnShape(x+CH_Wood_Fn_Dx(0, 0, z),y+CH_Wood_Fn_Dy(0, 0, z),z) }
#local fnShape2T=function{ fnShape2(x+CH_Wood_Fn_Dx(0, 0, z),y+CH_Wood_Fn_Dy(0, 0, z),z) }
#local fnShape3T=function{ fnShape3(x+CH_Wood_Fn_Dx(0, 0, z),y+CH_Wood_Fn_Dy(0, 0, z),z) }
#local fnShape4T=function{ fnShape4(x+CH_Wood_Fn_Dx(0, 0, z),y+CH_Wood_Fn_Dy(0, 0, z),z) }
#local fnBumpsT =function{ fnBumps(x+CH_Wood_Fn_Dx(0, 0, z),y+CH_Wood_Fn_Dy(0, 0, z),z) }

#local T_WoodA = texture {
   pigment {
     function { fnBumpsT }
     color_map { C_Map }
   }
   normal { Norm }
   finish { Fin }
}

isosurface{
  function {((fnShapeT (x,y,z)&fnShape3T(x,y,z))+fnBumpsT(x,y,z)*Strength) &
            ((fnShape2T(x,y,z)&fnShape4T(x,y,z))+fnBumpsT(x,y,z)*Strength2)}

  contained_by{
     box{ <-xSize*0.5-1.1,-ySize*0.5-1.1,-zSize*0.5-1.1>,
          < xSize*0.5+1.1, ySize*0.5+1.1, zSize*0.5+1.1> }
  }

  method CH_Wood_ISOmethod
  accuracy CH_Wood_ISOaccuracy
  #if (CH_Wood_ISOeval)
     eval
  #else
     max_gradient CH_Wood_ISOmaxgrad
  #end

  texture {
    T_WoodA
  }
  translate <xSize*0.5, ySize*0.5, zSize*0.5>
}


#end  //------------------------------------------------------------------------------------


//==========================================================================================
//===== Wood_Parquet_A - Parquet of Wood_Plank_X - pattern A ===============================
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------


#macro Wood_Parquet_A (Seed, Start, End, xSize, ySize, Height, Rotate, Translate,
                       Round, Round2, Strength, Strength2, Pigm, Fin, Norm, Text, WarpNbr, Typ)

  #if (ySize > xSize)
    #local buf = xSize;
    #local xSize = ySize;
    #local ySize = buf;
  #end

  #if (mod(xSize, ySize)> 0)
    #error "xSize and ySize in Wood_Parquet_A do not fit"
  #else
    #render "building parquet ...\n\n"

    #local Pl_Cnt=0;
    #local rd=seed(Seed);

    #local xASize = abs(End.x - Start.x);
    #local yASize = abs(End.y - Start.y);

    intersection {     //-------------------------------
     union {
      #local Acnt=0;
      #while ((Acnt < yASize+ySize) & (Acnt < xASize+ySize))

        //########################################################

        #local Ycnt=0;

        #while (Ycnt+Acnt < yASize+xSize)

          #local Xcnt=0;

          #while (Xcnt+Acnt < xASize+ySize)
            object { //===========================================
              #switch (CH_Wood_Quality)
                #case (1)
                  box { <0,0,0>, <Height, ySize, xSize>
                    texture { pigment { Pigm } }
                  }
                #break
                #case (2)
                  #switch (Typ)
                    #case (1)
                      Wood_Plank_01 (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                     Round, Strength, Pigm, Fin, Norm)
                      #break
                    #case (11)
                      Wood_Plank_01W (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                      Round, Strength, Pigm, Fin, Norm, WarpNbr)
                      #break
                    #case (12)
                      Wood_Plank_01T (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                      Round, Strength, Pigm, Text)
                      #break
                    #case (2)
                      Wood_Plank_02 (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                     Round, Strength, Strength2, Pigm, Fin, Norm)
                      #break
                    #case (20)
                      Wood_Plank_02E (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                      Round, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)
                      #break
                    #case (21)
                      Wood_Plank_02W (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                      Round, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)
                      #break
                    #case (22)
                      Wood_Plank_02T (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                      Round, Strength, Strength2, Pigm, Text)
                      #break
                    #case (3)
                      Wood_Plank_03 (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                     Round, Round2, Strength, Strength2, Pigm, Fin, Norm)
                      #break
                    #case (30)
                      Wood_Plank_03E (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                     Round, Round2, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)
                      #break
                    #case (31)
                      Wood_Plank_03W (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                     Round, Round2, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)
                      #break
                    #case (32)
                      Wood_Plank_03T (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                     Round, Round2, Strength, Strength2, Pigm, Text)
                  #end
                #break
                #else  //0
                  box { <0,0,0>, <Height, ySize, xSize> texture { pigment { color rgb <0,0,1> } }}
              #end


              rotate -90*y rotate 90*z
              translate < min(End.x, Start.x) + Acnt + Xcnt, min(End.y, Start.y) + Acnt + Ycnt, -Height >
            } //==================================================

            #local Pl_Cnt=Pl_Cnt + 1;
            #render concat(str(Pl_Cnt,0,0),"\r")

            #local Xcnt = Xcnt + xSize*2;
          #end
          #local Ycnt = Ycnt + xSize*2;
        #end

        //########################################################
        #local Ycnt=0;

        #while (Ycnt+Acnt < yASize+ySize)

          #local Xcnt=0;

          #while (Xcnt+Acnt < xASize+xSize)

            object { //===========================================
              #switch (CH_Wood_Quality)
                #case (1)
                  box { <0,0,0>, <Height, ySize, xSize>
                    texture { pigment { Pigm } }
                  }
                #break
                #case (2)
                  #switch (Typ)
                    #case (1)
                      Wood_Plank_01 (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                     Round, Strength, Pigm, Fin, Norm)
                      #break
                    #case (11)
                      Wood_Plank_01W (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                      Round, Strength, Pigm, Fin, Norm, WarpNbr)
                      #break
                    #case (12)
                      Wood_Plank_01T (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                      Round, Strength, Pigm, Text)
                      #break
                    #case (2)
                      Wood_Plank_02 (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                     Round, Strength, Strength2, Pigm, Fin, Norm)
                      #break
                    #case (20)
                      Wood_Plank_02E (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                      Round, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)
                      #break
                    #case (21)
                      Wood_Plank_02W (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                      Round, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)
                      #break
                    #case (22)
                      Wood_Plank_02T (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                      Round, Strength, Strength2, Pigm, Text)
                      #break
                    #case (3)
                      Wood_Plank_03 (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                     Round, Round2, Strength, Strength2, Pigm, Fin, Norm)
                      #break
                    #case (30)
                      Wood_Plank_03E (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                     Round, Round2, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)
                      #break
                    #case (31)
                      Wood_Plank_03W (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                     Round, Round2, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)
                      #break
                    #case (32)
                      Wood_Plank_03T (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                     Round, Round2, Strength, Strength2, Pigm, Text)
                  #end
                #break
                #else  //0
                  box { <0,0,0>, <Height, ySize, xSize> texture { pigment { color rgb <1,0,0> } }}
              #end

              rotate -90*y
              translate < min(End.x, Start.x) + Acnt + Xcnt, min(End.y, Start.y) + Acnt + Ycnt, -Height >
            } //==================================================

            #local Pl_Cnt=Pl_Cnt + 1;
            #render concat(str(Pl_Cnt,0,0),"\r")

            #local Xcnt = Xcnt + xSize*2;
          #end
          #local Ycnt = Ycnt + xSize*2;
        #end
        #local Acnt = Acnt + ySize;
      #end

      //########################################################

     }
     box { <Start.x, Start.y, Height>, <End.x, End.y, -Height*2>
       texture { pigment { color rgb <1,0,0> } }
     }
    }

    #render concat("parquet created (",str(Pl_Cnt,0,0) ," planks)\n")
  #end

#end  //------------------------------------------------------------------------------------


//==========================================================================================
//===== Wood_Parquet_B - Parquet of Wood_Plank_X - pattern B ===============================
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------


#macro Wood_Parquet_B (Seed, Start, End, xSize, ySize, Height, Rotate, Translate,
                       Round, Round2, Strength, Strength2, Pigm, Fin, Norm, Text, WarpNbr, Typ)

  #if (ySize > xSize)
    #local buf = xSize;
    #local xSize = ySize;
    #local ySize = buf;
  #end

  #if (mod(xSize, ySize)> 0)
    #error "xSize and ySize in Wood_Parquet_B do not fit"
  #else
    #render "building parquet ...\n\n"

    #local Pl_Cnt=0;
    #local rd=seed(Seed);

    #local xASize = abs(End.x - Start.x);
    #local yASize = abs(End.y - Start.y);

    intersection {     //-------------------------------
      union {
        #local Ycnt=0;
        #while (Ycnt < yASize)

          #local Xcnt=0;
          #while (Xcnt < xASize)

            #local Acnt=0;
            #while (Acnt < div(xSize, ySize))

              #if (mod(div(Xcnt,xSize)+div(Ycnt,xSize), 2)=0)

                object { //===========================================

                  #switch (CH_Wood_Quality)
                    #case (1)
                      box { <0,0,0>, <Height, ySize, xSize>
                        texture { pigment { Pigm } }
                      }
                    #break
                    #case (2)
                      #switch (Typ)
                        #case (1)
                          Wood_Plank_01 (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                         Round, Strength, Pigm, Fin, Norm)
                          #break
                        #case (11)
                          Wood_Plank_01W (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                          Round, Strength, Pigm, Fin, Norm, WarpNbr)
                          #break
                        #case (12)
                          Wood_Plank_01T (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                          Round, Strength, Pigm, Text)
                          #break
                        #case (2)
                          Wood_Plank_02 (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                         Round, Strength, Strength2, Pigm, Fin, Norm)
                          #break
                        #case (20)
                          Wood_Plank_02E (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                          Round, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)
                          #break
                        #case (21)
                          Wood_Plank_02W (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                          Round, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)
                          #break
                        #case (22)
                          Wood_Plank_02T (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                          Round, Strength, Strength2, Pigm, Text)
                          #break
                        #case (3)
                          Wood_Plank_03 (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                         Round, Round2, Strength, Strength2, Pigm, Fin, Norm)
                          #break
                        #case (30)
                          Wood_Plank_03E (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                         Round, Round2, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)
                          #break
                        #case (31)
                          Wood_Plank_03W (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                         Round, Round2, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)
                          #break
                        #case (32)
                          Wood_Plank_03T (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                         Round, Round2, Strength, Strength2, Pigm, Text)
                      #end

                    #break
                    #else  //0
                      box { <0,0,0>, <Height, ySize, xSize> texture { pigment { color rgb <0,0,Acnt/5> } }}
                  #end


                  rotate 90*y
                  translate < min(End.x, Start.x) + Xcnt, min(End.y, Start.y) + Ycnt + Acnt*ySize, -Height >
                } //==================================================
              #else
                object { //===========================================

                  #switch (CH_Wood_Quality)
                    #case (1)
                      box { <0,0,0>, <Height, ySize, xSize>
                        texture { pigment { Pigm } }
                      }
                    #break
                    #case (2)
                      #switch (Typ)
                        #case (1)
                          Wood_Plank_01 (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                         Round, Strength, Pigm, Fin, Norm)
                          #break
                        #case (11)
                          Wood_Plank_01W (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                          Round, Strength, Pigm, Fin, Norm, WarpNbr)
                          #break
                        #case (12)
                          Wood_Plank_01T (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                          Round, Strength, Pigm, Text)
                          #break
                        #case (2)
                          Wood_Plank_02 (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                         Round, Strength, Strength2, Pigm, Fin, Norm)
                          #break
                        #case (20)
                          Wood_Plank_02E (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                          Round, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)
                          #break
                        #case (21)
                          Wood_Plank_02W (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                          Round, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)
                          #break
                        #case (22)
                          Wood_Plank_02T (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                          Round, Strength, Strength2, Pigm, Text)
                          #break
                        #case (3)
                          Wood_Plank_03 (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                         Round, Round2, Strength, Strength2, Pigm, Fin, Norm)
                          #break
                        #case (30)
                          Wood_Plank_03E (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                         Round, Round2, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)
                          #break
                        #case (31)
                          Wood_Plank_03W (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                         Round, Round2, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)
                          #break
                        #case (32)
                          Wood_Plank_03T (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                         Round, Round2, Strength, Strength2, Pigm, Text)
                      #end

                    #break
                    #else  //0
                      box { <0,0,0>, <Height, ySize, xSize> texture { pigment { color rgb <Acnt/5,0,0> } }}
                  #end


                  rotate 90*y rotate 90*z
                  translate < min(End.x, Start.x) + Xcnt + (Acnt+1)*ySize, min(End.y, Start.y) + Ycnt, -Height >
                } //==================================================
              #end

              #local Pl_Cnt=Pl_Cnt + 1;
              #render concat(str(Pl_Cnt,0,0),"\r")

              #local Acnt = Acnt + 1;
            #end

            #local Xcnt = Xcnt + xSize;
          #end
          #local Ycnt = Ycnt + xSize;
        #end
      }
      box { <Start.x, Start.y, Height>, <End.x, End.y, -Height*2>
         texture { pigment { color rgb <1,0,0> } }
      }
    }



    #render concat("parquet created (",str(Pl_Cnt,0,0) ," planks)\n")
  #end

#end  //------------------------------------------------------------------------------------

