// Marsna.pov   1999 Neil Alexander

// Control file for Mars Conquest animation 
// This animation uses the following include files:
// Marstext.inc
// Capsule.inc
// Rovot.inc
// Marsenvr.inc
// Also, image maps and height field images are generated from
// the following files:
// MarsIM.pov
// MarsIM2.pov
// MarsHF.pov
// MarsHF2.pov
// Mars.gif 
// Flag.gif

#include "marstext.inc" 

#declare Scene = 1;  // (1 to 19)

#switch (Scene)

#case (1) //***************** Scene 1 ***********************
// 8 seconds, frames 1 to 192 
// Capsule drifts past camera, Mars in background
// at 3s, capsule & orbitter separate, orbitter moves up
// out of frame.  Capsule aligns itself with hydrogen jets
    
#include "capsule.inc"

camera {
  location  <0, 0, -600>
  angle 55
  look_at   <0, 0, 0>  
  } 
 
light_source {
  <0, 0, -9000000000000>
  color rgb 1
  rotate y*30
  }
  
sky_sphere {
  pigment { bozo
    color_map {
      [0.999 color rgb 0 ]
      [1 color rgb 1]
      }
    scale 0.003
    }
  }

#declare Fade = clock*32;
#if (Fade<1)
  plane {z,0
    translate z*-300
    texture {
      pigment { color rgbt <0,0,0,Fade>}
      }
    }
  #end  // if - fade in
  
sphere {0,132   // Mars
  texture { 
    pigment { bozo
      color_map {
        [0 Marscol]
        [0.1 Marscol*0.8]
        [0.3 rgb <1,0.6,0.45>]
        [1 Marscol]
        }
      frequency 1.2
      scale 8
      turbulence 0.5
      omega 0.8 
      lambda 2.5
      }
    normal { wrinkles 0.2 scale 10 }
    finish { ambient 0.4 diffuse 0.8}
    }
  translate <120-10*clock,-60,100>
  scale 10000
  }
  
  #declare Orbup = clock*8-3;
  #if (Orbup<0)
    #declare Orbup = 0;
    #end
    
  #declare Jet1 = clock*8-4;
  #declare Noz1 = Jet1;
  #if (Jet1<0)
    #declare Jet1 = 0;
    #end
  #if (Jet1<1)
    #declare Jet1 = Jet1*Jet1;
    #end
  
  #declare Jet2 = clock*8-5;
  #declare Noz2 = Jet2;
  #if (Jet2<0)
    #declare Jet2 = 0;
    #end
  #if (Jet2<1)
    #declare Jet2 = Jet2*Jet2;
    #end
  
  union {
    union {  // lower part
      object { Capsule (0)}
      object { Heatshield }
      object { Casing }
      #if ((Jet1>0)&(Jet1<1)) // first jet
        sphere {0,1
        scale <12,12,120>
        texture { pigment { color rgbf 1 }}
        hollow
        interior {
          media { emission 0.5 //absorption <1,1,0>
            density { spherical
              color_map {
                [0.0 rgb <0,0,0>]
                [0.625 rgb <0.25,0.25,0.312>]
                [1.0 rgb <1,1,1>]
                }
              scale <12,12,120>
              scale sin(Noz1*pi)
              translate -z*80
              }
            density { bozo poly_wave 0.3
              color_map {
                [0.0 rgb <0,0,0>]
                [0.625 rgb <0.25,0.25,0.312>]
                [1.0 rgb <1,1,1>]
                }
              phase clock
              scale 10
              translate <0,0,400*Noz1>
              } 
            density { bozo poly_wave 0.3
              color_map {
                [0.0 rgb <0,0,0>]
                [0.625 rgb <0.25,0.25,0.312>]
                [1.0 rgb <1,1,1>]
                }
              phase clock
              scale 10
              translate <0,0,200+400*Noz1>
              }
            }
          }
        translate <0,130,150>
        rotate y*300
        }
        #end  // if - jet1
      
      #if ((Jet2>0)&(Jet2<1)) // second jets
        sphere {0,1
        scale <12,12,120>
        texture { pigment { color rgbf 1 }}
        hollow
        interior {
          media { emission 0.5 //absorption <1,1,0>
            density { spherical
              color_map {
                [0.0 rgb <0,0,0>]
                [0.625 rgb <0.25,0.25,0.312>]
                [1.0 rgb <1,1,1>]
                }
              scale <12,12,120>
              scale sin(Noz2*pi)
              translate -z*80
              }
            density { bozo poly_wave 0.3
              color_map {
                [0.0 rgb <0,0,0>]
                [0.625 rgb <0.25,0.25,0.312>]
                [1.0 rgb <1,1,1>]
                }
              phase clock
              scale 10
              translate <0,0,400*Noz2>
              } 
            density { bozo poly_wave 0.3
              color_map {
                [0.0 rgb <0,0,0>]
                [0.625 rgb <0.25,0.25,0.312>]
                [1.0 rgb <1,1,1>]
                }
              phase clock
              scale 10
              translate <0,0,200+400*Noz2>
              }
            }
          }
        translate <0,130,150>
        rotate y*180
        } 
        
      sphere {0,1
        scale <12,12,120>
        texture { pigment { color rgbf 1 }}
        hollow
        interior {
          media { emission 0.5 //absorption <1,1,0>
            density { spherical
              color_map {
                [0.0 rgb <0,0,0>]
                [0.625 rgb <0.25,0.25,0.312>]
                [1.0 rgb <1,1,1>]
                }
              scale <12,12,120>
              scale sin(Noz2*pi)
              translate -z*80
              }
            density { bozo poly_wave 0.3
              color_map {
                [0.0 rgb <0,0,0>]
                [0.625 rgb <0.25,0.25,0.312>]
                [1.0 rgb <1,1,1>]
                }
              phase clock
              scale 10
              translate <0,0,100+400*Noz2>
              } 
            density { bozo poly_wave 0.3
              color_map {
                [0.0 rgb <0,0,0>]
                [0.625 rgb <0.25,0.25,0.312>]
                [1.0 rgb <1,1,1>]
                }
              phase clock
              scale 10
              translate <0,0,400+400*Noz2>
              }
            }
          }
        translate <0,130,150>
        rotate y*60
        }
        
        #end  // if - jet2
      
      rotate y*(30+120*clock)
      rotate <-50,40,0>
        +<-40*Jet1,23*Jet1,0>
        +<40*Jet2,-33*Jet2,0>
      }   // end lower part
    object { Orbitter 
      translate y*(154+Orbup*100)
      rotate y*(30+120*clock)
      rotate <-50,40,0>
      }
    #if ((clock>0.375)&(clock<0.75))
    difference { // puff
      sphere {0,80}
      sphere {0,1+28*Orbup} // disappears after 3s
      texture { pigment { color rgbf 1 }}
      hollow
      interior {
        media { emission 0.5 //absorption <1,1,0>
          density { spherical
            color_map {
              [0.0 rgb <0,0,0>]
              [0.625 rgb <0.25,0.25,0.312>]
              [1.0 rgb <1,1,1>]
              }
            scale 10+25*Orbup
            turbulence 0.4
            }
          density { bozo poly_wave 0.5-0.1*Orbup
            color_map {
              [0.0 rgb <0,0,0>]
              [0.625 rgb <0.25,0.25,0.312>]
              [1.0 rgb <1,1,1>]
              }
            phase clock
            scale 5+15*Orbup
            translate <0,10*Orbup,0>
            }
          density { bozo poly_wave 0.5
            color_map {
              [0.0 rgb <0,0,0>]
              [0.625 rgb <0.25,0.25,0.312>]
              [1.0 rgb <1,1,1>]
              }
            phase clock
            scale 5+15*Orbup
            translate <0,-10*Orbup,0>
            }
          }
        }
      scale <1.5,0.75,1.5>
      translate y*(144+Orbup*50)
      rotate y*(30+120*clock)
      rotate <-50,40,0>
      }
      #end  // if - puff
      
    translate <-180+180*clock,-60,400*clock>
    } 
  
#break  // end Scene 1
  
#case (2) //***************** Scene 2 ***********************
// 2 seconds, frames 193 to 240 
// Martian landscape, capsule going through atmosphere
// like a comet

#declare Sunrot = <-70,150,0>;

camera {
  location  <0, 60, -100>
  angle 55
  look_at   <0, 60, 0>
  }

light_source {
  <0, 0, 1000000000>
  color rgb Suncol
  rotate Sunrot
  }

sky_sphere {
  pigment { gradient y
    color_map {
      [0 rgb Suncol ]
      [1 rgb <0.125,0.5,0.6>]
      }
    }
  //rotate Sunrot
  }

height_field { 
  tga
  "MarsHF.tga"
  smooth
  texture {
    pigment {
      image_map {
        tga "MarsIM.tga"
        map_type 0
        interpolate 2
        } // image_map
      }
    finish { diffuse 0.8 }
    rotate <90,0,0>
    }
  translate <-0.5,0,-0.5>
  scale <2000,60,2000>
  rotate x*-3     // raise far edge, bury near edge
  translate <0,0,1500>
  } 
 
plane { y,0
  texture {
    pigment { granite
      color_map {
        [0 rgb Suncol]
        [1 rgb Suncol*0.75]
        }
      scale 50
      }
    normal { bumps 0.5 scale 10 }
    finish { diffuse 0.8 }
    }
  }


#declare C1 = 0;
#declare S1 = seed(10);
#while (C1<180)
  blob {        // random-shaped rock
    threshold 0.56
    sphere {0,1,1} // base part to make sure rock is grounded
    sphere {0,1,1  // random parts
      scale <0.5+rand(S1),0.5+rand(S1),0.5+rand(S1)>
      translate <rand(S1),rand(S1),rand(S1)>/2
      }
    sphere {0,1,1
      scale <0.5+rand(S1),0.5+rand(S1),0.5+rand(S1)>
      translate <rand(S1),rand(S1),rand(S1)>/2
      }
    sphere {0,1,1
      scale <0.5+rand(S1),0.5+rand(S1),0.5+rand(S1)>
      translate <rand(S1),rand(S1),rand(S1)>/2
      } 
    
    scale 12*rand(S1)
    #declare Rot = 55*(rand(S1)-0.5); // 27.5
    translate z*700/cos(radians(Rot))*sqrt(rand(S1))
    rotate y*Rot
    translate z*-100
    texture { //  rock texture
      pigment { bozo
        color_map {
          [0 rgb Suncol*0.75]
          [1 rgb <0.4,0.4,0.38>]
          }
        turbulence 0.4
        omega 0.8
        scale 0.5
        }
      finish { diffuse 0.8 }
      normal { granite 0.7 scale 2 }
      }
    }
  #declare C1 = C1 + 1;
  #end // while loop  
  
sphere { 0,1
  scale <15,1,1>
  texture { 
    pigment {rgbt 1}
    }
  hollow
  interior {
    media { emission <1,0.9,0.7>
      density { spherical 
        scale <20,0.5,0.5>
        translate <15,0,0>
        }
      density { bozo poly_wave 0.7
        phase clock
        scale 0.4
        translate <-20*clock,0,0>
        }
      }
    }
  translate <-20+20*clock,0,0>
  rotate z*-20
  scale 5000
  translate <0,100000,400000>
  }
  
#break  // end Scene 2

#case (3) //***************** Scene 3 ***********************
// 1.5 seconds, frames 241 to 276 
// Close-up, capsule going through atmosphere

#include "capsule.inc"

camera {
  location  <0, 0, -450>
  angle 40
  look_at   <0, 40, 0>  
  } 
 
light_source {
  <-1000, 1000, -900>
  color rgb <1,0.5,0.25>
  }
  
sky_sphere {
  pigment { gradient y
    color_map {
      [0 color rgb <0.25,0.281,0.5> ]
      [0.75 color rgb 0]
      }
    scale 2
    translate y*-1
    }
  }
  
#declare Fade = clock*12-11;
#if (Fade>0)
  plane {z,0
    translate z*-300
    texture {
      pigment { color rgbt <0,0,0,1-Fade>}
      }
    }
  #end  // if - fade out

union {
  object { Capsule (0)}
  object { Heatshield
    texture {
      pigment { wood
        color_map {
          [0.375 rgb <1,0.25,0>]
          [1 rgb <0.25,0,0>]
          }
        rotate x*90
        scale 170
        frequency 0.281
        turbulence 0.1
        omega 0.2
        octaves 8
        translate y*1000*clock
        }
      finish { ambient 1 }
      }
    }
  object { Casing }
  merge {
    cone {<0,-15,0>,170,<0,2500,0>,1200}
    sphere {0,1
      scale <170,25,170>
      translate y*-15
      }
    texture {
      pigment { radial
        color_map {
          [0 rgbf 1]
          [0.25 rgbf <1,0.5,0,1>]
          [0.375 rgbf <1,0.625,0,0.6>]
          [0.5 rgbf <1,0.75,0,0.2>]
          [0.625 rgbf <1,0.625,0,0.6>]
          [0.75 rgbf <1,0.5,0,1>]
          [1 rgbf 1]
          }
        scale <60,400,60>
        turbulence 2
        lambda 4
        omega 0.3
        octaves 8
        frequency 10
        translate y*2000*clock
        } 
      finish { ambient 1 diffuse 0 }
      }
    }
  rotate y*-60
  rotate <30,-40,0>
  }
  
#break  // end Scene 3  

#case (4) //***************** Scene 4 ***********************
// 1.5 seconds, frames 277 to 312
// parachute opens, capsule slows and swings,
// heatshield detaches and falls away

#include "capsule.inc"

camera {
  location  <0, 0, -2000>
  angle 55
  look_at   <0, 40, 0>  
  } 
 
light_source {
  <-6000, 120000, -90000>
  color rgb Suncol
  }
  
sky_sphere {
  pigment { gradient y
    color_map {
      [0 rgb Suncol ]
      [1 rgb <0.125,0.5,0.6>]
      }
    scale 2
    translate y*-1
    }
  }

#declare Fade1 = clock*12;  // fade in
#if (Fade1<1)
  plane {z,0
    translate z*-1500
    texture {
      pigment { color rgbt <0,0,0,Fade1>}
      }
    }
  #end  // if - fade in

  
#declare Fade2 = clock*12-11;  // fade out
#if (Fade2>0)
  plane {z,0
    translate z*-1500
    texture {
      pigment { color rgbt <0,0,0,1-Fade2>}
      }
    }
  #end  // if - fade out

#declare Par = clock*2;
#if (Par>1)
  #declare Par = 1;
  #end  
  
#declare Spin = clock*2-1;
#if (Spin<0)
  #declare Spin = 0;
  #end
  
#declare Spin = Spin*Spin;

#declare Slow = clock*16-7;
#if (Slow<0)
  #declare Slow = 0;
  #end 
#if (Slow<1)
  #declare Slow = Slow*Slow;
  #end
  
union {
  object { Capsule (0)}
  object { Casing }
  blob {
    threshold 0.56
    sphere { 0,1,1
      scale <0.01*1000*Par,10,0.01*1000*Par>
      }
    sphere {0,1,1
      scale <0.5,5,1>
      rotate <0,-30,35>
      }  
    sphere {0,1,1
      scale <2,2,3>
      translate <1,0,0>
      }
    sphere {0,1,1
      scale <2,2,0.3>
      translate <-1.5,0,0>
      }
    clipped_by {
      plane {-y,0}
        translate -y*2*(1-Par)
      }
    texture { 
      pigment { radial
        color_map {
          [0 color rgbf <0.95,0.95,0.95,0.4>]
          [0.5 color rgbf <1,1,1,0.4>]
          }
        scallop_wave
        frequency 30
        }
      finish { diffuse 0.8 }
      }
    scale <(40+40*Par),(60-20*Par),(40+40*Par)>
    translate y*480
    }
  
  translate y*-480
  rotate x*Spin*40
  translate y*480
      
  translate y*(100*Slow)
  translate y*(900-2000*clock)
  rotate <0,50,30>
  }
  
object { Heatshield
  texture {
    pigment { bozo
      color_map {
        [0 rgb 0.1]
        [1 rgb 0.3]
        }
      scale 20
      turbulence 0.3
      omega 0.2
      octaves 8
      }
    finish { diffuse 0.7 }
    }
  translate y*26
  rotate x*Spin*130
  translate y*-26
  translate y*(900-2000*clock)
  rotate <0,50,30>
  }
  
#break  // end Scene 4 

#case (5) //***************** Scene 5 ***********************
// 1.5 seconds, frames 313 to 348
// capsule drifting over martian landscape,
// seen from above

#include "capsule.inc"

camera {
  location  <0, 1800, -1600>
  angle 30
  look_at   <0, 0, 0>  
  } 
 
light_source {
  <-20000, 120000, -140000>
  color rgb Suncol*0.5
  } 
light_source {
  <-20000, 120000, -140000>
  color rgb Suncol*0.5
  shadowless
  }
  
sky_sphere {
  pigment { gradient y
    color_map {
      [0 rgb Suncol ]
      [1 rgb <0.125,0.5,0.6>]
      }
    }
  }

#declare Fade1 = clock*12;  // fade in
#if (Fade1<1)
  plane {y,0
    translate y*1000
    texture {
      pigment { color rgbt <0,0,0,Fade1>}
      }
    }
  #end  // if - fade in

  
#declare Fade2 = clock*12-11;  // fade out
#if (Fade2>0)
  plane {y,0
    translate y*1000
    texture {
      pigment { color rgbt <0,0,0,1-Fade2>}
      }
    }
  #end  // if - fade out

height_field { 
  tga
  "MarsHF.tga"
  texture {
    pigment {
      image_map {
        tga "MarsIM.tga"
        map_type 0
        interpolate 2
        } // image_map
      }
    finish { diffuse 0.8 }
    rotate <90,0,0>
    }
  translate <-0.5,0,-0.5>
  scale <2500,100,2500>
  translate <0,-1000,1000>
  }
  
union {
  object { Capsule (0)}
  object { Casing }
  merge { // parachute
    #declare PC = 0;
    #while (PC<15)
      sphere {0,1
        scale <400,200,200>
        rotate y*12*PC
        }
      #declare PC = PC+1;
      #end
    clipped_by {
      plane {-y,0}
      }
    texture { 
      pigment { color rgbf <1,1,1,0.1>}
      finish { diffuse 0.8 }
      }
    translate y*480
    } 
  rotate y*30
  translate <-120+180*clock,-60*clock,100-100*clock>
  }
      
#break  // end Scene 5  

#case (6) //***************** Scene 6 ***********************
// 1.5 seconds, frames 349 to 384
// capsule drops from parachute
// seen from below

#include "capsule.inc"

camera {
  location  <0, -300, -500>
  angle 60
  look_at   <0, 0, 0>  
  } 
 
light_source {
  <-20000, 120000, -140000>
  color rgb Suncol*0.5
  } 
light_source {
  <-20000, 120000, -140000>
  color rgb Suncol*0.5
  shadowless
  }  
  
light_source {
  <0, -12000, 0>
  color rgb Suncol*0.3
  shadowless
  }
  
sky_sphere {
  pigment { gradient y
    color_map {
      [0.5 rgb Suncol ]
      [1 rgb <0.125,0.5,0.6>]
      }
    }
  } 
  
plane {y,-2000 
  texture {
    pigment { granite
      color_map {
        [0 rgb Suncol]
        [1 rgb Suncol*0.75]
        }
      scale 50
      }
    normal { bumps 0.5 scale 10 }
    finish { diffuse 0.8 }
    }
  } 

#declare Fade1 = clock*12;  // fade in
#if (Fade1<1)
  plane {z,0
    translate z*-450
    texture {
      pigment { color rgbt <0,0,0,Fade1>}
      }
    }
  #end  // if - fade in

  
#declare Fade2 = clock*12-11;  // fade out
#if (Fade2>0)
  plane {z,0
    translate z*-450
    texture {
      pigment { color rgbt <0,0,0,1-Fade2>}
      }
    }
  #end  // if - fade out
  
#declare Drop = clock*clock;

union {
  object { Capsule (0)
    translate y*(-200*Drop)
    }
  object { Casing }
  merge { // parachute
    #declare PC = 0;
    #while (PC<15)
      sphere {0,1
        scale <400,200,200>
        rotate y*12*PC
        }
      #declare PC = PC+1;
      #end
    clipped_by {
      plane {-y,0}
      }
    texture { 
      pigment { color rgbf <1,1,1,0.4>}
      finish { diffuse 0.8 }
      }
    translate y*480
    }
  union { // wires
    #declare WR = 0;
    #while (WR<30)
      cylinder {<0,0,0>,<0,509,0>,0.5
        rotate z*47
        translate <-22,136,0>
        rotate y*(6+12*WR)
        }
      #declare WR = WR+1;
      #end
    texture { Alumtext }
    }  // end wires
     
  rotate y*30
  translate <60*clock,-60*clock,-60*clock>
  }
      
#break  // end Scene 6   

#case (7) //***************** Scene 7 ***********************
// 1.5 seconds, frames 385 to 420
// rockets ignite
// seen from below

global_settings { max_trace_level 6 }

#include "capsule.inc"

camera {
  location  <0, -120, -180>
  angle 50
  look_at   <0, 0, 0>  
  } 
 
light_source {
  <-20000, 120000, -140000>
  color rgb Suncol*0.5
  } 
  
light_source {
  <0, -12000, 0>
  color rgb Suncol*0.3
  shadowless
  } 
  
sky_sphere {
  pigment { gradient y
    color_map {
      [0 rgb <0.875,0.5,0.375> ]
      [1 rgb Suncol]
      }
    }
  } 
  
plane {y,-2000 
  texture {
    pigment { granite
      color_map {
        [0 rgb Suncol]
        [1 rgb Suncol*0.75]
        }
      scale 50
      }
    normal { bumps 0.5 scale 10 }
    finish { diffuse 0.8 }
    }
  } 

#declare Fade1 = clock*12;  // fade in
#if (Fade1<1)
  plane {y,0
    translate y*-110
    texture {
      pigment { color rgbt <0,0,0,Fade1>}
      }
    }
  #end  // if - fade in

  
#declare Fade2 = clock*12-11;  // fade out
#if (Fade2>0)
  plane {y,0
    translate y*-110
    texture {
      pigment { color rgbt <0,0,0,1-Fade2>}
      }
    }
  #end  // if - fade out
 
#declare Flameon = clock*3;
#if (Flameon>1)
  #declare Flameon = 1;
  #end
  
#declare Flameon = Flameon*Flameon;

#declare Flame = 
  sphere { 0,1   
    scale <18,60,18>
    texture {
      pigment { color rgbt 1 } 
      }
    hollow
    interior {
      media { emission 1
        density { spherical
          color_map {
            [0.0 rgb <0,0,0>]
            [0.625 rgb <0.5,0.125,0>]
            [1.0 rgb <1,0.875,0>]
            }  
          scale <18,56,18>*Flameon
          }
        density { bozo poly_wave 0.3
          color_map {
            [0.0 rgb <0,0,0>]
            [0.625 rgb <0.5,0.125,0>]
            [1.0 rgb <1,0.875,0>]
            }
          phase clock
          scale <3,30,3>
          translate -y*500*clock
          } 
        density { bozo poly_wave 0.3
          color_map {
            [0.0 rgb <0,0,0>]
            [0.625 rgb <0.5,0.125,0>]
            [1.0 rgb <1,0.875,0>]
            }
          phase clock
          scale <3,30,3>
          translate -y*700*clock
          }
        }
      }
    translate <25,-42*Flameon,0>
    } // end Flame 
  
union {
  object { Capsule (0)}
  object { Flame }
  object { Flame
    rotate y*120
    } 
  object { Flame
    rotate y*240
    }
  light_source {
    <0, -80, 0>
    color rgb <1,0.375,0>*Flameon
    shadowless
    }       
  translate y*24
  rotate <0,30,0>
  }
      
#break  // end Scene 7  

#case (8) //***************** Scene 8 ***********************
// 1.5 seconds, frames 421 to 456
// capsule lands
// dust flies

#include "capsule.inc"
#include "marsenvr.inc"

global_settings { max_trace_level 6 }

camera {
  location  <0, 130, -500>
  angle 50
  look_at   <0, 28, 0>  
  } 
 
light_source {
  <-20000, 140000, -120000>
  color rgb Suncol
  } 
sky_sphere {
  pigment { gradient y
    color_map {
      [0 rgb Suncol]
      [1 rgb <0.125,0.5,0.6>]
      }
    }
  }
  
torus { 400,200    // dust
  scale <1,0.5,1>
  hollow
  texture { 
    pigment { color rgbt 1 }
    }
  interior {
    media {
      emission 0.2
      density { wood
        color_map {
          [0.25 color rgb 0]
          [1 color rgb Suncol*0.2] 
          }
        rotate x*90
        scale 90
        turbulence 0.2
        phase clock*4.5
        translate y*clock*60
        }  
      density { bozo
        color_map {
          [0.25 color rgb 0]
          [1 color rgb Suncol*0.2] 
          }
        scale 40
        turbulence 0.4
        phase clock
        translate y*clock*120
        }
    
      }  // end media
    }  // end interior  
  }  // end dust
    
plane {y,0    // ground
  texture {
    pigment { granite
      color_map {
        [0 rgb Suncol]
        [1 rgb Suncol*0.75]
        }
      scale 100
      }
    normal { bumps 0.3 scale 20 }
    finish { diffuse 0.85 }
    }
  }

#declare C1 = 0;
#declare S1 = seed(8);
#while (C1<250)             // rocks
  blob {        // random-shaped rock
    threshold 0.56
    sphere {0,1,1} // base part to make sure rock is grounded
    sphere {0,1,1  // random parts
      scale <0.5+rand(S1),0.5+rand(S1),0.5+rand(S1)>
      translate <rand(S1),rand(S1),rand(S1)>/2
      }
    sphere {0,1,1
      scale <0.5+rand(S1),0.5+rand(S1),0.5+rand(S1)>
      translate <rand(S1),rand(S1),rand(S1)>/2
      }
    sphere {0,1,1
      scale <0.5+rand(S1),0.5+rand(S1),0.5+rand(S1)>
      translate <rand(S1),rand(S1),rand(S1)>/2
      } 
    
    scale 24*rand(S1)
    
    translate <(rand(S1)-0.5)*3000,0,rand(S1)*2000-500>
    
    
    texture { //  rock texture
      pigment { bozo
        color_map {
          [0 rgb Suncol*0.75]
          [1 rgb <0.4,0.4,0.38>]
          }
        turbulence 0.4
        omega 0.8
        scale 0.5
        }
      finish { diffuse 0.8 }
      normal { granite 0.7 scale 2 }
      }
    }
  #declare C1 = C1 + 1;
  #end // while loop - rocks
  
height_field {  // distant hills
  tga
  "MarsHF.tga"
  //smooth
  texture {
    pigment {
      image_map {
        tga "MarsIM.tga"
        map_type 0
        interpolate 2
        } // image_map
      }
    finish { diffuse 0.8 }
    rotate <90,0,0>
    }
  translate <-0.5,0,-0.5>
  scale <3000,100,3000>
  rotate y*90
  rotate x*-5     // raise far edge, bury near edge
  translate <0,0,2000>
  }  

#declare Fade1 = clock*12;  // fade in
#if (Fade1<1)
  plane {z,0
    translate z*-450
    texture {
      pigment { color rgbt <0,0,0,Fade1>}
      }
    }
  #end  // if - fade in

  
#declare Fade2 = clock*6-5;  // fade out - slower
#if (Fade2>0)
  plane {z,0
    translate z*-450
    texture {
      pigment { color rgbt <0,0,0,1-Fade2>}
      }
    }
  #end  // if - fade out

#declare Flame = 
  sphere { 0,1   
    scale <18,60,18>
    texture {
      pigment { color rgbt 1 } 
      }
    hollow
    interior {
      media { emission 1
        density { spherical
          color_map {
            [0.0 rgb <0,0,0>]
            [0.625 rgb <0.5,0.125,0>]
            [1.0 rgb <1,0.875,0>]
            }  
          scale <18,56,18>
          }
        density { bozo poly_wave 0.3
          color_map {
            [0.0 rgb <0,0,0>]
            [0.625 rgb <0.5,0.125,0>]
            [1.0 rgb <1,0.875,0>]
            }
          phase clock
          scale <3,30,3>
          translate -y*500*clock
          } 
        density { bozo poly_wave 0.3
          color_map {
            [0.0 rgb <0,0,0>]
            [0.625 rgb <0.5,0.125,0>]
            [1.0 rgb <1,0.875,0>]
            }
          phase clock
          scale <3,30,3>
          translate -y*700*clock
          }
        }
      }
    translate <25,-42,0>
    } // end Flame 
  
union {
  object { Capsule (0)}
  object { Flame }
  object { Flame
    rotate y*120
    } 
  object { Flame
    rotate y*240
    }
  light_source {
    <0, -20, 0>
    color rgb <1,0.5,0>
    fade_distance 80
    fade_power 3
    }       
  translate y*(100-50*clock)
  rotate <0,20,0>
  }
      
#break  // end Scene 8

#case (9) //***************** Scene 9 ***********************
// 4 seconds, frames 457 to 552
// dust settling, rovot raises head, turns

#include "capsule.inc"
#include "marsenvr.inc"
#include "rovot.inc"

camera {
  location  <0, 100, -100>
  rotate y*30
  angle 20
  look_at   <0, 115, 0>  
  } 
 
light_source {
  <-8000000, 12000000, -3000000>
  color rgb Suncol
  } 

sky_sphere {
  pigment { gradient y
    color_map {
      [0.25 rgb Suncol*0.875 ]
      [1 rgb <0.125,0.5,0.6>]
      }
    }
  }
  
#declare Fade1 = clock*16;  // fade in, 1/2 sec
#if (Fade1<1)
  plane {z,0
    translate z*-90
    rotate y*30
    texture {
      pigment { color rgbt <0,0,0,Fade1>}
      }
    }
  #end  // if - fade in

#declare NR1 = clock*4-1.5;
  #if (NR1<0)
    #declare NR1 = 0;
    #end
  #if (NR1>1)
    #declare NR1 = 1;
    #end   
    
#declare HR = clock*4-3;
  #if (HR<0)
    #declare HR = 0;
    #end
  
object { Capsule(1)
  rotate y*30
  translate <0,28,0>
  }
object { Rovot (NR1,90-60*HR,0,0)
  rotate y*30
  translate y*40
  }  
  
#break  // end Scene 9  

#case (10) //***************** Scene 10 ***********************
// 2 seconds, frames 553 to 600
// camera view, rotating

#include "marsenvr.inc"

camera {
  location  <0, 115, 0>
  angle 60
  look_at   <0, 115, -100>
  rotate y*-120*clock
  } 
 
light_source {
  <-8000000, 12000000, -3000000>
  color rgb Suncol
  } 

sky_sphere {
  pigment { gradient y
    color_map {
      [0.25 rgb Suncol*0.875 ]
      [1 rgb <0.125,0.5,0.6>]
      }
    }
  }

union {
  plane {z,0
    translate -z*10
    texture {
      pigment { bozo 
        color_map {
          [ 0.95 rgbf <1,1,1,0.8>]
          [ 0.95 rgbf <1,1,1,0.3>]
          [1 rgb 1 ]
          }
        scale <1,0.005,0.005>
        translate z*100*clock
        }
      }
    }
  box {<-0.025,-0.5,-10>,<0.025,0.5,-9>}
  box {<-0.5,-0.025,-10>,<0.5,0.025,-9>}
  difference {
    cylinder {<0,0,-10>,<0,0,-9.99>,0.75}
    cylinder {<0,0,-11>,<0,0,-8>,0.7}
    }
  difference {
    box {<-4.8,-3.6,-10>,<4.8,3.6,-9.99>}
    box {<-4.75,-3.55,-11>,<4.75,3.55,-9>}
    box {<-5,-2.6,-11>,<5,2.6,-9>}
    box {<-3.8,-4,-11>,<3.8,4,-9>}
    }
  text { ttf "crystal" "REC" 0.01,0
    scale 0.5
    translate <-4.1,3.8,9.9>
    rotate y*180 
    } 
    
  #if (clock<0.5)
    text { ttf "crystal" "STATUS CHECK:" 0.01,0
      scale 0.5
      translate <-4.8,-4.15,9.9>
      rotate y*180 
      }
    #end
    
  #if ((clock>0.25)&(clock<0.5))
    text { ttf "crystal" "ALL SYSTEMS OPERATIONAL" 0.01,0
      scale 0.5
      translate <-1.125,-4.15,9.9>
      rotate y*180 
      }
    #end
  
  #if (clock>0.625)
    text { ttf "crystal" "BATTERY CHARGE:" 0.01,0
      scale 0.5
      translate <-4.8,-4.15,9.9>
      rotate y*180 
      }
    #end
    
  #if (clock>0.75)
    text { ttf "crystal" "96.3%" 0.01,0
      scale 0.5
      translate <-0.5,-4.15,9.9>
      rotate y*180 
      }
    #end
    
  #declare RS = abs(sin(clock*pi*4)); // flashing REC light
  #if (RS>0.5)
    sphere { 0 ,0.2
      texture {
        pigment { color rgb <1,0.3,0.3>}
        finish { ambient 1 }
        }
      scale <1,1,0.01>
      translate <4.6,4,-10>
      }
    #end
    
  texture {
    pigment { color rgb 1 }
    finish { ambient 1 }
    }
  translate y*115
  no_shadow
  rotate y*-120*clock
  }
  
#break  // end Scene 10

#case (11) //***************** Scene 11 ***********************
// 3 seconds, frames 601 to 672
// rovot walking from lander

#include "capsule.inc"
#include "marsenvr.inc"
#include "rovot.inc"

camera {
  location  <400, 100, -50>
  angle 60 
  look_at   <200, 60, 0>
  } 
 
light_source {
  <-8000000, 12000000, -3000000>
  color rgb Suncol
  } 

sky_sphere {
  pigment { gradient y
    color_map {
      [0.25 rgb Suncol*0.875 ]
      [1 rgb <0.125,0.5,0.6>]
      }
    }
  }
  
object { Capsule(1)
  rotate y*30
  translate <0,28,0>
  }
object { Rovot (1,0,clock*3,1)
  translate x*(250+18*clock*3)
  }  
  
#break  // end Scene 11       

#case (12) //***************** Scene 12 ***********************
// 2.5 seconds, frames 673 to 732
// rovot walking from behind

#include "marsenvr.inc"
#include "rovot.inc"

camera {
  location  <200, 10, 0>
  angle 60 
  look_at   <300, 50, 0>
  } 
 
light_source {
  <-8000000, 12000000, -3000000>
  color rgb Suncol
  } 

sky_sphere {
  pigment { gradient y
    color_map {
      [0.25 rgb Suncol*0.875 ]
      [1 rgb <0.125,0.5,0.6>]
      }
    }
  }
  
object { Rovot (1,0,clock*2.5,1)
  translate x*(250+18*clock*2.5)
  }  
  
#break  // end Scene 12  

#case (13) //***************** Scene 13 ***********************
// 2.5 seconds, frames 733 to 792
// rovot walking from above

#include "marsenvr.inc"
#include "rovot.inc"

camera {
  location  <340+clock*18*2.5, 90, -40>
  angle 60 
  look_at   <300+clock*18*2.5, 56, 0>
  } 
 
light_source {
  <-4000000, 8000000, -5000000>
  color rgb Suncol
  } 

sky_sphere {
  pigment { gradient y
    color_map {
      [0.25 rgb Suncol*0.875 ]
      [1 rgb <0.125,0.5,0.6>]
      }
    }
  }
  
object { Rovot (1,0,clock*2.5,1)
  translate x*(300+18*clock*2.5)
  }  
  
#break  // end Scene 13    

#case (14) //***************** Scene 14 ***********************
// 2 seconds, frames 793 to 840
// long shot, sky twds evening

#include "marsenvr.inc"
#include "rovot.inc"

camera {
  location  <3000, 800, 0>
  angle 40 
  look_at   <9000, 3000, 8000>
  } 
 
light_source {
  <-1000000, 4000000, -10000000>
  color rgb Suncol
  } 

sky_sphere {
  pigment { gradient y
    color_map {
      [0.562 rgb <1,0.75,0.4> ] 
      [0.812 rgb <0.75,0.2,0.1>]
      [1 rgb <0.2,0,0.3>]
      }
    scale 2
    translate y*-1
    turbulence 0.2
    lambda 2.5 
    // rotate x*5
    }
  }
  
object { Rovot (1,0,clock*2,1)
  translate x*(36*clock*2)
  rotate z*30
  rotate y*-60 
  scale 3
  translate <9000,800,8000>
  }   
  
#break  // end Scene 14  

#case (15) //***************** Scene 15 ***********************
// 3 seconds, frames 841 to 912
// silouette against evening sky

#include "rovot.inc"

camera {
  location  <0, -40, 0>
  angle 40 
  look_at   <0, 10, -300>
  } 
 
light_source {  
  <-1000000, 1000000, -10000000> 
  color rgb Suncol
  } 

sky_sphere {
  pigment { gradient y
    color_map {
      [0.594 rgb <1,0.75,0.4> ] 
      [0.688 rgb <0.75,0.2,0.1>]
      [0.75 rgb <0.2,0,0.4>]
      }
    scale 2
    translate y*-1
    turbulence 0.15
    lambda 2.125 
    // rotate x*5
    }
  }
  
height_field { 
  tga
  "MarsHF2.tga"
  //smooth
  texture {
    pigment {  
      image_map {
        tga "MarsIM2.tga"
        map_type 0
        //interpolate 2
        } // image_map
      }
    finish { diffuse 0.8 }
    rotate <90,0,0>
    }
  translate <-0.5,0,-0.5>
  scale <1500,250,1500>
  rotate 200*y
  translate <300,-100,-1000>
  } 
  
object { Rovot (1,0,clock*3,1)
  translate x*(18*clock*3)
  rotate z*15 
  rotate y*-30
  scale 3
  translate <-30,-40,-900>
  }  
  
#break  // end Scene 15 

#case (16) //***************** Scene 16 ***********************
// 5 seconds, frames 913 to 1032
// look around, deploy & drop

#include "rovot.inc"

camera {
  location  <0, 60, 200>  
  angle 40  
  look_at   <0, 40, 0>
  } 
 
light_source { 
  <-1000000, 1000000, -10000000> 
  color rgb <0.7,0.45,0.3>
  } 
  
light_source {  
  <-1000000, 3000000, 10000000> 
  color rgb <0.7,0.45,0.3>*0.75
  shadowless
  } 

sky_sphere {
  pigment { gradient y
    color_map {
      [0.438 rgb <1,0.75,0.4> ] 
      [0.562 rgb <0.75,0.2,0.1>]
      [0.688 rgb <0.3,0,0.4>]
      }
    scale 2
    translate y*-1
    turbulence 0.1
    //lambda 2.125 
    // rotate x*5
    }
  }
  
height_field { 
  tga
  "MarsHF2.tga"
  smooth
  texture {
    pigment {  
      image_map {
        tga "MarsIM2.tga"
        map_type 0
        //interpolate 2
        } // image_map 
      }
    finish { diffuse 0.8 ambient 0.15 }
    rotate <90,0,0>
    }
  translate <-0.5,0,-0.5>
  scale <1200,150,1200>
  translate <190,-140,0>
  }
  
height_field { 
  gif
  "Mars.gif"
  smooth
  translate <-0.5,0,-0.5>
  scale <40000,5000,40000>
  translate <-5000,-4400,0>
  rotate y*150
  
  texture {
    pigment { color rgb <0.7,0.45,0.3>}
    finish { diffuse 0.4 ambient 0.5 }
    }
  } 

#declare HRR = clock*5;

#declare HR = (HRR/0.375)*75;
#if (HR>75)
  #declare HR = 75;
  #end
  
#if (HRR>0.625)
  #declare HR = HR - (HRR-0.625)/0.75*150;
  #if (HR<-75)
    #declare HR = -75;
    #end
  #end
#if (HRR>1.625)
  #declare HR = HR + (HRR-1.625)/0.375*75;
  #if (HR>0)
    #declare HR = 0;
    #end
  #end

#declare DD = (clock*5-2)/3;
#if (DD<0)
  #declare DD = 0;
  #end
  
object { Rovot (1,HR,DD,2)
  rotate y*-110
  }
    
  
#break  // end Scene 16  

#case (17) //***************** Scene 17 ***********************
// 1.5 seconds, frames 1033 to 1068
// gadget unfurl

#include "rovot.inc"

camera {
  location  <-50, 50, -80>  
  angle 50  
  look_at   <20, 40, 0>
  rotate y*-110
  } 
 
light_source { 
  <-1000000, 1000000, -10000000> 
  color rgb <0.7,0.45,0.3>
  } 
  
light_source {  
  <-1000000, 3000000, 10000000> 
  color rgb <0.7,0.45,0.3>*0.75
  shadowless
  } 

sky_sphere {
  pigment { gradient y
    color_map {
      [0.438 rgb <1,0.75,0.4> ] 
      [0.562 rgb <0.75,0.2,0.1>]
      [0.688 rgb <0.3,0,0.4>]
      }
    scale 2
    translate y*-1
    turbulence 0.1
    //lambda 2.125 
    // rotate x*5
    }
  }
  
height_field { 
  tga
  "MarsHF2.tga"
  smooth
  texture {
    pigment {  
      image_map {
        tga "MarsIM2.tga"
        map_type 0
        //interpolate 2
        } // image_map 
      }
    finish { diffuse 0.8 ambient 0.15 }
    rotate <90,0,0>
    }
  translate <-0.5,0,-0.5>
  scale <1200,150,1200>
  translate <190,-140,0>
  }
  
height_field { 
  gif
  "Mars.gif"
  smooth
  translate <-0.5,0,-0.5>
  scale <40000,5000,40000>
  translate <-5000,-4400,0>
  rotate y*150
  
  texture {
    pigment { color rgb <0.7,0.45,0.3>}
    finish { diffuse 0.4 ambient 0.5 }
    }
  } 
  
object { Rovot (1,0,1,2)
  rotate y*-110
  }
  
#declare GK = clock*1.5-0.5;
#if (GK<0)
  #declare GK = 0;
  #end

object { Gadget (GK*0.8,0)
  translate <40,2,0>
  rotate y*-110
  }  
  
#break  // end Scene 17 

#case (18) //***************** Scene 18 ***********************
// 4 seconds, frames 1069 to 1164
// gadget unfurl and activate

#include "rovot.inc"

camera {
  location  <130, 80, 0>  
  angle 50  
  look_at   <40, 80, 0>
  rotate y*-110
  } 
 
light_source { 
  <-1000000, 1000000, -10000000> 
  color rgb <0.7,0.45,0.3>
  } 
  
light_source {  
  <-1000000, 3000000, 10000000> 
  color rgb <0.7,0.45,0.3>*0.75
  shadowless
  } 

sky_sphere {
  pigment { gradient y
    color_map {
      [0.438 rgb <1,0.75,0.4> ] 
      [0.562 rgb <0.75,0.2,0.1>]
      [0.688 rgb <0.3,0,0.4>]
      }
    scale 2
    translate y*-1
    turbulence 0.1
    //lambda 2.125 
    // rotate x*5
    }
  }
  
height_field { 
  tga
  "MarsHF2.tga"
  smooth
  texture {
    pigment {  
      image_map {
        tga "MarsIM2.tga"
        map_type 0
        //interpolate 2
        } // image_map 
      }
    finish { diffuse 0.8 ambient 0.15 }
    rotate <90,0,0>
    }
  translate <-0.5,0,-0.5>
  scale <1200,150,1200>
  translate <190,-140,0>
  }
  
height_field { 
  gif
  "Mars.gif"
  smooth
  translate <-0.5,0,-0.5>
  scale <40000,5000,40000>
  translate <-5000,-4400,0>
  rotate y*150
  
  texture {
    pigment { color rgb <0.7,0.45,0.3>}
    finish { diffuse 0.4 ambient 0.5 }
    }
  } 
  
object { Rovot (1,0,1,2)
  rotate y*-110
  }

#declare UF = clock*16;
#if (UF>1)
  #declare UF = 1;
  #end
  
#declare AV = clock-.062;
#if (AV<0)
  #declare AV = 0;
  #end
  
object { Gadget (0.8+UF*0.2,AV)
  translate <40,2,0>
  rotate y*-110
  }  
  
#break  // end Scene 18  

#case (19) //***************** Scene 19 ***********************
// 4 seconds, frames 1165 to 1260
// closing credits

#include "rovot.inc"

camera {
  location  <530, -30, 400>  
  angle 10  
  look_at   <70, 70, 0>
  rotate y*-110
  } 
 
//light_source { 
//  <-1000000, 1000000, -10000000> 
//  color rgb <0.7,0.45,0.3>
//  } 
  
sky_sphere {
  pigment { gradient y
    color_map {
      [0.438 rgb <1,0.75,0.4> ] 
      [0.5 rgb <0.75,0.2,0.1>]
      [0.594 rgb <0.3,0,0.4>]
      }
    scale 2
    translate y*-1
    turbulence 0.1
    //lambda 2.125 
    // rotate x*5
    }
  }
  
height_field { 
  tga
  "MarsHF2.tga"
  smooth
  texture {
    pigment {  
      image_map {
        tga "MarsIM2.tga"
        map_type 0
        //interpolate 2
        } // image_map 
      }
    finish { diffuse 0.8 ambient 0.15 }
    rotate <90,0,0>
    }
  translate <-0.5,0,-0.5>
  scale <1200,150,1200>
  translate <190,-140,0>
  }
  
#declare AV = clock+0.25;
#if (AV>1)
  #declare AV = AV-1;
  #end
  
object { Gadget (1,AV)
  translate <40,2,0>
  rotate y*-110
  }  

#if (clock>0.25) 
  text { ttf "frkgotn" "MARS CONQUEST" 0.01,0
    scale <0.4,0.5,0.5>
    translate <-5,-3.1,60>
    rotate<-9.325,-131,0>
    translate <530,-30,400>
    rotate y*-110 
    texture {
      pigment { color rgb 1}
      finish { ambient 1 }
      }
    }
  #end
  
#if (clock>0.5) 
  text { ttf "frkgotn" "Neil Alexander  1999" 0.01,0
    scale <0.4,0.5,0.5>
    translate <-5,-3.7,60>
    rotate<-9.325,-131,0>
    translate <530,-30,400>
    rotate y*-110 
    texture {
      pigment { color rgb 1}
      finish { ambient 1 }
      }
    }
  #end 
  
#if (clock>0.9375)
  #declare FK = clock*32-30;
  #if (FK>1)
    #declare FK = 1;
    #end 
    
  plane {z,0
    texture {
      pigment { gradient y
        color_map {
          [(0.875-FK*0.875) rgbt <0,0,0,1>]
          [(1-FK) rgbt 0]
          }
        scale 4.25
        }
      }
    translate <0,0,50>
    rotate<-9.325,-131,0>
    translate <530,-30,400>
    rotate y*-110 
    }
  #end
  
#break  // end Scene 19

#end  // end switch scene
