#!/usr/bin/env python3
#
# Check for the first line in a file generated with gmmproc,
# to see which gmmproc version was used, to see whether
# to enable __declspec(dllexport to export symbols).  This
# is *not* intended for source files that are not generated
# with gmmproc.
#
# Author: Chun-wei Fan April 2, 2020

import argparse
import os
import sys

min_required_gmmproc_ver = '2.64.3'

parser = argparse.ArgumentParser(description='Check gmmproc version used.')
parser.add_argument('--file',
                    dest='file',
                    help='Generated .cc/.h file to check gmmproc version')
parser.add_argument('--gmmprocdir',
                    dest='gmmprocdir',
                    help='Directory where gmmproc is located')
args = parser.parse_args()

if args.file is None and args.gmmprocdir is None:
    raise ValueError('Either --file or --gmmprocdir must be specified')

if args.gmmprocdir is not None:
    # gmmprocdir is specified: Check version string in gmmproc
    gmmproc_path = os.path.join(args.gmmprocdir, 'gmmproc')
    if not os.path.exists(gmmproc_path):
        raise ValueError('A valid directory to locate gmmproc must be ' \
                         'specified with --gmmprocdir=<directory>')

    gmmproc_ver_str = None
    with open(gmmproc_path, 'r') as f:
        for line in f:
            if line.startswith('  $main::glibmm_version = '):
                gmmproc_ver_str = line[line.find('\"') + 1:line.rfind('\"')]

    if gmmproc_ver_str is None:
        raise ValueError('The gmmproc at %s is invalid' % gmmproc_path)

    gmmproc_ver = gmmproc_ver_str.split('.')
else:
    # A pre-generated file is specified via --file
    if not os.path.exists(args.file):
        raise FileNotFoundError('File specified with --file does not exist')

    # We only allow .h/.cc files to run this check
    if not args.file.endswith('.cc') and \
       not args.file.endswith('.h'):
        raise ValueError('Only .cc/.h files are accepted here')

    # Now grab the first line of the file we are checking for
    f = open(args.file)
    firstline = f.readline()
    f.close()

    # Check for gmmproc signature...
    if not firstline.startswith('// Generated by gmmproc '):
       raise ValueError('Specified file is not generated by gmmproc')

    tokens = firstline.split()
    gmmproc_ver = tokens[tokens.index('gmmproc') + 1].split('.')

# Now compare the gmmproc version against the one we want
# (2.64.3 or later)
gmmproc_major = int(gmmproc_ver[0])
gmmproc_minor = int(gmmproc_ver[1])
gmmproc_micro = int(gmmproc_ver[2])

min_required_ver = min_required_gmmproc_ver.split('.')
min_major_ver = int(min_required_ver[0])
min_minor_ver = int(min_required_ver[1])
min_micro_ver = int(min_required_ver[2])

if gmmproc_major > min_major_ver or \
   (gmmproc_major == min_major_ver and \
    gmmproc_minor > min_minor_ver) or \
   (gmmproc_major == min_major_ver and \
    gmmproc_minor == min_minor_ver and \
    gmmproc_micro >= min_micro_ver):
    sys.exit(0)
else:
    sys.exit(1)
