"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.IntegrationInstanceBuilder = void 0;

var _uuid = require("uuid");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; } /*
                                                                                                                                                                                                                   * Copyright OpenSearch Contributors
                                                                                                                                                                                                                   * SPDX-License-Identifier: Apache-2.0
                                                                                                                                                                                                                   */

class IntegrationInstanceBuilder {
  constructor(client) {
    _defineProperty(this, "client", void 0);

    this.client = client;
  }

  build(integration, options) {
    const instance = integration.deepCheck().then(result => {
      if (!result.ok) {
        return Promise.reject(result.error);
      }

      return integration.getAssets();
    }).then(assets => {
      if (!assets.ok) {
        return Promise.reject(assets.error);
      }

      return assets.value;
    }).then(assets => this.remapIDs(assets.savedObjects)).then(assets => this.remapDataSource(assets, options.dataSource)).then(assets => this.postAssets(assets)).then(refs => this.buildInstance(integration, refs, options));
    return instance;
  }

  remapDataSource(assets, dataSource) {
    if (!dataSource) return assets;
    assets = assets.map(asset => {
      if (asset.type === 'index-pattern') {
        asset.attributes.title = dataSource;
      }

      return asset;
    });
    return assets;
  }

  remapIDs(assets) {
    const toRemap = assets.filter(asset => asset.id);
    const idMap = new Map();
    return toRemap.map(item => {
      if (!idMap.has(item.id)) {
        idMap.set(item.id, (0, _uuid.v4)());
      }

      item.id = idMap.get(item.id);

      for (let ref = 0; ref < item.references.length; ref++) {
        const refId = item.references[ref].id;

        if (!idMap.has(refId)) {
          idMap.set(refId, (0, _uuid.v4)());
        }

        item.references[ref].id = idMap.get(refId);
      }

      return item;
    });
  }

  async postAssets(assets) {
    try {
      const response = await this.client.bulkCreate(assets);
      const refs = response.saved_objects.map(obj => {
        var _obj$attributes;

        return {
          assetType: obj.type,
          assetId: obj.id,
          status: 'available',
          // Assuming a successfully created object is available
          isDefaultAsset: obj.type === 'dashboard',
          // Assuming for now that dashboards are default
          description: (_obj$attributes = obj.attributes) === null || _obj$attributes === void 0 ? void 0 : _obj$attributes.title
        };
      });
      return Promise.resolve(refs);
    } catch (err) {
      return Promise.reject(err);
    }
  }

  async buildInstance(integration, refs, options) {
    const config = await integration.getConfig();

    if (!config.ok) {
      return Promise.reject(new Error('Attempted to create instance with invalid template', config.error));
    }

    return Promise.resolve({
      name: options.name,
      templateName: config.value.name,
      dataSource: options.dataSource,
      creationDate: new Date().toISOString(),
      assets: refs
    });
  }

}

exports.IntegrationInstanceBuilder = IntegrationInstanceBuilder;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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