"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerNoteRoute = registerNoteRoute;

var _configSchema = require("@osd/config-schema");

var _notebooks = require("../../../common/constants/notebooks");

var _notebooks2 = require("../../adaptors/notebooks");

/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
function registerNoteRoute(router) {
  // Fetch all the notebooks available
  router.get({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/`,
    validate: {}
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    let notebooksData = [];

    try {
      notebooksData = await _notebooks2.BACKEND.viewNotes(opensearchNotebooksClient, _notebooks.wreckOptions);
      return response.ok({
        body: {
          data: notebooksData
        }
      });
    } catch (error) {
      console.log('Notebook:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  }); // Get all paragraphs of notebooks

  router.get({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note/{noteId}`,
    validate: {
      params: _configSchema.schema.object({
        noteId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const notebookinfo = await _notebooks2.BACKEND.fetchNote(opensearchNotebooksClient, request.params.noteId, _notebooks.wreckOptions);
      return response.ok({
        body: notebookinfo
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  }); // Add a Notebook

  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note`,
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const addResponse = await _notebooks2.BACKEND.addNote(opensearchNotebooksClient, request.body, _notebooks.wreckOptions);
      return response.ok({
        body: addResponse.message.objectId
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  }); // Rename a notebook

  router.put({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note/rename`,
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        noteId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const renameResponse = await _notebooks2.BACKEND.renameNote(opensearchNotebooksClient, request.body, _notebooks.wreckOptions);
      return response.ok({
        body: renameResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  }); // Clone a notebook

  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note/clone`,
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        noteId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const cloneResponse = await _notebooks2.BACKEND.cloneNote(opensearchNotebooksClient, request.body, _notebooks.wreckOptions);
      return response.ok({
        body: cloneResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  }); // Delete notebooks

  router.delete({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note/{noteList}`,
    validate: {
      params: _configSchema.schema.object({
        noteList: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const delResponse = await _notebooks2.BACKEND.deleteNote(opensearchNotebooksClient, request.params.noteList, _notebooks.wreckOptions);
      return response.ok({
        body: delResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  }); // Add sample notebooks

  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note/addSampleNotebooks`,
    validate: {
      body: _configSchema.schema.object({
        visIds: _configSchema.schema.arrayOf(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const addSampleNotesResponse = await _notebooks2.BACKEND.addSampleNotes(opensearchNotebooksClient, request.body.visIds, _notebooks.wreckOptions);
      return response.ok({
        body: addSampleNotesResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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