"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getSampleNotebooks = void 0;

var _uuid = require("uuid");

/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
const getDemoNotebook = (dateString, visId) => {
  const uuids = [(0, _uuid.v4)()];
  const oneWeekAgo = new Date();
  oneWeekAgo.setDate(oneWeekAgo.getDate() - 7);
  return {
    name: 'OpenSearch Notebooks Quick Start Guide',
    dateCreated: dateString,
    dateModified: dateString,
    backend: 'Default',
    paragraphs: [{
      output: [{
        result: `An OpenSearch Dashboards notebook is an interface that lets you easily combine code snippets, live visualizations, and narrative text in a single notebook interface.

Notebooks let you interactively explore data by running different visualizations that you can share with team members to collaborate on a project.

A notebook is a document composed of two elements: code blocks (Markdown/SQL/PPL) and visualizations.

Common use cases include creating postmortem reports, designing runbooks, building live infrastructure reports, and writing documentation.

You can also generate [reports](https://opensearch.org/docs/dashboards/reporting/) directly from your notebooks.

For more information, refer to the [documentation](https://opensearch.org/docs/dashboards/notebooks/).`,
        outputType: 'MARKDOWN',
        execution_time: '0 ms'
      }],
      input: {
        inputText: `%md
An OpenSearch Dashboards notebook is an interface that lets you easily combine code snippets, live visualizations, and narrative text in a single notebook interface.

Notebooks let you interactively explore data by running different visualizations that you can share with team members to collaborate on a project.

A notebook is a document composed of two elements: code blocks (Markdown/SQL/PPL) and visualizations.

Common use cases include creating postmortem reports, designing runbooks, building live infrastructure reports, and writing documentation.

You can also generate [reports](https://opensearch.org/docs/dashboards/reporting/) directly from your notebooks.

For more information, refer to the [documentation](https://opensearch.org/docs/dashboards/notebooks/).`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'Notebooks combine code blocks and visualizations for describing data. Code blocks support markdown, SQL, and PPL languages. Specify the input language on the first line using %[language type] syntax. For example, type %md for markdown, %sql for SQL, and %ppl for PPL. A sample visualization is shown below:',
        outputType: 'MARKDOWN',
        execution_time: '0 ms'
      }],
      input: {
        inputText: `%md
Notebooks combine code blocks and visualizations for describing data. Code blocks support markdown, SQL, and PPL languages. Specify the input language on the first line using %\[language type\] syntax. For example, type %md for markdown, %sql for SQL, and %ppl for PPL. A sample visualization is shown below:`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: '',
        outputType: 'VISUALIZATION',
        execution_time: '0.017 ms'
      }],
      input: {
        inputText: `{"viewMode":"view","panels":{"1":{"gridData":{"x":0,"y":0,"w":50,"h":20,"i":"1"},"type":"visualization","explicitInput":{"id":"1","savedObjectId":"${visId}"}}},"isFullScreenMode":false,"filters":[],"useMargins":false,"id":"${uuids[0]}","timeRange":{"to":"${dateString}","from":"${oneWeekAgo}"},"title":"embed_viz_${uuids[0]}","query":{"query":"","language":"lucene"},"refreshConfig":{"pause":true,"value":15}}`,
        inputType: 'VISUALIZATION'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'You can add a new paragraph from the **Paragraph actions** menu on the top right of the page, the context menu on the top right of each paragraph, or the **Add paragraph** button on the bottom of the page.',
        outputType: 'MARKDOWN',
        execution_time: '0 ms'
      }],
      input: {
        inputText: `%md
You can add a new paragraph from the **Paragraph actions** menu on the top right of the page, the context menu on the top right of each paragraph, or the **Add paragraph** button on the bottom of the page.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'You can also reorder, duplicate, or delete paragraphs from these menus.',
        outputType: 'MARKDOWN',
        execution_time: '0 ms'
      }],
      input: {
        inputText: `%md
You can also reorder, duplicate, or delete paragraphs from these menus.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: "To execute a paragraph, choose **Run**. To make changes to the input block, choose the downward arrow that's next to the paragraph title.",
        outputType: 'MARKDOWN',
        execution_time: '0 ms'
      }],
      input: {
        inputText: `%md
To execute a paragraph, choose **Run**. To make changes to the input block, choose the downward arrow that's next to the paragraph title.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `There are three view modes:
- **View both**: allows you to edit paragraphs and view outputs on the same page.
- **Input only**: only shows input of each paragraph, allowing easier editing.
- **Output only**: only shows output of each paragraph and hides panels.`,
        outputType: 'MARKDOWN',
        execution_time: '0 ms'
      }],
      input: {
        inputText: `%md
There are three view modes:
- **View both**: allows you to edit paragraphs and view outputs on the same page.
- **Input only**: only shows input of each paragraph, allowing easier editing.
- **Output only**: only shows output of each paragraph and hides panels.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'To go back, choose **Notebooks** on the top left of your screen.',
        outputType: 'MARKDOWN',
        execution_time: '0 ms'
      }],
      input: {
        inputText: `%md
To go back, choose **Notebooks** on the top left of your screen.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }]
  };
};

const getRootCauseNotebook = (dateString, visIds) => {
  const uuids = [(0, _uuid.v4)(), (0, _uuid.v4)()];
  const oneWeekAgo = new Date();
  oneWeekAgo.setDate(oneWeekAgo.getDate() - 7);
  return {
    name: '[Logs] Sample Root Cause Event Analysis',
    dateCreated: dateString,
    dateModified: dateString,
    backend: 'Default',
    paragraphs: [{
      output: [{
        result: 'This notebook goes through a sample root cause event analysis using PPL and visualizations with the OpenSearch Dashboards sample web logs data.',
        outputType: 'MARKDOWN',
        execution_time: '0.016 ms'
      }],
      input: {
        inputText: `%md
This notebook goes through a sample root cause event analysis using PPL and visualizations with the OpenSearch Dashboards sample web logs data.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: "First, let's bring up a few visualizations for a quick overview.",
        outputType: 'MARKDOWN',
        execution_time: '0.046 ms'
      }],
      input: {
        inputText: `%md
First, let's bring up a few visualizations for a quick overview.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: '',
        outputType: 'VISUALIZATION',
        execution_time: '0.017 ms'
      }],
      input: {
        inputText: `{"viewMode":"view","panels":{"1":{"gridData":{"x":0,"y":0,"w":50,"h":20,"i":"1"},"type":"visualization","explicitInput":{"id":"1","savedObjectId":"${visIds[0]}"}}},"isFullScreenMode":false,"filters":[],"useMargins":false,"id":"${uuids[0]}","timeRange":{"to":"${dateString}","from":"${oneWeekAgo}"},"title":"embed_viz_${uuids[0]}","query":{"query":"","language":"lucene"},"refreshConfig":{"pause":true,"value":15}}`,
        inputType: 'VISUALIZATION'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: '',
        outputType: 'VISUALIZATION',
        execution_time: '0.017 ms'
      }],
      input: {
        inputText: `{"viewMode":"view","panels":{"1":{"gridData":{"x":0,"y":0,"w":50,"h":20,"i":"1"},"type":"visualization","explicitInput":{"id":"1","savedObjectId":"${visIds[1]}"}}},"isFullScreenMode":false,"filters":[],"useMargins":false,"id":"${uuids[1]}","timeRange":{"to":"${dateString}","from":"${oneWeekAgo}"},"title":"embed_viz_${uuids[1]}","query":{"query":"","language":"lucene"},"refreshConfig":{"pause":true,"value":15}}`,
        inputType: 'VISUALIZATION'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `Let's take a look at the source data by the selected fields (search and fields).`,
        outputType: 'MARKDOWN',
        execution_time: '0.013 ms'
      }],
      input: {
        inputText: `%md
Let's take a look at the source data by the selected fields (search and fields).`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, 
  response, bytes | head 20

`,
        outputType: 'QUERY',
        execution_time: '0.008 ms'
      }],
      input: {
        inputText: `%ppl
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, 
  response, bytes | head 20

`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'Check for any error log with response code 404 or 503 (filter).',
        outputType: 'MARKDOWN',
        execution_time: '0.009 ms'
      }],
      input: {
        inputText: `%md
Check for any error log with response code 404 or 503 (filter).`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response, bytes | where response='503' or response='404' | head 20
`,
        outputType: 'QUERY',
        execution_time: '0.007 ms'
      }],
      input: {
        inputText: `%ppl
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response, bytes | where response='503' or response='404' | head 20
`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: "We see too many events. Let's quickly check which host has the issue (dedup).",
        outputType: 'MARKDOWN',
        execution_time: '0.014 ms'
      }],
      input: {
        inputText: `%md
We see too many events. Let's quickly check which host has the issue (dedup).`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' | dedup host | head 20
`,
        outputType: 'QUERY',
        execution_time: '0.010 ms'
      }],
      input: {
        inputText: `%ppl
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' | dedup host | head 20
`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: "We get too few events. Let's dedup in consecutive mode (dedup).",
        outputType: 'MARKDOWN',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%md
We get too few events. Let's dedup in consecutive mode (dedup).`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' |
    dedup host consecutive=true | head 20
`,
        outputType: 'QUERY',
        execution_time: '0.007 ms'
      }],
      input: {
        inputText: `%ppl
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' |
    dedup host consecutive=true | head 20
`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'How many IP addresses for each response (stats).',
        outputType: 'MARKDOWN',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%md
How many IP addresses for each response (stats).`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' | 
   stats count() as ip_count by response | head 20
`,
        outputType: 'QUERY',
        execution_time: '0.008 ms'
      }],
      input: {
        inputText: `%ppl
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' | 
   stats count() as ip_count by response | head 20
`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: "To dive deep, let's group by host and response, count, and sum (stats).",
        outputType: 'MARKDOWN',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%md
To dive deep, let's group by host and response, count, and sum (stats).`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' | 
   stats count() as ip_count, sum(bytes) 
   as sum_bytes by host, response | head 20
`,
        outputType: 'QUERY',
        execution_time: '0.007 ms'
      }],
      input: {
        inputText: `%ppl
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' | 
   stats count() as ip_count, sum(bytes) 
   as sum_bytes by host, response | head 20
`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `We don't see a meaningful response. Let's change to resp_code (rename).`,
        outputType: 'MARKDOWN',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%md
We don't see a meaningful response. Let's change to resp_code (rename).`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' |
    stats count() as ip_count, sum(bytes) 
    as sum_bytes by host, response |
     rename response as resp_code | head 20

`,
        outputType: 'QUERY',
        execution_time: '0.009 ms'
      }],
      input: {
        inputText: `%ppl
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' |
    stats count() as ip_count, sum(bytes) 
    as sum_bytes by host, response |
     rename response as resp_code | head 20

`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: "The data looks better now. Let's sort by `DESC count` and `ASC sum_bytes` (sort).",
        outputType: 'MARKDOWN',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%md
The data looks better now. Let's sort by \`DESC count\` and \`ASC sum_bytes\` (sort).`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' | 
   stats count() as ip_count, sum(bytes) 
   as sum_bytes by host, response | 
   rename response as resp_code | 
   sort - ip_count, + sum_bytes | head 20
`,
        outputType: 'QUERY',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%ppl
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' | 
   stats count() as ip_count, sum(bytes) 
   as sum_bytes by host, response | 
   rename response as resp_code | 
   sort - ip_count, + sum_bytes | head 20
`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: "Let's check if we can perform aggregations after stats (eval).",
        outputType: 'MARKDOWN',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%md
Let's check if we can perform aggregations after stats (eval).`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' |
    stats count() as ip_count, sum(bytes) 
    as sum_bytes by host, response | 
    rename response as resp_code | 
    sort - ip_count, + sum_bytes | 
    eval per_ip_bytes=sum_bytes/ip_count  | head 20
`,
        outputType: 'QUERY',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%ppl
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' |
    stats count() as ip_count, sum(bytes) 
    as sum_bytes by host, response | 
    rename response as resp_code | 
    sort - ip_count, + sum_bytes | 
    eval per_ip_bytes=sum_bytes/ip_count  | head 20
`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `Wait, what's meant by an evaluation. Can we really perform an evaluation?`,
        outputType: 'MARKDOWN',
        execution_time: '0.007 ms'
      }],
      input: {
        inputText: `%md
Wait, what's meant by an evaluation. Can we really perform an evaluation?`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' | 
   stats count() as ip_count, sum(bytes) 
   as sum_bytes by host, response | 
   rename response as resp_code | 
   sort - ip_count, + sum_bytes | 
   eval per_ip_bytes=sum_bytes/ip_count, 
    double_per_ip_bytes = 2 * per_ip_bytes | head 20
`,
        outputType: 'QUERY',
        execution_time: '0.010 ms'
      }],
      input: {
        inputText: `%ppl
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' | 
   stats count() as ip_count, sum(bytes) 
   as sum_bytes by host, response | 
   rename response as resp_code | 
   sort - ip_count, + sum_bytes | 
   eval per_ip_bytes=sum_bytes/ip_count, 
    double_per_ip_bytes = 2 * per_ip_bytes | head 20
`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }]
  };
};

const getSQLNotebook = dateString => {
  return {
    name: '[Flights] OpenSearch SQL Quick Start Guide',
    dateCreated: dateString,
    dateModified: dateString,
    backend: 'Default',
    paragraphs: [{
      output: [{
        result: `OpenSearch SQL lets you write queries in SQL rather than the [OpenSearch query domain-specific language (DSL)](https://opensearch.org/docs/opensearch/query-dsl/full-text/). If you’re already familiar with SQL and don’t want to learn the query DSL, this feature is a great option.

For more information, please refer to the [documentation](https://opensearch.org/docs/search-plugins/sql/index/).`,
        outputType: 'MARKDOWN',
        execution_time: '0.013 ms'
      }],
      input: {
        inputText: `%md
OpenSearch SQL lets you write queries in SQL rather than the [OpenSearch query domain-specific language (DSL)](https://opensearch.org/docs/opensearch/query-dsl/full-text/). If you’re already familiar with SQL and don’t want to learn the query DSL, this feature is a great option.

For more information, please refer to the [documentation](https://opensearch.org/docs/search-plugins/sql/index/).`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'This notebook shows you how to use SQL with the [OpenSearch Dashboards sample flight data](/app/home#/tutorial_directory).',
        outputType: 'MARKDOWN',
        execution_time: '0.010 ms'
      }],
      input: {
        inputText: `%md
This notebook shows you how to use SQL with the [OpenSearch Dashboards sample flight data](/app/home#/tutorial_directory).`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'To use SQL, add a code paragraph, type %sql on the first line, and then add SQL queries on the next line.',
        outputType: 'MARKDOWN',
        execution_time: '0.007 ms'
      }],
      input: {
        inputText: `%md
To use SQL, add a code paragraph, type %sql on the first line, and then add SQL queries on the next line.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'Select * from opensearch_dashboards_sample_data_flights limit 20;',
        outputType: 'QUERY',
        execution_time: '0.007 ms'
      }],
      input: {
        inputText: `%sql
Select * from opensearch_dashboards_sample_data_flights limit 20;`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: "You can specify fields in the `SELECT` clause and use the `WHERE` clause to filter results. The following query finds flights heading to countries that start with 'A' that are more than 5000 miles away.",
        outputType: 'MARKDOWN',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%md
You can specify fields in the \`SELECT\` clause and use the \`WHERE\` clause to filter results. The following query finds flights heading to countries that start with 'A' that are more than 5000 miles away.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `SELECT FlightNum,OriginCountry,OriginCityName,DestCountry,DestCityName,DistanceMiles FROM opensearch_dashboards_sample_data_flights WHERE DistanceMiles > 5000 AND DestCountry LIKE 'A%' LIMIT 20;`,
        outputType: 'QUERY',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%sql
SELECT FlightNum,OriginCountry,OriginCityName,DestCountry,DestCityName,DistanceMiles FROM opensearch_dashboards_sample_data_flights WHERE DistanceMiles > 5000 AND DestCountry LIKE 'A%' LIMIT 20;`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'OpenSearch SQL also supports subqueries:',
        outputType: 'MARKDOWN',
        execution_time: '0.007 ms'
      }],
      input: {
        inputText: `%md
OpenSearch SQL also supports subqueries:`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `SELECT opensearch_dashboards_sample_data_flights.FlightNum,opensearch_dashboards_sample_data_flights.OriginCountry,opensearch_dashboards_sample_data_flights.OriginCityName,opensearch_dashboards_sample_data_flights.DestCountry,opensearch_dashboards_sample_data_flights.DestCityName,opensearch_dashboards_sample_data_flights.DistanceMiles FROM opensearch_dashboards_sample_data_flights WHERE FlightNum IN (SELECT FlightNum FROM opensearch_dashboards_sample_data_flights WHERE DistanceMiles > 5000 AND DestCountry = 'AU') LIMIT 20;`,
        outputType: 'QUERY',
        execution_time: '0.009 ms'
      }],
      input: {
        inputText: `%sql
SELECT opensearch_dashboards_sample_data_flights.FlightNum,opensearch_dashboards_sample_data_flights.OriginCountry,opensearch_dashboards_sample_data_flights.OriginCityName,opensearch_dashboards_sample_data_flights.DestCountry,opensearch_dashboards_sample_data_flights.DestCityName,opensearch_dashboards_sample_data_flights.DistanceMiles FROM opensearch_dashboards_sample_data_flights WHERE FlightNum IN (SELECT FlightNum FROM opensearch_dashboards_sample_data_flights WHERE DistanceMiles > 5000 AND DestCountry = 'AU') LIMIT 20;`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'OpenSearch SQL supports inner joins, cross joins, and left outer joins. The following query joins the flights index with itself to find flights departed from countries that are both origins and destinations.',
        outputType: 'MARKDOWN',
        execution_time: '0.007 ms'
      }],
      input: {
        inputText: `%md
OpenSearch SQL supports inner joins, cross joins, and left outer joins. The following query joins the flights index with itself to find flights departed from countries that are both origins and destinations.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `
SELECT a.FlightNum,a.OriginCountry,a.OriginCityName,a.DestCountry,a.DestCityName,a.DistanceMiles FROM opensearch_dashboards_sample_data_flights a JOIN opensearch_dashboards_sample_data_flights b on a.OriginCountry = b.DestCountry LIMIT 20`,
        outputType: 'QUERY',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%sql
SELECT a.FlightNum,a.OriginCountry,a.OriginCityName,a.DestCountry,a.DestCityName,a.DistanceMiles FROM opensearch_dashboards_sample_data_flights a JOIN opensearch_dashboards_sample_data_flights b on a.OriginCountry = b.DestCountry LIMIT 20`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'For aggregations, use the `GROUP BY` clause. The following query finds the countries with more than 500 flights departed.',
        outputType: 'MARKDOWN',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%md
For aggregations, use the \`GROUP BY\` clause. The following query finds the countries with more than 500 flights departed.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `SELECT OriginCountry,COUNT(1) FROM opensearch_dashboards_sample_data_flights GROUP BY OriginCountry HAVING COUNT(1) > 500 LIMIT 20;`,
        outputType: 'QUERY',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%sql
SELECT OriginCountry,COUNT(1) FROM opensearch_dashboards_sample_data_flights GROUP BY OriginCountry HAVING COUNT(1) > 500 LIMIT 20;`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'OpenSearch SQL supports expressions.',
        outputType: 'MARKDOWN',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%md
OpenSearch SQL supports expressions.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `SELECT abs(-1.234), abs(-1 * abs(-5)), dayofmonth(DATE '2021-07-07');`,
        outputType: 'QUERY',
        execution_time: '0.005 ms'
      }],
      input: {
        inputText: `%sql
SELECT abs(-1.234), abs(-1 * abs(-5)), dayofmonth(DATE '2021-07-07');`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }]
  };
};

const getPPLNotebook = dateString => {
  return {
    name: '[Logs] OpenSearch Piped Processing Language (PPL) Quick Start Guide',
    dateCreated: dateString,
    dateModified: dateString,
    backend: 'Default',
    paragraphs: [{
      output: [{
        result: `Query DSL is powerful and fast, but it has a steep learning curve because it doesn’t have a human-readable interface. It's also difficult to create ad hoc queries and explore your data. SQL lets you extract and analyze data in a declarative manner.

OpenSearch makes its search and query engine robust by introducing Piped Processing Language (PPL).

PPL enables developers, DevOps engineers, support engineers, site reliability engineers (SREs), and IT managers to effectively discover and explore log data stored in OpenSearch.

With PPL, you can extract insights from your data with a sequence of commands delimited by pipes (|). PPL supports a comprehensive set of commands including search, where, fields, rename, dedup, sort, eval, head, top, and rare. PPL also supports functions, operators and expressions.

Even if you're new to OpenSearch and are only familiar with the pipe (|) syntax, you can still be productive from day one.`,
        outputType: 'MARKDOWN',
        execution_time: '0.009 ms'
      }],
      input: {
        inputText: `%md
Query DSL is powerful and fast, but it has a steep learning curve because it doesn’t have a human-readable interface. It's also difficult to create ad hoc queries and explore your data. SQL lets you extract and analyze data in a declarative manner.

OpenSearch makes its search and query engine robust by introducing Piped Processing Language (PPL).

PPL enables developers, DevOps engineers, support engineers, site reliability engineers (SREs), and IT managers to effectively discover and explore log data stored in OpenSearch.

With PPL, you can extract insights from your data with a sequence of commands delimited by pipes (|). PPL supports a comprehensive set of commands including search, where, fields, rename, dedup, sort, eval, head, top, and rare. PPL also supports functions, operators and expressions.

Even if you're new to OpenSearch and are only familiar with the pipe (|) syntax, you can still be productive from day one.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `Piped Processing Language (PPL) is a query language that lets you use pipe (|) syntax to explore, discover, and query data stored in OpenSearch.

For more information, refer to the [documentation](https://opensearch.org/docs/search-plugins/ppl/index/).`,
        outputType: 'MARKDOWN',
        execution_time: '0.009 ms'
      }],
      input: {
        inputText: `%md
Piped Processing Language (PPL) is a query language that lets you use pipe (|) syntax to explore, discover, and query data stored in OpenSearch.

For more information, refer to the [documentation](https://opensearch.org/docs/search-plugins/ppl/index/).`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'In this notebook, we will demonstrate some simple PPL queries with the [OpenSearch Dashboards sample web logs](/app/home#/tutorial_directory). ',
        outputType: 'MARKDOWN',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%md
In this notebook, we will demonstrate some simple PPL queries with the [OpenSearch Dashboards sample web logs](/app/home#/tutorial_directory). `,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'To use PPL, add a code paragraph, type `%ppl` on the first line, and add your PPL query on the next line. PPL uses `source` to specify indices and `head` to limit results.',
        outputType: 'MARKDOWN',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%md
To use PPL, add a code paragraph, type \`%ppl\` on the first line, and add your PPL query on the next line. PPL uses \`source\` to specify indices and \`head\` to limit results.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `
source=opensearch_dashboards_sample_data_logs | head 20`,
        outputType: 'QUERY',
        execution_time: '0.026 ms'
      }],
      input: {
        inputText: `%ppl
source=opensearch_dashboards_sample_data_logs | head 20`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'To specify fields to include and filter results, use the `field` and `where` commands. The next query returns hosts with failed responses.',
        outputType: 'MARKDOWN',
        execution_time: '0.007 ms'
      }],
      input: {
        inputText: `%md
To specify fields to include and filter results, use the \`field\` and \`where\` commands. The next query returns hosts with failed responses.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `
source=opensearch_dashboards_sample_data_logs | fields host, clientip, response, bytes | where response='503' or response='404'`,
        outputType: 'QUERY',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%ppl
source=opensearch_dashboards_sample_data_logs | fields host, clientip, response, bytes | where response='503' or response='404'`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'To see most common hosts from the previous result, use the `top` command.',
        outputType: 'MARKDOWN',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%md
To see most common hosts from the previous result, use the \`top\` command.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `
source=opensearch_dashboards_sample_data_logs | where response='503' or response='404' | top host`,
        outputType: 'QUERY',
        execution_time: '0.008 ms'
      }],
      input: {
        inputText: `%ppl
source=opensearch_dashboards_sample_data_logs | where response='503' or response='404' | top host`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'To perform aggregations on search results, use the `stats` command.',
        outputType: 'MARKDOWN',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%md
To perform aggregations on search results, use the \`stats\` command.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `
source=opensearch_dashboards_sample_data_logs | where response='503' or response='404' | stats count(1) by host`,
        outputType: 'QUERY',
        execution_time: '0.011 ms'
      }],
      input: {
        inputText: `%ppl
source=opensearch_dashboards_sample_data_logs | where response='503' or response='404' | stats count(1) by host`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `For more information, refer to the [documentation](https://opensearch.org/docs/search-plugins/ppl/index/).

To see how to perform sample root cause event analysis using PPL and visualizations, see the "\[Logs\] Sample Root Cause Event Analysis" notebook.`,
        outputType: 'MARKDOWN',
        execution_time: '0.009 ms'
      }],
      input: {
        inputText: `%md
For more information, refer to the [documentation](https://opensearch.org/docs/search-plugins/ppl/index/).

To see how to perform sample root cause event analysis using PPL and visualizations, see the "\[Logs\] Sample Root Cause Event Analysis" notebook.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }]
  };
};

const getSampleNotebooks = visIds => {
  const dateString = new Date().toISOString();
  return [{
    notebook: getDemoNotebook(dateString, visIds[2]),
    dateModified: dateString,
    dateCreated: dateString
  }, {
    notebook: getSQLNotebook(dateString),
    dateModified: dateString,
    dateCreated: dateString
  }, {
    notebook: getPPLNotebook(dateString),
    dateModified: dateString,
    dateCreated: dateString
  }, {
    notebook: getRootCauseNotebook(dateString, visIds),
    dateModified: dateString,
    dateCreated: dateString
  }];
};

exports.getSampleNotebooks = getSampleNotebooks;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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