"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.loadSampleData = void 0;

var _moment = _interopRequireDefault(require("moment"));

var _readline = _interopRequireDefault(require("readline"));

var _fs = _interopRequireDefault(require("fs"));

var _zlib = require("zlib");

var _lodash = require("lodash");

var _helpers = require("../../utils/helpers");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 *
 * Modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */
//@ts-ignore
const BULK_INSERT_SIZE = 500;

const loadSampleData = (filePath, indexName, client, request) => {
  return new Promise((resolve, reject) => {
    let count = 0;
    let docs = [];
    let isPaused = false;
    let offset = 0;
    const startTime = (0, _moment.default)(new Date().getTime()).subtract({
      days: 7
    }); // Create the read stream for the file. Set a smaller buffer size here to prevent it from
    // getting too large, to prevent inserting too many docs at once into the index.

    const readStream = _fs.default.createReadStream(filePath, {
      highWaterMark: 1024 * 4
    });

    const lineStream = _readline.default.createInterface({
      input: readStream.pipe((0, _zlib.createUnzip)())
    }); // This is only ran when the end of lineStream closes normally. It is used to
    // bulk insert the final batch of lines that are < BULK_INSERT_SIZE


    const onClose = async () => {
      if (docs.length > 0) {
        try {
          await bulkInsert(docs);
        } catch (err) {
          reject(err);
          return;
        }
      }

      resolve(count);
    };

    lineStream.on('close', onClose);
    lineStream.on('pause', async () => {
      isPaused = true;
    });
    lineStream.on('resume', async () => {
      isPaused = false;
    });
    lineStream.on('line', async doc => {
      // for the initial doc, get the timestamp to properly set an offset
      if (count === 0) {
        const initialTime = (0, _moment.default)(JSON.parse(doc).timestamp);
        offset = startTime.diff(initialTime);
      }

      count++;
      docs.push(doc); // If not currently paused: pause the stream to prevent concurrent bulk inserts
      // on the cluster which could cause performance issues.
      // Also, empty the current docs[] before performing the bulk insert to prevent
      // buffered docs from being dropped.

      if (docs.length >= BULK_INSERT_SIZE && !isPaused) {
        lineStream.pause(); // save the docs to insert, and empty out current docs list

        const docsToInsert = docs.slice();
        docs = [];

        try {
          await bulkInsert(docsToInsert);
          lineStream.resume();
        } catch (err) {
          lineStream.removeListener('close', onClose);
          lineStream.close();
          reject(err);
        }
      }
    });

    const bulkInsert = async docs => {
      try {
        const bulkBody = prepareBody(docs, offset);
        const resp = await client.asScoped(request).callAsCurrentUser('bulk', {
          body: bulkBody
        });

        if (resp.errors) {
          const errorItems = resp.items;
          const firstErrorReason = (0, _lodash.isEmpty)(errorItems) ? 'Error while bulk inserting. Please try again.' : (0, _helpers.prettifyErrorMessage)(errorItems[0].index.error.reason);
          return Promise.reject(new Error(firstErrorReason));
        }
      } catch (err) {
        console.log('Error while bulk inserting. ', err);
        return Promise.reject(new Error('Error while bulk inserting. Please try again.'));
      }
    };

    const prepareBody = (docs, offset) => {
      const bulkBody = [];
      let docIdCount = count - docs.length;
      docs.forEach(doc => {
        bulkBody.push(getDocDetails(docIdCount));
        bulkBody.push(updateTimestamp(doc, offset));
        docIdCount++;
      });
      return bulkBody;
    };

    const updateTimestamp = (doc, offset) => {
      let docAsJSON = JSON.parse(doc);
      const updatedTimestamp = docAsJSON.timestamp + offset;
      docAsJSON.timestamp = updatedTimestamp;
      return docAsJSON;
    };

    const getDocDetails = docIdCount => {
      return `{ "index": { "_index": "${indexName}", "_id": ${docIdCount} } }`;
    };
  });
};

exports.loadSampleData = loadSampleData;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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