/*	$NetBSD: micphy.c,v 1.4.10.2 2019/09/04 08:33:39 martin Exp $	*/

/*-
 * Copyright (c) 1998, 1999, 2000 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Jason R. Thorpe of the Numerical Aerospace Simulation Facility,
 * NASA Ames Research Center, and by Frank van der Linden.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * Copyright (c) 1997 Manuel Bouyer.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * Driver for Micrel KSZ9021RN PHYs
 */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: micphy.c,v 1.4.10.2 2019/09/04 08:33:39 martin Exp $");

#include "opt_mii.h"

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/device.h>
#include <sys/socket.h>
#include <sys/errno.h>

#include <net/if.h>
#include <net/if_media.h>

#include <dev/mii/mii.h>
#include <dev/mii/miivar.h>
#include <dev/mii/miidevs.h>

static int	micphymatch(device_t, cfdata_t, void *);
static void	micphyattach(device_t, device_t, void *);
static void	micphy_reset(struct mii_softc *);
static int	micphy_service(struct mii_softc *, struct mii_data *, int);

CFATTACH_DECL3_NEW(micphy, sizeof(struct mii_softc),
    micphymatch, micphyattach, mii_phy_detach, mii_phy_activate, NULL, NULL,
    DVF_DETACH_SHUTDOWN);

static int	micphy_service(struct mii_softc *, struct mii_data *, int);
static void	micphy_fixup(struct mii_softc *, int, int, device_t);

static const struct mii_phy_funcs micphy_funcs = {
	micphy_service, ukphy_status, micphy_reset,
};

static const struct mii_phydesc micphys[] = {
	{ MII_OUI_MICREL,		MII_MODEL_MICREL_KSZ8081,
	  MII_STR_MICREL_KSZ8081 },

	{ MII_OUI_MICREL,		MII_MODEL_MICREL_KSZ9021RNI,
	  MII_STR_MICREL_KSZ9021RNI },

	{ 0,				0,
	  NULL },
};

#define	MII_KSZ8081_PHYCTL2			0x1f

static int
micphymatch(device_t parent, cfdata_t match, void *aux)
{
	struct mii_attach_args *ma = aux;

	if (mii_phy_match(ma, micphys) != NULL)
		return 10;

	return 1;
}

static void
micphyattach(device_t parent, device_t self, void *aux)
{
	struct mii_softc *sc = device_private(self);
	struct mii_attach_args *ma = aux;
	struct mii_data *mii = ma->mii_data;
	int model = MII_MODEL(ma->mii_id2);
	int rev = MII_REV(ma->mii_id2);
	const struct mii_phydesc *mpd;

	mpd = mii_phy_match(ma, micphys);
	aprint_naive(": Media interface\n");
	aprint_normal(": %s, rev. %d\n", mpd->mpd_name, rev);

	sc->mii_dev = self;
	sc->mii_inst = mii->mii_instance;
	sc->mii_phy = ma->mii_phyno;
	sc->mii_funcs = &micphy_funcs;
	sc->mii_pdata = mii;
	sc->mii_flags = ma->mii_flags;
	sc->mii_anegticks = MII_ANEGTICKS;

	PHY_RESET(sc);

	micphy_fixup(sc, model, rev, parent);

	sc->mii_capabilities = PHY_READ(sc, MII_BMSR) & ma->mii_capmask;
	if (sc->mii_capabilities & BMSR_EXTSTAT)
		sc->mii_extcapabilities = PHY_READ(sc, MII_EXTSR);
	aprint_normal_dev(self, "");
	if ((sc->mii_capabilities & BMSR_MEDIAMASK) == 0 &&
	    (sc->mii_extcapabilities & EXTSR_MEDIAMASK) == 0)
		aprint_error("no media present");
	else
		mii_phy_add_media(sc);
	aprint_normal("\n");
}

static void
micphy_reset(struct mii_softc *sc)
{
	int reg = 0; /* XXX gcc */

	/*
	 * The 8081 has no "sticky bits" that survive a soft reset; several bits
	 * in the Phy Control Register 2 must be preserved across the reset.
	 * These bits are set up by the bootloader; they control how the phy
	 * interfaces to the board (such as clock frequency and LED behavior).
	 */
	if (sc->mii_mpd_model == MII_MODEL_MICREL_KSZ8081)
		reg = PHY_READ(sc, MII_KSZ8081_PHYCTL2);
	mii_phy_reset(sc);
	if (sc->mii_mpd_model == MII_MODEL_MICREL_KSZ8081)
		PHY_WRITE(sc, MII_KSZ8081_PHYCTL2, reg);
}

static int
micphy_service(struct mii_softc *sc, struct mii_data *mii, int cmd)
{
	struct ifmedia_entry *ife = mii->mii_media.ifm_cur;
	int reg;

	switch (cmd) {
	case MII_POLLSTAT:
		/*
		 * If we're not polling our PHY instance, just return.
		 */
		if (IFM_INST(ife->ifm_media) != sc->mii_inst)
			return 0;
		break;

	case MII_MEDIACHG:
		/*
		 * If the media indicates a different PHY instance,
		 * isolate ourselves.
		 */
		if (IFM_INST(ife->ifm_media) != sc->mii_inst) {
			reg = PHY_READ(sc, MII_BMCR);
			PHY_WRITE(sc, MII_BMCR, reg | BMCR_ISO);
			return 0;
		}

		/*
		 * If the interface is not up, don't do anything.
		 */
		if ((mii->mii_ifp->if_flags & IFF_UP) == 0)
			break;

		mii_phy_setmedia(sc);
		break;

	case MII_TICK:
		/*
		 * If we're not currently selected, just return.
		 */
		if (IFM_INST(ife->ifm_media) != sc->mii_inst)
			return 0;

		if (mii_phy_tick(sc) == EJUSTRETURN)
			return 0;
		break;

	case MII_DOWN:
		mii_phy_down(sc);
		return 0;
	}

	/* Update the media status. */
	mii_phy_status(sc);

	/* Callback if something changed. */
	mii_phy_update(sc, cmd);
	return 0;
}

#define XREG_CONTROL	0x0b
#define XREG_WRITE	0x0c
#define XREG_READ	0x0d
#define XREG_CTL_SEL_READ	0x0000
#define XREG_CTL_SEL_WRITE	0x8000

#define REG_RGMII_CLOCK_AND_CONTROL	0x104
#define REG_RGMII_RX_DATA		0x105

static void micphy_writexreg(struct mii_softc *sc, uint32_t reg, uint32_t wval)
{
	int rval __debugused;

	PHY_WRITE(sc, XREG_CONTROL, XREG_CTL_SEL_WRITE | reg);
	PHY_WRITE(sc, XREG_WRITE, wval);
	PHY_WRITE(sc, XREG_CONTROL, XREG_CTL_SEL_READ | reg);
	rval = PHY_READ(sc, XREG_READ);
	KDASSERT(wval == rval);
}

static void
micphy_fixup(struct mii_softc *sc, int model, int rev, device_t parent)
{
	switch (model) {
	case MII_MODEL_MICREL_KSZ9021RNI:
		if (!device_is_a(parent, "cpsw"))
			break;

		aprint_normal_dev(sc->mii_dev, "adjusting RGMII signal timing for cpsw\n");

		// RGMII RX Data Pad Skew
		micphy_writexreg(sc, REG_RGMII_RX_DATA, 0x0000);

		// RGMII Clock and Control Pad Skew
		micphy_writexreg(sc, REG_RGMII_CLOCK_AND_CONTROL, 0x9090);

		break;
	}

	return;
}
