#!/usr/bin/python
###############################################################################
#                                                                             #
# Fireinfo                                                                    #
# Copyright (C) 2010, 2011 IPFire Team (www.ipfire.org)                       #
#                                                                             #
# This program is free software: you can redistribute it and/or modify        #
# it under the terms of the GNU General Public License as published by        #
# the Free Software Foundation, either version 3 of the License, or           #
# (at your option) any later version.                                         #
#                                                                             #
# This program is distributed in the hope that it will be useful,             #
# but WITHOUT ANY WARRANTY; without even the implied warranty of              #
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               #
# GNU General Public License for more details.                                #
#                                                                             #
# You should have received a copy of the GNU General Public License           #
# along with this program.  If not, see <http://www.gnu.org/licenses/>.       #
#                                                                             #
###############################################################################

import _fireinfo
import system

class Hypervisor(object):
	def __init__(self):
		self.__info = _fireinfo.get_hypervisor()

	@property
	def system(self):
		"""
			Return the current instance of the System class.

			We need to do that as a property because otherwise
			we get a recursion.
		"""
		return system.System()

	@property
	def vendor(self):
		"""
			Returns the name of the hypervisor vendor.
		"""
		if not self.virtual:
			return None

		# Some of the hypervisors can be detected in a right way.
		# We can return them at this place.
		if self.__info["hypervisor"] in ("Xen", "VMWare", "KVM"):
			return self.__info["hypervisor"]

		# Citrix Xen says it is Microsoft Hv.
		if self.__info["hypervisor"] == "Microsoft" and \
				self.system.bios_vendor == "Xen":
			return "Xen"

		if not self.__info["hypervisor"]:
			# On VMWare systems, the bios vendor string contains "VMWare".
			if self.__is_hypervisor_vmware():
				return "VMWare"

			# VirtualBox got "innotek GmbH" as bios vendor.
			elif self.__is_hypervisor_virtualbox():
				return "VirtualBox"

			# Check for qemu.
			elif self.__is_hypervisor_qemu():
				return "Qemu"

			# Check for Microsoft.
			elif self.__is_hypervisor_microsoft():
				return "Microsoft"

		return "unknown"

	@property
	def type(self):
		"""
			Returns if the host is running in full virt mode or
			if it is running in a paravirtualized environment.
		"""
		if not self.virtual:
			return None

		if self.__info["virtype"]:
			return self.__info["virtype"]

		if self.vendor in ("Qemu", "KVM", "VirtualBox", "VMWare"):
			return "full"

		return "unknown"

	@property
	def virtual(self):
		"""
			Returns true if the host is running in a virtual environment.
			Otherwise: false.
		"""
		return _fireinfo.is_virtualized() or \
			"hypervisor" in self.system.cpu.flags or \
			self.__is_hypervisor_virtualbox() or \
			self.__is_hypervisor_vmware() or \
			self.__is_hypervisor_qemu() or \
			self.__is_hypervisor_microsoft()

	def __is_hypervisor_virtualbox(self):
		"""
			Check for virtualbox hypervisor by comparing the bios vendor string
			to "innotek GmbH".
		"""
		return self.system.bios_vendor == "innotek GmbH"

	def __is_hypervisor_vmware(self):
		"""
			Check for the VMWare hypervisor by the VMWare Hypervisor port check.

			http://kb.vmware.com/selfservice/microsites/search.do?language=en_US&cmd=displayKC&externalId=1009458
		"""
		if self.system.vendor:
			return self.system.vendor.startswith("VMware")

			# XXX We should use _fireinfo.vmware_hypervisor_port_check() here, too.
			# This currently segfaults (and I have no clue why) on VMware player.

	def __is_hypervisor_qemu(self):
		"""
			Check for old qemu emulator.
		"""
		if self.system.bios_vendor:
			return self.system.bios_vendor == "Bochs"

		return False

	def __is_hypervisor_microsoft(self):
		"""
			Check for Microsoft hypervisor.
		"""
		if self.system.vendor:
			return "Microsoft" in self.system.vendor

		return False


if __name__ == "__main__":
	h = Hypervisor()

	print "Vendor:", h.vendor
	print "Type:", h.type
	print "Virtual:", h.virtual
