/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  AmountString,
  NotificationType,
  TransactionMajorState,
  TransactionMinorState,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { GlobalTestState, generateRandomPayto } from "../harness/harness.js";
import {
  createSimpleTestkudosEnvironmentV3,
  withdrawViaBankV3,
} from "../harness/helpers.js";
import { defaultCoinConfig } from "../harness/denomStructures.js";

/**
 * Run test for basic, bank-integrated withdrawal and payment.
 */
export async function runAgeRestrictionsDepositTest(t: GlobalTestState) {
  // Set up test environment

  const { walletClient, bankClient, exchange } =
    await createSimpleTestkudosEnvironmentV3(
      t,
      defaultCoinConfig.map((x) => x("TESTKUDOS")),
      {
        ageMaskSpec: "8:10:12:14:16:18:21",
      },
    );

  // Withdraw digital cash into the wallet.

  const withdrawalResult = await withdrawViaBankV3(t, {
    walletClient,
    bankClient,
    exchange,
    amount: "TESTKUDOS:20",
  });

  await withdrawalResult.withdrawalFinishedCond;

  const dgIdResp = await walletClient.client.call(
    WalletApiOperation.GenerateDepositGroupTxId,
    {},
  );

  const depositTxId = dgIdResp.transactionId;

  const depositTrack = walletClient.waitForNotificationCond(
    (n) =>
      n.type == NotificationType.TransactionStateTransition &&
      n.transactionId == depositTxId &&
      n.newTxState.major == TransactionMajorState.Pending &&
      n.newTxState.minor == TransactionMinorState.Track,
  );

  const depositDone = walletClient.waitForNotificationCond(
    (n) =>
      n.type == NotificationType.TransactionStateTransition &&
      n.transactionId == depositTxId &&
      n.newTxState.major == TransactionMajorState.Done,
  );

  const depositGroupResult = await walletClient.client.call(
    WalletApiOperation.CreateDepositGroup,
    {
      amount: "TESTKUDOS:10" as AmountString,
      depositPaytoUri: generateRandomPayto("foo"),
      transactionId: depositTxId,
    },
  );

  t.assertDeepEqual(depositGroupResult.transactionId, depositTxId);

  await depositTrack;

  await exchange.runAggregatorOnceWithTimetravel({
    timetravelMicroseconds: 1000 * 1000 * 60 * 60 * 3,
  });

  await depositDone;

  const transactions = await walletClient.client.call(
    WalletApiOperation.GetTransactions,
    {},
  );

  console.log("transactions", JSON.stringify(transactions, undefined, 2));
  t.assertDeepEqual(transactions.transactions[0].type, "withdrawal");
  t.assertDeepEqual(transactions.transactions[1].type, "deposit");
  // The raw amount is what ends up on the bank account, which includes
  // deposit and wire fees.
  t.assertDeepEqual(transactions.transactions[1].amountRaw, "TESTKUDOS:9.79");
}

runAgeRestrictionsDepositTest.suites = ["wallet"];
