"======================================================================
|
|   Virtual File System layer definitions
|
|
 ======================================================================"


"======================================================================
|
| Copyright 2002 Free Software Foundation, Inc.
| Written by Paolo Bonzini.
|
| This file is part of the GNU Smalltalk class library.
|
| The GNU Smalltalk class library is free software; you can redistribute it
| and/or modify it under the terms of the GNU Lesser General Public License
| as published by the Free Software Foundation; either version 2.1, or (at
| your option) any later version.
| 
| The GNU Smalltalk class library is distributed in the hope that it will be
| useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
| MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser
| General Public License for more details.
| 
| You should have received a copy of the GNU Lesser General Public License
| along with the GNU Smalltalk class library; see the file COPYING.LIB.
| If not, write to the Free Software Foundation, 59 Temple Place - Suite
| 330, Boston, MA 02111-1307, USA.  
|
 ======================================================================"

Smalltalk addSubspace: #VFS!
Namespace current: VFS!

Object subclass: #VFSHandler
       instanceVariableNames: 'name'
       classVariableNames: 'Registry'
       poolDictionaries: ''
       category: 'Streams-Files'
! 

VFSHandler subclass: #RealFileHandler
       instanceVariableNames: 'stat'
       classVariableNames: 'Epoch'
       poolDictionaries: ''
       category: 'Streams-Files'
! 

RealFileHandler subclass: #DecodedFileHandler
       instanceVariableNames: 'realFileName'
       classVariableNames: 'FileTypes'
       poolDictionaries: ''
       category: 'Streams-Files'
! 

VFSHandler subclass: #ArchiveFileHandler
       instanceVariableNames: ''
       classVariableNames: ''
       poolDictionaries: ''
       category: 'Streams-Files'
! 

ArchiveFileHandler subclass: #ExternalArchiveFileHandler
       instanceVariableNames: ''
       classVariableNames: 'FileTypes'
       poolDictionaries: ''
       category: 'Streams-Files'
! 

CStruct
    subclass: #CStatStruct
    declaration: #( 
	(#stMode #uShort) " protection "
	(#stSize #long)   " total size, in bytes "
	(#stAtime #long)  " time of last access "
	(#stMtime #long)  " time of last (contents) modification "
	(#stCtime #long)  " time of last (attribute) change "
    )
    classVariableNames: ''
    poolDictionaries: ''
    category: 'Streams-Files'
!


RealFileHandler
	defineCFunc: 'stat'
	withSelectorArgs: 'statOn: fileName into: statStruct'
	returning: #int
	args: #(#string #cObject).

"opendir and closedir needed to test for directories"

RealFileHandler
	defineCFunc: 'opendir'
	withSelectorArgs: 'openDir: dirName'
	returning: #cObject
	args: #(#string).

RealFileHandler
	defineCFunc: 'closedir'
	withSelectorArgs: 'closeDir: dirObject'
	returning: #int
	args: #(#cObject).

RealFileHandler
	defineCFunc: 'fileIsReadable'
	withSelectorArgs: 'primIsReadable: name'
	returning: #boolean
	args: #(#string).

RealFileHandler
	defineCFunc: 'fileIsWriteable'
	withSelectorArgs: 'primIsWriteable: name'
	returning: #boolean
	args: #(#string).

RealFileHandler
	defineCFunc: 'fileIsExecutable'
	withSelectorArgs: 'primIsExecutable: name'
	returning: #boolean
	args: #(#string).

RealFileHandler
	defineCFunc: 'unlink'
	withSelectorArgs: 'primUnlink: fileName'
	returning: #void
	args: #(#string).

RealFileHandler
	defineCFunc: 'rename'
	withSelectorArgs: 'primRename: oldFileName to: newFileName'
	returning: #void
	args: #(#string #string).

RealFileHandler
	 defineCFunc: 'rmdir'
	 withSelectorArgs: 'primRemoveDir: fileName'
	 returning: #void
	 args: #(#string).

RealFileHandler
	 defineCFunc: 'mkdir'
	 withSelectorArgs: 'primCreateDir: dirName mode: mode'
	 returning: #void
	 args: #(#string #int).

RealFileHandler class
	 defineCFunc: 'getCurDirName'
	 withSelectorArgs: 'working'
	 returning: #stringOut
	 args: #().

RealFileHandler
	 defineCFunc: 'extractDirentName'
	 withSelectorArgs: 'extractDirentName: dirent'
	 returning: #string
	 args: #(#cObject).

RealFileHandler
	 defineCFunc: 'readdir'
	 withSelectorArgs: 'readDir: dirObject'
	 returning: #cObject
	 args: #(#cObject).

RealFileHandler
	 defineCFunc: 'rewinddir'
	 withSelectorArgs: 'rewindDir: dirObject'
	 returning: #void
	 args: #(#cObject)!



!VFSHandler class methodsFor: 'instance creation'!

for: fileName
    "Answer the (real or virtual) file handler for the file named fileName"

    | pos1 fsName pos2 subPath file result |
    file := fileName.
    pos1 := file indexOf: $#.
    pos1 = 0 ifTrue: [ ^RealFileHandler new name: file ].
    [
	"Extract the file name and path, and resolve the first virtual
         file path (for example abc#uzip/def in abc#uzip/def#ugz)"

	fsName := file
	    copyFrom: pos1 + 1
	    to: (file indexOf: $/ startingAt: pos1 ifAbsent: [ file size + 1 ]) - 1.

	pos2 := file indexOf: $# startingAt: pos1 + 1 ifAbsent: [ file size + 1 ].
	subPath := pos1 + fsName size + 2 >= pos2
	    ifTrue: [ nil ]
	    ifFalse: [ file copyFrom: pos1 + fsName size + 2 to: pos2 - 1 ].

	pos2 > file size
    ] whileFalse: [
	result := self
	    vfsFor: (file copyFrom: 1 to: pos1 - 1)
	    name: fsName
	    subPath: (file copyFrom: pos1 + fsName size + 2 to: pos2 - 1).

	file := result realFileName, (file copyFrom: pos2 to: file size).
	pos1 := file indexOf: $#
    ].

    "Resolve the last virtual file path"    
    ^self
	vfsFor: (file copyFrom: 1 to: pos1 - 1)
	name: fsName
	subPath: subPath
! !

!VFSHandler class methodsFor: 'initializing'!

initialize
    "Register the receiver with ObjectMemory"
    ObjectMemory addDependent: self.
    self update: #returnFromSnapshot!

update: aspect
    "Private - Remove the files before quitting, and register the virtual
     filesystems specified by the subclasses upon image load."

    aspect == #returnFromSnapshot ifTrue: [
	Registry := LookupTable new.
	self allSubclassesDo: [ :each |
	    each fileSystems do: [ :fs | self register: fs forClass: each ]
	]
    ].
    aspect == #aboutToQuit ifTrue: [
	self allSubinstancesDo: [ :each | each release ]
    ].
!

fileSystems
    "Answer the virtual file systems that can be processed by this subclass.
     The default is to answer an empty array, but subclasses can override
     this.  If you do so, you should override #vfsFor:name:subPath: as well
     or you risk infinite loops."

    ^#()!

register: fileSystem forClass: vfsHandlerClass
    "Register the given file system to be handled by an instance of
     vfsHandlerClass.  This is automatically called if the class overrides
     #fileSystems."
    Registry at: fileSystem put: vfsHandlerClass! !

!VFSHandler class methodsFor: 'private'!

vfsFor: fileName name: fsName subPath: subPath
    "Create an instance of a subclass of the receiver, implementing the virtual
     file `subPath' inside the `fileName' archive.  fsName is the virtual
     filesystem name and is used to determine the subclass to be instantiated."
    ^(Registry at: fsName) 
	vfsFor: fileName name: fsName subPath: subPath!


!VFSHandler methodsFor: 'accessing'!

name
    "Answer the name of the file identified by the receiver"
    ^name
!

name: aName
    "Private - Initialize the receiver's instance variables"
    name := aName
!

realFileName
    "Answer the real file name which holds the file contents,
     or nil if it does not apply."
    ^name
! !


!RealFileHandler class methodsFor: 'initialization'!

initialize
    "Initialize the receiver's class variables"
    Epoch := DateTime year: 2000 day: 2 hour: 0 minute: 0 second: 0.
! !


!RealFileHandler methodsFor: 'accessing'!

name: aName
    "Private - Initialize the receiver's instance variables"
    name := File fullNameFor: aName
!

size
    "Answer the size of the file identified by the receiver"
    ^self stat stSize value
!

lastAccessTime
    "Answer the last access time of the file identified by the receiver"
    ^self getDateAndTime: self stat stAtime value
!

lastChangeTime
    "Answer the last change time of the file identified by the receiver
    (the `last change time' has to do with permissions, ownership and the
    like). On some operating systems, this could actually be the
    file creation time."
    ^self getDateAndTime: self stat stCtime value
!

creationTime
    "Answer the creation time of the file identified by the receiver.
    On some operating systems, this could actually be the last change time
    (the `last change time' has to do with permissions, ownership and the
    like)."
    ^self getDateAndTime: self stat stCtime value
!

lastModifyTime
    "Answer the last modify time of the file identified by the receiver
    (the `last modify time' has to do with the actual file contents)."
    ^self getDateAndTime: self stat stMtime value
!

refresh
    "Refresh the statistics for the receiver"
    stat isNil ifTrue: [
	stat := CStatStruct new.
	stat addToBeFinalized
    ].
    self statOn: self realFileName into: stat.
    File checkError
! !



!RealFileHandler methodsFor: 'testing'!

exists
    "Answer whether a file with the name contained in the receiver does exist."
    stat isNil ifTrue: [
	stat := CStatStruct new.
	stat addToBeFinalized.
    ].
    self statOn: self realFileName into: stat.
    ^File errno == 0
!

isDirectory
    "Answer whether a file with the name contained in the receiver does exist
    and identifies a directory."
    | dir errno |
    dir := self openDir: self realFileName.
    errno := File errno.
    (errno = 0) ifTrue: [
	self closeDir: dir.
	^true
    ].
    errno = 20 ifTrue: [ ^false ].
    errno = 13 ifTrue: [ ^true ].
    File checkError: errno
!

isReadable
    "Answer whether a file with the name contained in the receiver does exist
     and is readable"
    ^self primIsReadable: self realFileName!

isWriteable
    "Answer whether a file with the name contained in the receiver does exist
     and is writeable"
    ^self primIsWriteable: self realFileName!

isExecutable
    "Answer whether a file with the name contained in the receiver does exist
     and is executable"
    ^self primIsExecutable: self realFileName!

isAccessible
    "Answer whether a directory with the name contained in the receiver does
     exist and can be accessed"
    ^self primIsExecutable: self realFileName! !


!RealFileHandler methodsFor: 'file operations'!

open: class mode: mode ifFail: aBlock
    "Open the receiver in the given mode (as answered by FileStream's
    class constant methods)"
    ^class fopen: self realFileName mode: mode ifFail: aBlock
!

open: mode ifFail: aBlock
    "Open the receiver in the given mode (as answered by FileStream's
    class constant methods)"
    ^self open: FileStream mode: mode ifFail: aBlock
!

openDescriptor: mode ifFail: aBlock
    "Open the receiver in the given mode (as answered by FileStream's
    class constant methods)"
    ^self open: FileDescriptor mode: mode ifFail: aBlock
!

remove
    "Remove the file with the given path name"
    self isDirectory
	ifTrue: [ self primRemoveDir: self realFileName ]
	ifFalse: [ self primUnlink: self realFileName ].
    File checkError
!

renameTo: newFileName
    "Rename the file with the given path name oldFileName to newFileName"
    self primRename: self realFileName to: newFileName.
    File checkError
! !


!RealFileHandler methodsFor: 'private'!

getDateAndTime: time
    "Private - Convert a time expressed in seconds from 1/1/2000 to
     an array of two Smalltalk Date and Time objects"

    ^Epoch + (Duration seconds: time)
!

stat
    "Private - Answer the receiver's statistics' C struct"
    stat isNil ifTrue: [ self refresh ].
    ^stat
! !
    

!RealFileHandler methodsFor: 'directory operations'!

createDir: dirName
    "Change the current working directory to dirName."
    self
	primCreateDir: (Directory append: dirName to: self realFileName)
	mode: 8r777.

    File checkError
! !


!RealFileHandler methodsFor: 'enumerating'!

do: aBlock
    "Evaluate aBlock once for each file in the directory represented by the
    receiver, passing its name. aBlock should not return."
    | dir entry |
    dir := self openDir: self realFileName.
    File checkError.

    [ entry := self readDir: dir.
      File checkError.
      entry notNil ] whileTrue:
	  [ aBlock value: (self extractDirentName: entry) ].
    self closeDir: dir.
! !


!DecodedFileHandler class methodsFor: 'registering'!

initialize
    "Initialize the default virtual filesystems and the associated
     filter commands."
    FileTypes := LookupTable new
	at: 'Z' put: 'compress -cf < %1 > %2';
	at: 'uZ' put: 'zcat -f < %1 > %2';
	at: 'gz' put: 'gzip -cf < %1 > %2';
	at: 'ugz' put: 'gzip -cdf < %1 > %2';
	at: 'bz2' put: 'bzip2 < %1 > %2';
	at: 'ubz2' put: 'bzip2 -d < %1 > %2';
	at: 'tar' put: 'tar chof %2 %1';
	at: 'tgz' put: 'tar chof - %1 | gzip -cf > %2';
	at: 'nop' put: 'cat %1 > %2';
	at: 'strings' put: 'strings %1 > %2';
	yourself!

fileSystems
    "Answer the virtual file systems that can be processed by this subclass.
     These are #gz (gzip a file), #ugz (uncompress a gzipped file),
     #Z (compress a file via Unix compress), #uZ (uncompress a compressed
     file), #bz2 (compress a file via bzip2), #ubz2 (uncompress a file via
     bzip2), #tar (make a tar archive out of a directory), #tgz (make a
     gzipped tar archive out of a directory), #nop (do nothing, used for
     testing) and #strings (use the `strings' utility to extract printable
     strings from a file)."
    ^FileTypes keys!

vfsFor: file name: fsName subPath: subPath
    "Create a temporary file and use it to construct the contents of the given
     file, under the virtual filesystem fsName. subPath must be nil because
     this class supports single-file virtual filesystems only."
    | temp command |
    subPath isNil
	ifFalse: [ SystemExceptions FileError signal: 'not a tree-shaped filesystem' ].

    command := FileTypes at: fsName.
    temp := FileStream openTemporaryFile: Directory temporary, '/sfs'.
    Smalltalk system: (command bindWith: file with: temp name).
    ^self new name: file realFileName: temp name! !


!DecodedFileHandler methodsFor: 'files'!

name: virtualFileName realFileName: temporaryFileName
    "Private - Initialize a new object storing the contents of the
     virtualFileName file into temporaryFileName."
    self addToBeFinalized.
    self name: virtualFileName.
    realFileName := temporaryFileName!

finalize
    "Upon finalization, we remove the file that was temporarily holding the file
     contents"
    self release!

realFileName
    "Answer the real file name which holds the file contents,
     or nil if it does not apply."
    ^realFileName!

release
    "Remove the file that was temporarily holding the file contents"
    realFileName isNil ifTrue: [ ^self ].
    self remove.
    realFileName := nil! !

!ExternalArchiveFileHandler class methodsFor: 'registering'!

fileSystems
    "Answer the virtual file systems that can be processed by this subclass.
     These are given by the names of the executable files in the vfs
     subdirectory of the image directory, of the parent of the kernel
     directory and of the parent of the system kernel directory."
    FileTypes := LookupTable new. 
    self fileSystemsIn: Directory kernel, '/../vfs'.
    self fileSystemsIn: Directory image, '/vfs'.
    self fileSystemsIn: Directory systemKernel, '/../vfs'.
    ^FileTypes keys asSet!

fileSystemsIn: path
    "Registers the executable files in the given directory to be used
     to resolve a virtual file system."
    | dir |
    dir := RealFileHandler for: path.
    dir exists ifFalse: [ ^self ].
    dir do: [ :each |
	(File isExecutable: path, '/', each)
	    ifTrue: [ FileTypes at: each put: path, '/', each ]
    ]!

vfsFor: file name: fsName subPath: subPath
    "Create a temporary file and use it to construct the contents of the given
     file, under the virtual filesystem fsName. subPath must be nil because
     this class supports single-file virtual filesystems only."
    self notYetImplemented! !

RealFileHandler initialize!
DecodedFileHandler initialize!
VFSHandler initialize!
Namespace current: Smalltalk!

