/* -*- Mode: c++ -*- 
 *
 *  Copyright 1997 Massachusetts Institute of Technology
 * 
 *  Permission to use, copy, modify, distribute, and sell this software and its
 *  documentation for any purpose is hereby granted without fee, provided that
 *  the above copyright notice appear in all copies and that both that
 *  copyright notice and this permission notice appear in supporting
 *  documentation, and that the name of M.I.T. not be used in advertising or
 *  publicity pertaining to distribution of the software without specific,
 *  written prior permission.  M.I.T. makes no representations about the
 *  suitability of this software for any purpose.  It is provided "as is"
 *  without express or implied warranty.
 * 
 */ 

#ifndef _VRTV_H_
#define _VRTV_H_

#include <sys/types.h>
#include <fcntl.h>	/* for file i/o */
#include <VrSigProc.h>
#include <math.h>
#include <qapplication.h>
#include <qlayout.h>
#include <qimage.h>
#include <qwidget.h>
#include <qpainter.h>
#define NeedFunctionPrototypes 1
#include <X11/Xlib.h>
#include <X11/Intrinsic.h>
#include <X11/extensions/XShm.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <VrDecimatingSigProc.h>

#if defined (ENABLE_MMX)
#include <VrMMX.h>
#endif

//#define AUDIO_CENTER (10680000.0 - 4500000.0)
#define AUDIO_CENTER (10673000.0 - 4500000.0)
#define MAX_TAPS	1000 
#define HORIZ_OFFSET	150
#define SYNC_LOW	225		/* Number of samples with 'low' value */
#define SYNC_HIGH	37		/* Number of samples with 'high' value */
#define SYNC_LENGTH	512 //(SYNC_LOW+SYNC_HIGH+SYNC_LOW)

#define SYNC_SCALE	10
#define SYNC_ORDER	(numTaps * SYNC_SCALE)
#define FREE_RUN	300		/* Number of fields between search for vertical sync*/
#define TOTAL_SIZE	((int)((33000000 / 59.9411) / 4)) /* number of samples in 1 field*/
#define PER_LINE	((33000000 / 60 / 4) / 262) /* number of samples in 1 scan line */
#define MY_LINE_SIZE	500		/* Max number of pixels in 1 scan line */
#define MY_HEIGHT	400		/* Max number of lines in 1 video field */
#define DEBUG_COUNT	1638400		/* interval for displaying debug information */

#define MAX_COLOR	256		/* Max pixel value in color map for image display */
#define SKIP_FRAMES	5 //3		/* Number of frames to skip after a frame is displayed*/
#define BIG		999999999	/* used as initial value in max/min calculations */
#define TAPTYPE	int			/* Data type of filter taps */
#define SHIFT_INC 10

QWidget *master_widget;
QBoxLayout *master_layout;
QApplication *master_application;
int qtemp_started = 0;
int qtemp_argc = 1;
char *qtemp_argv[] = {"main"};

int debug_limit = 1;
int noShm = 1;
XShmSegmentInfo shmInfo = {0, -1, NULL, 0};
WId my_local_hd;
Display *my_local_dpy;
XImage *my_local_ximage;
uchar *local_newbits;
uint pix[256];				// pixel translation table 
int dc_base, video_shift_amount, debug_counter;
int frame_count, frame_total, counts_between, weight_count, started_skip;
int vsync_value_low, vsync_value_high, vsync_sample;
int hsync_left, sync_next, hsync_adjust, vsize_extra;
int vsync_left, vsync_frame, vsync_search_count;
int decimate_count, window_count, vsync_start, sync_input, vsync_offset, vsync_offset_new=0;
int sync_window[SYNC_LENGTH];
int sync_high_start = SYNC_LOW, sync_high_end = SYNC_LOW + SYNC_HIGH-1;
int sync_average, sync_low_average, sync_high_average, sync_average_max;
int video_too_high, video_too_low;
int free_run_field_count = 1;
int TVinSampFreq;
VrComplex audio_phase_correction, audio_phase_corr_incr;
float audio_center_freq, audio_gain;
mmxTaps* audio_processedTaps; //Precomputed constants, shifted four times
VrComplex* audio_tap_values;
int audio_decimate, audio_taps, process_audio, video_skip, current_decimation;
void initimage(void);
void shm_put();

template<class iType> 
class VrTV : public VrSigProc<iType,complex> {
protected:
  int numTaps;
  int tapincrement;
#if defined (ENABLE_MMX)
  mmxTaps* mmxtaps, *mmxsynctaps, *mmxvideo;
#endif
  TAPTYPE* taps, *synctaps, *tapstart, *tapend;
  float cutoff, center_freq, gain, arg;
  void calculate_taps(TAPTYPE *arg_taps, mmxTaps **arg_mmxtaps, int arg_order,
    float arg_center_freq, int arg_normalize, float arg_gain);
  int decimation;
  int hdata, packetc;
  int current_line, current_decim, decim_max;
  int current_column;
  unsigned char *current_pointer;
public: 
  virtual void work(int n);
  virtual void initialize();
  void setContr(double contr){ dc_base = (int)(contr*(-10000.0));}
  void setBright(double bright){ video_shift_amount = (int)bright;}
  void setVsync(double vsync){ vsync_offset = (int)(vsync*TOTAL_SIZE/400.0);}
  VrTV(int c,int t,int d ,int f, float g, int arg_audio_decimate, int arg_audio_taps);
  ~VrTV();
};
 
#define AUDIO_PROCESSING() { \
    if (process_audio <= 0) { \
	/* process_audio counts down the number of input samples that are \
	 * skipped before the next audio output point is to be calculated. \
	 * Since this number has gone to 0 when we reach the interior of the \
	 * 'if' statement, we calculate the next audio sample using the \
	 * audio FIR filter. \
	 */ \
        inputp = inputreadptr; \
        audio_result = 0;  \
/*#if defined (ENABLE_MMX)*/ \
          if(audio_processedTaps->mmxReady()) \
	      audio_result = audio_processedTaps->mmxCVDProduct(inputp); \
          else \
/*#endif*/  \
          {  \
	      VrComplex *taps_tmp = audio_tap_values; \
	      for (int j=0; j < audio_taps; j++) \
	        audio_result += taps_tmp[j] * inputp[j]; \
          } \
          audio_phase_correction *= audio_phase_corr_incr; \
          audio_result *= audio_phase_correction; \
    } /* end of audio processing */ }

#define CALC_TAP() \
	this_input = *inputp++; \
  	result_re += *tapp++ * this_input; \
  	result_im += *tapp++ * this_input;

/* Perform the video processing. */
#define VIDEO_AM_DEMOD() { \
    current_decimation = decimation; \
    inputp = inputreadptr; \
/*#if defined (XXENABLE_MMX) */\
/*    if(mmxvideo->mmxReady()) { */\
/*        video_result = mmxvideo->mmxCVDProduct(inputp); */\
/*        current = (int) (real(video_result) * real(video_result) +  imag(video_result) * imag(video_result)); */\
/*        current = (65000 - current) >> 2; */\
/*    } */\
/*    else*/ \
/*#endif */ \
    { \
    result_re = 0; \
    result_im = 0; \
    tapp = tapstart; \
    lasttap = tapend; \
    /* Perform the calculation of the FIR filter for video. \
     * This filter performs both frequency translation from 10.7 MHz to DC \
     * and filtering of the input signal. \
     */ \
    while (tapp != lasttap) { \
	/* expand inline for better optimization - tap count must be a multiple of 8 */ \
	CALC_TAP(); CALC_TAP(); CALC_TAP(); CALC_TAP(); \
	CALC_TAP(); CALC_TAP(); CALC_TAP(); CALC_TAP(); \
    } \
    result_re >>= (SHIFT_INC/2); \
    result_im >>= (SHIFT_INC/2); \
    /* Get the square of the absolute value of the video signal to convert it \
     * back from the complex domain into a 'real' number. \
     */ \
    current = ((int) (result_re * result_re +  result_im * result_im)) >> (1 * SHIFT_INC); \
    current = (65000 - current) >> 2; \
    } }

    /* Output debug data and statistics to the screen every few seconds.
     */
#define DEBUG_OUTPUT() { \
    if (++decimate_count > DEBUG_COUNT) { \
        decimate_count = 0; \
        if (packetc > 0) \
	    hdata /= packetc; \
        decim_max = hdata/300;  \
        if (debug_counter++ > debug_limit) {	/* display random debug stuff every so often */ \
	    debug_limit = 2;			/* move the interval out */ \
	    debug_counter = 0; \
	    if (frame_count == 0) frame_count = 1; \
	    printf ("frame %d between %d decim %d\n", \
	        frame_count, frame_total / frame_count, decim_max); \
	    printf ("vsync low %d high %d\n", vsync_value_low, vsync_value_high); \
	    printf("dc base %d shift %d packet %d avg %d\n", dc_base, video_shift_amount, packetc, hdata); \
	    printf ("too high %d low %d\n", video_too_high, video_too_low); \
	    frame_count = 0; hdata = 0; packetc = 0; frame_total = 0; \
	    video_too_high = 0; video_too_low = 0; \
	} \
    } /* DEBUG_COUNT */ }

#define START_NEW_VSYNC() { \
	    vsync_search_count =0; \
	    /* dc_base is used normalize the black level of the */ \
	    /** signal to 0 (DC restoration). */ \
            dc_base = vsync_value_low; \
	    /* video_shift_amount is used to scale the brightest part */ \
            /* of the picture to be MAX_COLORS */ \
	    tempvid = 7 * (vsync_value_high - vsync_value_low); \
            if (tempvid > 0) { \
                video_shift_amount = -10; \
                while (tempvid) { \
	            video_shift_amount++; \
	            tempvid >>= 1; \
	        } \
            } \
	    tapincrement = numTaps; \
	    mmxvideo = mmxtaps; \
	    tapstart = taps; \
	    tapend = &taps[numTaps*2]; \
	    sync_next = 0; \
	    while (vsync_offset > TOTAL_SIZE) \
		vsync_offset -= TOTAL_SIZE; \
	    /*vsync_left = TOTAL_SIZE - vsync_offset; */ \
	    vsync_left = TOTAL_SIZE - vsync_offset_new; \
	    vsync_offset = 0; \
	    if (vsize_extra++ > 1) { \
		vsize_extra = 0; \
		vsync_left -= 1; \
	    } \
	    hsync_left = PER_LINE; \
	    hsync_adjust = 0; \
	    if (current_line > 120 && current_line < 300 /*170*/) { \
		frame_total += counts_between; \
		frame_count++; \
		shm_put(); \
		started_skip = 0; \
	    } \
	    counts_between = 0; \
	    current_line = 0; \
	    if ((free_run_field_count > 1) && vsync_frame == 0) current_line = 600; \
	    if (!started_skip) { \
		started_skip = 1; \
		video_skip = SKIP_FRAMES * TOTAL_SIZE * decimation; \
		begin_skip = 1; \
	    } }

#define CHECK_END_OF_LINE() { \
    /* Check for the number of pixels in one horizontal scan line and wrap */\
    /* video image if necessary. */\
    if (--hsync_left <= 0) {	/* we got an horizontal sync */ \
	/* when we get an horizontal sync, we reset the colum counter and */\
	/* set 'current_pointer' to point to the frame buffer for the */\
	/* next video line. */\
	packetc++; \
	if (current_line >= MY_HEIGHT) \
	    current_pointer = NULL; \
	else \
	    current_pointer = &local_newbits[current_line++ * MY_LINE_SIZE * 2]; \
	current_column = 0; \
	current_decim = 0; \
	hsync_left = PER_LINE; \
    	increment_amount += 1; \
    	inputvalid -= 1; \
	inputreadptr += 1; \
	if (hsync_adjust++ > 1) { \
	    hsync_adjust = 0; \
    	    increment_amount += 1; \
    	    inputvalid -= 1; \
	    inputreadptr += 1; \
	    vsync_left--; \
	} \
    } }

#define CHECK_END_OF_FIELD() { \
    /* If we are nearing the end of the video field, increase the number of */ \
    /* taps in the filter to obtain a higher quality image for the vertical */ \
    /* sync recognition process. */ \
    if (vsync_frame > free_run_field_count && vsync_left < SYNC_LENGTH) { \
	tapincrement = SYNC_ORDER; \
/*#if defined (ENABLE_MMX) */\
	mmxvideo = mmxsynctaps; \
/*#endif*/ \
	tapstart = synctaps; \
	tapend = &synctaps[SYNC_ORDER*2]; \
    } \
    /* We have hit the end of the current video field (as calculated by */ \
    /* number of input samples).  If we are not in free run, then start */ \
    /* calculating correlation values for vertical sync detection. */ \
    if (--vsync_left <= 0 && (!sync_next)) { \
	if (vsync_frame++ > free_run_field_count) { \
	    /* start looking for a new Vertical Sync */ \
	    sync_next = 1; \
	    vsync_search_count =0; \
	    vsync_frame = 0; \
 	    vsync_offset = HORIZ_OFFSET; \
	    sync_average_max = -BIG; \
	    free_run_field_count <<= 1; \
	    if (free_run_field_count > FREE_RUN) \
		free_run_field_count = FREE_RUN; \
	} \
	else \
	    vsync_start = 1; \
    }   \
    /* Calculate correlation values for vertical sync detection. \
     */ \
    if (sync_next) { \
	if (vsync_search_count++ > TOTAL_SIZE + 1000) \
		vsync_start = 1; \
        sync_low_average += current - sync_window[sync_input]; \
        sync_window[sync_input] = current; \
        if (++sync_input > SYNC_LENGTH-1) sync_input = 0; \
        sync_high_average += sync_window[sync_high_start] - sync_window[sync_high_end]; \
        if (++sync_high_start > SYNC_LENGTH-1) sync_high_start = 0; \
        if (++sync_high_end > SYNC_LENGTH-1) sync_high_end = 0; \
        sync_average = (sync_high_average << 1) - sync_low_average / SYNC_LENGTH; \
	vsync_offset++; \
        if (sync_average > sync_average_max) { \
	    sync_average_max = sync_average; \
	    vsync_offset = HORIZ_OFFSET; \
	    vsync_sample = 0; \
	    vsync_value_low = BIG; \
	    vsync_value_high = -BIG; \
	} \
    } \
    /* Remember highest and lowest input values in area near correlation \
     * peak.  These values are then used to calculate DC offset and scale \
     * factors for the video image. \
     */ \
    if (vsync_sample < SYNC_HIGH + SYNC_LOW) { \
        vsync_sample++; \
        if (current < vsync_value_low) vsync_value_low = current; \
        if (current > vsync_value_high) vsync_value_high = current; \
    } }

/* Work function for performing actual processing of data.
 */
template<class iType> void
VrTV<iType>::work(int n)
{
register char *inputp, *inputreadptr;
int increment_amount, inputvalid;
register int this_input;
register TAPTYPE *tapp, *lasttap;
register TAPTYPE result_re, result_im;
int current, tempvid, video_value;
int increment_output, outputvalid;
complex *outputptr;
VrComplex audio_result, video_result;
int begin_skip;

  inputreadptr = inputReadPtr(-tapincrement + 1);
  increment_amount = 0;
  inputvalid = validUnits();
  outputptr = getWritePtr();
  outputvalid = bufferLeftTillWrap();
  increment_output = 0;
  begin_skip = 0;
  for (int work_index=0;work_index<n;) { 
    /* Perform the audio processing.
     */
	AUDIO_PROCESSING();

    if (video_skip > 0)
	goto skip_video_processing;
    /* video_skip counts down the number of input samples that are
     * skipped before the next video output point is to be calculated.
     * Since this number has gone to 0, we calculate either the next
     * video sample (using the 'short' FIR filter) or calculate a 
     * data point that is used in the convolution calculation when
     * looking for vertical sync (using the 'long' FIR filter).
     */

	VIDEO_AM_DEMOD();
	CHECK_END_OF_LINE(); 
	CHECK_END_OF_FIELD();

    /* Perform the X window processing for the start of a new video field
     */
    if (vsync_start) {		/* we got a vertical sync */
	    vsync_start = 0;
		START_NEW_VSYNC();
    }
    /* Perform the X window processing for successive pixels in a video image
     */
    else {			/* here we add pixels to the output image */
	counts_between++;
        if (current_decim++ > decim_max && current_pointer) {
	    hdata++;
	    video_value = (current - dc_base) >> video_shift_amount;	/* make max value = color MAX_COLOR */
            if (video_value > 255) {
		video_value = 255;
		video_too_high++;
	    }
            else if (video_value < 0) {
		video_value = 0;
		video_too_low++;
	    }
	    if (current_column++ < MY_LINE_SIZE) {
		*current_pointer++ = video_value;
		*current_pointer++ = video_value;		/* to double width of image */
	    }
	    current_decim = 0;
	}
    }
	DEBUG_OUTPUT();

skip_video_processing:
    /* Output demodulated data to the buffer for passing to the audio
     * driver.
     */
    if (process_audio <= 0) {
        process_audio = audio_decimate;
	if (outputvalid-- > 0) {
	    *outputptr++ = audio_result;
	    increment_output++;
	}
	else {
	    incWritePtr(increment_output);
	    outputWrite(audio_result);
  	    outputptr = getWritePtr();
  	    outputvalid = bufferLeftTillWrap();
  	    increment_output = 0;
	}
	work_index++;
    }
    /* Increment input pointers to get to the next sample that must be
     * processed.
     */
    increment_amount += current_decimation;
    inputvalid -= current_decimation;
    if (inputvalid > 0)
	inputreadptr += current_decimation;
    else {
	incInput(increment_amount);
	inputreadptr = inputReadPtr(-tapincrement + 1); 
	inputvalid = validUnits();
	increment_amount = 0;
    }
#if 1
    if (begin_skip) {
	begin_skip = 0;
	current_decimation = audio_decimate;
	}
    if (video_skip < current_decimation)
        current_decimation = decimation;
#endif
    process_audio -= current_decimation;
    video_skip -= current_decimation;
  }
  if (increment_amount > 0)
    incInput(increment_amount);
  if (increment_output > 0)
    incWritePtr(increment_output);
}

/* Calculate tap values for a Hamming window of the requested order.
 *
 *  FIR filter definition: 
 *  VrTV( cutoff freq., Num of taps, decimation factor, center frequency) 
 *  cutoff (Hz) = 0.0  => LPF using Hamming window, o/w LPF transformed to have higher cutoff freq
 *  decimation factor => set to one unless some integer >1 specified
 *  center_freq (Hz) => used to specify a composite frequency-shifting filter (i.e. channel filter)
 */ 
template<class iType> void
VrTV<iType>::calculate_taps(TAPTYPE *arg_taps, mmxTaps **arg_mmxtaps, int arg_order,
    float arg_center_freq, int arg_normalize, float arg_gain)
{
  int index;
  float M = arg_order-1; /* filter Order */
  float ftemp, my_gain;
  float taptemp_re, taptemp_im;
  VrComplex temptaps[MAX_TAPS];

  TVinSampFreq = getInputSamplingFrequencyN(0); 
  if (arg_center_freq == 0.0){  // produces a low-pass filter using a real Hamming window
    for ( index=0 ; index < arg_order ; index++)
      temptaps[index] = VrComplex(arg_gain * (0.54-0.46*
		cos(2*M_PI*((float)index)/M)), 0.0);
  } else {	// Build composite Complex Filter => adds freq-shifting part
    if (arg_normalize) {
        ftemp = 0.0;
        for ( index=0 ; index < arg_order ; index++)
          ftemp += 0.54-0.46*cos(2*M_PI*(float)index/(M));
        /* Normalize gain of FIR filter to be independant of number of taps */
    	my_gain =  arg_gain * 4.94 / ftemp;
    }
    else
    	my_gain =  arg_gain;
    arg = (2*M_PI*arg_center_freq / (float)TVinSampFreq);
    //printf ("total of all taps %f arg %f gain %f\n", ftemp, arg, my_gain);
    for ( index=0 ; index < arg_order ; index++) {
      ftemp = index;
      temptaps[index] = VrComplex(my_gain*cos(arg*ftemp)
		* (0.54-0.46*cos(2*M_PI*ftemp /M) ),
      	  -my_gain*sin(arg*ftemp)
		* (0.54-0.46*cos(2*M_PI*ftemp/M) ) );
    }
    //phase_corr_incr_re = (TAPTYPE) (cos(arg*decimation) * (1 << SHIFT_INC));
    //phase_corr_incr_im = (TAPTYPE) (-sin(arg*decimation) * (1 << SHIFT_INC));
    for ( index=0 ; index < arg_order ; index++) {
      arg_taps[2 * index] = (TAPTYPE) (real(temptaps[index]) * (1 << SHIFT_INC));
      arg_taps[2 * index + 1] = (TAPTYPE) (imag(temptaps[index]) * (1 << SHIFT_INC));
    }
  }
#if defined (ENABLE_MMX)
   *arg_mmxtaps=new mmxTaps(temptaps,arg_order);
#endif
}

template<class iType> 
VrTV<iType>::VrTV(int c,int t,int dec,int freq, float g, int arg_audio_decimate, int arg_audio_taps)
  :VrSigProc<iType,complex>(2),numTaps(t),cutoff(c),center_freq(freq),gain(g),decimation(dec)
{
   audio_decimate = arg_audio_decimate;
   audio_taps = arg_audio_taps;
   current_decimation = decimation;
}

void init_application()
{
   if (!master_application) {
  	master_application = new QApplication(qtemp_argc, qtemp_argv);
  	master_widget = new QWidget();
  	master_widget->setMinimumSize(570, 590);
  	master_layout = new QVBoxLayout(master_widget);
  	master_application->setMainWidget(master_widget);
  	master_widget->resize(570, 640);
   }
}

void begin_application()
{
   if (!qtemp_started) {
	qtemp_started = 1;
  	master_widget->show();
  	master_application->exit_loop();		/* tell a.exec() to not stay forever */
  	master_application->exec();			/* start up windowing package */
  	master_application->processEvents(0);		/* we need to call this to finish initialization! */
   }
}
/* Initialize any variables needed by the module.
 */
template<class iType> 
void VrTV<iType>::initialize()
{
int shmMajorv;
int shmMinorv;
int shmSharedPixmaps;
int i;
QPainter *master_painter;
float M, arg;

   init_application();
   taps = new TAPTYPE[2 * MAX_TAPS];
   synctaps = new TAPTYPE[2 * MAX_TAPS];
   setHistory(SYNC_ORDER);
   calculate_taps(synctaps, &mmxsynctaps, SYNC_ORDER, center_freq, 1, gain); /* calculate first */
   calculate_taps(taps, &mmxtaps, numTaps, center_freq, 1, gain);
   dc_base = 1;
   video_shift_amount = 7;

   /* Initialize the audio filter */
   audio_phase_correction = VrComplex(1,0);
   audio_center_freq = AUDIO_CENTER;
   audio_gain = 2.0;
   M = audio_taps-1; /* filter Order */
   arg = 2*M_PI*audio_center_freq / (float)TVinSampFreq; 
   audio_tap_values = new VrComplex[audio_taps];
   for ( i=0 ; i < audio_taps ; i++)
      audio_tap_values[i] = VrComplex(audio_gain*cos(arg*i)*(0.54-0.46*cos(2*M_PI*i/(M))),
         audio_gain*(-1)*sin(arg*i)*(0.54-0.46*cos(2*M_PI*i/(M)))); 
   audio_phase_corr_incr = VrComplex(cos(arg*(float)audio_decimate),
				(-1)*sin(arg*(float)audio_decimate));
   tapincrement = numTaps;
#if defined (ENABLE_MMX)
   audio_processedTaps=new mmxTaps(audio_tap_values,audio_taps);
   mmxvideo = mmxtaps;
#endif
   tapstart = taps;
   tapend = &taps[numTaps*2];

   /* Initialize the X window interface for (if possible) displaying
    * images using the shared memory X interface.  Note that this interface
    * runs correctly on Linux x86 machines, but does not appear to run correctly
    * on Linux Alpha.
    */ 
   master_painter = new QPainter(master_widget); 
   my_local_hd = master_painter->get_hd();
   my_local_dpy = master_painter->get_dpy();
   noShm = XDisplayString(my_local_dpy)[0] != ':'
	|| !XShmQueryVersion(my_local_dpy, &shmMajorv, &shmMinorv, &shmSharedPixmaps);
   printf ("do we not have shared memory ? shm = %d\n", noShm);
   //noShm = 1;			/* no shared memory present */
   if (!noShm) {
	my_local_ximage = 
	    XShmCreateImage(my_local_dpy, (Visual *)NULL,
		8 /* depth */, ZPixmap, 0, &shmInfo, MY_LINE_SIZE, MY_HEIGHT); 
    	shmInfo.shmid = shmget(IPC_PRIVATE, MY_LINE_SIZE * MY_HEIGHT, IPC_CREAT | 0777);
    	local_newbits = (uchar *) shmat (shmInfo.shmid, (char *)NULL, 0);
    	shmInfo.shmaddr = (char *)local_newbits;
    	shmInfo.readOnly = False;
    	XShmAttach(my_local_dpy, &shmInfo);
	shmctl(shmInfo.shmid, IPC_RMID, 0);
   }
   else {
	my_local_ximage = 
	   XCreateImage(my_local_dpy, (Visual *)NULL,
		8 /* depth */, ZPixmap, 0, 0, MY_LINE_SIZE, MY_HEIGHT, 32, MY_LINE_SIZE);
        local_newbits = (uchar *)malloc(MY_LINE_SIZE * MY_HEIGHT);
   }
   my_local_ximage->data = (char *)local_newbits;
   for ( i=0; i<256; i++ ) {		// allocate colors
    	QColor c( i, i, i );
       	pix[i] = c.pixel();
   }
   shm_put();
}
 
/* Update the X window video image using stored data.
 */
void shm_put()
{
register uchar *p;
int  i;

    p = local_newbits;
    i = MY_LINE_SIZE * MY_HEIGHT;
    while (i-- > 0) {
	*p = pix[*p];
	p++;
    }
    if (!noShm)
      XShmPutImage(my_local_dpy, my_local_hd, qt_xget_readonly_gc(), my_local_ximage, 
		   0, 0, 50, 250, my_local_ximage->width, my_local_ximage->height, 0);
    else
      XPutImage(my_local_dpy, my_local_hd, qt_xget_readonly_gc(), my_local_ximage,
		0, 0, 50, 250, my_local_ximage->width, my_local_ximage->height);
    XFlush(my_local_dpy);
} 

/* Free the shared memory used by the X window interface.  This function
 * is never called and appears to not be necessary.  (No memory leaks
 * occur even though it is not used).
 */
void shm_stop()
{
    if (shmInfo.shmid >= 0) {
      XShmDetach(my_local_dpy, &shmInfo);
      shmInfo.shmid = -1;
    }
}

/* Delete any variables used by the module.
 */
template<class iType> 
VrTV<iType>::~VrTV()
{
  delete taps;
  delete synctaps;
  delete audio_tap_values;
}
#endif
