#include <stdio.h>
#include "tile_merge.h"


TileManager *
dup_buffer (TileManager * src)
{
  GimpBuffer * dst;
  GimpArea roi;
  
  dst = tile_manager_new (gimp_buffer_width (src),
                          gimp_buffer_height (src),
                          gimp_buffer_bpp (src));

  /* get image area expanded to whole tiles */
  roi.a.x = GET_MIN (0, 64);
  roi.a.y = GET_MIN (0, 64);
  roi.b.x = GET_MAX (gimp_buffer_width (src), 64);
  roi.b.y = GET_MAX (gimp_buffer_height (src), 64);

  /* convert to tile coords */
  ga_image_to_buffer (src, &roi);
  ga_buffer_to_tile (src, &roi);

  /* copy all the tiles over */
  g_return_val_if_fail (gimp_buffer_map (dst, &roi, src, &roi) == TRUE, NULL);

  return dst;
}



Tile*
tile_manager_get_tile (TileManager *tm,
		       int          xpixel,
		       int          ypixel,
		       int          wantread,
		       int          wantwrite)
{
  Tile * t;
  
  t = g_new (Tile, 1);

  t->b = tm;

  t->bounds.a.x = GET_MIN (xpixel, tm->_x.tile.size);
  t->bounds.a.y = GET_MIN (ypixel, tm->_y.tile.size);
  t->bounds.b.x = GET_MAX (xpixel, tm->_x.tile.size);
  t->bounds.b.y = GET_MAX (ypixel, tm->_y.tile.size);
  if (t->bounds.b.x == t->bounds.a.x)
    t->bounds.b.x = t->bounds.a.x + tm->_x.tile.size;
  if (t->bounds.b.y == t->bounds.a.y)
    t->bounds.b.y = t->bounds.a.y + tm->_y.tile.size;

  t->focus.a.x = xpixel;
  t->focus.a.y = ypixel;
  t->focus.b.x = MIN (xpixel, t->bounds.b.x);
  t->focus.b.y = MIN (ypixel, t->bounds.b.y);

  t->tile.x = t->bounds.a.x / 64;
  t->tile.y = t->bounds.a.y / 64;
  
  t->used = FALSE;

  t->d = NULL;
  t->rowstride = 0;
  t->pixstride = 0;
  
  if (wantwrite == TRUE)
    {
      t->d = gimp_buffer_use (t->b, &t->tile, USE_WRITE);
    }
  else if (wantread == TRUE)
    {
      t->d = gimp_buffer_use (t->b, &t->tile, USE_READ);
    }
  else
    {
      return t;
    }

  if (t->d == NULL)
    {
      tile_release (t, FALSE);
      return NULL;
    }

  t->used = TRUE;
  t->rowstride = (t->bounds.b.x - t->bounds.a.x) * t->b->_z.bpp;
  t->pixstride = t->b->_z.bpp;
  
  return t;
}


void
tile_release (Tile *t,
              int dirty)
{
  if (t->used == TRUE)
    gimp_buffer_use (t->b, &t->tile, USE_RELEASE);
  g_free (t);
}


void *
tile_data_pointer (Tile *t,
                   int x,
                   int y)
{
  guchar * data;

  data = guchar_d (t->d)
    + (y * t->rowstride)
    + (x * t->pixstride);

  return (void*) data;
}

