/* The GIMP -- an image manipulation program
 * Copyright (C) 1995 Spencer Kimball and Peter Mattis
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <string.h>
#include "buffer.h"

#define K GIMP_BUFFER_CLASS(buffer->klass)


static GimpBufferClass my_class =
{
  BUFFER_NONE,
  NULL, /* delete */
  NULL, /* alloc */
  NULL, /* map */
  NULL, /* validate */
  NULL, /* use */
  NULL, /* query */
  NULL  /* data */
};


void
gimp_buffer_init (GimpBuffer * buffer,
                  Tag tag,
                  gint x_len,
                  gint x_period,
                  gint x_offset,
                  gint y_len,
                  gint y_period,
                  gint y_offset)
{
  g_return_if_fail (buffer != NULL);
  g_return_if_fail (x_len > 0);
  g_return_if_fail (y_len > 0);
  g_return_if_fail (tag_valid (tag) == TRUE);

  buffer->_x.len = x_len;
  buffer->_x.period = x_period;
  buffer->_x.offset = x_offset;

  buffer->_y.len = y_len;
  buffer->_y.period = y_period;
  buffer->_y.offset = y_offset;

  buffer->_z.len = tag_bytes (tag);
  buffer->_z.tag = tag;

  buffer->vfunc = NULL;
  buffer->vdata = NULL;

  buffer->qq_x = 0;
  buffer->qq_y = 0;

  buffer->klass = (void*) &my_class;
}


void
gimp_buffer_uninit (GimpBuffer * buffer)
{
}


gint
gimp_buffer_width (GimpBuffer *buffer)
{
  g_return_val_if_fail (buffer != NULL, 0);

  return buffer->_x.len;
}


gint
gimp_buffer_height (GimpBuffer *buffer)
{
  g_return_val_if_fail (buffer != NULL, 0);

  return buffer->_y.len;
}


gint
gimp_buffer_depth (GimpBuffer *buffer)
{
  g_return_val_if_fail (buffer != NULL, 0);

  return buffer->_z.len;
}


gint
gimp_buffer_tilenum (GimpBuffer *buffer,
                     GimpPoint  *point)
{
  g_return_val_if_fail (buffer != NULL, -1);
  g_return_val_if_fail (point != NULL, -1);

  {
    gint xtotal = (buffer->_x.len +
                   buffer->_x.offset +
                   buffer->_x.period - 1) / buffer->_x.period;

    gint xtiles = (point->x +
                   buffer->_x.offset) / buffer->_x.period;
    
    gint ytiles = (point->y +
                   buffer->_y.offset) / buffer->_y.period;
    
    return (ytiles * xtotal) + xtiles;
  }
}


gboolean
gimp_buffer_focus (GimpBuffer   *buffer,
                   GimpPortion  *portion,
                   GimpArea     *roi)
{
  GimpArea *p;

  g_return_val_if_fail (buffer != NULL, FALSE);
  g_return_val_if_fail (buffer->klass != NULL, FALSE);
  g_return_val_if_fail (portion != NULL, FALSE);
  g_return_val_if_fail (roi != NULL, FALSE);

  /* consider buffer height and width */
  if (roi->a.x <  0 ||
      roi->a.y <  0 ||
      roi->b.x <= roi->a.x ||
      roi->b.y <= roi->a.y ||
      roi->a.x >= buffer->_x.len ||
      roi->a.y >= buffer->_y.len ||
      roi->b.x >  buffer->_x.len ||
      roi->b.y >  buffer->_y.len)
    return FALSE;

  /* decrease val to tile boundary */
#define GET_MIN(val,period) ((val) - ((val) % (period)))

  /* increase val to tile boundary */
#define GET_MAX(val,period) ((val) % (period) \
                             ? (val) + (period) - ((val) % (period)) \
                             : (val))
  
  /* remember the original area */
  portion->orig = *roi;

  /* trim to whole tiles */
  p = &portion->min;
  p->a.x = GET_MAX (roi->a.x, buffer->_x.period);
  p->a.y = GET_MAX (roi->a.y, buffer->_y.period);
  p->b.x = GET_MIN (roi->b.x, buffer->_x.period);
  p->b.y = GET_MIN (roi->b.y, buffer->_y.period);
  if (p->b.x < p->a.x) p->b.x = p->a.x;
  if (p->b.y < p->a.y) p->b.y = p->a.y;
  
  /* expand to whole tiles */
  p = &portion->max;
  p->a.x = GET_MIN (roi->a.x, buffer->_x.period);
  p->a.y = GET_MIN (roi->a.y, buffer->_y.period);
  p->b.x = GET_MAX (roi->b.x, buffer->_x.period);
  p->b.y = GET_MAX (roi->b.y, buffer->_y.period);
  
  /* expand to single tile */
  p = &portion->bounds;
  p->a.x = GET_MIN (roi->a.x, buffer->_x.period);
  p->a.y = GET_MIN (roi->a.y, buffer->_y.period);
  p->b.x = GET_MAX (roi->a.x, buffer->_x.period);
  p->b.y = GET_MAX (roi->a.y, buffer->_y.period);
  if (p->b.x == p->a.x) p->b.x = p->a.x + buffer->_x.period;
  if (p->b.y == p->a.y) p->b.y = p->a.y + buffer->_y.period;

  /* trim to single tile */
  p = &portion->focus;
  p->a.x = roi->a.x;
  p->a.y = roi->a.y;
  p->b.x = MIN (roi->b.x, portion->bounds.b.x);
  p->b.y = MIN (roi->b.y, portion->bounds.b.y);

  return TRUE;
}


void
gimp_buffer_delete (GimpBuffer *buffer)
{
  g_return_if_fail (buffer != NULL);
  g_return_if_fail (buffer->klass != NULL);
  g_return_if_fail (K->delete != NULL);

  K->delete (buffer);
}


gboolean
gimp_buffer_alloc (GimpBuffer        *buffer,
                   GimpArea          *area,
                   Alloc              how)
{
  g_return_val_if_fail (buffer != NULL, FALSE);
  g_return_val_if_fail (buffer->klass != NULL, FALSE);
  g_return_val_if_fail (K->alloc != NULL, FALSE);
  g_return_val_if_fail (area != NULL, FALSE);

  return K->alloc (buffer, area, how);
}


gboolean
gimp_buffer_map (GimpBuffer    *buffer,
                 GimpArea      *darea,
                 GimpBuffer    *src,
                 GimpArea      *sarea)
{
  g_return_val_if_fail (buffer != NULL, FALSE);
  g_return_val_if_fail (buffer->klass != NULL, FALSE);  
  g_return_val_if_fail (K->map != NULL, FALSE);
  g_return_val_if_fail (src != NULL, FALSE);
  g_return_val_if_fail (darea != NULL, FALSE);
  g_return_val_if_fail (sarea != NULL, FALSE);

  /* same subtypes */
  g_return_val_if_fail (K->type == GIMP_BUFFER_CLASS(src->klass)->type, FALSE);

  /* compatible tiling structures */
  g_return_val_if_fail (buffer->_x.period == src->_x.period, FALSE);
  g_return_val_if_fail (buffer->_x.offset == src->_x.offset, FALSE);
  g_return_val_if_fail (buffer->_y.period == src->_y.period, FALSE);
  g_return_val_if_fail (buffer->_y.offset == src->_y.offset, FALSE);

  /* areas aligned to tile boundaries */
  {
    gint x =
      darea->a.x + darea->b.x +
      sarea->a.x + sarea->b.x -
      4 * buffer->_x.offset;
    gint y =
      darea->a.y + darea->b.y +
      sarea->a.y + sarea->b.y -
      4 * buffer->_y.offset;

    g_return_val_if_fail (x % buffer->_x.period == 0, FALSE);
    g_return_val_if_fail (y % buffer->_y.period == 0, FALSE);
  }
  
  return K->map (buffer, darea, src, sarea);
}


gboolean
gimp_buffer_validate (GimpBuffer        *buffer,
                      GimpArea          *area,
                      Validate           how)
{
  g_return_val_if_fail (buffer != NULL, FALSE);
  g_return_val_if_fail (buffer->klass != NULL, FALSE);
  g_return_val_if_fail (K->validate != NULL, FALSE);
  g_return_val_if_fail (area != NULL, FALSE);

  return K->validate (buffer, area, how);
}


gboolean
gimp_buffer_use (GimpBuffer        *buffer,
                 GimpPortion       *portion,
                 Use                how)
{
  g_return_val_if_fail (buffer != NULL, FALSE);
  g_return_val_if_fail (buffer->klass != NULL, FALSE);
  g_return_val_if_fail (K->use != NULL, FALSE);
  g_return_val_if_fail (portion != NULL, FALSE);
  
  return K->use (buffer, portion, how);
}


gboolean
gimp_buffer_query (GimpBuffer    *buffer,
                   GimpPortion   *portion,
                   GimpMemStatus *status)
{
  g_return_val_if_fail (buffer != NULL, FALSE);
  g_return_val_if_fail (buffer->klass != NULL, FALSE);
  g_return_val_if_fail (K->query != NULL, FALSE);
  g_return_val_if_fail (portion != NULL, FALSE);
  g_return_val_if_fail (status != NULL, FALSE);
  
  return K->query (buffer, portion, status);
}


gboolean
gimp_buffer_data (GimpBuffer     *buffer,
                  GimpPortion    *portion,
                  GimpPixelArray *array)
{
  g_return_val_if_fail (buffer != NULL, FALSE);
  g_return_val_if_fail (buffer->klass != NULL, FALSE);
  g_return_val_if_fail (K->data != NULL, FALSE);
  g_return_val_if_fail (portion != NULL, FALSE);
  g_return_val_if_fail (array != NULL, FALSE);
  
  return K->data (buffer, portion, array);
}


