/* GTK - The GIMP Toolkit
 * Copyright (C) 1995-1997 Peter Mattis, Spencer Kimball and Josh MacDonald
 * Copyright (C) 1998-1999 Adrian E. Feiguin <adrian@ifir.ifir.edu.ar>
 *
 * GtkSheet widget for Gtk+, by Adrian E. Feiguin, July 1998.
 * Based on GtkClist widget by Jay Painter, but major changes.
 * Memory allocation routines inspired on SC (Spreadsheet Calculator)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <string.h>
#include "gtksheet.h"

#define CELL_SPACING 1
#define DRAG_WIDTH 6
#define TIMEOUT_SCROLL 20
#define TIMEOUT_FLASH 100
#define TIME_INTERVAL 8
#define COLUMN_MIN_WIDTH 10
#define MINROWS 1
#define MINCOLS 1
#define MAXLENGTH 30
#define CELLOFFSET 4
#define MAX_TEXT_LENGTH 120
#define DEFAULT_COLUMN_WIDTH 80
#define DEFAULT_ROW_HEIGHT(widget) (widget->style->font->ascent+\
				   2 * widget->style->font->descent+\
				   2 * CELLOFFSET)  
				   

/* scrollbar spacing class macro */
#define SCROLLBAR_SPACING(w) (GTK_SHEET_CLASS (GTK_OBJECT (w)->klass)->scrollbar_spacing)

/* gives the top pixel of the given row in context of
 * the sheet's voffset */
static inline gint
ROW_TOP_YPIXEL(GtkSheet *sheet, gint nrow)
{
   return (sheet->voffset + sheet->row[nrow].top_ypixel);
}


/* returns the row index from a y pixel location in the 
 * context of the sheet's voffset */
static inline gint 
ROW_FROM_YPIXEL(GtkSheet *sheet, gint y)
{
  gint i, cy;

  cy = sheet->voffset + sheet->column_title_area.height;
  if(y < cy) return 0;
  for (i = 0; i <= sheet->maxrow; i++)
    {
      if (y >= cy  && y <= (cy + sheet->row[i].height) && sheet->row[i].is_visible)
	return i;
      if(sheet->row[i].is_visible) cy += sheet->row[i].height;

    }

  /* no match */
  return sheet->maxrow;
}


/* gives the left pixel of the given column in context of
 * the sheet's hoffset */
static inline gint
COLUMN_LEFT_XPIXEL(GtkSheet *sheet, gint ncol)
{
   return (sheet->hoffset + sheet->column[ncol].left_xpixel);
}

/* returns the column index from a x pixel location in the 
 * context of the sheet's hoffset */
static inline gint
COLUMN_FROM_XPIXEL (GtkSheet * sheet,
		    gint x)
{
  gint i, cx;

  cx = sheet->hoffset + sheet->row_title_area.width;
  if(x < cx) return 0;
  for (i = 0; i <= sheet->maxcol; i++)
    {
      if (x >= cx  && x <= (cx + sheet->column[i].width) && sheet->column[i].is_visible)
	return i;
      if(sheet->column[i].is_visible) cx += sheet->column[i].width;

    }

  /* no match */
  return sheet->maxcol;
}

/* returns the top pixel of the given row in the context of
 * the sheet height */
static inline gint
ROW_TOP(GtkSheet *sheet, gint row) 
{
  gint i,cy;
 
  cy=sheet->column_title_area.height;
  for (i=0;i<row; i++)
   if(sheet->row[i].is_visible) cy += sheet->row[i].height;
  
  return cy;
}


/* returns the left pixel of the given column in the context of
 * the sheet width */
static inline gint
COLUMN_LEFT(GtkSheet *sheet, gint column) 
{
  gint i,cx;
 
  cx=sheet->row_title_area.width;
  for (i=0;i<column; i++)
   if(sheet->column[i].is_visible) cx += sheet->column[i].width;
  
  return cx;
}


/* returns the total height of the sheet */
static inline SHEET_HEIGHT(GtkSheet *sheet)
{
  gint i,cx;
 
  cx=sheet->column_title_area.height;
  for (i=0;i<=sheet->maxrow; i++)
   if(sheet->row[i].is_visible) cx += sheet->row[i].height;
  
  return cx;
}


/* returns the total width of the sheet */
static inline SHEET_WIDTH(GtkSheet *sheet)
{
  gint i,cx;
 
  cx=sheet->row_title_area.width;
  for (i=0;i<=sheet->maxcol; i++)
   if(sheet->column[i].is_visible) cx += sheet->column[i].width;
  
  return cx;
}

#define MIN_VISIBLE_ROW(sheet) sheet->view.row0
#define MAX_VISIBLE_ROW(sheet) sheet->view.rowi
#define MIN_VISIBLE_COLUMN(sheet) sheet->view.col0
#define MAX_VISIBLE_COLUMN(sheet) sheet->view.coli


static inline POSSIBLE_XDRAG(GtkSheet *sheet, gint x, gint *drag_column)
{
 gint i;
 gint column, xdrag;

 column=COLUMN_FROM_XPIXEL(sheet, x);
 *drag_column=column;

 xdrag=COLUMN_LEFT_XPIXEL(sheet,column)+CELL_SPACING;
 if(x <= xdrag+DRAG_WIDTH/2 && column != 0){
   while(!sheet->column[column-1].is_visible && column>0) column--;
   *drag_column=column-1;
   return sheet->column[column-1].is_sensitive;
 }

 xdrag+=sheet->column[column].width;
 if(x >= xdrag-DRAG_WIDTH/2)
   return sheet->column[column].is_sensitive;

 return FALSE;
} 

static inline POSSIBLE_YDRAG(GtkSheet *sheet, gint y, gint *drag_row)
{
 gint i;
 gint row, ydrag;

 row=ROW_FROM_YPIXEL(sheet, y);
 *drag_row=row;

 ydrag=ROW_TOP_YPIXEL(sheet,row)+CELL_SPACING;
 if(y <= ydrag+DRAG_WIDTH/2 && row != 0){
   while(!sheet->row[row-1].is_visible && row>0) row--;
   *drag_row=row-1;
   return sheet->row[row-1].is_sensitive;
 }

 ydrag+=sheet->row[row].height;

 if(y >= ydrag-DRAG_WIDTH/2)
   return sheet->row[row].is_sensitive;
 
 
 return FALSE;
}        

static inline POSSIBLE_DRAG(GtkSheet *sheet, gint x, gint y,
                            gint *drag_row, gint *drag_column)
{
  gint ydrag, xdrag;

  *drag_column=COLUMN_FROM_XPIXEL(sheet,x);
  *drag_row=ROW_FROM_YPIXEL(sheet,y);

  if(x>=COLUMN_LEFT_XPIXEL(sheet,sheet->range.col0)-DRAG_WIDTH/2 &&
     x<=COLUMN_LEFT_XPIXEL(sheet,sheet->range.coli)+
        sheet->column[sheet->range.coli].width+DRAG_WIDTH/2){
     ydrag=ROW_TOP_YPIXEL(sheet,sheet->range.row0);
     if(y>=ydrag-DRAG_WIDTH/2 && y<=ydrag+DRAG_WIDTH/2){
        *drag_row=sheet->range.row0;
        return TRUE;
     }
     ydrag=ROW_TOP_YPIXEL(sheet,sheet->range.rowi)+
           sheet->row[sheet->range.rowi].height;
     if(y>=ydrag-DRAG_WIDTH/2 && y<=ydrag+DRAG_WIDTH/2){
        *drag_row=sheet->range.rowi;
        return TRUE;
     }
  }

  if(y>=ROW_TOP_YPIXEL(sheet,sheet->range.row0)-DRAG_WIDTH/2 &&
     y<=ROW_TOP_YPIXEL(sheet,sheet->range.rowi)+
        sheet->row[sheet->range.rowi].height+DRAG_WIDTH/2){
     xdrag=COLUMN_LEFT_XPIXEL(sheet,sheet->range.col0);
     if(x>=xdrag-DRAG_WIDTH/2 && x<=xdrag+DRAG_WIDTH/2){
        *drag_column=sheet->range.col0;
        return TRUE;
     }
     xdrag=COLUMN_LEFT_XPIXEL(sheet,sheet->range.coli)+
           sheet->column[sheet->range.coli].width;
     if(x>=xdrag-DRAG_WIDTH/2 && x<=xdrag+DRAG_WIDTH/2){
        *drag_column=sheet->range.coli;
        return TRUE;
     }
  }
  return FALSE;
}

static inline POSSIBLE_RESIZE(GtkSheet *sheet, gint x, gint y,
                            gint *drag_row, gint *drag_column)
{
  gint xdrag, ydrag;
  
  xdrag=COLUMN_LEFT_XPIXEL(sheet,sheet->range.coli)+
           sheet->column[sheet->range.coli].width;

  ydrag=ROW_TOP_YPIXEL(sheet,sheet->range.rowi)+
           sheet->row[sheet->range.rowi].height;

  *drag_column=COLUMN_FROM_XPIXEL(sheet,x);
  *drag_row=ROW_FROM_YPIXEL(sheet,y);

  if(x>=xdrag-DRAG_WIDTH/2 && x<=xdrag+DRAG_WIDTH/2 &&
     y>=ydrag-DRAG_WIDTH/2 && y<=ydrag+DRAG_WIDTH/2) return TRUE;

  return FALSE;  
}

typedef void (*GtkSheetSignal1) (GtkObject *object,
	                         gint arg1,
	                         gint arg2,
	                         gpointer data);

typedef void (*GtkSheetSignal2) (GtkObject * object,
				 gpointer arg1,
				 gpointer data);

typedef void (*GtkSheetSignal3) (GtkObject * object,
				 gpointer arg1, gpointer arg2,
				 gpointer data);


static void gtk_sheet_class_init (GtkSheetClass * klass);
static void gtk_sheet_init (GtkSheet * sheet);
static void
gtk_sheet_class_init (GtkSheetClass * klass);
static void
gtk_sheet_destroy (GtkObject * object);
static void
gtk_sheet_realize (GtkWidget * widget);
static void
gtk_sheet_unrealize (GtkWidget * widget);
static void
gtk_sheet_map (GtkWidget * widget);
static void
gtk_sheet_unmap (GtkWidget * widget);
static void
gtk_sheet_draw (GtkWidget * widget,
		GdkRectangle * area);
static gint
gtk_sheet_expose (GtkWidget * widget,
		  GdkEventExpose * event);
static gint
gtk_sheet_range_isvisible (GtkSheet * sheet,
			 GtkSheetRange range);
static gint
gtk_sheet_cell_isvisible (GtkSheet * sheet,
			  gint row, gint column);
static GtkSheetCell *
gtk_sheet_cell_new (void);

static gint
gtk_sheet_scroll(gpointer data);
static gint
gtk_sheet_flash(gpointer data);

static gint
gtk_sheet_button_press (GtkWidget * widget,
			GdkEventButton * event);
static gint
gtk_sheet_button_release (GtkWidget * widget,
			GdkEventButton * event);
static gint
gtk_sheet_motion (GtkWidget * widget,
		  GdkEventMotion * event);
static gint
gtk_sheet_move_query(GtkSheet *sheet, gint row, gint column);
static void
gtk_sheet_extend_selection(GtkSheet *sheet, gint row, gint column);
static void
gtk_sheet_new_selection(GtkSheet *sheet, GtkSheetRange *range);
static void
gtk_sheet_draw_border (GtkSheet *sheet, GtkSheetRange range);
static void
gtk_sheet_draw_corners(GtkSheet *sheet, GtkSheetRange range);
static gint
gtk_sheet_key_press(GtkWidget *widget,
		    GdkEventKey *key);
static void
gtk_sheet_size_request (GtkWidget * widget,
			GtkRequisition * requisition);
static void
gtk_sheet_size_allocate (GtkWidget * widget,
			 GtkAllocation * allocation);
static void
gtk_sheet_foreach (GtkContainer * container,
		   GtkCallback callback,
		   gpointer callback_data);
static void
gtk_sheet_deactivate_cell(GtkSheet *sheet);
static void
gtk_sheet_activate_cell(GtkSheet *sheet, gint row, gint col);
static void
gtk_sheet_draw_active_cell(GtkSheet *sheet);
static void
gtk_sheet_show_active_cell(GtkSheet *sheet);
static void
gtk_sheet_click_cell(GtkSheet *sheet, gint row, gint column);
static void
gtk_sheet_make_backing_pixmap (GtkSheet *sheet, gint width, gint height);
static void
gtk_sheet_draw_backing_pixmap(GtkSheet *sheet, GtkSheetRange range);
static void
create_scrollbars (GtkSheet * sheet);
static void
adjust_scrollbars (GtkSheet * sheet);
static void
vadjustment_changed (GtkAdjustment * adjustment,
			       gpointer data);
static void
hadjustment_changed (GtkAdjustment * adjustment,
			       gpointer data);
static void
vadjustment_value_changed (GtkAdjustment * adjustment,
				     gpointer data);
static void
hadjustment_value_changed (GtkAdjustment * adjustment,
				     gpointer data);
static void                          
draw_xor_vline (GtkSheet * sheet);
static void                          
draw_xor_hline (GtkSheet * sheet);
static void
draw_xor_rectangle(GtkSheet *sheet, GtkSheetRange range);
static void
gtk_sheet_draw_flashing_range(GtkSheet *sheet, GtkSheetRange range);
static gint
new_column_width (GtkSheet * sheet,
		  gint column,
		  gint * x);
static gint
new_row_height (GtkSheet * sheet,
		  gint row,
		  gint * y);
static void
create_sheet_entry(GtkSheet *sheet);
static void
create_global_button(GtkSheet *sheet);

static void
gtk_sheet_size_allocate_entry(GtkSheet *sheet);
static void
gtk_sheet_entry_set_max_size(GtkSheet *sheet);

static void
size_allocate_column_title_buttons (GtkSheet * sheet);
static void
size_allocate_row_title_buttons (GtkSheet * sheet);

static void
gtk_sheet_recalc_top_ypixels(GtkSheet *sheet, gint row);
static void
gtk_sheet_recalc_left_xpixels(GtkSheet *sheet, gint column);

static void
row_button_set (GtkSheet *sheet, gint row);
static void
column_button_set (GtkSheet *sheet, gint column);
static void
row_button_release (GtkSheet *sheet, gint row);
static void
column_button_release (GtkSheet *sheet, gint column);
static void
gtk_sheet_button_draw(GtkSheet *sheet, gint row, gint column);


static void 
size_allocate_global_button (GtkSheet *sheet);
static void
global_button_clicked(GtkWidget *widget, gpointer data);

static void
gtk_sheet_set_range_attributes(GtkSheet *sheet, GtkSheetAttr attributes);
static void
gtk_sheet_set_cell_attributes(GtkSheet *sheet, gint row, gint col, GtkSheetAttr attributes);

/* Signals */

enum {
      SELECT_ROW, 
      SELECT_COLUMN, 
      SELECT_RANGE,
      CLIP_RANGE,
      RESIZE_RANGE,
      MOVE_RANGE, 
      ACTIVATE_CELL,
      SET_CELL,
      CHANGED,
      LAST_SIGNAL
};


static void
gtk_sheet_marshal_signal_1 (GtkObject * object,
			    GtkSignalFunc func,
			    gpointer func_data,
			    GtkArg * args);
static void 
gtk_sheet_marshal_signal_2 (GtkObject * object,
			    GtkSignalFunc func,
			    gpointer func_data,
			    GtkArg * args);

static void 
gtk_sheet_marshal_signal_3 (GtkObject * object,
			    GtkSignalFunc func,
			    gpointer func_data,
			    GtkArg * args);

static GtkContainerClass *parent_class = NULL;
static gint sheet_signals[LAST_SIGNAL] = {0};


guint
gtk_sheet_get_type ()
{
  static guint sheet_type = 0;
  if(!sheet_type){
	GtkTypeInfo sheet_info =
        {
		"GtkSheet",
	        sizeof(GtkSheet),
	        sizeof(GtkSheetClass),
	        (GtkClassInitFunc) gtk_sheet_class_init,
	        (GtkObjectInitFunc) gtk_sheet_init,
		(GtkArgSetFunc) NULL,
	        (GtkArgGetFunc) NULL,
	};
	sheet_type = gtk_type_unique (gtk_container_get_type (), &sheet_info);
  }
  return sheet_type;
}

static void
gtk_sheet_class_init (GtkSheetClass * klass)
{
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;
  GtkContainerClass *container_class;

  object_class = (GtkObjectClass *) klass;
  widget_class = (GtkWidgetClass *) klass;
  container_class = (GtkContainerClass *) klass;

  parent_class = gtk_type_class (gtk_container_get_type ());

  sheet_signals[SELECT_ROW] =
    gtk_signal_new ("select_row",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GtkSheetClass, select_row),
		    gtk_sheet_marshal_signal_1,
            GTK_TYPE_NONE, 2, GTK_TYPE_INT, GTK_TYPE_INT);
  sheet_signals[SELECT_COLUMN] =
    gtk_signal_new ("select_column",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GtkSheetClass, select_column),
		    gtk_sheet_marshal_signal_1,
            GTK_TYPE_NONE, 2, GTK_TYPE_INT, GTK_TYPE_INT);
  sheet_signals[SELECT_RANGE] =
    gtk_signal_new ("select_range",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GtkSheetClass, select_range),
		    gtk_sheet_marshal_signal_2,
	    GTK_TYPE_NONE, 1, GTK_TYPE_POINTER);
  sheet_signals[CLIP_RANGE] =
    gtk_signal_new ("clip_range",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GtkSheetClass, select_range),
		    gtk_sheet_marshal_signal_2,
	    GTK_TYPE_NONE, 1, GTK_TYPE_POINTER);
  sheet_signals[RESIZE_RANGE] =
    gtk_signal_new ("resize_range",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GtkSheetClass, select_range),
		    gtk_sheet_marshal_signal_3,
	    GTK_TYPE_NONE, 2, GTK_TYPE_POINTER, GTK_TYPE_POINTER);
  sheet_signals[MOVE_RANGE] =
    gtk_signal_new ("move_range",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GtkSheetClass, select_range),
		    gtk_sheet_marshal_signal_3,
	    GTK_TYPE_NONE, 2, GTK_TYPE_POINTER, GTK_TYPE_POINTER);
  sheet_signals[ACTIVATE_CELL] =
    gtk_signal_new ("activate_cell",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GtkSheetClass, activate_cell),
		    gtk_sheet_marshal_signal_1,
            GTK_TYPE_NONE, 2, GTK_TYPE_INT, GTK_TYPE_INT);
  sheet_signals[SET_CELL] =
    gtk_signal_new ("set_cell",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GtkSheetClass, set_cell),
		    gtk_sheet_marshal_signal_1,
            GTK_TYPE_NONE, 2, GTK_TYPE_INT, GTK_TYPE_INT);
  sheet_signals[CHANGED] =
    gtk_signal_new ("changed",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GtkSheetClass, changed),
		    gtk_sheet_marshal_signal_1,
            GTK_TYPE_NONE, 2, GTK_TYPE_INT, GTK_TYPE_INT);

  gtk_object_class_add_signals (object_class, sheet_signals, LAST_SIGNAL);

  object_class->destroy = gtk_sheet_destroy;

  widget_class->realize = gtk_sheet_realize;
  widget_class->unrealize = gtk_sheet_unrealize;
  widget_class->draw_focus = NULL;
  widget_class->map = gtk_sheet_map;
  widget_class->unmap = gtk_sheet_unmap;
  widget_class->draw = gtk_sheet_draw;
  widget_class->button_press_event = gtk_sheet_button_press;
  widget_class->button_release_event = gtk_sheet_button_release;
  widget_class->motion_notify_event = gtk_sheet_motion;
  widget_class->key_press_event = gtk_sheet_key_press;
  widget_class->expose_event = gtk_sheet_expose;
  widget_class->size_request = gtk_sheet_size_request;
  widget_class->size_allocate = gtk_sheet_size_allocate;
  widget_class->focus_in_event = NULL;
  widget_class->focus_out_event = NULL;

  container_class->add = NULL;
  container_class->remove = NULL;
  container_class->foreach = NULL;

  klass->select_row = NULL;
  klass->select_column = NULL;
  klass->select_range = NULL;
  klass->clip_range = NULL;
  klass->resize_range = NULL;
  klass->move_range = NULL;
  klass->activate_cell = NULL;
  klass->set_cell = NULL;
  klass->changed = NULL;

  klass->scrollbar_spacing = 5;

}

static void
gtk_sheet_marshal_signal_1 (GtkObject * object,
			    GtkSignalFunc func,
			    gpointer func_data,
			    GtkArg * args)
{
  GtkSheetSignal1 rfunc;

  rfunc = (GtkSheetSignal1) func;

  (*rfunc) (object, GTK_VALUE_INT (args[0]),
	    GTK_VALUE_INT (args[1]),
	    func_data);
}

static void
gtk_sheet_marshal_signal_2 (GtkObject * object,
			    GtkSignalFunc func,
			    gpointer func_data,
			    GtkArg * args)
{
  GtkSheetSignal2 rfunc;

  rfunc = (GtkSheetSignal2) func;

  (*rfunc) (object, GTK_VALUE_POINTER (args[0]),
	    func_data);
}

static void
gtk_sheet_marshal_signal_3 (GtkObject * object,
			    GtkSignalFunc func,
			    gpointer func_data,
			    GtkArg * args)
{
  GtkSheetSignal3 rfunc;

  rfunc = (GtkSheetSignal3) func;

  (*rfunc) (object, GTK_VALUE_POINTER (args[0]), GTK_VALUE_POINTER(args[1]),
	    func_data);
}

static void gtk_sheet_init(GtkSheet *sheet)
{
  sheet->flags = 0;

  GTK_WIDGET_UNSET_FLAGS (sheet, GTK_NO_WINDOW);
  GTK_WIDGET_SET_FLAGS (sheet, GTK_CAN_FOCUS);


  sheet->maxrow = 0;
  sheet->maxcol = 0;

  sheet->view.row0 = 0;
  sheet->view.col0 = 0;
  sheet->view.rowi = 0;
  sheet->view.coli = 0;

  sheet->maxallocrow = 0;
  sheet->maxalloccol = 0;

  sheet->column_title_window=NULL;
  sheet->column_title_area.x=0;
  sheet->column_title_area.y=0;
  sheet->column_title_area.width=0;
  sheet->column_title_area.height=DEFAULT_ROW_HEIGHT(GTK_WIDGET(sheet));
 
  sheet->row_title_window=NULL;
  sheet->row_title_area.x=0;
  sheet->row_title_area.y=0;
  sheet->row_title_area.width=DEFAULT_COLUMN_WIDTH;
  sheet->row_title_area.height=0;

  sheet->active_cell.row=0;
  sheet->active_cell.col=0;
  sheet->selection_cell.row=0;
  sheet->selection_cell.col=0;

  sheet->sheet_entry=NULL;
  sheet->pixmap=NULL;

  sheet->range.row0=0;
  sheet->range.rowi=0;
  sheet->range.col0=0;
  sheet->range.coli=0;

  sheet->state=GTK_SHEET_NORMAL;

  sheet->attributes = NULL;
  sheet->maxrange = 0;

  sheet->sheet_window = NULL;
  sheet->sheet_window_width = 0;
  sheet->sheet_window_height = 0;

  sheet->hoffset = 0;
  sheet->voffset = 0;

  sheet->shadow_type = GTK_SHADOW_IN;
  sheet->hscrollbar_policy = GTK_POLICY_ALWAYS;
  sheet->vscrollbar_policy = GTK_POLICY_ALWAYS;

  sheet->cursor_drag = gdk_cursor_new(GDK_PLUS);
  sheet->xor_gc = NULL;
  sheet->fg_gc = NULL;
  sheet->bg_gc = NULL;
  sheet->x_drag = 0;
  sheet->y_drag = 0;

}

GtkWidget *
gtk_sheet_new (int rows, int columns, gchar *title)
{
  GtkSheet *sheet;

  /* sanity check */
  if (columns < MINCOLS || rows < MINROWS)
    return NULL;
  sheet = gtk_type_new (gtk_sheet_get_type ());

  sheet->row=(GtkSheetRow *)malloc(sizeof(GtkSheetRow));
  sheet->column=(GtkSheetColumn *)malloc(sizeof(GtkSheetColumn));
  sheet->data=(GtkSheetCell ***)malloc(sizeof(GtkSheetCell **));

  /* set number of rows and columns */
  GrowSheet(sheet, MINROWS, MINCOLS);
  AddRow(sheet,rows-1);
  AddColumn(sheet,columns-1);
  /* create scrollbars */
  create_scrollbars (sheet);
  /* create sheet entry */
  create_sheet_entry(sheet);
  /* create global selection button */
  create_global_button(sheet);

  sheet->name=NULL;
  gtk_sheet_set_title(sheet, title);

  return GTK_WIDGET (sheet);
}


GtkWidget *
gtk_sheet_new_browser(int rows, int columns, gchar *title)
{
  GtkWidget *sheet;
  
  sheet=gtk_sheet_new(rows, columns, title);
  GTK_SHEET_SET_FLAGS(sheet, SHEET_IS_LOCKED);
  GTK_SHEET_SET_FLAGS(sheet, SHEET_AUTORESIZE);
 
  return sheet;
}

void
gtk_sheet_set_title(GtkSheet *sheet, gchar *title)
{
  GtkWidget *old_widget;
  GtkWidget *label;

  g_return_if_fail (sheet != NULL);
  g_return_if_fail (GTK_IS_SHEET (sheet));

  if (sheet->name)
    g_free (sheet->name);

  sheet->name = g_strdup (title);

  if(!GTK_WIDGET_REALIZED(GTK_WIDGET(sheet)) || !title) return;

  /* remove and destroy the old widget */
  old_widget = GTK_BUTTON (sheet->button)->child;
  if (old_widget)
    {
      gtk_container_remove (GTK_CONTAINER (sheet->button), old_widget);
      gtk_widget_destroy (old_widget);
    }

  label = gtk_label_new (title);
  gtk_misc_set_alignment(GTK_MISC(label), 0.5, 0.);
  gtk_container_add (GTK_CONTAINER (sheet->button), label);
  gtk_widget_show (label);
  size_allocate_global_button(sheet);
  gtk_signal_emit(GTK_OBJECT(sheet),sheet_signals[CHANGED], -1, -1);

}

void
gtk_sheet_set_border (GtkSheet * sheet,
		      GtkShadowType border)
{
  g_return_if_fail (sheet != NULL);
  g_return_if_fail (GTK_IS_SHEET (sheet));

  sheet->shadow_type = border;

  if (GTK_WIDGET_VISIBLE (sheet))
    gtk_widget_queue_resize (GTK_WIDGET (sheet));
}

void
gtk_sheet_freeze (GtkSheet *sheet)
{
  g_return_if_fail (sheet != NULL);
  g_return_if_fail (GTK_IS_SHEET (sheet));

  GTK_SHEET_SET_FLAGS(sheet, SHEET_IS_FROZEN);
}

void
gtk_sheet_thaw(GtkSheet *sheet)
{
  g_return_if_fail (sheet != NULL);
  g_return_if_fail (GTK_IS_SHEET (sheet));

  GTK_SHEET_UNSET_FLAGS(sheet, SHEET_IS_FROZEN);

  adjust_scrollbars(sheet);
}

void
gtk_sheet_set_column_title (GtkSheet * sheet,
			    gint column,
			    gchar * title)
{
  g_return_if_fail (sheet != NULL);
  g_return_if_fail (GTK_IS_SHEET (sheet));

  if (sheet->column[column].name)
    g_free (sheet->column[column].name);

  sheet->column[column].name = g_strdup (title);
}

void
gtk_sheet_set_row_title (GtkSheet * sheet,
			 gint row,
			 gchar * title)
{
  g_return_if_fail (sheet != NULL);
  g_return_if_fail (GTK_IS_SHEET (sheet));

  if (sheet->row[row].name)
    g_free (sheet->row[row].name);

  sheet->row[row].name = g_strdup (title);
}

void
gtk_sheet_row_button_add_label(GtkSheet *sheet, gint row, char *label)
{
  g_return_if_fail (sheet != NULL);
  g_return_if_fail (GTK_IS_SHEET (sheet));

  if(row < 0 || row > sheet->maxrow) return;

  if (sheet->row[row].button.label)
    g_free (sheet->row[row].button.label);
 
  sheet->row[row].button.label = g_strdup (label);

  if(!GTK_SHEET_IS_FROZEN(sheet)){  
    gtk_sheet_button_draw(sheet, row, -1);
    gtk_signal_emit(GTK_OBJECT(sheet),sheet_signals[CHANGED], row, -1);
  }
}  

void
gtk_sheet_column_button_add_label(GtkSheet *sheet, gint column, char *label)
{
  g_return_if_fail (sheet != NULL);
  g_return_if_fail (GTK_IS_SHEET (sheet));

  if(column <0 || column >sheet->maxrow) return;

  if (sheet->column[column].button.label)
    g_free (sheet->column[column].button.label);
  

  sheet->column[column].button.label = g_strdup (label);

  if(!GTK_SHEET_IS_FROZEN(sheet)){
    gtk_sheet_button_draw(sheet, -1, column);
    gtk_signal_emit(GTK_OBJECT(sheet),sheet_signals[CHANGED], -1, column);
  }
}  


void
gtk_sheet_moveto (GtkSheet * sheet,
		  gint row,
		  gint column,
	          gfloat row_align,
                  gfloat col_align)
{
  gint x, y;

  g_return_if_fail (sheet != NULL);
  g_return_if_fail (GTK_IS_SHEET (sheet));

  if (row < 0 || row > sheet->maxrow)
    return;
  if (column < 0 || column > sheet->maxcol)
    return;

  /* adjust vertical scrollbar */
  if (row >= 0 && row_align >=0.)
    {
      y = ROW_TOP_YPIXEL(sheet, row) - sheet->voffset -
          row_align*(sheet->sheet_window_height - 
                     sheet->column_title_area.height)-
          (1.-row_align)*sheet->row[row].height;

      if (y < 0)
	GTK_RANGE (sheet->vscrollbar)->adjustment->value = 0.0;
      else if (y > SHEET_HEIGHT (sheet) - sheet->sheet_window_height + sheet->column_title_area.height)
	GTK_RANGE (sheet->vscrollbar)->adjustment->value = SHEET_HEIGHT (sheet) - sheet->sheet_window_height + sheet->column_title_area.height;
      else
	GTK_RANGE (sheet->vscrollbar)->adjustment->value = y;
      
      gtk_signal_emit_by_name (GTK_OBJECT (GTK_RANGE (sheet->vscrollbar)->adjustment), 
			       "value_changed");
    } 
     
  /* adjust horizontal scrollbar */
  if (column >= 0 && col_align >= 0.)
    {
      x = COLUMN_LEFT_XPIXEL (sheet, column) - sheet->hoffset -
          col_align*(sheet->sheet_window_width -
                     sheet->row_title_area.width) -
          (1.-col_align)*sheet->column[column].width;
      
      if (x < 0)
	GTK_RANGE (sheet->hscrollbar)->adjustment->value = 0.0;
      else if (x > SHEET_WIDTH (sheet) - sheet->sheet_window_width)
	GTK_RANGE (sheet->hscrollbar)->adjustment->value = SHEET_WIDTH (sheet) - 	  sheet->sheet_window_width;
      else
	GTK_RANGE (sheet->hscrollbar)->adjustment->value = x;
      
      gtk_signal_emit_by_name (GTK_OBJECT (GTK_RANGE (sheet->hscrollbar)->adjustment), 
			       "value_changed");
    }
}

void 
gtk_sheet_column_set_sensitivity(GtkSheet *sheet, gint column, gint sensitive)
{
  g_return_if_fail (sheet != NULL);
  g_return_if_fail (GTK_IS_SHEET (sheet));

  sheet->column[column].is_sensitive=sensitive;
  sheet->column[column].button.state=GTK_STATE_INSENSITIVE;
}

void
gtk_sheet_columns_set_sensitivity(GtkSheet *sheet, gint sensitive)
{
  gint i;

  g_return_if_fail (sheet != NULL);
  g_return_if_fail (GTK_IS_SHEET (sheet));

  for(i=0; i<=sheet->maxcol; i++)
     gtk_sheet_column_set_sensitivity(sheet, i, sensitive);
}

void 
gtk_sheet_row_set_sensitivity(GtkSheet *sheet, gint row,  gint sensitive)
{

  g_return_if_fail (sheet != NULL);
  g_return_if_fail (GTK_IS_SHEET (sheet));

  sheet->row[row].is_sensitive=sensitive;
  sheet->row[row].button.state=GTK_STATE_INSENSITIVE;
}

void
gtk_sheet_rows_set_sensitivity(GtkSheet *sheet, gint sensitive)
{
  gint i;

  g_return_if_fail (sheet != NULL);
  g_return_if_fail (GTK_IS_SHEET (sheet));

  for(i=0; i<=sheet->maxrow; i++)
     gtk_sheet_row_set_sensitivity(sheet, i, sensitive);
}

void
gtk_sheet_column_set_visibility(GtkSheet *sheet, gint column, gint visible)
{
  gint i;

  g_return_if_fail (sheet != NULL);
  g_return_if_fail (GTK_IS_SHEET (sheet));

  if(sheet->column[column].is_visible == visible) return;

  sheet->column[column].is_visible = visible;

  gtk_sheet_recalc_left_xpixels(sheet, column);

  if(!GTK_SHEET_IS_FROZEN(sheet) && 
    gtk_sheet_cell_isvisible(sheet, MIN_VISIBLE_ROW(sheet), column)){
      gtk_sheet_range_draw(sheet, NULL);
      size_allocate_column_title_buttons(sheet);
  }
}

void
gtk_sheet_row_set_visibility(GtkSheet *sheet, gint row, gint visible)
{
  gint i;

  g_return_if_fail (sheet != NULL);
  g_return_if_fail (GTK_IS_SHEET (sheet));

  if(sheet->row[row].is_visible == visible) return;

  sheet->row[row].is_visible = visible;

  gtk_sheet_recalc_top_ypixels(sheet, row);

  if(!GTK_SHEET_IS_FROZEN(sheet) && 
    gtk_sheet_cell_isvisible(sheet, row, MIN_VISIBLE_COLUMN(sheet))){
      gtk_sheet_range_draw(sheet, NULL);
      size_allocate_row_title_buttons(sheet);
  }
}

void
gtk_sheet_select_row (GtkSheet * sheet,
		      gint row)
{
  g_return_if_fail (sheet != NULL);
  g_return_if_fail (GTK_IS_SHEET (sheet));

  if (row < 0 || row > sheet->maxrow)
    return;

  if(sheet->state != GTK_SHEET_NORMAL) gtk_sheet_unselect_range(sheet, NULL);

  sheet->state=GTK_SHEET_ROW_SELECTED;                     
  sheet->range.row0=row;
  sheet->range.col0=0;
  sheet->range.rowi=row;
  sheet->range.coli=sheet->maxcol;
  sheet->active_cell.row=row;
  sheet->active_cell.col=0;

  gtk_signal_emit (GTK_OBJECT (sheet), sheet_signals[SELECT_ROW], row, 0);
  gtk_sheet_select_range(sheet, NULL);

}


void
gtk_sheet_select_column (GtkSheet * sheet,
		         gint column)
{
  
  g_return_if_fail (sheet != NULL);
  g_return_if_fail (GTK_IS_SHEET (sheet));

  if (column < 0 || column > sheet->maxcol)
    return;

  if(sheet->state != GTK_SHEET_NORMAL) gtk_sheet_unselect_range(sheet, NULL);

  sheet->state=GTK_SHEET_COLUMN_SELECTED;                     
  sheet->range.row0=0;
  sheet->range.col0=column;
  sheet->range.rowi=sheet->maxrow;
  sheet->range.coli=column;
  sheet->active_cell.row=0;
  sheet->active_cell.col=column;

  gtk_signal_emit (GTK_OBJECT (sheet), sheet_signals[SELECT_COLUMN], 0, column);
  gtk_sheet_select_range(sheet, NULL);

}

void
gtk_sheet_clip_range (GtkSheet *sheet, GtkSheetRange range)
{
  gint x,y,width,height;

  g_return_if_fail (sheet != NULL);
  g_return_if_fail (GTK_IS_SHEET (sheet));

  if(GTK_SHEET_IN_CLIP(sheet)) return;

  GTK_SHEET_SET_FLAGS(sheet, SHEET_IN_CLIP);

  sheet->clip_range=range;
  sheet->interval=0;
  sheet->clip_timer=gtk_timeout_add(TIMEOUT_FLASH, gtk_sheet_flash, sheet); 

  gtk_signal_emit(GTK_OBJECT(sheet), sheet_signals[CLIP_RANGE],
                                     &sheet->clip_range);

}

void
gtk_sheet_unclip_range(GtkSheet *sheet)
{

  if(!GTK_SHEET_IN_CLIP(sheet)) return;

  GTK_SHEET_UNSET_FLAGS(sheet, SHEET_IN_CLIP);
  gtk_timeout_remove(sheet->clip_timer);
  gtk_sheet_range_draw(sheet, &sheet->clip_range);

  if(gtk_sheet_range_isvisible(sheet, sheet->range))
    gtk_sheet_range_draw(sheet, &sheet->range);
}

static gint
gtk_sheet_flash(gpointer data)
{
  GtkSheet *sheet;
  gint x,y,width,height;
  GdkRectangle clip_area;

  sheet=GTK_SHEET(data);

  if(!gtk_sheet_range_isvisible(sheet, sheet->clip_range)) return TRUE;
  if(GTK_SHEET_IN_XDRAG(sheet)) return TRUE; 
  if(GTK_SHEET_IN_YDRAG(sheet)) return TRUE; 
 
  x=COLUMN_LEFT_XPIXEL(sheet,sheet->clip_range.col0)+1;
  y=ROW_TOP_YPIXEL(sheet,sheet->clip_range.row0)+1;
  width=COLUMN_LEFT_XPIXEL(sheet,sheet->clip_range.coli)-x+ 
             sheet->column[sheet->clip_range.coli].width-1;
  height=ROW_TOP_YPIXEL(sheet,sheet->clip_range.rowi)-y+
             sheet->row[sheet->clip_range.rowi].height-1;

  clip_area.x=sheet->row_title_area.width;
  clip_area.y=sheet->column_title_area.height;
  clip_area.width=sheet->sheet_window_width;
  clip_area.height=sheet->sheet_window_height;

  if(x<0) {
     width=width+x;
     x=0;
  }
  if(width>clip_area.width) width=clip_area.width+10;
  if(y<0) {
     height=height+y;
     y=0;
  }
  if(height>clip_area.height) height=clip_area.height+10;

  gdk_draw_pixmap(sheet->sheet_window,
                  GTK_WIDGET(sheet)->style->fg_gc[GTK_STATE_NORMAL],
                  sheet->pixmap,
                  x, y,
                  x, y,
                  1, height);

  gdk_draw_pixmap(sheet->sheet_window,
                  GTK_WIDGET(sheet)->style->fg_gc[GTK_STATE_NORMAL],
                  sheet->pixmap,
                  x, y,
                  x, y,
                  width, 1);

  gdk_draw_pixmap(sheet->sheet_window,
                  GTK_WIDGET(sheet)->style->fg_gc[GTK_STATE_NORMAL],
                  sheet->pixmap,
                  x, y+height,
                  x, y+height,
                  width, 1);

  gdk_draw_pixmap(sheet->sheet_window,
                  GTK_WIDGET(sheet)->style->fg_gc[GTK_STATE_NORMAL],
                  sheet->pixmap,
                  x+width, y,
                  x+width, y,
                  1, height);


  sheet->interval=sheet->interval+1;
  if(sheet->interval==TIME_INTERVAL) sheet->interval=0;

  gtk_sheet_draw_flashing_range(sheet, sheet->clip_range);

  return TRUE;

}

static void
gtk_sheet_draw_flashing_range(GtkSheet *sheet, GtkSheetRange range)
{
  GdkRectangle clip_area;
  gint x,y,width,height;

  if(!gtk_sheet_range_isvisible(sheet, sheet->clip_range)) return;
  
  clip_area.x=sheet->row_title_area.width;
  clip_area.y=sheet->column_title_area.height;
  clip_area.width=sheet->sheet_window_width;
  clip_area.height=sheet->sheet_window_height;

  gdk_gc_set_clip_rectangle(sheet->xor_gc, &clip_area);  

  x=COLUMN_LEFT_XPIXEL(sheet,sheet->clip_range.col0)+1;
  y=ROW_TOP_YPIXEL(sheet,sheet->clip_range.row0)+1;
  width=COLUMN_LEFT_XPIXEL(sheet,sheet->clip_range.coli)-x+ 
             sheet->column[sheet->clip_range.coli].width-1;
  height=ROW_TOP_YPIXEL(sheet,sheet->clip_range.rowi)-y+
             sheet->row[sheet->clip_range.rowi].height-1;

  if(x<0) {
     width=width+x;
     x=0;
  }
  if(width>clip_area.width) width=clip_area.width+10;
  if(y<0) {
     height=height+y;
     y=0;
  }
  if(height>clip_area.height) height=clip_area.height+10;

  gdk_gc_set_line_attributes(sheet->xor_gc, 1, 1, 0 ,0 );


  if(sheet->interval > TIME_INTERVAL/2) {
  gdk_draw_line(sheet->sheet_window, sheet->xor_gc,
                x, y,
                x, y+sheet->interval-TIME_INTERVAL/2);

  gdk_draw_line(sheet->sheet_window, sheet->xor_gc,
                x, y+height,
                x+sheet->interval-TIME_INTERVAL/2, y+height);

  gdk_draw_line(sheet->sheet_window, sheet->xor_gc,
                x+width, y+height,
                x+width, y+height-sheet->interval+TIME_INTERVAL/2);

  gdk_draw_line(sheet->sheet_window, sheet->xor_gc,
                x+width, y,
                x+width-sheet->interval+TIME_INTERVAL/2, y);
  }

  gdk_draw_line(sheet->sheet_window, sheet->xor_gc,
                x, y+sheet->interval,
                x, y+height);

  gdk_draw_line(sheet->sheet_window, sheet->xor_gc,
                x+sheet->interval, y+height,
                x+width, y+height);

  gdk_draw_line(sheet->sheet_window, sheet->xor_gc,
                x+width, y+height-sheet->interval,
                x+width, y);

  gdk_draw_line(sheet->sheet_window, sheet->xor_gc,
                x+width-sheet->interval, y,
                x, y);

  gdk_gc_set_line_attributes (sheet->xor_gc, 1, 0, 0, 0);

  gdk_gc_set_clip_rectangle(sheet->xor_gc, NULL);

}

static gint
gtk_sheet_range_isvisible (GtkSheet * sheet,
			 GtkSheetRange range)
{
  g_return_val_if_fail (sheet != NULL, FALSE);

  if (range.row0 < 0 || range.row0 > sheet->maxrow)
    return FALSE;

  if (range.rowi < 0 || range.rowi > sheet->maxrow)
    return FALSE;

  if (range.col0 < 0 || range.col0 > sheet->maxcol)
    return FALSE;

  if (range.coli < 0 || range.coli > sheet->maxcol)
    return FALSE;

  if (range.rowi < ROW_FROM_YPIXEL (sheet, sheet->column_title_area.height + CELL_SPACING))
    return FALSE;

  if (range.row0 > ROW_FROM_YPIXEL (sheet, sheet->sheet_window_height))
    return FALSE;

  if (range.coli < COLUMN_FROM_XPIXEL (sheet, sheet->row_title_area.width + CELL_SPACING))
    return FALSE;

  if (range.col0 > COLUMN_FROM_XPIXEL (sheet, sheet->sheet_window_width))
    return FALSE;

  return TRUE;
}

static gint
gtk_sheet_cell_isvisible (GtkSheet * sheet,
			  gint row, gint column)
{
  GtkSheetRange range;

  range.row0 = row;
  range.col0 = column;
  range.rowi = row;
  range.coli = column;

  return gtk_sheet_range_isvisible(sheet, range);
}


GtkAdjustment *
gtk_sheet_get_vadjustment (GtkSheet * sheet)
{
  g_return_val_if_fail (sheet != NULL, NULL);
  g_return_val_if_fail (GTK_IS_SHEET (sheet), NULL);

  return gtk_range_get_adjustment (GTK_RANGE (sheet->vscrollbar));
}

GtkAdjustment *
gtk_sheet_get_hadjustment (GtkSheet * sheet)
{
  g_return_val_if_fail (sheet != NULL, NULL);
  g_return_val_if_fail (GTK_IS_SHEET (sheet), NULL);

  return gtk_range_get_adjustment (GTK_RANGE (sheet->hscrollbar));
}


void
gtk_sheet_set_policy (GtkSheet * sheet,
		      GtkPolicyType vscrollbar_policy,
		      GtkPolicyType hscrollbar_policy)
{
  g_return_if_fail (sheet != NULL);
  g_return_if_fail (GTK_IS_SHEET (sheet));

  if (sheet->vscrollbar_policy != vscrollbar_policy)
    {
      sheet->vscrollbar_policy = vscrollbar_policy;
      if(GTK_WIDGET(sheet)->parent)
          gtk_widget_queue_resize(GTK_WIDGET(sheet));
    }
  if (sheet->hscrollbar_policy != hscrollbar_policy)
    {
      sheet->hscrollbar_policy = hscrollbar_policy;
      if(GTK_WIDGET(sheet)->parent);
          gtk_widget_queue_resize(GTK_WIDGET(sheet));
    }
}

void
gtk_sheet_set_update_policy (GtkSheet * sheet,
		             GtkUpdateType vscrollbar_policy,
		             GtkUpdateType hscrollbar_policy)
{
  g_return_if_fail (sheet != NULL);
  g_return_if_fail (GTK_IS_SHEET (sheet));

  if (sheet->vscrollbar_policy != GTK_RANGE(sheet->vscrollbar)->policy)
      GTK_RANGE(sheet->vscrollbar)->policy = vscrollbar_policy;
    
  if (sheet->hscrollbar_policy != GTK_RANGE(sheet->hscrollbar)->policy)
      GTK_RANGE(sheet->hscrollbar)->policy = hscrollbar_policy;
}

static void
gtk_sheet_destroy (GtkObject * object)
{
  gint i;
  GtkSheet *sheet;

  g_return_if_fail (object != NULL);
  g_return_if_fail (GTK_IS_SHEET (object));

  sheet = GTK_SHEET (object);

  /* get rid of all the rows */
/*  gtk_sheet_clear (sheet);
*/

  /* destroy the scrollbars */
  gtk_widget_destroy (sheet->vscrollbar);
  gtk_widget_destroy (sheet->hscrollbar);

  /* destroy the entry */
  gtk_widget_destroy (sheet->sheet_entry);

  if (GTK_OBJECT_CLASS (parent_class)->destroy)
    (*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

static void
gtk_sheet_realize (GtkWidget * widget)
{
  GtkSheet *sheet;
  GdkWindowAttr attributes;
  GtkAllocation allocation;
  gint attributes_mask;
  GdkGCValues values, auxvalues;
  gchar *name;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_SHEET (widget));

  sheet = GTK_SHEET (widget);

  GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);

  attributes.window_type = GDK_WINDOW_CHILD;
  attributes.x = widget->allocation.x;
  attributes.y = widget->allocation.y;
  attributes.width = widget->allocation.width;
  attributes.height = widget->allocation.height;
  attributes.wclass = GDK_INPUT_OUTPUT;

  attributes.visual = gtk_widget_get_visual (widget);
  attributes.colormap = gtk_widget_get_colormap (widget);

  attributes.event_mask = gtk_widget_get_events (widget);
  attributes.event_mask |= (GDK_EXPOSURE_MASK |
			    GDK_BUTTON_PRESS_MASK |
			    GDK_BUTTON_RELEASE_MASK |
			    GDK_KEY_PRESS_MASK |
			    GDK_POINTER_MOTION_MASK |
			    GDK_POINTER_MOTION_HINT_MASK);
  attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP |
                    GDK_WA_CURSOR;

  attributes.cursor = gdk_cursor_new(GDK_TOP_LEFT_ARROW);

  /* main window */
  widget->window = gdk_window_new (gtk_widget_get_parent_window (widget), &attributes, attributes_mask);

  gdk_window_set_user_data (widget->window, sheet);

  widget->style = gtk_style_attach (widget->style, widget->window);

  gtk_style_set_background (widget->style, widget->window, GTK_STATE_NORMAL);

  /* column-title window */
  sheet->column_title_window = gdk_window_new (widget->window, &attributes, attributes_mask);
  gdk_window_set_user_data (sheet->column_title_window, sheet);

  gtk_style_set_background (widget->style, sheet->column_title_window, GTK_STATE_NORMAL);
  gdk_window_show (sheet->column_title_window);

  /* row-title window */
  sheet->row_title_window = gdk_window_new (widget->window, &attributes, attributes_mask);
  gdk_window_set_user_data (sheet->row_title_window, sheet);

  gtk_style_set_background (widget->style, sheet->row_title_window, GTK_STATE_NORMAL);
  gdk_window_show (sheet->row_title_window);

  /* sheet-window */
  attributes.cursor = gdk_cursor_new(GDK_PLUS);

  sheet->sheet_window = gdk_window_new (widget->window, &attributes, attributes_mask);
  gdk_window_set_user_data (sheet->sheet_window, sheet);

  gdk_window_set_background (sheet->sheet_window, &widget->style->white);
  gdk_window_show (sheet->sheet_window);
  gdk_window_get_size (sheet->sheet_window, &sheet->sheet_window_width,
		       &sheet->sheet_window_height);

  size_allocate_row_title_buttons(sheet);
  size_allocate_column_title_buttons(sheet);

  /* backing_pixmap */
  gtk_sheet_make_backing_pixmap(sheet, 0, 0);  

  /* GCs */
  sheet->fg_gc = gdk_gc_new (widget->window);
  sheet->bg_gc = gdk_gc_new (widget->window);

  gdk_color_white(widget->style->colormap, &widget->style->white);
  gdk_color_black(widget->style->colormap, &widget->style->black);

  gdk_gc_get_values(sheet->fg_gc, &auxvalues);

/*  values.foreground = widget->style->white;
*/
  values.foreground=auxvalues.background;
  values.function = GDK_XOR;
  values.subwindow_mode = GDK_INCLUDE_INFERIORS;
  sheet->xor_gc = gdk_gc_new_with_values (widget->window,
					  &values,
					  GDK_GC_FOREGROUND |
					  GDK_GC_FUNCTION |
					  GDK_GC_SUBWINDOW);

  gtk_widget_set_parent_window (sheet->button, sheet->sheet_window);
  gtk_widget_set_parent_window (sheet->sheet_entry, sheet->sheet_window);

  name = g_strdup(sheet->name);
  gtk_sheet_set_title(sheet, name);

  gtk_sheet_activate_cell(sheet, sheet->active_cell.row, 
                                 sheet->active_cell.col);

  

}

static void
create_global_button(GtkSheet *sheet)
{
   sheet->button=gtk_button_new();
   gtk_widget_set_parent (sheet->button, GTK_WIDGET(sheet));
   gtk_widget_size_request(sheet->button, &sheet->button->requisition);
   gtk_widget_set_parent_window (sheet->button, sheet->sheet_window);
   gtk_signal_connect (GTK_OBJECT (sheet->button),
		      "pressed",
		      (GtkSignalFunc) global_button_clicked,
		      (gpointer) sheet);
 
}
static void
size_allocate_global_button(GtkSheet *sheet)
{
  GtkAllocation allocation;

  allocation.x=0;
  allocation.y=0;
  allocation.width=DEFAULT_COLUMN_WIDTH;
  allocation.height=DEFAULT_ROW_HEIGHT(GTK_WIDGET(sheet));

  gtk_widget_size_allocate(sheet->button, &allocation);
  gtk_widget_queue_draw(sheet->button);
  gtk_widget_show(sheet->button);
}

static void
global_button_clicked(GtkWidget *widget, gpointer data)
{
 gtk_sheet_click_cell(GTK_SHEET(data), -1, -1);
 gtk_widget_grab_focus(GTK_WIDGET(data));
}


static void
gtk_sheet_unrealize (GtkWidget * widget)
{
  gint i;
  GtkSheet *sheet;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_SHEET (widget));

  sheet = GTK_SHEET (widget);
  GTK_WIDGET_UNSET_FLAGS (widget, GTK_REALIZED | GTK_MAPPED);

  gdk_cursor_destroy (sheet->cursor_drag);
  gdk_gc_destroy (sheet->xor_gc);
  gdk_gc_destroy (sheet->fg_gc);
  gdk_gc_destroy (sheet->bg_gc);

  gtk_style_detach (widget->style);

  gdk_window_destroy (sheet->sheet_window);
  gdk_window_destroy (sheet->column_title_window);
  gdk_window_destroy (sheet->row_title_window);
  gdk_window_set_user_data (widget->window, NULL);
  gdk_window_destroy (widget->window);

  if (sheet->pixmap)
    gdk_pixmap_unref (sheet->pixmap);

  widget->window = NULL;
  sheet->column_title_window=NULL;
  sheet->sheet_window = NULL;
  sheet->cursor_drag = NULL;
  sheet->xor_gc = NULL;
  sheet->fg_gc = NULL;
  sheet->bg_gc = NULL;

}

static void
gtk_sheet_map (GtkWidget * widget)
{
  GtkSheet *sheet;
  gint i;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_SHEET (widget));

  sheet = GTK_SHEET (widget);

  if (!GTK_WIDGET_MAPPED (widget))
    {
      GTK_WIDGET_SET_FLAGS (widget, GTK_MAPPED);

      gdk_window_show (widget->window);
      gdk_window_show (sheet->column_title_window);
      gdk_window_show (sheet->row_title_window);
      gdk_window_show (sheet->sheet_window);

      /* map vscrollbars */
      if (GTK_WIDGET_VISIBLE (sheet->vscrollbar) &&
	  !GTK_WIDGET_MAPPED (sheet->vscrollbar))
	gtk_widget_map (sheet->vscrollbar);

      if (GTK_WIDGET_VISIBLE (sheet->hscrollbar) &&
	  !GTK_WIDGET_MAPPED (sheet->hscrollbar))
	gtk_widget_map (sheet->hscrollbar);



      if (GTK_WIDGET_VISIBLE (sheet->sheet_entry) &&
	  !GTK_WIDGET_MAPPED (sheet->sheet_entry)){
	gtk_widget_map (sheet->sheet_entry);
        gdk_window_hide(sheet->sheet_entry->window);
      }

      if (GTK_WIDGET_VISIBLE (sheet->button) &&
	  !GTK_WIDGET_MAPPED (sheet->button))
	gtk_widget_map (sheet->button);

    }
}

static void
gtk_sheet_unmap (GtkWidget * widget)
{
  GtkSheet *sheet;
  gint i;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_SHEET (widget));

  sheet = GTK_SHEET (widget);

  if (GTK_WIDGET_MAPPED (widget))
    {
      GTK_WIDGET_UNSET_FLAGS (widget, GTK_MAPPED);

      gdk_window_hide (sheet->sheet_window);
      gdk_window_hide (sheet->column_title_window);
      gdk_window_hide (sheet->row_title_window);
      gdk_window_hide (widget->window);

      /* unmap scrollbars */
      if (GTK_WIDGET_MAPPED (sheet->vscrollbar))
	gtk_widget_unmap (sheet->vscrollbar);

      if (GTK_WIDGET_MAPPED (sheet->hscrollbar))
	gtk_widget_unmap (sheet->hscrollbar);

      if (GTK_WIDGET_MAPPED (sheet->sheet_entry))
	gtk_widget_unmap (sheet->sheet_entry);

      if (GTK_WIDGET_MAPPED (sheet->button))
	gtk_widget_unmap (sheet->button);

    }
}


void
gtk_sheet_cell_draw_default (GtkSheet *sheet, gint row, gint col)
{
  GtkWidget *widget;
  GdkGC *fg_gc, *bg_gc;
  GtkSheetCellAttr attributes;
  GdkRectangle area;

  g_return_if_fail (sheet != NULL);

  /* bail now if we arn't drawable yet */
  if (!GTK_WIDGET_DRAWABLE (sheet)) return;

  if (row < 0 || row > sheet->maxrow) return;
  if (col < 0 || col > sheet->maxcol) return;
  if (!sheet->column[col].is_visible) return;
  if (!sheet->row[row].is_visible) return;

  widget = GTK_WIDGET (sheet);

  gtk_sheet_get_attributes(sheet, row, col, &attributes);

  /* select GC for background rectangle */
  gdk_gc_set_foreground (sheet->fg_gc, &attributes.foreground);
  gdk_gc_set_foreground (sheet->bg_gc, &attributes.background);

  fg_gc = sheet->fg_gc;
  bg_gc = sheet->bg_gc;

  area.x=COLUMN_LEFT_XPIXEL(sheet,col);
  area.y=ROW_TOP_YPIXEL(sheet,row);
  area.width=sheet->column[col].width;
  area.height=sheet->row[row].height;

  gdk_draw_rectangle (sheet->pixmap,
  	              bg_gc,
	              TRUE,
	              area.x,area.y,
	              area.width,area.height);

  gdk_gc_set_line_attributes (sheet->fg_gc, 1, 0, 0, 0);

  if(attributes.border.width == 0)
     if(attributes.background.pixel == GTK_WIDGET(sheet)->style->white.pixel)
        gdk_draw_rectangle (sheet->pixmap,
  	                    GTK_WIDGET(sheet)->style->bg_gc[GTK_STATE_NORMAL],
	                    FALSE,
	                    area.x, area.y,
	                    area.width, area.height);
}

void
gtk_sheet_cell_draw_border (GtkSheet *sheet, gint row, gint col, gint mask)
{
  GtkWidget *widget;
  GdkGC *fg_gc, *bg_gc;
  GtkSheetCellAttr attributes;
  GdkRectangle area;
  gint width;

  g_return_if_fail (sheet != NULL);

  /* bail now if we arn't drawable yet */
  if (!GTK_WIDGET_DRAWABLE (sheet)) return;

  if (row < 0 || row > sheet->maxrow) return;
  if (col < 0 || col > sheet->maxcol) return;
  if (!sheet->column[col].is_visible) return;
  if (!sheet->row[row].is_visible) return;

  widget = GTK_WIDGET (sheet);

  gtk_sheet_get_attributes(sheet, row, col, &attributes);

  /* select GC for background rectangle */
  gdk_gc_set_foreground (sheet->fg_gc, &attributes.border.color);
  gdk_gc_set_foreground (sheet->bg_gc, &attributes.background);

  fg_gc = sheet->fg_gc;
  bg_gc = sheet->bg_gc;

  area.x=COLUMN_LEFT_XPIXEL(sheet,col);
  area.y=ROW_TOP_YPIXEL(sheet,row);
  area.width=sheet->column[col].width;
  area.height=sheet->row[row].height;

  width = attributes.border.width;
  gdk_gc_set_line_attributes(sheet->fg_gc, attributes.border.width,
                                           attributes.border.line_style,
                                           attributes.border.cap_style,
                                           attributes.border.join_style);

  if(width>0){
   if(attributes.border.mask & CELL_LEFT_BORDER & mask)
      gdk_draw_line(sheet->pixmap, sheet->fg_gc,
                    area.x, area.y,
                    area.x, area.y+area.height);

   if(attributes.border.mask & CELL_RIGHT_BORDER & mask)
      gdk_draw_line(sheet->pixmap, sheet->fg_gc,
                    area.x+area.width, area.y,
                    area.x+area.width, 
                    area.y+area.height);

   if(attributes.border.mask & CELL_TOP_BORDER & mask)
      gdk_draw_line(sheet->pixmap, sheet->fg_gc,
                    area.x,area.y,
                    area.x+area.width, 
                    area.y);

   if(attributes.border.mask & CELL_BOTTOM_BORDER & mask)
      gdk_draw_line(sheet->pixmap, sheet->fg_gc,
                    area.x, area.y+area.height,
                    area.x+area.width, 
                    area.y+area.height);

  }

}


void
gtk_sheet_cell_draw_label (GtkSheet *sheet, gint row, gint col)
{
  GtkWidget *widget;
  GdkRectangle area;
  gint i;
  gint text_width, text_height, y;
  gint xoffset=1;  /* cursor space */
  int space, size, sizel, sizer;
  GdkGC *fg_gc, *bg_gc;
  GtkSheetCellAttr attributes;
  char *label;

  g_return_if_fail (sheet != NULL);

   /* bail now if we arn't drawable yet */
   if (!GTK_WIDGET_DRAWABLE (sheet))
    return;

  if (row > sheet->maxallocrow) return;
  if (col > sheet->maxalloccol) return;
  if (!sheet->data[row][col]) return;
  if (!sheet->data[row][col]->text || strlen(sheet->data[row][col]->text)==0)
      return;

  if (row < 0 || row > sheet->maxrow) return;
  if (col < 0 || col > sheet->maxcol) return;
  if (!sheet->column[col].is_visible) return;
  if (!sheet->row[row].is_visible) return;


  widget = GTK_WIDGET(sheet);

  label = sheet->data[row][col]->text;

  gtk_sheet_get_attributes(sheet, row, col, &attributes);

  /* select GC for background rectangle */
  gdk_gc_set_foreground (sheet->fg_gc, &attributes.foreground);
  gdk_gc_set_foreground (sheet->bg_gc, &attributes.background);
  gdk_gc_set_font(sheet->fg_gc, attributes.font);

  fg_gc = sheet->fg_gc;
  bg_gc = sheet->bg_gc;

  area.x=COLUMN_LEFT_XPIXEL(sheet,col);
  area.y=ROW_TOP_YPIXEL(sheet,row);
  area.width=sheet->column[col].width;
  area.height=sheet->row[row].height;

  text_width = gdk_string_width (attributes.font, label)+4;
  text_height = attributes.font->ascent + attributes.font->descent;
  y = area.y + area.height - CELLOFFSET - 1;
  y = y - text_height + attributes.font->ascent;

  switch(attributes.justification){
     case GTK_JUSTIFY_LEFT:
          if(!GTK_SHEET_ADJUST_TEXT(sheet)){          
           size=area.width;
           for(i=col+1; i<=MAX_VISIBLE_COLUMN(sheet); i++){
             if(gtk_sheet_cell_get_text(sheet, row, i)) break;
             if(size>=text_width) break;
             size+=sheet->column[i].width;
             sheet->column[i].left_text_column = MIN(col, sheet->column[i].left_text_column);
           }
           area.width=size;
          }
          xoffset += attributes.border.width/2;
          break;
     case GTK_JUSTIFY_RIGHT:
          if(!GTK_SHEET_ADJUST_TEXT(sheet)){          
           size=area.width;
           area.x+=area.width;
           for(i=col-1; i>=MIN_VISIBLE_COLUMN(sheet); i--){
             if(gtk_sheet_cell_get_text(sheet, row, i)) break;
             if(size>=text_width) break;
             size+=sheet->column[i].width;
             sheet->column[i].right_text_column = MAX(col, sheet->column[i].right_text_column);
           }
           area.width=size;
          }
          area.x-=size;
          xoffset+=area.width-text_width - 2 * CELLOFFSET +
                   attributes.border.width/2+4;
          break;
     case GTK_JUSTIFY_CENTER:
          if(!GTK_SHEET_ADJUST_TEXT(sheet)){          
           sizel=area.width/2;
           sizer=area.width/2;
	   area.x+=area.width/2;
           for(i=col+1; i<=MAX_VISIBLE_COLUMN(sheet); i++){
             if(gtk_sheet_cell_get_text(sheet, row, i)) break;
             if(sizer>=text_width/2) break;
             sizer+=sheet->column[i].width;
             sheet->column[i].left_text_column = MIN(col, sheet->column[i].left_text_column);
           }
           for(i=col-1; i>=MIN_VISIBLE_COLUMN(sheet); i--){
             if(gtk_sheet_cell_get_text(sheet, row, i)) break;
             if(sizel>=text_width/2) break;
             sizel+=sheet->column[i].width;
             sheet->column[i].right_text_column = MAX(col, sheet->column[i].right_text_column);
           }
           size=MIN(sizel, sizer);
          }
	  area.x-=sizel;
          xoffset+= sizel - text_width/2 - CELLOFFSET+2;
	  area.width=sizel+sizer;
          break;
   }

  gdk_gc_set_clip_rectangle(fg_gc, &area);

  gdk_draw_string (sheet->pixmap, 
		   attributes.font,
		   fg_gc,
                   area.x+xoffset+CELLOFFSET,
		   y,
	      	   label);

  gdk_gc_set_clip_rectangle(fg_gc, NULL);

}

void
gtk_sheet_draw_cell(GtkSheet *sheet, gint row, gint column)
{
  GtkSheetRange range;
  range.col0=column;
  range.coli=column;
  range.row0=row;
  range.rowi=row;
  gtk_sheet_range_draw(sheet, &range);
}

void
gtk_sheet_range_draw(GtkSheet *sheet, GtkSheetRange *range)
{
 int i,j;
 int state;
 GtkSheetRange drawing_range;
 GdkRectangle area;

 g_return_if_fail(sheet != NULL);
 g_return_if_fail(GTK_SHEET(sheet));
 
 if(!GTK_WIDGET_DRAWABLE(GTK_WIDGET(sheet))) return;
 if(!GTK_WIDGET_REALIZED(GTK_WIDGET(sheet))) return;

 if(range == (GtkSheetRange *)NULL)
 {
   drawing_range.row0=MIN_VISIBLE_ROW(sheet);
   drawing_range.col0=MIN_VISIBLE_COLUMN(sheet);
   drawing_range.rowi=MAX_VISIBLE_ROW(sheet);
   drawing_range.coli=MAX_VISIBLE_COLUMN(sheet);
   gdk_draw_rectangle (sheet->pixmap,
	               GTK_WIDGET(sheet)->style->white_gc,
	               TRUE,
	               0,0,
	               sheet->sheet_window_width,sheet->sheet_window_height);

 }
 else
 {
   drawing_range.row0=MAX(range->row0, MIN_VISIBLE_ROW(sheet));
   drawing_range.col0=MAX(range->col0, MIN_VISIBLE_COLUMN(sheet));
   drawing_range.rowi=MIN(range->rowi, MAX_VISIBLE_ROW(sheet));
   drawing_range.coli=MIN(range->coli, MAX_VISIBLE_COLUMN(sheet));
 }

 if(drawing_range.coli == sheet->maxcol){
  area.x=COLUMN_LEFT_XPIXEL(sheet,sheet->maxcol)+
         sheet->column[sheet->maxcol].width+1;
  area.y=0;

  gdk_draw_rectangle (sheet->pixmap,
	              GTK_WIDGET(sheet)->style->white_gc,
	              TRUE,
	              area.x,area.y,
	              sheet->row_title_area.width,sheet->sheet_window_height);
              
  gdk_draw_pixmap(sheet->sheet_window,
                  GTK_WIDGET(sheet)->style->fg_gc[GTK_STATE_NORMAL],
                  sheet->pixmap,
                  area.x,
                  area.y,
                  area.x,
                  area.y,
                  sheet->row_title_area.width,
                  sheet->sheet_window_height);                  
 }
 if(drawing_range.rowi == sheet->maxrow){
  area.x=0;
  area.y=ROW_TOP_YPIXEL(sheet,sheet->maxrow)+sheet->row[sheet->maxrow].height+1;

  gdk_draw_rectangle (sheet->pixmap,
	              GTK_WIDGET(sheet)->style->white_gc,
	              TRUE,
	              area.x,area.y,
	              sheet->sheet_window_width,
                      sheet->column_title_area.height);

  gdk_draw_pixmap(sheet->sheet_window,
                  GTK_WIDGET(sheet)->style->fg_gc[GTK_STATE_NORMAL],
                  sheet->pixmap,
                  area.x,
                  area.y,
                  area.x,
                  area.y,
                  sheet->sheet_window_width,
                  sheet->column_title_area.height);      
 }

 for(i=drawing_range.row0; i<=drawing_range.rowi; i++)
  for(j=drawing_range.col0; j<=drawing_range.coli; j++){
     gtk_sheet_cell_draw_default(sheet, i, j);
  }

 for(i=drawing_range.row0; i<=drawing_range.rowi; i++)
  for(j=drawing_range.col0; j<=drawing_range.coli; j++){
     gtk_sheet_cell_draw_border(sheet, i-1, j, CELL_BOTTOM_BORDER);
     gtk_sheet_cell_draw_border(sheet, i+1, j, CELL_TOP_BORDER);
     gtk_sheet_cell_draw_border(sheet, i, j-1, CELL_RIGHT_BORDER);
     gtk_sheet_cell_draw_border(sheet, i, j+1, CELL_LEFT_BORDER);
     gtk_sheet_cell_draw_border(sheet, i, j, 15);
  }

 for(i=drawing_range.row0; i<=drawing_range.rowi; i++)
  for(j=drawing_range.col0; j<=drawing_range.coli; j++){
     if(i<=sheet->maxallocrow && j<=sheet->maxalloccol && sheet->data[i][j])
       gtk_sheet_cell_draw_label (sheet, i, j);
     
  }

 for(i=drawing_range.row0; i<=drawing_range.rowi; i++)
  for(j=sheet->column[drawing_range.col0].left_text_column; j<drawing_range.col0; j++){
     if(i<=sheet->maxallocrow && j<=sheet->maxalloccol && sheet->data[i][j])
       gtk_sheet_cell_draw_label (sheet, i, j);
  }  
 for(i=drawing_range.row0; i<=drawing_range.rowi; i++)
  for(j=drawing_range.coli+1; j<=sheet->column[drawing_range.coli].right_text_column; j++){
     if(i<=sheet->maxallocrow && j<=sheet->maxalloccol && sheet->data[i][j])
       gtk_sheet_cell_draw_label (sheet, i, j);
  }  



  gtk_sheet_draw_backing_pixmap(sheet, drawing_range);


  if(sheet->state != GTK_SHEET_NORMAL && gtk_sheet_range_isvisible(sheet, sheet->range)){
      gtk_sheet_range_draw_selection(sheet, drawing_range);
  }
 
  if(sheet->state == GTK_SHEET_NORMAL && gtk_sheet_range_isvisible(sheet, sheet->range)){ 
      if(sheet->active_cell.row>=drawing_range.row0 &&
         sheet->active_cell.row<=drawing_range.rowi &&
         sheet->active_cell.col>=drawing_range.col0 &&
         sheet->active_cell.col<=drawing_range.coli)
                   gtk_sheet_show_active_cell(sheet);
  } 

}

void
gtk_sheet_range_draw_selection(GtkSheet *sheet, GtkSheetRange range)
{
  GdkRectangle area;
  gint i,j;
  GtkSheetRange aux;

  if(range.col0 > sheet->range.coli || range.coli < sheet->range.col0 ||
     range.row0 > sheet->range.rowi || range.rowi < sheet->range.row0)
     return;

  if(!gtk_sheet_range_isvisible(sheet, range)) return;
  if(!GTK_WIDGET_REALIZED(GTK_WIDGET(sheet))) return;

  aux=range;

  range.col0=MAX(sheet->range.col0, range.col0);
  range.coli=MIN(sheet->range.coli, range.coli);
  range.row0=MAX(sheet->range.row0, range.row0);
  range.rowi=MIN(sheet->range.rowi, range.rowi);

  range.col0=MAX(range.col0, MIN_VISIBLE_COLUMN(sheet));
  range.coli=MIN(range.coli, MAX_VISIBLE_COLUMN(sheet));
  range.row0=MAX(range.row0, MIN_VISIBLE_ROW(sheet));
  range.rowi=MIN(range.rowi, MAX_VISIBLE_ROW(sheet));

  for(i=range.row0; i<=range.rowi; i++){
   for(j=range.col0; j<=range.coli; j++){

    if(gtk_sheet_cell_get_state(sheet, i, j)==GTK_STATE_SELECTED && 
       sheet->column[j].is_visible && sheet->row[i].is_visible){

      area.x=COLUMN_LEFT_XPIXEL(sheet,j);
      area.y=ROW_TOP_YPIXEL(sheet,i);
      area.width=sheet->column[j].width;
      area.height=sheet->row[i].height;

      if(i==sheet->range.row0){
            area.y=area.y+2;
            area.height=area.height-2;
      }
      if(i==sheet->range.rowi) area.height=area.height-3;
      if(j==sheet->range.col0){
            area.x=area.x+2;
            area.width=area.width-2;
      }
      if(j==sheet->range.coli) area.width=area.width-3;

      if(i!=sheet->active_cell.row || j!=sheet->active_cell.col){
       gdk_draw_rectangle (sheet->sheet_window,
  	                   sheet->xor_gc,
	   	           TRUE,
	                   area.x+1,area.y+1,
	                   area.width,area.height);
      }
    }

   }
  }

  gtk_sheet_draw_border(sheet, sheet->range);

}

static void
gtk_sheet_draw_backing_pixmap(GtkSheet *sheet, GtkSheetRange range)
{
  gint x,y,width,height;
 
  if(!GTK_WIDGET_REALIZED(GTK_WIDGET(sheet))) return;
 
  x=COLUMN_LEFT_XPIXEL(sheet,range.col0);
  y=ROW_TOP_YPIXEL(sheet, range.row0);  
  width=COLUMN_LEFT_XPIXEL(sheet, range.coli)-x+sheet->column[range.coli].width;
  height=ROW_TOP_YPIXEL(sheet, range.rowi)-y+sheet->row[range.rowi].height;

  if(range.row0==sheet->range.row0){
          y=y-3;
          height=height+3;
  }
  if(range.rowi==sheet->range.rowi) height=height+3;
  if(range.col0==sheet->range.col0){
            x=x-3;
            width=width+3;
  }
  if(range.coli==sheet->range.coli) width=width+3;

  
  x=MAX(x, 0);
  y=MAX(y, 0);
  width=MIN(width, sheet->sheet_window_width-x);
  height=MIN(height, sheet->sheet_window_height-y);

  if(range.coli==sheet->maxcol) width=sheet->sheet_window_width-x;
  if(range.rowi==sheet->maxrow) height=sheet->sheet_window_height-y;

  gdk_draw_pixmap(sheet->sheet_window,
                  GTK_WIDGET(sheet)->style->fg_gc[GTK_STATE_NORMAL],
                  sheet->pixmap,
                  x,
                  y,
                  x,
                  y,
                  width+1,
                  height+1);                  
      
}

static GtkSheetCell *
gtk_sheet_cell_new()
{
 GtkSheetCell *cell;
 cell=(GtkSheetCell *)malloc(sizeof(GtkSheetCell));
 cell->text=NULL;
 return cell;
}

void 
gtk_sheet_set_cell(GtkSheet *sheet, int row, int col, int justification,
                   gchar *text)
{
 GtkSheetCell **cell;
 GtkSheetRange range;
 gint text_width;
 GtkSheetCellAttr attributes;
 GdkColor color;
 gint i,j;

 CheckBounds(sheet, row, col);

 cell=&sheet->data[row][col];

 if(*cell==NULL){
  gtk_sheet_get_attributes(sheet, row, col, &attributes);
  (*cell) = gtk_sheet_cell_new();
  (*cell)->attributes = attributes;
 }
 else
  attributes=(*cell)->attributes;

 (*cell)->row=row;
 (*cell)->col=col;
 (*cell)->attributes.justification = justification;

 if((*cell)->text){
    g_free((*cell)->text);
 }

 (*cell)->text=g_strdup(text);

 if(attributes.is_visible){

   text_width = gdk_string_width (attributes.font, (*cell)->text);

   range.row0 = row;
   range.rowi = row;
   range.col0 = sheet->view.col0;
   range.coli = sheet->view.coli;

   if(GTK_SHEET_AUTORESIZE(sheet) && text_width > sheet->column[col].width-2*CELLOFFSET-(*cell)->attributes.border.width)
      gtk_sheet_set_column_width(sheet, col, text_width+2*CELLOFFSET+(*cell)->attributes.border.width);
   else
     if(!GTK_SHEET_IS_FROZEN(sheet))
      gtk_sheet_range_draw(sheet, &range);
     
 }
     
 gtk_signal_emit(GTK_OBJECT(sheet),sheet_signals[CHANGED], row, col);

}

void
gtk_sheet_cell_clear (GtkSheet *sheet, gint row, gint column)
{
  gchar *text;
  GtkSheetRange range;

  if(GTK_SHEET_IS_LOCKED(sheet)) return;

  text=gtk_sheet_cell_get_text(sheet, row, column); 
  if(!text) return;

  g_free(text);
  g_free(sheet->data[row][column]);

  sheet->data[row][column]=NULL;

  range.row0 = row;
  range.rowi = row;
  range.col0 = sheet->view.col0;
  range.coli = sheet->view.coli;

  if(!GTK_SHEET_IS_FROZEN(sheet)){
    gtk_sheet_range_draw(sheet, &range);
  }

  gtk_signal_emit(GTK_OBJECT(sheet),sheet_signals[CHANGED], row, column);
}
    
void
gtk_sheet_range_clear (GtkSheet *sheet, GtkSheetRange *range)
{
  gint i, j;
  gchar *text;
  GtkSheetRange clear;

  if(GTK_SHEET_IS_LOCKED(sheet)) return;

  if(!range){
    clear.row0=0;
    clear.rowi=sheet->maxallocrow;
    clear.col0=0;
    clear.coli=sheet->maxalloccol;
  }else
    clear=*range;  

  clear.row0=MAX(clear.row0, 0);
  clear.col0=MAX(clear.col0, 0);
  clear.rowi=MIN(clear.rowi, sheet->maxrow);
  clear.coli=MIN(clear.coli, sheet->maxcol);

  for(i=clear.row0; i<=clear.rowi; i++)
    for(j=clear.col0; j<=clear.coli; j++){
      text=gtk_sheet_cell_get_text(sheet, i, j); 
      if(text){
        g_free(text);
        g_free(sheet->data[i][j]);
        sheet->data[i][j]=NULL;
        gtk_signal_emit(GTK_OBJECT(sheet),sheet_signals[CHANGED], i, j);
      }
    }

  gtk_sheet_range_draw(sheet, NULL);
}


gchar *     
gtk_sheet_cell_get_text (GtkSheet *sheet, gint row, gint col)
{

  if(row > sheet->maxallocrow || col > sheet->maxalloccol) return NULL;
  if(!sheet->data[row][col]) return NULL;
  if(!sheet->data[row][col]->text) return NULL;
  if(strlen(sheet->data[row][col]->text) == 0) return NULL;

  return (sheet->data[row][col]->text);
}

int
gtk_sheet_cell_get_state (GtkSheet *sheet, int row, int col)
{
 int state=sheet->state;
 GtkSheetRange *range=&sheet->range;

 switch (state){
                case GTK_SHEET_NORMAL:
                     return GTK_STATE_NORMAL;
		     break;
		case GTK_SHEET_ROW_SELECTED:
                     if(row>=range->row0 && row<=range->rowi) 
                                        return GTK_STATE_SELECTED;
		     break;
                case GTK_SHEET_COLUMN_SELECTED:
                     if(col>=range->col0 && col<=range->coli) 
                                        return GTK_STATE_SELECTED;
		     break;
		case GTK_SHEET_RANGE_SELECTED:
                     if(row >= range->row0 && row <= range->rowi && \
                        col >= range->col0 && col <= range->coli)
                                        return GTK_STATE_SELECTED;
		     break;
 }
 return GTK_STATE_NORMAL;
}

gint
gtk_sheet_get_pixel_info (GtkSheet * sheet,
			  gint x,
			  gint y,
			  gint * row,
			  gint * column)
{
  gint trow, tcol;

  g_return_val_if_fail (sheet != NULL, 0);

  /* bounds checking, return false if the user clicked 
   * on a blank area */
  trow = ROW_FROM_YPIXEL (sheet, y);
  if (trow > sheet->maxrow)
    return FALSE;

  *row = trow;

  tcol = COLUMN_FROM_XPIXEL (sheet, x);
  if (tcol > sheet->maxcol)
    return FALSE;

 *column = tcol;

  return TRUE;
}

gint
gtk_sheet_get_cell_info (GtkSheet * sheet,
			  gint row,
                          gint column,
			  GdkRectangle *area)
{
  if(row > sheet->maxrow || column > sheet->maxcol) return FALSE;
  if(row < 0 || column < 0) return FALSE;

  area->x= COLUMN_LEFT_XPIXEL(sheet, column);
  area->y= ROW_TOP_YPIXEL(sheet, row);
  area->width=sheet->column[column].width;
  area->height=sheet->row[row].height;  

  return TRUE;
}

void 
gtk_sheet_set_active_cell (GtkSheet *sheet, gint row, gint column)
{
 if(row < 0 || column < 0) return;
 if(row > sheet->maxrow || column > sheet->maxcol) return;

 sheet->active_cell.row=row;
 sheet->active_cell.col=column;
  
 if(GTK_WIDGET_REALIZED(GTK_WIDGET(sheet))){
   gtk_sheet_click_cell(sheet, row, column);
   gtk_sheet_activate_cell(sheet, row, column);
 }
}

static void
gtk_sheet_deactivate_cell(GtkSheet *sheet)
{
 GtkSheetRange range;
 char *text;
 gint row,col;
 char label[MAX_TEXT_LENGTH+1];
 gint justification;
 GtkSheetCellAttr attributes;

 if(!GTK_WIDGET_DRAWABLE(GTK_WIDGET(sheet))) return;
 if(!GTK_WIDGET_REALIZED(GTK_WIDGET(sheet))) return;

 if(sheet->active_cell.row < 0 || sheet->active_cell.col < 0) return;
 row=sheet->active_cell.row;
 col=sheet->active_cell.col;

 text=gtk_shentry_get_text(GTK_SHENTRY(sheet->sheet_entry));

 gtk_sheet_get_attributes(sheet, row, col, &attributes); 
 justification=attributes.justification;

 if(text && strlen(text)!=0){
      gtk_sheet_set_cell(sheet, row, col, justification, text);
      gtk_signal_emit(GTK_OBJECT(sheet),sheet_signals[SET_CELL], row, col);
 }
 else
      gtk_sheet_cell_clear(sheet, row, col);      
 
 row=sheet->active_cell.row;
 col=sheet->active_cell.col;

 sheet->active_cell.row=-1;
 sheet->active_cell.col=-1;

 column_button_release(sheet, col);
 row_button_release(sheet, row);

 if(gtk_sheet_cell_isvisible(sheet, row, col)){
   gdk_draw_pixmap(sheet->sheet_window,
                   GTK_WIDGET(sheet)->style->fg_gc[GTK_STATE_NORMAL],
                   sheet->pixmap,
                   COLUMN_LEFT_XPIXEL(sheet,col)-1,
                   ROW_TOP_YPIXEL(sheet,row)-1,
                   COLUMN_LEFT_XPIXEL(sheet,col)-1,
                   ROW_TOP_YPIXEL(sheet,row)-1,
                   sheet->column[col].width+4,
                   sheet->row[row].height+4);   
   gdk_window_hide(sheet->sheet_entry->window);
 }

 gtk_shentry_set_text(GTK_SHENTRY(sheet->sheet_entry),"", GTK_JUSTIFY_LEFT);

 GTK_WIDGET_UNSET_FLAGS(sheet->sheet_entry, GTK_HAS_FOCUS);
 GTK_WIDGET_SET_FLAGS(GTK_WIDGET(sheet), GTK_HAS_FOCUS);
 gtk_widget_grab_focus(GTK_WIDGET(sheet));
}	

static void
gtk_sheet_activate_cell(GtkSheet *sheet, gint row, gint col)
{
 char text[MAX_TEXT_LENGTH+1]="";
 GtkSheetCell *cell;
 gint size;
 gint i;
 gint justification;
 GtkSheetCellAttr attributes;

 if(sheet->state != GTK_SHEET_NORMAL){
        sheet->state=GTK_SHEET_NORMAL;
        gtk_sheet_unselect_range(sheet, NULL);
 }
 sheet->range.row0=row;
 sheet->range.col0=col;
 sheet->range.rowi=row;
 sheet->range.coli=col;
 sheet->active_cell.row=row;
 sheet->active_cell.col=col;
 sheet->selection_cell.row=row;
 sheet->selection_cell.col=col;
 row_button_set(sheet, row);
 column_button_set(sheet, col); 

 if(!GTK_WIDGET_REALIZED(GTK_WIDGET(sheet))) return;
         
 gtk_sheet_get_attributes(sheet, row, col, &attributes); 

 justification=attributes.justification;

 if(row <= sheet->maxallocrow && col <= sheet->maxalloccol){
  if(sheet->data[row][col]!=NULL){
    cell=sheet->data[row][col];
    sprintf(text,"%s",cell->text);
  }
 }

 gtk_shentry_set_text(GTK_SHENTRY(sheet->sheet_entry), text,
                      justification);
 gtk_sheet_entry_set_max_size(sheet);
 if(GTK_SHEET_IS_LOCKED(sheet) || !attributes.is_editable) 
            gtk_shentry_set_editable(GTK_SHENTRY(sheet->sheet_entry), FALSE);
 else
            gtk_shentry_set_editable(GTK_SHENTRY(sheet->sheet_entry), TRUE);

 gtk_shentry_set_visibility(GTK_SHENTRY(sheet->sheet_entry), attributes.is_visible);

 gtk_sheet_show_active_cell(sheet);

 gtk_signal_emit(GTK_OBJECT(sheet),sheet_signals[ACTIVATE_CELL], row, col);
}

static void
gtk_sheet_show_active_cell(GtkSheet *sheet)
{
 if(!GTK_WIDGET_REALIZED(GTK_WIDGET(sheet))) return;

 if (gtk_sheet_cell_isvisible(sheet, sheet->active_cell.row, sheet->active_cell.col))
  {
      gtk_sheet_size_allocate_entry(sheet);
      if(GTK_WIDGET_REALIZED(sheet->sheet_entry)){
	  gtk_widget_queue_draw(sheet->sheet_entry);
          gdk_window_show(sheet->sheet_entry->window);
	  gtk_widget_grab_focus(sheet->sheet_entry);
      }
      gtk_sheet_draw_active_cell(sheet);
      GTK_WIDGET_SET_FLAGS(sheet->sheet_entry, GTK_HAS_FOCUS);
      GTK_SHEET_UNSET_FLAGS(sheet, SHEET_IN_SELECTION);
  }
}

static void
gtk_sheet_draw_active_cell(GtkSheet *sheet)
{
    gint row, col;

    if(!GTK_WIDGET_DRAWABLE(GTK_WIDGET(sheet))) return;
    if(!GTK_WIDGET_REALIZED(GTK_WIDGET(sheet))) return;

    row = sheet->active_cell.row;
    col = sheet->active_cell.col;

    if(!gtk_sheet_cell_isvisible(sheet, row, col)) return;
 
    row_button_set(sheet, row);
    column_button_set(sheet, col);

    gtk_sheet_draw_backing_pixmap(sheet, sheet->range);
    gtk_sheet_draw_border(sheet, sheet->range);

}


static void
gtk_sheet_make_backing_pixmap (GtkSheet *sheet, gint width, gint height)
{
  gint pixmap_width, pixmap_height;

  if(width == 0 && height == 0){
     width=sheet->sheet_window_width+80;
     height=sheet->sheet_window_height+80;
  }

  if (!sheet->pixmap)
    {
      /* allocate */
      sheet->pixmap = gdk_pixmap_new (sheet->sheet_window,
			              width, height,
				      -1);
    }
  else
    {
      /* reallocate if sizes don't match */
      gdk_window_get_size (sheet->pixmap,
			   &pixmap_width, &pixmap_height);
      if ((pixmap_width != width) || (pixmap_height != height))
	{
	  gdk_pixmap_unref (sheet->pixmap);
	  sheet->pixmap = gdk_pixmap_new (sheet->sheet_window,
					       width, height,
					       -1);
          gtk_sheet_range_draw(sheet, NULL);
	}
    }
}

static void
gtk_sheet_new_selection(GtkSheet *sheet, GtkSheetRange *range)
{
  gint i,j, mask1, mask2;
  gint state, selected;
  gint x,y,width,height;
  GtkSheetRange new_range, aux_range;
  GdkRectangle area;
  

  g_return_if_fail (sheet != NULL);

  if(range==NULL) range=&sheet->range;

  new_range=*range;

  range->row0=MIN(range->row0, sheet->range.row0);
  range->rowi=MAX(range->rowi, sheet->range.rowi);
  range->col0=MIN(range->col0, sheet->range.col0);
  range->coli=MAX(range->coli, sheet->range.coli);

  range->row0=MAX(range->row0, MIN_VISIBLE_ROW(sheet));
  range->rowi=MIN(range->rowi, MAX_VISIBLE_ROW(sheet));
  range->col0=MAX(range->col0, MIN_VISIBLE_COLUMN(sheet));
  range->coli=MIN(range->coli, MAX_VISIBLE_COLUMN(sheet));

  aux_range.row0=MAX(new_range.row0, MIN_VISIBLE_ROW(sheet));
  aux_range.rowi=MIN(new_range.rowi, MAX_VISIBLE_ROW(sheet));
  aux_range.col0=MAX(new_range.col0, MIN_VISIBLE_COLUMN(sheet));
  aux_range.coli=MIN(new_range.coli, MAX_VISIBLE_COLUMN(sheet));

  for(i=range->row0; i<=range->rowi; i++){
   for(j=range->col0; j<=range->coli; j++){     

    state=gtk_sheet_cell_get_state(sheet, i, j);
    selected=(i<=new_range.rowi && i>=new_range.row0 && 
        j<=new_range.coli && j>=new_range.col0) ? TRUE : FALSE;

    if(state==GTK_STATE_SELECTED && selected &&
       sheet->column[j].is_visible && sheet->row[i].is_visible &&
       (i==sheet->range.row0 || i==sheet->range.rowi ||
        j==sheet->range.col0 || j==sheet->range.coli ||
        i==new_range.row0 || i==new_range.rowi ||
        j==new_range.col0 || j==new_range.coli)){

       mask1 = i==sheet->range.row0 ? 1 : 0;
       mask1 = i==sheet->range.rowi ? mask1+2 : mask1;
       mask1 = j==sheet->range.col0 ? mask1+4 : mask1;
       mask1 = j==sheet->range.coli ? mask1+8 : mask1;

       mask2 = i==new_range.row0 ? 1 : 0;
       mask2 = i==new_range.rowi ? mask2+2 : mask2;
       mask2 = j==new_range.col0 ? mask2+4 : mask2;
       mask2 = j==new_range.coli ? mask2+8 : mask2;     

       if(mask1 != mask2){
         x=COLUMN_LEFT_XPIXEL(sheet,j);
         y=ROW_TOP_YPIXEL(sheet, i);  
         width=COLUMN_LEFT_XPIXEL(sheet, j)-x+sheet->column[j].width;
         height=ROW_TOP_YPIXEL(sheet, i)-y+sheet->row[i].height;

         if(i==sheet->range.row0){
            y=y-3;
            height=height+3;
         }
         if(i==sheet->range.rowi) height=height+3;
         if(j==sheet->range.col0){
            x=x-3;
            width=width+3;
         }
         if(j==sheet->range.coli) width=width+3;

         gdk_draw_pixmap(sheet->sheet_window,
                  GTK_WIDGET(sheet)->style->fg_gc[GTK_STATE_NORMAL],
                  sheet->pixmap,
                  x+1,
                  y+1,
                  x+1,
                  y+1,
                  width,
                  height);           

         if(i != sheet->active_cell.row || j != sheet->active_cell.col){
           x=COLUMN_LEFT_XPIXEL(sheet,j);
           y=ROW_TOP_YPIXEL(sheet, i);  
           width=COLUMN_LEFT_XPIXEL(sheet, j)-x+sheet->column[j].width;
           height=ROW_TOP_YPIXEL(sheet, i)-y+sheet->row[i].height;

           if(i==new_range.row0){
               y=y+2;
               height=height-2;
            }
            if(i==new_range.rowi) height=height-3;
            if(j==new_range.col0){
               x=x+2;
               width=width-2;
            }
            if(j==new_range.coli) width=width-3;

            gdk_draw_rectangle (sheet->sheet_window,
  	                   sheet->xor_gc,
	   	           TRUE,
	                   x+1,y+1,
	                   width,height);
          }
       }
    }
   }
  }

  for(i=range->row0; i<=range->rowi; i++){
   for(j=range->col0; j<=range->coli; j++){     

    state=gtk_sheet_cell_get_state(sheet, i, j);
    selected=(i<=new_range.rowi && i>=new_range.row0 && 
        j<=new_range.coli && j>=new_range.col0) ? TRUE : FALSE;

    if(state==GTK_STATE_SELECTED && !selected &&   
       sheet->column[j].is_visible && sheet->row[i].is_visible){

      x=COLUMN_LEFT_XPIXEL(sheet,j);
      y=ROW_TOP_YPIXEL(sheet, i);  
      width=COLUMN_LEFT_XPIXEL(sheet, j)-x+sheet->column[j].width;
      height=ROW_TOP_YPIXEL(sheet, i)-y+sheet->row[i].height;

      if(i==sheet->range.row0){
            y=y-3;
            height=height+3;
      }
      if(i==sheet->range.rowi) height=height+3;
      if(j==sheet->range.col0){
            x=x-3;
            width=width+3;
      }
      if(j==sheet->range.coli) width=width+3;

      gdk_draw_pixmap(sheet->sheet_window,
                  GTK_WIDGET(sheet)->style->fg_gc[GTK_STATE_NORMAL],
                  sheet->pixmap,
                  x+1,
                  y+1,
                  x+1,
                  y+1,
                  width,
                  height);           
    }
   }
  }

  for(i=range->row0; i<=range->rowi; i++){
   for(j=range->col0; j<=range->coli; j++){     

    state=gtk_sheet_cell_get_state(sheet, i, j);
    selected=(i<=new_range.rowi && i>=new_range.row0 && 
        j<=new_range.coli && j>=new_range.col0) ? TRUE : FALSE;

    if(state!=GTK_STATE_SELECTED && selected &&
       sheet->column[j].is_visible && sheet->row[i].is_visible &&
       (i != sheet->active_cell.row || j != sheet->active_cell.col)){

      x=COLUMN_LEFT_XPIXEL(sheet,j);
      y=ROW_TOP_YPIXEL(sheet, i);  
      width=COLUMN_LEFT_XPIXEL(sheet, j)-x+sheet->column[j].width;
      height=ROW_TOP_YPIXEL(sheet, i)-y+sheet->row[i].height;

      if(i==new_range.row0){
            y=y+2;
            height=height-2;
       }
       if(i==new_range.rowi) height=height-3;
       if(j==new_range.col0){
            x=x+2;
            width=width-2;
       }
       if(j==new_range.coli) width=width-3;

       gdk_draw_rectangle (sheet->sheet_window,
  	                   sheet->xor_gc,
	   	           TRUE,
	                   x+1,y+1,
	                   width,height);

    }   

   }
  }

  for(i=aux_range.row0; i<=aux_range.rowi; i++){
   for(j=aux_range.col0; j<=aux_range.coli; j++){     

    if(sheet->column[j].is_visible && sheet->row[i].is_visible){

       state=gtk_sheet_cell_get_state(sheet, i, j);

       mask1 = i==sheet->range.row0 ? 1 : 0;
       mask1 = i==sheet->range.rowi ? mask1+2 : mask1;
       mask1 = j==sheet->range.col0 ? mask1+4 : mask1;
       mask1 = j==sheet->range.coli ? mask1+8 : mask1;

       mask2 = i==new_range.row0 ? 1 : 0;
       mask2 = i==new_range.rowi ? mask2+2 : mask2;
       mask2 = j==new_range.col0 ? mask2+4 : mask2;
       mask2 = j==new_range.coli ? mask2+8 : mask2;    
       if(mask2!=mask1 || (mask2==mask1 && state!=GTK_STATE_SELECTED)){
         x=COLUMN_LEFT_XPIXEL(sheet,j);
         y=ROW_TOP_YPIXEL(sheet, i);  
         width=sheet->column[j].width;
         height=sheet->row[i].height;
         if(mask2 & 1)
               gdk_draw_rectangle (sheet->sheet_window,
  	                           sheet->xor_gc,
	   	                   TRUE,
	                           x+1,y-1,
	                           width,3);

           
         if(mask2 & 2)
               gdk_draw_rectangle (sheet->sheet_window,
  	                           sheet->xor_gc,
	   	                   TRUE,
	                           x+1,y+height-1,
	                           width,3);

         if(mask2 & 4)
               gdk_draw_rectangle (sheet->sheet_window,
  	                           sheet->xor_gc,
	   	                   TRUE,
	                           x-1,y+1,
	                           3,height);


         if(mask2 & 8)
               gdk_draw_rectangle (sheet->sheet_window,
  	                           sheet->xor_gc,
	   	                   TRUE,
	                           x+width-1,y+1,
	                           3,height);

       

       }         

    } 

   }
  } 


  *range=new_range;
  gtk_sheet_draw_corners(sheet, new_range);

}

static void
gtk_sheet_draw_border (GtkSheet *sheet, GtkSheetRange new_range)
{
  GdkRectangle area;
  gint i;
  gint x,y,width,height;

  x=COLUMN_LEFT_XPIXEL(sheet,new_range.col0);
  y=ROW_TOP_YPIXEL(sheet,new_range.row0);
  width=COLUMN_LEFT_XPIXEL(sheet,new_range.coli)-x+ 
             sheet->column[new_range.coli].width;
  height=ROW_TOP_YPIXEL(sheet,new_range.rowi)-y+
             sheet->row[new_range.rowi].height;

  area.x=sheet->row_title_area.width;
  area.y=sheet->column_title_area.height;
  area.width=sheet->sheet_window_width;
  area.height=sheet->sheet_window_height;

  if(x<0) {
      width=width+x;
      x=0;
  }
  if(width>area.width) width=area.width+10;
  if(y<0) {
      height=height+y;
      y=0;
  }
  if(height>area.height) height=area.height+10;

  gdk_gc_set_clip_rectangle(sheet->xor_gc, &area);

  for(i=-1; i<=1; i++)
     gdk_draw_rectangle (sheet->sheet_window,
  	                 sheet->xor_gc,
                         FALSE,
	                 x+i,y+i,
	                 width-2*i,height-2*i);

  gdk_gc_set_clip_rectangle(sheet->xor_gc, NULL);
  
  gtk_sheet_draw_corners(sheet, new_range);

}

static void
gtk_sheet_draw_corners(GtkSheet *sheet, GtkSheetRange range)
{
  GdkRectangle area;
  gint x,y;

  area.x=sheet->row_title_area.width;
  area.y=sheet->column_title_area.height;
  area.width=sheet->sheet_window_width;
  area.height=sheet->sheet_window_height;

  gdk_gc_set_clip_rectangle(sheet->xor_gc, &area);


  if(gtk_sheet_cell_isvisible(sheet, range.row0, range.col0)){
       x=COLUMN_LEFT_XPIXEL(sheet,range.col0);
       y=ROW_TOP_YPIXEL(sheet,range.row0);
       gdk_draw_pixmap(sheet->sheet_window,
                  GTK_WIDGET(sheet)->style->fg_gc[GTK_STATE_NORMAL],
                  sheet->pixmap,
                  x-1,
                  y-1,
                  x-1,
                  y-1,
                  3,
                  3);         
       gdk_draw_rectangle (sheet->sheet_window,
  	                 sheet->xor_gc,
                         TRUE,
	                 x-1,y-1,
	                 3,3);
  }

  if(gtk_sheet_cell_isvisible(sheet, range.row0, range.coli)){
       x=COLUMN_LEFT_XPIXEL(sheet,range.coli)+
         sheet->column[range.coli].width;
       y=ROW_TOP_YPIXEL(sheet,range.row0);
       gdk_draw_pixmap(sheet->sheet_window,
                  GTK_WIDGET(sheet)->style->fg_gc[GTK_STATE_NORMAL],
                  sheet->pixmap,
                  x-1,
                  y-1,
                  x-1,
                  y-1,
                  3,
                  3);         
       gdk_draw_rectangle (sheet->sheet_window,
  	                 sheet->xor_gc,
                         TRUE,
	                 x-1,y-1,
	                 3,3);
  }

  if(gtk_sheet_cell_isvisible(sheet, range.rowi, range.col0)){
       x=COLUMN_LEFT_XPIXEL(sheet,range.col0);
       y=ROW_TOP_YPIXEL(sheet,range.rowi)+
         sheet->row[range.rowi].height;
       gdk_draw_pixmap(sheet->sheet_window,
                  GTK_WIDGET(sheet)->style->fg_gc[GTK_STATE_NORMAL],
                  sheet->pixmap,
                  x-1,
                  y-1,
                  x-1,
                  y-1,
                  3,
                  3);         
       gdk_draw_rectangle (sheet->sheet_window,
  	                 sheet->xor_gc,
                         TRUE,
	                 x-1,y-1,
	                 3,3);
  }

  if(gtk_sheet_cell_isvisible(sheet, range.rowi, range.coli)){
       x=COLUMN_LEFT_XPIXEL(sheet,range.coli)+
         sheet->column[range.coli].width;
       y=ROW_TOP_YPIXEL(sheet,range.rowi)+
         sheet->row[range.rowi].height;
       gdk_draw_pixmap(sheet->sheet_window,
                  GTK_WIDGET(sheet)->style->fg_gc[GTK_STATE_NORMAL],
                  sheet->pixmap,
                  x-3,
                  y-3,
                  x-3,
                  y-3,
                  5,
                  5);         
       gdk_draw_rectangle (sheet->sheet_window,
  	                 sheet->xor_gc,
                         TRUE,
	                 x-2,y-2,
	                 5,5);
  }

  gdk_gc_set_clip_rectangle(sheet->xor_gc, &area);

}



static void
gtk_sheet_real_select_range (GtkSheet * sheet,
			GtkSheetRange * range)
{
  gint i;
  gint state;
  GdkRectangle area;
  GtkSheetRange select_range, unselect_range, aux_range;

  g_return_if_fail (sheet != NULL);

  if(range==NULL) range=&sheet->range;

  if(range->row0 < 0 || range->rowi < 0) return;
  if(range->col0 < 0 || range->coli < 0) return;

  state=sheet->state;

  if(state==GTK_SHEET_COLUMN_SELECTED || state==GTK_SHEET_RANGE_SELECTED){
   for(i=sheet->range.col0; i< range->col0; i++)
    column_button_release(sheet, i);
   for(i=range->coli+1; i<= sheet->range.coli; i++)
    column_button_release(sheet, i);
   for(i=range->col0; i<=range->coli; i++){
    column_button_set(sheet, i);
   }
  }
 
  if(state==GTK_SHEET_ROW_SELECTED || state==GTK_SHEET_RANGE_SELECTED){
   for(i=sheet->range.row0; i< range->row0; i++)
    row_button_release(sheet, i);
   for(i=range->rowi+1; i<= sheet->range.rowi; i++)
    row_button_release(sheet, i);
   for(i=range->row0; i<=range->rowi; i++){
    row_button_set(sheet, i);
   }
  }

  if(range->coli != sheet->range.coli || range->col0 != sheet->range.col0 ||
     range->rowi != sheet->range.rowi || range->row0 != sheet->range.row0)
         {

           gtk_sheet_new_selection(sheet, range);

	   sheet->range.col0=range->col0;
	   sheet->range.coli=range->coli;
	   sheet->range.row0=range->row0;
	   sheet->range.rowi=range->rowi;
	 }
  else
         {
	   gtk_sheet_draw_backing_pixmap(sheet, sheet->range);
           gtk_sheet_range_draw_selection(sheet, sheet->range);
         }

  gtk_signal_emit(GTK_OBJECT(sheet), sheet_signals[SELECT_RANGE], range);
	
}

void
gtk_sheet_select_range(GtkSheet * sheet, GtkSheetRange *range)
{
  gint i;
  GdkRectangle area;

  g_return_if_fail (sheet != NULL);

  if(range==NULL) range=&sheet->range;

  if(range->row0 < 0 || range->rowi < 0) return;
  if(range->col0 < 0 || range->coli < 0) return;

  sheet->range.row0=range->row0;
  sheet->range.rowi=range->rowi;
  sheet->range.col0=range->col0;
  sheet->range.coli=range->coli;

  if (gtk_sheet_range_isvisible (sheet, *range)){
    gtk_sheet_draw_backing_pixmap(sheet, sheet->range);
    gtk_sheet_range_draw_selection(sheet, sheet->range);
  }
  if(sheet->state==GTK_SHEET_COLUMN_SELECTED || 
                                     sheet->state==GTK_SHEET_RANGE_SELECTED){
     for(i=range->col0; i<=range->coli; i++)
        column_button_set(sheet, i);
  }

  if(sheet->state==GTK_SHEET_ROW_SELECTED || 
                                     sheet->state==GTK_SHEET_RANGE_SELECTED){
     for(i=range->row0; i<=range->rowi; i++)
        row_button_set(sheet, i);
  }

  gtk_signal_emit(GTK_OBJECT(sheet), sheet_signals[SELECT_RANGE], range);
}

void
gtk_sheet_unselect_range (GtkSheet * sheet,
			  GtkSheetRange *range)
{
  gint i;
  GdkRectangle area;
 
  g_return_if_fail (sheet != NULL);

  if(range==NULL) range=&sheet->range;

  if(range->row0 < 0 || range->rowi < 0) return;
  if(range->col0 < 0 || range->coli < 0) return;

  if (gtk_sheet_range_isvisible (sheet, *range)){
    gtk_sheet_draw_backing_pixmap(sheet, *range);
  }

  for(i=range->col0; i<=range->coli; i++){
     column_button_release(sheet, i);
  }

  for(i=range->row0; i<=range->rowi; i++){
     row_button_release(sheet, i);
  }

}


static void
gtk_sheet_draw (GtkWidget * widget,
		GdkRectangle * area)
{
  GtkSheet *sheet;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_SHEET (widget));
  g_return_if_fail (area != NULL);

  if (GTK_WIDGET_DRAWABLE (widget))
    {
      sheet = GTK_SHEET (widget);

      gdk_window_clear_area (widget->window,
			     area->x, area->y,
			     area->width, area->height);

      /* draw sheet shadow/border */
      gtk_draw_shadow (widget->style, widget->window,
		       GTK_STATE_NORMAL, sheet->shadow_type,
		       0, 0, 
		       sheet->sheet_window_width + (2 * widget->style->klass->xthickness),
		       sheet->sheet_window_height + (2 * widget->style->klass->ythickness));

      gdk_window_clear_area (sheet->sheet_window,
			     0, 0, -1, -1);

      gtk_sheet_range_draw (sheet, NULL);

    }
}


static gint
gtk_sheet_expose (GtkWidget * widget,
		  GdkEventExpose * event)
{
  GtkSheet *sheet;
  GtkSheetRange range;
  GdkRectangle area, intersect;
  gint pixmap_width, pixmap_height;

  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_SHEET (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  if (GTK_WIDGET_DRAWABLE (widget))
  {
      sheet = GTK_SHEET (widget);

      range.row0=ROW_FROM_YPIXEL(sheet,event->area.y);
      range.col0=COLUMN_FROM_XPIXEL(sheet,event->area.x);
      range.rowi=ROW_FROM_YPIXEL(sheet,event->area.y+event->area.height);
      range.coli=COLUMN_FROM_XPIXEL(sheet,event->area.x+event->area.width);

      /* draw border */
      if (event->window == widget->window)
	gtk_draw_shadow (widget->style, widget->window,
			 GTK_STATE_NORMAL, sheet->shadow_type,
			 0, 0,
			 sheet->sheet_window_width + (2 * widget->style->klass->xthickness),
			 sheet->sheet_window_height + (2 * widget->style->klass->ythickness));

      /* exposure events on the sheet */
      if (event->window == sheet->sheet_window){
        gtk_sheet_draw_backing_pixmap(sheet, range);
              
        if(sheet->state != GTK_SHEET_NORMAL && gtk_sheet_range_isvisible(sheet, sheet->range)){          
          gtk_sheet_draw_backing_pixmap(sheet, sheet->range );
          gtk_sheet_range_draw_selection(sheet, sheet->range);
        }

        if((!GTK_SHEET_IN_XDRAG(sheet)) && (!GTK_SHEET_IN_YDRAG(sheet))){
          if(sheet->state == GTK_SHEET_NORMAL && !GTK_SHEET_IN_SELECTION(sheet)){

           if(sheet->active_cell.row >= 0 && sheet->active_cell.col >= 0){
            gchar text[MAX_TEXT_LENGTH];
            gchar *entry_text;
            gint justification;

            entry_text = gtk_shentry_get_text(GTK_SHENTRY(sheet->sheet_entry));

            if(strlen(entry_text)>0){
              strcpy(text, entry_text);
              justification = GTK_SHENTRY(sheet->sheet_entry)->justification;
	      gtk_sheet_set_cell(sheet, sheet->active_cell.row, sheet->active_cell.col, justification, text);
            }
 
	    gtk_sheet_activate_cell(sheet, sheet->active_cell.row, sheet->active_cell.col);
           } 
          }

          if(sheet->state == GTK_SHEET_NORMAL && GTK_SHEET_IN_SELECTION(sheet))
            gtk_sheet_draw_active_cell(sheet);
       }
      }
      if(event->window == sheet->row_title_window){
                     size_allocate_row_title_buttons(sheet);
                     gdk_window_show(sheet->row_title_window);
      }

      if(event->window == sheet->column_title_window){
                     size_allocate_column_title_buttons(sheet);
                     gdk_window_show(sheet->column_title_window);
      }

  }

  return FALSE;
}


static gint
gtk_sheet_button_press (GtkWidget * widget,
			GdkEventButton * event)
{
  GtkSheet *sheet;
  GtkSheetCell **cell;
  GdkModifierType mods;
  gint x, y, row, column;
  char *text;
  gdouble dval;

  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_SHEET (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  gdk_window_get_pointer(widget->window, NULL, NULL, &mods);
  if(!(mods&GDK_BUTTON1_MASK)) return;

  sheet = GTK_SHEET (widget);

  /* press on resize windows */
  if (event->window == sheet->column_title_window && !GTK_SHEET_COLUMN_FROZEN(sheet))
      {
	gtk_widget_get_pointer (widget, &sheet->x_drag, NULL);
        if(POSSIBLE_XDRAG(sheet, sheet->x_drag, &sheet->drag_cell.col)){

	  GTK_SHEET_SET_FLAGS (sheet, SHEET_IN_XDRAG);
	  gdk_pointer_grab (sheet->column_title_window, FALSE,
			  GDK_POINTER_MOTION_HINT_MASK |
			  GDK_BUTTON1_MOTION_MASK |
			  GDK_BUTTON_RELEASE_MASK,
			  NULL, NULL, event->time);

	  draw_xor_vline (sheet);
	  return FALSE;
        }
      }

  if (event->window == sheet->row_title_window && !GTK_SHEET_ROW_FROZEN(sheet))
      {
	gtk_widget_get_pointer (widget, NULL, &sheet->y_drag);

        if(POSSIBLE_YDRAG(sheet, sheet->y_drag, &sheet->drag_cell.row)){
	  GTK_SHEET_SET_FLAGS (sheet, SHEET_IN_YDRAG);
	  gdk_pointer_grab (sheet->row_title_window, FALSE,
			  GDK_POINTER_MOTION_HINT_MASK |
			  GDK_BUTTON1_MOTION_MASK |
			  GDK_BUTTON_RELEASE_MASK,
			  NULL, NULL, event->time);

	  draw_xor_hline (sheet);
	  return FALSE;
        }
      }

  /* selections on the sheet */
    if(event->window == sheet->sheet_window){
     gint i,j;
     gtk_widget_get_pointer (widget, &x, &y);
     gtk_sheet_get_pixel_info (sheet, x, y, &row, &column);
     gtk_grab_add(GTK_WIDGET(sheet));
     sheet->timer=gtk_timeout_add(TIMEOUT_SCROLL, gtk_sheet_scroll, sheet); 
     GTK_WIDGET_UNSET_FLAGS(sheet->sheet_entry, GTK_HAS_FOCUS);
     GTK_WIDGET_SET_FLAGS(GTK_SHEET(sheet), GTK_HAS_FOCUS);
     gtk_widget_grab_focus(GTK_WIDGET(sheet));
     if(sheet->cursor_drag->type==GDK_SIZING &&
        !GTK_SHEET_IN_SELECTION(sheet) && !GTK_SHEET_IN_RESIZE(sheet)){
        GTK_SHEET_SET_FLAGS(sheet, SHEET_IN_RESIZE);
        sheet->x_drag=x;
        sheet->y_drag=y;
        sheet->drag_cell.row=row;
        sheet->drag_cell.col=column;
        sheet->drag_range=sheet->range;
        draw_xor_rectangle(sheet, sheet->drag_range);
     }
     else if(sheet->cursor_drag->type==GDK_TOP_LEFT_ARROW &&
            !GTK_SHEET_IN_SELECTION(sheet) && !GTK_SHEET_IN_DRAG(sheet)) {
            GTK_SHEET_SET_FLAGS(sheet, SHEET_IN_DRAG);
            sheet->x_drag=x;
            sheet->y_drag=y;
            sheet->drag_cell.row=row;
            sheet->drag_cell.col=column;
            sheet->drag_range=sheet->range;
            draw_xor_rectangle(sheet, sheet->drag_range);
          }
          else 
          {
           gtk_sheet_click_cell(sheet, row, column);
           GTK_SHEET_SET_FLAGS(sheet, SHEET_IN_SELECTION);
          }
    }

    if(event->window == sheet->column_title_window){
     gtk_widget_get_pointer (widget, &x, &y);
     column = COLUMN_FROM_XPIXEL(sheet, x);
     if(sheet->column[column].is_sensitive){;
       gtk_sheet_click_cell(sheet, -1, column);
       gtk_grab_add(GTK_WIDGET(sheet));
       sheet->timer=gtk_timeout_add(TIMEOUT_SCROLL, gtk_sheet_scroll, sheet); 
       GTK_WIDGET_UNSET_FLAGS(sheet->sheet_entry, GTK_HAS_FOCUS);
       GTK_WIDGET_SET_FLAGS(GTK_SHEET(sheet), GTK_HAS_FOCUS);
       gtk_widget_grab_focus(GTK_WIDGET(sheet));
       GTK_SHEET_SET_FLAGS(sheet,SHEET_IN_SELECTION);
     }
    }

    if(event->window == sheet->row_title_window){
     gtk_widget_get_pointer (widget, &x, &y);
     row = ROW_FROM_YPIXEL(sheet, y);
     if(sheet->row[row].is_sensitive){
       gtk_sheet_click_cell(sheet, row, -1);
       gtk_grab_add(GTK_WIDGET(sheet));
       sheet->timer=gtk_timeout_add(TIMEOUT_SCROLL, gtk_sheet_scroll, sheet); 
       GTK_WIDGET_UNSET_FLAGS(sheet->sheet_entry, GTK_HAS_FOCUS);
       GTK_WIDGET_SET_FLAGS(GTK_SHEET(sheet), GTK_HAS_FOCUS);
       gtk_widget_grab_focus(GTK_WIDGET(sheet));
       GTK_SHEET_SET_FLAGS(sheet,SHEET_IN_SELECTION);
     }
    }

    return FALSE;
}

static gint
gtk_sheet_scroll(gpointer data)
{
 GtkSheet *sheet;
 gint x,y,row,column;
 gint move;
  
 sheet=GTK_SHEET(data);
 gtk_widget_get_pointer (GTK_WIDGET(sheet), &x, &y);
 gtk_sheet_get_pixel_info (sheet, x, y, &row, &column);

 move=TRUE;

 if(GTK_SHEET_IN_SELECTION(sheet))
      gtk_sheet_extend_selection(sheet, row, column);
 if(GTK_SHEET_IN_DRAG(sheet) || GTK_SHEET_IN_RESIZE(sheet)){
       move=gtk_sheet_move_query(sheet, row, column);
       if(move) draw_xor_rectangle(sheet, sheet->drag_range);      
 }       
 return TRUE;
      
}

static void
gtk_sheet_click_cell(GtkSheet *sheet, gint row, gint column)
{
      if(sheet->state!=GTK_SHEET_NORMAL){
       sheet->state=GTK_SHEET_NORMAL;
       gtk_sheet_unselect_range(sheet, NULL);
      }
      else
         gtk_sheet_deactivate_cell(sheet);
      

      gtk_sheet_move_query(sheet, row, column);

      if(row == -1 && column >= 0){
	  gtk_sheet_select_column(sheet, column);
          return;
      }
      if(column == -1 && row >= 0){
 	  gtk_sheet_select_row(sheet, row);
          return;
      }
      if(row!=-1 && column !=-1){
          sheet->active_cell.row=row;
          sheet->active_cell.col=column;
	  sheet->selection_cell.row=row;
          sheet->selection_cell.col=column;
          sheet->range.row0=row;
          sheet->range.col0=column;
          sheet->range.rowi=row;
          sheet->range.coli=column;
	  sheet->state=GTK_SHEET_NORMAL;
	  GTK_SHEET_SET_FLAGS(sheet, SHEET_IN_SELECTION);
	  gtk_sheet_draw_active_cell(sheet);
	  return;
      }
      if(row==-1 && column ==-1){
          sheet->state=GTK_SHEET_RANGE_SELECTED;                     
          sheet->range.row0=0;
          sheet->range.col0=0;
          sheet->range.rowi=sheet->maxrow;
          sheet->range.coli=sheet->maxcol;
	  sheet->active_cell.row=0;
	  sheet->active_cell.col=0;
	  gtk_sheet_select_range(sheet, NULL);
	  return;
      }
      gtk_sheet_activate_cell(sheet, sheet->active_cell.row,
                                     sheet->active_cell.col);
}

static gint
gtk_sheet_button_release (GtkWidget * widget,
			GdkEventButton * event)
{
  GtkSheet *sheet;
  gint x,y;
 
  sheet=GTK_SHEET(widget);
  gtk_grab_remove(widget);

  /* release on resize windows */
  if (GTK_SHEET_IN_XDRAG (sheet)){
	  GTK_SHEET_UNSET_FLAGS (sheet, SHEET_IN_XDRAG);
          GTK_SHEET_UNSET_FLAGS (sheet, SHEET_IN_SELECTION);
	  gtk_widget_get_pointer (widget, &x, NULL);
	  gdk_pointer_ungrab (event->time);
	  draw_xor_vline (sheet);
	  
	  gtk_sheet_set_column_width (sheet, sheet->drag_cell.col, new_column_width (sheet, sheet->drag_cell.col, &x));
          gtk_signal_emit_by_name (GTK_OBJECT (GTK_RANGE (sheet->hscrollbar)->adjustment), "value_changed");
	  return FALSE;
  }

  if (GTK_SHEET_IN_YDRAG (sheet)){
	  GTK_SHEET_UNSET_FLAGS (sheet, SHEET_IN_YDRAG);
          GTK_SHEET_UNSET_FLAGS (sheet,SHEET_IN_SELECTION);
	  gtk_widget_get_pointer (widget, NULL, &y);
	  gdk_pointer_ungrab (event->time);
	  draw_xor_hline (sheet);
	  
	  gtk_sheet_set_row_height (sheet, sheet->drag_cell.row, new_row_height (sheet, sheet->drag_cell.row, &y));
          gtk_signal_emit_by_name (GTK_OBJECT (GTK_RANGE (sheet->vscrollbar)->adjustment), "value_changed");
	  return FALSE;
  }

  
  if (GTK_SHEET_IN_DRAG(sheet)){
      GtkSheetRange old_range;
      GTK_SHEET_UNSET_FLAGS(sheet, SHEET_IN_DRAG);
      GTK_SHEET_UNSET_FLAGS(sheet, SHEET_IN_SELECTION);
      draw_xor_rectangle(sheet, sheet->drag_range);
      gtk_sheet_unselect_range(sheet, NULL);
      sheet->active_cell.row=sheet->active_cell.row+
                             (sheet->drag_range.row0-sheet->range.row0);
      sheet->active_cell.col=sheet->active_cell.col+
                             (sheet->drag_range.col0-sheet->range.col0);
      sheet->selection_cell.row=sheet->selection_cell.row+
                             (sheet->drag_range.row0-sheet->range.row0);
      sheet->selection_cell.col=sheet->selection_cell.col+
                             (sheet->drag_range.col0-sheet->range.col0);
      old_range=sheet->range;
      sheet->range=sheet->drag_range;
      sheet->drag_range=old_range;

      if(sheet->state==GTK_STATE_NORMAL) sheet->state=GTK_SHEET_RANGE_SELECTED;
      gtk_signal_emit(GTK_OBJECT(sheet),sheet_signals[MOVE_RANGE],
                      &sheet->drag_range, &sheet->range);
      gtk_sheet_select_range(sheet, &sheet->range);
  }

  if (GTK_SHEET_IN_RESIZE(sheet)){
      GtkSheetRange old_range;
      GTK_SHEET_UNSET_FLAGS(sheet, SHEET_IN_RESIZE);
      GTK_SHEET_UNSET_FLAGS(sheet, SHEET_IN_SELECTION);
      draw_xor_rectangle(sheet, sheet->drag_range);
      gtk_sheet_unselect_range(sheet, NULL);
      sheet->active_cell.row=sheet->active_cell.row+
                             (sheet->drag_range.row0-sheet->range.row0);
      sheet->active_cell.col=sheet->active_cell.col+
                             (sheet->drag_range.col0-sheet->range.col0);
      if(sheet->drag_range.row0<sheet->range.row0)
                     sheet->selection_cell.row=sheet->drag_range.row0;
      if(sheet->drag_range.rowi>=sheet->range.rowi)
                     sheet->selection_cell.row=sheet->drag_range.rowi;
      if(sheet->drag_range.col0<sheet->range.col0)
                     sheet->selection_cell.col=sheet->drag_range.col0;
      if(sheet->drag_range.coli>=sheet->range.coli)
                     sheet->selection_cell.col=sheet->drag_range.coli;
      old_range=sheet->range;
      sheet->range=sheet->drag_range;
      sheet->drag_range=old_range;

      if(sheet->state==GTK_STATE_NORMAL) sheet->state=GTK_SHEET_RANGE_SELECTED;
      gtk_signal_emit(GTK_OBJECT(sheet),sheet_signals[RESIZE_RANGE],
                      &sheet->drag_range, &sheet->range);
      gtk_sheet_select_range(sheet, &sheet->range);
  }

  if(sheet->state==GTK_SHEET_NORMAL && GTK_SHEET_IN_SELECTION(sheet)){
      gtk_sheet_activate_cell(sheet, sheet->active_cell.row, 
                                     sheet->active_cell.col);
  }

  gtk_timeout_remove(sheet->timer);
  gtk_grab_remove(GTK_WIDGET(sheet));
  GTK_SHEET_UNSET_FLAGS(sheet, SHEET_IN_SELECTION);
  return FALSE;
}

static gint
gtk_sheet_motion (GtkWidget * widget,
		  GdkEventMotion * event)
{
  GtkSheet *sheet;
  GdkModifierType mods;
  GdkCursorType new_cursor;
  gint x, y, row, column;
  gint i;

  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_SHEET (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);


  sheet = GTK_SHEET (widget);


  /* selections on the sheet */
  x = event->x;
  y = event->y;

  if(event->window == sheet->column_title_window && !GTK_SHEET_COLUMN_FROZEN(sheet)){
    gtk_widget_get_pointer(widget, &x, &y);
    if(!GTK_SHEET_IN_SELECTION(sheet) && POSSIBLE_XDRAG(sheet, x, &column)){
      new_cursor=GDK_SB_H_DOUBLE_ARROW;
      if(new_cursor != sheet->cursor_drag->type){
        gdk_cursor_destroy(sheet->cursor_drag);
        sheet->cursor_drag=gdk_cursor_new(GDK_SB_H_DOUBLE_ARROW);
        gdk_window_set_cursor(sheet->column_title_window,sheet->cursor_drag);
      }
    }else{
      new_cursor=GDK_TOP_LEFT_ARROW;
      if(!GTK_SHEET_IN_XDRAG(sheet) && new_cursor != sheet->cursor_drag->type){
        gdk_cursor_destroy(sheet->cursor_drag);
        sheet->cursor_drag=gdk_cursor_new(GDK_TOP_LEFT_ARROW);
        gdk_window_set_cursor(sheet->column_title_window,sheet->cursor_drag);
      }
    }
  }      

  if(event->window == sheet->row_title_window && !GTK_SHEET_ROW_FROZEN(sheet)){
    gtk_widget_get_pointer(widget, &x, &y);
    if(!GTK_SHEET_IN_SELECTION(sheet) && POSSIBLE_YDRAG(sheet,y, &column)){
      new_cursor=GDK_SB_V_DOUBLE_ARROW;
      if(new_cursor != sheet->cursor_drag->type){
        gdk_cursor_destroy(sheet->cursor_drag);
        sheet->cursor_drag=gdk_cursor_new(GDK_SB_V_DOUBLE_ARROW);
        gdk_window_set_cursor(sheet->row_title_window,sheet->cursor_drag);
      }
    }else{
      new_cursor=GDK_TOP_LEFT_ARROW;
      if(!GTK_SHEET_IN_YDRAG(sheet) && new_cursor != sheet->cursor_drag->type){
        gdk_cursor_destroy(sheet->cursor_drag);
        sheet->cursor_drag=gdk_cursor_new(GDK_TOP_LEFT_ARROW);
        gdk_window_set_cursor(sheet->row_title_window,sheet->cursor_drag);
      }
    }
  }      

  new_cursor=GDK_PLUS;
  if(!POSSIBLE_DRAG(sheet,x,y,&row,&column) && !GTK_SHEET_IN_DRAG(sheet) &&
     !POSSIBLE_RESIZE(sheet,x,y,&row,&column) && !GTK_SHEET_IN_RESIZE(sheet) &&
     event->window == sheet->sheet_window && 
     new_cursor != sheet->cursor_drag->type){
         gdk_cursor_destroy(sheet->cursor_drag);
         sheet->cursor_drag=gdk_cursor_new(GDK_PLUS);
         gdk_window_set_cursor(sheet->sheet_window,sheet->cursor_drag);
  }

  new_cursor=GDK_TOP_LEFT_ARROW;
  if(!(POSSIBLE_RESIZE(sheet,x,y,&row,&column) || GTK_SHEET_IN_RESIZE(sheet)) &&
     (POSSIBLE_DRAG(sheet, x,y,&row,&column) || GTK_SHEET_IN_DRAG(sheet)) && 
     event->window == sheet->sheet_window && 
     new_cursor != sheet->cursor_drag->type){
         gdk_cursor_destroy(sheet->cursor_drag);
         sheet->cursor_drag=gdk_cursor_new(GDK_TOP_LEFT_ARROW);
         gdk_window_set_cursor(sheet->sheet_window,sheet->cursor_drag);
  }

  new_cursor=GDK_SIZING;
  if(!GTK_SHEET_IN_DRAG(sheet) &&
     (POSSIBLE_RESIZE(sheet,x,y,&row,&column) || GTK_SHEET_IN_RESIZE(sheet)) &&
     event->window == sheet->sheet_window && 
     new_cursor != sheet->cursor_drag->type){
         gdk_cursor_destroy(sheet->cursor_drag);
         sheet->cursor_drag=gdk_cursor_new(GDK_SIZING);
         gdk_window_set_cursor(sheet->sheet_window,sheet->cursor_drag);
  }

  gdk_window_get_pointer (widget->window, &x, &y, &mods);
  if(!(mods & GDK_BUTTON1_MASK)) return FALSE;

  if (GTK_SHEET_IN_XDRAG (sheet)){
	if (event->is_hint || event->window != widget->window)
	    gtk_widget_get_pointer (widget, &x, NULL);
	  else
	    x = event->x;

	  new_column_width (sheet, sheet->drag_cell.col, &x);
	  if (x != sheet->x_drag)
	    {
	      draw_xor_vline (sheet);
	      sheet->x_drag = x;
	      draw_xor_vline (sheet);
	    }
          return FALSE;
  }

  if (GTK_SHEET_IN_YDRAG (sheet)){
	  if (event->is_hint || event->window != widget->window)
	    gtk_widget_get_pointer (widget, NULL, &y);
	  else
	    y = event->y;

	  new_row_height (sheet, sheet->drag_cell.row, &y);
	  if (y != sheet->y_drag)
	    {
	      draw_xor_hline (sheet);
	      sheet->y_drag = y;
	      draw_xor_hline (sheet);
	    }
          return FALSE;
  }

  if (GTK_SHEET_IN_DRAG(sheet)){
       GtkSheetRange aux;
       column=COLUMN_FROM_XPIXEL(sheet,x)-sheet->drag_cell.col;
       row=ROW_FROM_YPIXEL(sheet,y)-sheet->drag_cell.row;
       if(sheet->state==GTK_SHEET_COLUMN_SELECTED) row=0;
       if(sheet->state==GTK_SHEET_ROW_SELECTED) column=0;
       sheet->x_drag=x;
       sheet->y_drag=y;
       aux=sheet->range;
       if(aux.row0+row >= 0 && aux.rowi+row <= sheet->maxrow &&
          aux.col0+column >= 0 && aux.coli+column <= sheet->maxcol){
             aux=sheet->drag_range;
             sheet->drag_range.row0=sheet->range.row0+row;
             sheet->drag_range.col0=sheet->range.col0+column;
             sheet->drag_range.rowi=sheet->range.rowi+row;
             sheet->drag_range.coli=sheet->range.coli+column;
             if(aux.row0 != sheet->drag_range.row0 ||
                aux.col0 != sheet->drag_range.col0){
                draw_xor_rectangle (sheet, aux);
                draw_xor_rectangle (sheet, sheet->drag_range);
             }
       }
       return FALSE;
  }

  if (GTK_SHEET_IN_RESIZE(sheet)){
       GtkSheetRange aux;
       gint v_h;
       v_h=1;
       if(abs(x-COLUMN_LEFT_XPIXEL(sheet,sheet->drag_cell.col)) >
          abs(y-ROW_TOP_YPIXEL(sheet,sheet->drag_cell.row))) v_h=2;

       column=COLUMN_FROM_XPIXEL(sheet,x)-sheet->drag_cell.col;
       row=ROW_FROM_YPIXEL(sheet,y)-sheet->drag_cell.row;
       if(sheet->state==GTK_SHEET_COLUMN_SELECTED) row=0;
       if(sheet->state==GTK_SHEET_ROW_SELECTED) column=0;
       sheet->x_drag=x;
       sheet->y_drag=y;
       aux=sheet->range;

       if(row < sheet->range.row0 - sheet->range.rowi - 1) 
          row=row+(sheet->range.rowi-sheet->range.row0 + 1);
       else if(row<0) row=0;

       if(column < sheet->range.col0 - sheet->range.coli - 1)
          column=column+(sheet->range.coli-sheet->range.col0 + 1);
       else if(column<0) column=0;

       if(v_h==1) 
           column=0;
       else
           row=0;

       if(aux.row0+row >= 0 && aux.rowi+row <= sheet->maxrow &&
          aux.col0+column >= 0 && aux.coli+column <= sheet->maxcol){

             aux=sheet->drag_range;
             sheet->drag_range=sheet->range;

             if(row<0) sheet->drag_range.row0=sheet->range.row0+row;
             if(row>0) sheet->drag_range.rowi=sheet->range.rowi+row;
             if(column<0) sheet->drag_range.col0=sheet->range.col0+column;
             if(column>0) sheet->drag_range.coli=sheet->range.coli+column;
             
             if(aux.row0 != sheet->drag_range.row0 ||
                aux.rowi != sheet->drag_range.rowi ||
                aux.col0 != sheet->drag_range.col0 ||
                aux.coli != sheet->drag_range.coli){
                     draw_xor_rectangle (sheet, aux);
                     draw_xor_rectangle (sheet, sheet->drag_range);
             }
       }
       return FALSE;
  }

  

  gtk_sheet_get_pixel_info (sheet, x, y, &row, &column);

  if(sheet->state==GTK_SHEET_NORMAL && row==sheet->active_cell.row &&
     column==sheet->active_cell.col) return FALSE;

  if(GTK_SHEET_IN_SELECTION(sheet) && mods&GDK_BUTTON1_MASK)
                          gtk_sheet_extend_selection(sheet, row, column);

}

static gint
gtk_sheet_move_query(GtkSheet *sheet, gint row, gint column)
{
  gint move;
  gfloat row_align, col_align;

  move=FALSE;
  row_align=-1.;
  col_align=-1.;

  if(row>MAX_VISIBLE_ROW(sheet) && sheet->state!=GTK_SHEET_COLUMN_SELECTED) {
          row_align = 1.;
          move = TRUE;
  }
  if(row<MIN_VISIBLE_ROW(sheet) && sheet->state!=GTK_SHEET_COLUMN_SELECTED) {
          row_align= 0.;
          move = TRUE;
  }
  if(column>MAX_VISIBLE_COLUMN(sheet) && sheet->state!=GTK_SHEET_ROW_SELECTED) {
          col_align = 1.;
          move = TRUE;
  } 
  if(column<MIN_VISIBLE_COLUMN(sheet) && sheet->state!=GTK_SHEET_ROW_SELECTED) {
	  col_align = 0.;
          move = TRUE;
  }

  if(move){
        gtk_sheet_moveto(sheet, row, column, row_align, col_align);
  }

  return(move);
}

static void
gtk_sheet_extend_selection(GtkSheet *sheet, gint row, gint column)
{
   GtkSheetRange range;
   GdkRectangle area;
   gint state;
   gint r,c;

   if(row == sheet->selection_cell.row && column == sheet->selection_cell.col)
        return;

   gtk_sheet_move_query(sheet, row, column);
   gtk_widget_grab_focus(GTK_WIDGET(sheet));

   if(GTK_SHEET_IN_DRAG(sheet)) return;

   state=sheet->state;

   switch(sheet->state){
    case GTK_SHEET_ROW_SELECTED:
	 column = sheet->maxcol;
         break;
    case GTK_SHEET_COLUMN_SELECTED:
	 row = sheet->maxrow;
         break; 
    case GTK_SHEET_NORMAL:
	 sheet->state=GTK_SHEET_RANGE_SELECTED;
         GTK_SHEET_SET_FLAGS(sheet, SHEET_IN_SELECTION);
         r=sheet->active_cell.row;
         c=sheet->active_cell.col;
         sheet->range.col0=c;
         sheet->range.row0=r;
         sheet->range.coli=c;
         sheet->range.rowi=r;
         gdk_draw_pixmap(sheet->sheet_window,
                   GTK_WIDGET(sheet)->style->fg_gc[GTK_STATE_NORMAL],
                   sheet->pixmap,
                   COLUMN_LEFT_XPIXEL(sheet,c)-1,
                   ROW_TOP_YPIXEL(sheet,r)-1,
                   COLUMN_LEFT_XPIXEL(sheet,c)-1,
                   ROW_TOP_YPIXEL(sheet,r)-1,
                   sheet->column[c].width+4,
                   sheet->row[r].height+4);   
         gtk_sheet_range_draw_selection(sheet, sheet->range);
    case GTK_SHEET_RANGE_SELECTED:
         sheet->state=GTK_SHEET_RANGE_SELECTED;
   }

   sheet->selection_cell.row = row;
   sheet->selection_cell.col = column;

   range.col0=MIN(column,sheet->active_cell.col);
   range.coli=MAX(column,sheet->active_cell.col);
   range.row0=MIN(row,sheet->active_cell.row);
   range.rowi=MAX(row,sheet->active_cell.row);

   if(range.row0 != sheet->range.row0 || range.rowi != sheet->range.rowi ||
      range.col0 != sheet->range.col0 || range.coli != sheet->range.coli ||
      state==GTK_SHEET_NORMAL)
               gtk_sheet_real_select_range(sheet, &range);

}

static gint
gtk_sheet_key_press(GtkWidget *widget,
		    GdkEventKey *key)
{
  GtkSheet *sheet;
  GtkSheetCell **cell;
  gint row, col;
  char *text;
  gdouble dval;
  gint state;
  gint extend_selection=FALSE;
  gint force_move=FALSE;
  gint scroll=1;

  sheet = GTK_SHEET(widget);

  if(key->state & GDK_CONTROL_MASK || key->keyval==GDK_Control_L ||
     key->keyval==GDK_Control_R) return;
/*
  {
    if(key->keyval=='c' || key->keyval == 'C' && sheet->state != GTK_STATE_NORMAL)
            gtk_sheet_clip_range(sheet, sheet->range);
    if(key->keyval=='x' || key->keyval == 'X')
            gtk_sheet_unclip_range(sheet);    
    return;
  }
*/
  extend_selection = (key->state & GDK_SHIFT_MASK) || key->keyval==GDK_Shift_L 
|| key->keyval==GDK_Shift_R;

  state=sheet->state;

  if(!extend_selection && sheet->state != GTK_SHEET_NORMAL){  
         sheet->state=GTK_SHEET_NORMAL;
         gtk_sheet_unselect_range(sheet, NULL);
  }
  GTK_SHEET_UNSET_FLAGS(sheet, SHEET_IN_SELECTION);


  switch(key->keyval){
    case GDK_Return:
      if(sheet->active_cell.row >= 0 && sheet->active_cell.col >= 0){
       if(sheet->active_cell.row < sheet->maxrow){
            col = sheet->active_cell.col;
	    row = sheet->active_cell.row + scroll;
            while(!sheet->row[row].is_visible && row<sheet->maxrow) row++;
	    gtk_sheet_click_cell(sheet, row, col);
       }
      } 
      break;
    case GDK_Tab:
      if(sheet->active_cell.row >= 0 && sheet->active_cell.col >= 0){
       if(sheet->active_cell.col < sheet->maxcol){
            col = sheet->active_cell.col + scroll; 
	    row = sheet->active_cell.row;
            while(!sheet->column[col].is_visible && col<sheet->maxcol) col++;
	    gtk_sheet_click_cell(sheet, row, col);
       }       
      }
      break;
/*    case GDK_BackSpace:
      if(sheet->active_cell.row >= 0 && sheet->active_cell.col >= 0){
       if(sheet->active_cell.col > 0){
            col = sheet->active_cell.col - scroll; 
	    row = sheet->active_cell.row;
            while(!sheet->column[col].is_visible && col > 0) col--;
	    gtk_sheet_click_cell(sheet, row, col);
       }       
      }
      break;
*/
    case GDK_Page_Up:
      scroll=MAX_VISIBLE_ROW(sheet)-MIN_VISIBLE_ROW(sheet)+1;
    case GDK_Up:
      if(extend_selection){
        if(sheet->selection_cell.row > 0){
          row = sheet->selection_cell.row - scroll;
          while(!sheet->row[row].is_visible && row > 0) row--;
          row = MAX(0, row);
          gtk_sheet_extend_selection(sheet, row, sheet->selection_cell.col);
        }
        return TRUE;
      }
      col = sheet->active_cell.col;
      if(state==GTK_SHEET_COLUMN_SELECTED) 
             sheet->active_cell.row = MIN_VISIBLE_ROW(sheet);
      row = sheet->active_cell.row - scroll;
      while(!sheet->row[row].is_visible && row > 0) row--;
      row = MAX(0,row);
      gtk_sheet_click_cell(sheet, row, col);
      break;
    case GDK_Page_Down:
      scroll=MAX_VISIBLE_ROW(sheet)-MIN_VISIBLE_ROW(sheet)+1;
    case GDK_Down:
      if(extend_selection){
        if(sheet->selection_cell.row < sheet->maxrow){
          row = sheet->selection_cell.row + scroll;
          while(!sheet->row[row].is_visible && row < sheet->maxrow) row++;
          row = MIN(sheet->maxrow, row);
          gtk_sheet_extend_selection(sheet, row, sheet->selection_cell.col);
        }
        return TRUE;
      }
      if(sheet->active_cell.row < sheet->maxrow){
           col = sheet->active_cell.col;
	   if(state==GTK_SHEET_COLUMN_SELECTED) 
                sheet->active_cell.row = MIN_VISIBLE_ROW(sheet);
	   row = sheet->active_cell.row + scroll;
           while(!sheet->row[row].is_visible && row < sheet->maxrow) row++;
           row = MIN(sheet->maxrow, row);
	   gtk_sheet_click_cell(sheet, row, col);
      }
      break;
    case GDK_Right:
      if(extend_selection){
        if(sheet->selection_cell.col < sheet->maxcol){
          col = sheet->selection_cell.col + 1;
          while(!sheet->column[col].is_visible && col < sheet->maxcol) col++;
          gtk_sheet_extend_selection(sheet, sheet->selection_cell.row, col);
        }
        return TRUE;
      }
      if(sheet->active_cell.col < sheet->maxcol){
	   if(state==GTK_SHEET_ROW_SELECTED) 
                sheet->active_cell.col = MIN_VISIBLE_COLUMN(sheet);
           col = sheet->active_cell.col + 1;
	   row = sheet->active_cell.row;
           while(!sheet->column[col].is_visible && col < sheet->maxcol) col++;
           if(strlen(gtk_shentry_get_text(GTK_SHENTRY(sheet->sheet_entry))) == 0 
              || force_move){
	        gtk_sheet_click_cell(sheet, row, col);
           }
      }
      break;
    case GDK_Left:
      if(extend_selection){
        if(sheet->selection_cell.col > 0){
          col = sheet->selection_cell.col - 1;
          while(!sheet->column[col].is_visible && col > 0) col--;          
          gtk_sheet_extend_selection(sheet, sheet->selection_cell.row, col);
        }
	return TRUE;
      }
      if(state==GTK_SHEET_ROW_SELECTED) 
                sheet->active_cell.col = MIN_VISIBLE_COLUMN(sheet);
      col = sheet->active_cell.col - 1;
      row = sheet->active_cell.row;
      while(!sheet->column[col].is_visible && col > 0) col--;
      col = MAX(0, col);

      if(strlen(gtk_shentry_get_text(GTK_SHENTRY(sheet->sheet_entry))) == 0
         || force_move){
                gtk_sheet_click_cell(sheet, row, col);
      }
      break;
    case GDK_Home:
      row=0;
      while(!sheet->row[row].is_visible && row < sheet->maxrow) row++;
      gtk_sheet_click_cell(sheet, row, sheet->active_cell.col);
      break;
    case GDK_End:
      row=sheet->maxrow;
      while(!sheet->row[row].is_visible && row > 0) row--;
      gtk_sheet_click_cell(sheet, row, sheet->active_cell.col);
      break;
    default:
      if(extend_selection) return TRUE;
      if(state == GTK_SHEET_ROW_SELECTED) 
        sheet->active_cell.col=MIN_VISIBLE_COLUMN(sheet);
      if(state == GTK_SHEET_COLUMN_SELECTED)
        sheet->active_cell.row=MIN_VISIBLE_ROW(sheet);
      gtk_sheet_click_cell(sheet, sheet->active_cell.row, sheet->active_cell.col);
  }

  if(extend_selection) return TRUE;
 
  gtk_sheet_activate_cell(sheet, sheet->active_cell.row,
                                 sheet->active_cell.col);
} 





static void
gtk_sheet_size_request (GtkWidget * widget,
			GtkRequisition * requisition)
{
  gint i;
  GtkSheet *sheet;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_SHEET (widget));
  g_return_if_fail (requisition != NULL);

  sheet = GTK_SHEET (widget);

  requisition->width = 3*DEFAULT_COLUMN_WIDTH;
  requisition->height = 3*DEFAULT_ROW_HEIGHT(widget);

  /* compute the size of the column title area */
  sheet->column_title_area.height = DEFAULT_ROW_HEIGHT(widget);
  requisition->height += sheet->column_title_area.height;

  /* compute the size of the row title area */
  sheet->row_title_area.width = DEFAULT_COLUMN_WIDTH;
  requisition->width += sheet->row_title_area.width;


  /* add the vscrollbar space */
  if ((sheet->vscrollbar_policy == GTK_POLICY_AUTOMATIC) ||
      GTK_WIDGET_VISIBLE (sheet->vscrollbar))
    {
      gtk_widget_size_request (sheet->vscrollbar, &sheet->vscrollbar->requisition);

      requisition->width += sheet->vscrollbar->requisition.width + SCROLLBAR_SPACING (sheet);
      requisition->height = MAX (requisition->height,
				 sheet->vscrollbar->requisition.height);
    }

  /* add the hscrollbar space */
  if ((sheet->hscrollbar_policy == GTK_POLICY_AUTOMATIC) ||
      GTK_WIDGET_VISIBLE (sheet->hscrollbar))
    {
      gtk_widget_size_request (sheet->hscrollbar, &sheet->hscrollbar->requisition);

      requisition->height += sheet->hscrollbar->requisition.height + SCROLLBAR_SPACING (sheet);
      requisition->width = MAX (sheet->hscrollbar->requisition.width, 
				requisition->width - 
				sheet->vscrollbar->requisition.width);

    }

  requisition->width += widget->style->klass->xthickness * 2 +
    GTK_CONTAINER (widget)->border_width * 2;
  requisition->height += widget->style->klass->ythickness * 2 +
    GTK_CONTAINER (widget)->border_width * 2;

  sheet->view.row0=ROW_FROM_YPIXEL(sheet, sheet->column_title_area.height+1);
  sheet->view.rowi=ROW_FROM_YPIXEL(sheet, sheet->sheet_window_height);
  sheet->view.col0=COLUMN_FROM_XPIXEL(sheet, sheet->row_title_area.width+1);
  sheet->view.coli=COLUMN_FROM_XPIXEL(sheet, sheet->sheet_window_width);

}

 
static void
gtk_sheet_size_allocate (GtkWidget * widget,
			 GtkAllocation * allocation)
{
  GtkSheet *sheet;
  GtkAllocation sheet_allocation;
  GtkAllocation child_allocation;
  gint i, vscrollbar_vis, hscrollbar_vis;
  GtkSheetRange range;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_SHEET (widget));
  g_return_if_fail (allocation != NULL);

  sheet = GTK_SHEET (widget);
  widget->allocation = *allocation;

  if (GTK_WIDGET_REALIZED (widget))
   {
    gdk_window_move_resize (widget->window,
	      allocation->x + GTK_CONTAINER (widget)->border_width,
	      allocation->y + GTK_CONTAINER (widget)->border_width,
              allocation->width - GTK_CONTAINER (widget)->border_width * 2,
	      allocation->height - GTK_CONTAINER (widget)->border_width * 2);

    /* use internal allocation structure for all the math
     * because it's easier than always subtracting the container
     * border width */
    sheet->internal_allocation.x = 0;
    sheet->internal_allocation.y = 0;
    sheet->internal_allocation.width = allocation->width -
        GTK_CONTAINER (widget)->border_width * 2 ;
    sheet->internal_allocation.height = allocation->height -
	GTK_CONTAINER (widget)->border_width * 2 ;
	
    /* allocate sheet window assuming no scrollbars */
    sheet_allocation.x = sheet->internal_allocation.x + widget->style->klass->xthickness;
    sheet_allocation.y = sheet->internal_allocation.y + widget->style->klass->ythickness;
    sheet_allocation.width = sheet->internal_allocation.width - 
	(2 * widget->style->klass->xthickness);
    sheet_allocation.height = sheet->internal_allocation.height -
	(2 * widget->style->klass->ythickness);
    /* 
     * here's where we decide to show/not show the scrollbars
     */
    vscrollbar_vis = 0;
    hscrollbar_vis = 0;

    for (i = 0; i <= 1; i++)
	{
	  if (SHEET_HEIGHT (sheet) <= sheet_allocation.height &&
	      sheet->vscrollbar_policy == GTK_POLICY_AUTOMATIC)
	    {
	      vscrollbar_vis = 0;
	    }
	  else
	    {
	      if (!vscrollbar_vis)
		{
		  vscrollbar_vis = 1;
		  sheet_allocation.width -= sheet->vscrollbar->requisition.width +
		    SCROLLBAR_SPACING (sheet);
		}  
	    }
	  
	  if (SHEET_WIDTH (sheet) <= sheet_allocation.width &&
	      sheet->hscrollbar_policy == GTK_POLICY_AUTOMATIC)
	    {
	      hscrollbar_vis = 0;
	    }
	  else
	    {
	      if (!hscrollbar_vis)
		{
		  hscrollbar_vis = 1;
		  sheet_allocation.height -= sheet->hscrollbar->requisition.height + 
		    SCROLLBAR_SPACING (sheet);	
		}  
	    }
	}

    sheet->sheet_window_width = sheet_allocation.width;
    sheet->sheet_window_height = sheet_allocation.height;

    gdk_window_move_resize (sheet->sheet_window,
			      sheet_allocation.x,
			      sheet_allocation.y,
			      sheet_allocation.width,
			      sheet_allocation.height);

    /* position the window which holds the column title buttons */
    sheet->column_title_area.x = widget->style->klass->xthickness +
                                   sheet->row_title_area.width;
    sheet->column_title_area.y = widget->style->klass->ythickness;
    sheet->column_title_area.width = sheet_allocation.width - 
                                       sheet->row_title_area.width;                         
    gdk_window_move_resize (sheet->column_title_window,
			      sheet->column_title_area.x,
			      sheet->column_title_area.y,
			      sheet->column_title_area.width,
			      sheet->column_title_area.height);

    /* column button allocation */
    size_allocate_column_title_buttons (sheet);

    /* position the window which holds the row title buttons */
    sheet->row_title_area.x = widget->style->klass->xthickness;
    sheet->row_title_area.y = widget->style->klass->ythickness +
                                sheet->column_title_area.height;
    sheet->row_title_area.height = sheet_allocation.height -
                                sheet->column_title_area.height;

    gdk_window_move_resize (sheet->row_title_window,
			      sheet->row_title_area.x,
			      sheet->row_title_area.y,
			      sheet->row_title_area.width,
			      sheet->row_title_area.height);

    /* row button allocation */
    size_allocate_row_title_buttons (sheet);

    adjust_scrollbars (sheet);

    /* allocate the vscrollbar */
    if (vscrollbar_vis)
	{
	  if (!GTK_WIDGET_VISIBLE (sheet->vscrollbar))
	    gtk_widget_show (sheet->vscrollbar);
	      
	  child_allocation.x = sheet->internal_allocation.x + 
	    sheet->internal_allocation.width -
	    sheet->vscrollbar->requisition.width;
	  child_allocation.y = sheet->internal_allocation.y;
	  child_allocation.width = sheet->vscrollbar->requisition.width;
	  child_allocation.height = sheet->internal_allocation.height -
	    (hscrollbar_vis ? (sheet->hscrollbar->requisition.height + SCROLLBAR_SPACING (sheet)) : 0);

	  gtk_widget_size_allocate (sheet->vscrollbar, &child_allocation);
	}
      else
	{
	   if (GTK_WIDGET_VISIBLE (sheet->vscrollbar))
		gtk_widget_hide (sheet->vscrollbar);
	}

    /* allocate the hscrollbar */
    if (hscrollbar_vis)
	{
	  if (!GTK_WIDGET_VISIBLE (sheet->hscrollbar))
	    gtk_widget_show (sheet->hscrollbar);
      
	  child_allocation.x = sheet->internal_allocation.x;
	  child_allocation.y = sheet->internal_allocation.y +
	    sheet->internal_allocation.height -
	    sheet->hscrollbar->requisition.height;
	  child_allocation.width = sheet->internal_allocation.width -
	    (vscrollbar_vis ? (sheet->vscrollbar->requisition.width + SCROLLBAR_SPACING (sheet)) : 0);
	  child_allocation.height = sheet->hscrollbar->requisition.height;

	  gtk_widget_size_allocate (sheet->hscrollbar, &child_allocation);
	}
      else
	{
	  if (GTK_WIDGET_VISIBLE (sheet->hscrollbar))
		gtk_widget_hide (sheet->hscrollbar);
	}
    

    sheet->view.row0=ROW_FROM_YPIXEL(sheet, sheet->column_title_area.height+1);
    sheet->view.rowi=ROW_FROM_YPIXEL(sheet, sheet->sheet_window_height);
    sheet->view.col0=COLUMN_FROM_XPIXEL(sheet, sheet->row_title_area.width+1);
    sheet->view.coli=COLUMN_FROM_XPIXEL(sheet, sheet->sheet_window_width);

    /* set the scrollbars adjustments */
    adjust_scrollbars (sheet);

    /* show sheet entry */
    if(sheet->state == GTK_SHEET_NORMAL){
     range.col0=sheet->active_cell.col;
     range.row0=sheet->active_cell.row;
     range.coli=sheet->active_cell.col;
     range.rowi=sheet->active_cell.row;
     if(gtk_sheet_range_isvisible(sheet, range))
       gdk_window_show(sheet->sheet_entry->window);
    }

    /* show column titles */
    gdk_window_show (sheet->column_title_window);

    /* show row titles */
    gdk_window_show (sheet->row_title_window);

    /* show global button */
    size_allocate_global_button(sheet);

    /* show backing pixmap */
    gtk_sheet_make_backing_pixmap(sheet, 0, 0); 

   }

}

static void
size_allocate_column_title_buttons (GtkSheet * sheet)
{
  gint i;

  if (!GTK_WIDGET_REALIZED (sheet))
    return;

  if(MAX_VISIBLE_COLUMN(sheet) == sheet->maxcol)
     gdk_window_clear_area (sheet->column_title_window,
	   		    0,0,
			    sheet->column_title_area.width, 
                            sheet->column_title_area.height);

  for (i = MIN_VISIBLE_COLUMN(sheet); i <= MAX_VISIBLE_COLUMN(sheet); i++)
    {
      gtk_sheet_button_draw(sheet,-1,i);
    }

}
	
static void
size_allocate_row_title_buttons (GtkSheet * sheet)
{
  gint i;

  if (!GTK_WIDGET_REALIZED (sheet))
    return;
  if(MAX_VISIBLE_ROW(sheet) == sheet->maxrow)
    gdk_window_clear_area (sheet->row_title_window,
			   0,0,
			   sheet->row_title_area.width, 
                           sheet->row_title_area.height);
  for(i = MIN_VISIBLE_ROW(sheet); i <= MAX_VISIBLE_ROW(sheet); i++)
    {
      gtk_sheet_button_draw(sheet,i,-1);
    }

}
	  
static void
gtk_sheet_recalc_top_ypixels(GtkSheet *sheet, gint row)
{
  gint i, cy;

  cy=sheet->column_title_area.height;
  for(i=0; i<=sheet->maxrow; i++){
      sheet->row[i].top_ypixel=cy;
      if(sheet->row[i].is_visible) cy+=sheet->row[i].height;
  }
}

static void
gtk_sheet_recalc_left_xpixels(GtkSheet *sheet, gint column)
{
  gint i, cx;

  cx=sheet->row_title_area.width;
  for(i=0; i<=sheet->maxcol; i++){
      sheet->column[i].left_xpixel=cx;
      if(sheet->column[i].is_visible) cx+=sheet->column[i].width;
  }

}



static void
gtk_sheet_size_allocate_entry(GtkSheet *sheet)
{
 GtkAllocation shentry_allocation;
 GtkSheetCellAttr attributes;
 gint i, row, col, sizel, sizer;
 gint size, max_size, text_size, column_width;

 max_size=GTK_SHENTRY(sheet->sheet_entry)->text_max_size;
 text_size=GTK_SHENTRY(sheet->sheet_entry)->text == NULL ? 0 :
           gdk_string_width(sheet->sheet_entry->style->font,
                          GTK_SHENTRY(sheet->sheet_entry)->text)+14;
 column_width=sheet->column[sheet->active_cell.col].width - 5;

 size=MIN(text_size, max_size);
 size=MAX(size,column_width);

 row=sheet->active_cell.row;
 col=sheet->active_cell.col;

 shentry_allocation.x=COLUMN_LEFT_XPIXEL(sheet,sheet->active_cell.col) + 3;
 shentry_allocation.y=ROW_TOP_YPIXEL(sheet,sheet->active_cell.row) + 2;
 shentry_allocation.width=column_width;
 shentry_allocation.height=sheet->row[sheet->active_cell.row].height - 5;

 if(!GTK_SHEET_ADJUST_TEXT(sheet)){
  switch(GTK_SHENTRY(sheet->sheet_entry)->justification){
   case GTK_JUSTIFY_CENTER:
/*     sizer=column_width/2;
     sizel=column_width/2;
     for(i=col+1; i<=MAX_VISIBLE_COLUMN(sheet); i++){
      if(gtk_sheet_cell_get_text(sheet, row, i)) break;
      if(sizer>text_size/2) break;
      sizer+=sheet->column[i].width;
     }
     for(i=col-1; i>=MIN_VISIBLE_COLUMN(sheet); i--){
      if(gtk_sheet_cell_get_text(sheet, row, i)) break;
      if(sizel>text_size/2) break;
      sizel+=sheet->column[i].width;
     }
     shentry_allocation.width=MIN(text_size,sizel+sizer);
     shentry_allocation.x=shentry_allocation.x-2+
                          column_width/2-MIN(sizel,text_size/2);
*/
     shentry_allocation.width=size;
     shentry_allocation.x=shentry_allocation.x+
                          column_width/2-size/2+1;
     break;
   case GTK_JUSTIFY_RIGHT:
     shentry_allocation.x=shentry_allocation.x+column_width-size+1;
   case GTK_JUSTIFY_LEFT:
     shentry_allocation.width=size;
  }
 }

 gtk_sheet_get_attributes(sheet, sheet->active_cell.row, sheet->active_cell.col, &attributes); 


 if(GTK_WIDGET_REALIZED(sheet->sheet_entry)){
  gdk_gc_set_foreground(GTK_SHENTRY(sheet->sheet_entry)->fg_gc, &attributes.foreground);
  gdk_gc_set_foreground(GTK_SHENTRY(sheet->sheet_entry)->bg_gc, &attributes.background);
  gdk_window_set_background(sheet->sheet_entry->window, &attributes.background);
  gdk_window_set_background(GTK_SHENTRY(sheet->sheet_entry)->text_area, &attributes.background);
  gdk_window_set_background(sheet->sheet_window, &attributes.background);
 }

 gtk_widget_size_allocate(sheet->sheet_entry, &shentry_allocation);
}

static void
gtk_sheet_entry_set_max_size(GtkSheet *sheet)
{
 gint i;
 gint size=0;
 gint sizel=0, sizer=0;
 gint row,col;

 row=sheet->active_cell.row;
 col=sheet->active_cell.col;

 if(GTK_SHEET_ADJUST_TEXT(sheet)) return; 

 switch(GTK_SHENTRY(sheet->sheet_entry)->justification){
  case GTK_JUSTIFY_LEFT:
    for(i=col+1; i<=MAX_VISIBLE_COLUMN(sheet); i++){
     if(gtk_sheet_cell_get_text(sheet, row, i)) break;
     size+=sheet->column[i].width;
    }
    break;
  case GTK_JUSTIFY_RIGHT:
    for(i=col-1; i>=MIN_VISIBLE_COLUMN(sheet); i--){
     if(gtk_sheet_cell_get_text(sheet, row, i)) break;
     size+=sheet->column[i].width;
    }
    break;
  case GTK_JUSTIFY_CENTER:
    for(i=col+1; i<=MAX_VISIBLE_COLUMN(sheet); i++){
     if(gtk_sheet_cell_get_text(sheet, row, i)) break;
     sizer+=sheet->column[i].width;
    }
    for(i=col-1; i>=MIN_VISIBLE_COLUMN(sheet); i--){
     if(gtk_sheet_cell_get_text(sheet, row, i)) break;
     sizel+=sheet->column[i].width;
    }
    size=2*MIN(sizel, sizer);
 }

 if(size!=0) size+=sheet->column[col].width;
 GTK_SHENTRY(sheet->sheet_entry)->text_max_size=size;

}

static void
create_sheet_entry(GtkSheet *sheet)
{
 GtkWidget *entry;
 GdkEventKey *key;

 entry=sheet->sheet_entry=gtk_shentry_new_with_max_length(MAX_TEXT_LENGTH);
 gtk_widget_set_parent(entry,GTK_WIDGET(sheet));
 gtk_widget_size_request(entry, &entry->requisition);
 gtk_sheet_size_allocate_entry(sheet);
 gtk_widget_set_parent_window (sheet->sheet_entry, sheet->sheet_window);
 gtk_signal_connect_object(GTK_OBJECT(entry),"activate",
                    GTK_SIGNAL_FUNC(gtk_sheet_key_press), 
                    GTK_OBJECT(GTK_WIDGET(sheet)));
 
 gtk_widget_show (entry);
}


/* container virtuals */
static void
gtk_sheet_foreach (GtkContainer * container,
		   GtkCallback callback,
		   gpointer callback_data)
{
  GtkSheet *sheet;
  gint i;

  g_return_if_fail (container != NULL);
  g_return_if_fail (GTK_IS_SHEET (container));
  g_return_if_fail (callback != NULL);

  sheet = GTK_SHEET (container);

}


/* BUTTONS */
static void
row_button_set (GtkSheet *sheet, gint row)
{
  if(sheet->row[row].button.state == GTK_STATE_ACTIVE) return;

  sheet->row[row].button.state = GTK_STATE_ACTIVE;
  gtk_sheet_button_draw(sheet, row, -1);
 
}

static void
column_button_set (GtkSheet *sheet, gint column)
{
  if(sheet->column[column].button.state == GTK_STATE_ACTIVE) return;

  sheet->column[column].button.state = GTK_STATE_ACTIVE;
  gtk_sheet_button_draw(sheet, -1, column);
 
}

static void
row_button_release (GtkSheet *sheet, gint row)
{
  if(sheet->row[row].button.state == GTK_STATE_NORMAL) return;

  sheet->row[row].button.state = GTK_STATE_NORMAL;
  gtk_sheet_button_draw(sheet, row, -1);
}

static void
column_button_release (GtkSheet *sheet, gint column)
{
  if(sheet->column[column].button.state == GTK_STATE_NORMAL) return;

  sheet->column[column].button.state = GTK_STATE_NORMAL;
  gtk_sheet_button_draw(sheet, -1, column);
}

static void
gtk_sheet_button_draw (GtkSheet *sheet, gint row, gint column)
{
  GdkWindow *window;
  GtkShadowType shadow_type;
  gint width, height;
  gint x,y;
  gint index;
  gchar label[10];
  int text_width, text_height;
  GtkSheetButton *button;
  GdkRectangle allocation;
  gint is_sensitive;

  if(!GTK_WIDGET_REALIZED(GTK_WIDGET(sheet))) return;

  if(row >= 0 && !sheet->row[row].is_visible) return;
  if(column >= 0 && !sheet->column[column].is_visible) return;
  if(column>=0 && column <MIN_VISIBLE_COLUMN(sheet)) return;
  if(column>=0 && column >MAX_VISIBLE_COLUMN(sheet)) return;
  if(row>=0 && row <MIN_VISIBLE_ROW(sheet)) return;
  if(row>=0 && row >MAX_VISIBLE_ROW(sheet)) return;

  if(row==-1){
     window=sheet->column_title_window;
     button=&sheet->column[column].button;
     index=column;
     x = COLUMN_LEFT_XPIXEL(sheet, column)+CELL_SPACING-sheet->row_title_area.width;
     y = 0;
     width = sheet->column[column].width;
     height = sheet->column_title_area.height;
     is_sensitive=sheet->column[column].is_sensitive;
  }
  if(column==-1){
     window=sheet->row_title_window;
     button=&sheet->row[row].button;
     index=row;
     x = 0;
     y = ROW_TOP_YPIXEL(sheet, row)+CELL_SPACING-sheet->column_title_area.height;
     width = sheet->row_title_area.width;
     height = sheet->row[row].height;
     is_sensitive=sheet->row[row].is_sensitive;
  }
  allocation.x=x;
  allocation.y=y;
  allocation.width=width;
  allocation.height=height;

  gtk_style_set_background (GTK_WIDGET(sheet)->style,window, button->state);

  gdk_window_clear_area (window,
                         x, y,
	                 width, height);

  gdk_draw_rectangle (window,
                      GTK_WIDGET(sheet)->style->bg_gc[GTK_STATE_NORMAL],
	              FALSE, x, y, width - 1, height - 1);
  gdk_draw_rectangle (window, 
                      GTK_WIDGET(sheet)->style->bg_gc[GTK_STATE_NORMAL],
	              FALSE, x + 1, y + 1, width - 3, height - 3);

  if (button->state == GTK_STATE_ACTIVE){
     shadow_type = GTK_SHADOW_IN;
     gdk_draw_rectangle (window,
		   GTK_WIDGET(sheet)->style->bg_gc[button->state], FALSE,
	           x + 1, y + 1, width - 4, height - 4);
  }else{
     shadow_type = GTK_SHADOW_OUT;
     gdk_draw_rectangle (window,
	           GTK_WIDGET(sheet)->style->bg_gc[button->state], FALSE,
		   x + 2, y + 2, width - 5, height - 5);
  }

  gtk_draw_shadow (GTK_WIDGET(sheet)->style, window,
		   button->state, shadow_type,
		   x, y, width, height);


  text_height=GTK_WIDGET(sheet)->style->font->ascent + 
              GTK_WIDGET(sheet)->style->font->descent;
    
  y += (height - text_height)/ 2 + GTK_WIDGET(sheet)->style->font->ascent;

  gdk_gc_set_clip_rectangle(GTK_WIDGET(sheet)->style->black_gc, &allocation);

  if(button->label && strlen(button->label)>0){
           text_width = gdk_string_width (GTK_WIDGET (sheet)->style->font,
   			      button->label);
           if(button->state==GTK_STATE_INSENSITIVE || !is_sensitive)
             gdk_draw_string (window, 
		      GTK_WIDGET(sheet)->style->font,
	  	      GTK_WIDGET(sheet)->style->white_gc,
		      x + (width - text_width) / 2 + 1  , y + 1,
	      	      button->label);

           gdk_draw_string (window, 
		    GTK_WIDGET(sheet)->style->font,
	  	    GTK_WIDGET(sheet)->style->fg_gc[button->state],
		    x + (width - text_width) / 2 , y,
	      	    button->label);
  }else{
           sprintf(label,"%d",index);
           text_width = gdk_string_width (GTK_WIDGET (sheet)->style->font,
   			      label);

           if(button->state==GTK_STATE_INSENSITIVE || !is_sensitive)
             gdk_draw_string (window, 
		      GTK_WIDGET(sheet)->style->font,
	  	      GTK_WIDGET(sheet)->style->white_gc,
		      x + (width - text_width) / 2 + 1  , y + 1,
	      	      label);

           gdk_draw_string (window, 
		    GTK_WIDGET(sheet)->style->font,
	  	    GTK_WIDGET(sheet)->style->fg_gc[button->state],
		    x + (width - text_width) / 2 , y,
	      	    label);
  }

  gdk_gc_set_clip_rectangle(GTK_WIDGET(sheet)->style->black_gc, NULL);
   
}


/* SCROLLBARS
 *
 * functions:
 *   create_scrollbars
 *   adjust_scrollbars
 *   vadjustment_changed
 *   hadjustment_changed
 *   vadjustment_value_changed
 *   hadjustment_value_changed */
static void
create_scrollbars (GtkSheet * sheet)
{
  GtkAdjustment *adjustment;

  sheet->vscrollbar = gtk_vscrollbar_new (NULL);

  adjustment = gtk_range_get_adjustment (GTK_RANGE (sheet->vscrollbar));
  GTK_RANGE(sheet->vscrollbar)->policy=GTK_UPDATE_DISCONTINUOUS;

  gtk_signal_connect (GTK_OBJECT (adjustment), "changed",
		      (GtkSignalFunc) vadjustment_changed,
		      (gpointer) sheet);

  gtk_signal_connect (GTK_OBJECT (adjustment), "value_changed",
		      (GtkSignalFunc) vadjustment_value_changed,
		      (gpointer) sheet);

  gtk_widget_set_parent (sheet->vscrollbar, GTK_WIDGET (sheet));
  gtk_widget_show (sheet->vscrollbar);

  sheet->hscrollbar = gtk_hscrollbar_new (NULL);

  adjustment = gtk_range_get_adjustment (GTK_RANGE (sheet->hscrollbar));
  GTK_RANGE(sheet->hscrollbar)->policy=GTK_UPDATE_DISCONTINUOUS;

  gtk_signal_connect (GTK_OBJECT (adjustment), "changed",
		      (GtkSignalFunc) hadjustment_changed,
		      (gpointer) sheet);

  gtk_signal_connect (GTK_OBJECT (adjustment), "value_changed",
		      (GtkSignalFunc) hadjustment_value_changed,
		      (gpointer) sheet);

  gtk_widget_set_parent (sheet->hscrollbar, GTK_WIDGET (sheet));
  gtk_widget_show (sheet->hscrollbar);
}

static void
adjust_scrollbars (GtkSheet * sheet)
{
  GTK_RANGE (sheet->vscrollbar)->adjustment->page_size = sheet->sheet_window_height;
  GTK_RANGE (sheet->vscrollbar)->adjustment->page_increment = sheet->sheet_window_height / 2;
  GTK_RANGE (sheet->vscrollbar)->adjustment->step_increment = DEFAULT_ROW_HEIGHT(GTK_WIDGET(sheet));
  GTK_RANGE (sheet->vscrollbar)->adjustment->lower = 0;
  GTK_RANGE (sheet->vscrollbar)->adjustment->upper = SHEET_HEIGHT (sheet)+
  DEFAULT_ROW_HEIGHT(GTK_WIDGET(sheet));

  if (sheet->sheet_window_height - sheet->voffset > SHEET_HEIGHT (sheet))
    {
      GTK_RANGE (sheet->vscrollbar)->adjustment->value = SHEET_HEIGHT (sheet) - 
	sheet->sheet_window_height;
      gtk_signal_emit_by_name (GTK_OBJECT (GTK_RANGE (sheet->vscrollbar)->adjustment), 
			       "value_changed");
    }

  GTK_RANGE (sheet->hscrollbar)->adjustment->page_size = sheet->sheet_window_width;
  GTK_RANGE (sheet->hscrollbar)->adjustment->page_increment = sheet->sheet_window_width / 2;
  GTK_RANGE (sheet->hscrollbar)->adjustment->step_increment = DEFAULT_COLUMN_WIDTH;
  GTK_RANGE (sheet->hscrollbar)->adjustment->lower = 0;
  GTK_RANGE (sheet->hscrollbar)->adjustment->upper = SHEET_WIDTH (sheet)+
  DEFAULT_COLUMN_WIDTH;

  if (sheet->sheet_window_width - sheet->hoffset > SHEET_WIDTH (sheet))
    {
      GTK_RANGE (sheet->hscrollbar)->adjustment->value = SHEET_WIDTH (sheet) - 
	sheet->sheet_window_width;
      gtk_signal_emit_by_name (GTK_OBJECT (GTK_RANGE (sheet->hscrollbar)->adjustment), 
			       "value_changed");
    }

  if (SHEET_HEIGHT (sheet) <= sheet->sheet_window_height &&
      sheet->vscrollbar_policy == GTK_POLICY_AUTOMATIC)
    {
      if (GTK_WIDGET_VISIBLE (sheet->vscrollbar))
	{
	  gtk_widget_hide (sheet->vscrollbar);
	  gtk_widget_queue_resize (GTK_WIDGET (sheet));
	}
    }
  else
    {
      if (!GTK_WIDGET_VISIBLE (sheet->vscrollbar))
	{
	  gtk_widget_show (sheet->vscrollbar);
	  gtk_widget_queue_resize (GTK_WIDGET (sheet));
	}
    }

  if (SHEET_WIDTH (sheet) <= sheet->sheet_window_width &&
      sheet->hscrollbar_policy == GTK_POLICY_AUTOMATIC)
    {
      if (GTK_WIDGET_VISIBLE (sheet->hscrollbar))
	{
	  gtk_widget_hide (sheet->hscrollbar);
	  gtk_widget_queue_resize (GTK_WIDGET (sheet));
	}
    }
  else
    {
      if (!GTK_WIDGET_VISIBLE (sheet->hscrollbar))
	{
	  gtk_widget_show (sheet->hscrollbar);
	  gtk_widget_queue_resize (GTK_WIDGET (sheet));
	}
    }

  gtk_signal_emit_by_name (GTK_OBJECT (GTK_RANGE (sheet->vscrollbar)->adjustment), "changed");
  gtk_signal_emit_by_name (GTK_OBJECT (GTK_RANGE (sheet->hscrollbar)->adjustment), "changed");
}


static void
vadjustment_changed (GtkAdjustment * adjustment,
			       gpointer data)
{
  GtkSheet *sheet;

  g_return_if_fail (adjustment != NULL);
  g_return_if_fail (data != NULL);

  sheet = GTK_SHEET (data);
}

static void
hadjustment_changed (GtkAdjustment * adjustment,
			       gpointer data)
{
  GtkSheet *sheet;

  g_return_if_fail (adjustment != NULL);
  g_return_if_fail (data != NULL);

  sheet = GTK_SHEET (data);
}


static void
vadjustment_value_changed (GtkAdjustment * adjustment,
				     gpointer data)
{
  GtkSheet *sheet;
  GdkRectangle area;
  gint diff, value, old_value;
  GtkSheetRange drawing_range;
  gint i;
  gint row, new_row;
  gint y=0;

  g_return_if_fail (adjustment != NULL);
  g_return_if_fail (data != NULL);
  g_return_if_fail (GTK_IS_SHEET (data));

  sheet = GTK_SHEET (data);


  row=ROW_FROM_YPIXEL(sheet,sheet->column_title_area.height + CELL_SPACING);
    
  old_value=0;
  for(i=0; i< row; i++)
    if(sheet->row[i].is_visible) old_value+=sheet->row[i].height;

  for(i=0; i<= sheet->maxrow; i++){
   if(sheet->row[i].is_visible) y+=sheet->row[i].height;
   if(y > adjustment->value) break;
  }
  y-=sheet->row[i].height;
  new_row=i;

  if (adjustment->value > -sheet->voffset){
   if(row == new_row){
    new_row+=1;
    y=y+sheet->row[row].height;
   }
  }
  adjustment->value=y;
 
  if(new_row == 0){
   GTK_RANGE(sheet->vscrollbar)->adjustment->step_increment=
   sheet->row[0].height;
  }else{
   GTK_RANGE(sheet->vscrollbar)->adjustment->step_increment=
   MIN(sheet->row[new_row].height, sheet->row[new_row-1].height);
  }

  GTK_RANGE(sheet->vscrollbar)->adjustment->value=adjustment->value;
  gtk_range_slider_update (GTK_RANGE(sheet->vscrollbar));
  gtk_widget_queue_draw(sheet->vscrollbar);

  value = adjustment->value;

  if (adjustment == gtk_range_get_adjustment (GTK_RANGE (sheet->vscrollbar)))
    {
   
      if (value >= -sheet->voffset)
	{
	  /* scroll down */
	  diff = value + sheet->voffset;
	}
       else
	{
	  /* scroll up */
	  diff = -sheet->voffset - value;
	}

      sheet->voffset = -value;
    }
 
  sheet->view.row0=ROW_FROM_YPIXEL(sheet, sheet->column_title_area.height+1);
  sheet->view.rowi=ROW_FROM_YPIXEL(sheet, sheet->sheet_window_height);

  gdk_window_hide(sheet->sheet_entry->window);
  gtk_sheet_range_draw(sheet, NULL);
  size_allocate_row_title_buttons(sheet);
}

static void
hadjustment_value_changed (GtkAdjustment * adjustment,
				     gpointer data)
{
  GtkSheet *sheet;
  GdkRectangle area;
  gint i, diff, value, old_value;
  GtkSheetRange drawing_range ;
  gint column, new_column;
  gint x=0;

  g_return_if_fail (adjustment != NULL);
  g_return_if_fail (data != NULL);
  g_return_if_fail (GTK_IS_SHEET (data));
  sheet = GTK_SHEET (data);

  column=COLUMN_FROM_XPIXEL(sheet,sheet->row_title_area.width + CELL_SPACING);

  old_value=0;
  for(i=0; i< column; i++)
    if(sheet->column[i].is_visible) old_value+=sheet->column[i].width;

  for(i=0; i<= sheet->maxcol; i++){
   if(sheet->column[i].is_visible) x+=sheet->column[i].width;
   if(x > adjustment->value) break;
  }
  x-=sheet->column[i].width;
  new_column=i;

  if (adjustment->value > -sheet->hoffset){
   if(column == new_column && column != sheet->maxcol){
    new_column+=1;
    x=x+sheet->column[column].width;
   }
  }

  adjustment->value=x;


  if(new_column == 0){
   GTK_RANGE(sheet->hscrollbar)->adjustment->step_increment=
   sheet->column[0].width;
  }else{
   GTK_RANGE(sheet->hscrollbar)->adjustment->step_increment=
   MIN(sheet->column[new_column].width, sheet->column[new_column-1].width);
  }

  GTK_RANGE(sheet->hscrollbar)->adjustment->value=adjustment->value;
  gtk_range_slider_update (GTK_RANGE(sheet->hscrollbar));
  gtk_widget_queue_draw(sheet->hscrollbar);

  value = adjustment->value;

  if (adjustment == gtk_range_get_adjustment (GTK_RANGE (sheet->hscrollbar)))
    {
      if (value >= -sheet->hoffset)
	{
	  /* scroll right */
	  diff = value + sheet->hoffset;
	}
      else
	{
	  /* scroll left */
	  diff = -sheet->hoffset - value;
	}

      sheet->hoffset = -value;
    }

  sheet->view.col0=COLUMN_FROM_XPIXEL(sheet, sheet->row_title_area.width+1);
  sheet->view.coli=COLUMN_FROM_XPIXEL(sheet, sheet->sheet_window_width);

  gdk_window_hide(sheet->sheet_entry->window);
  gtk_sheet_range_draw(sheet, NULL);
  size_allocate_column_title_buttons(sheet);

}

	

/* COLUMN RESIZING */
static void                          
draw_xor_vline (GtkSheet * sheet)
{
  GtkWidget *widget;
  
  g_return_if_fail (sheet != NULL);
  
  widget = GTK_WIDGET (sheet);

  gdk_draw_line (widget->window, sheet->xor_gc,  
                 sheet->x_drag,                                       
                 widget->style->klass->ythickness+sheet->column_title_area.height,                               
                 sheet->x_drag,                                             
                 sheet->sheet_window_height + 1);
}

/* ROW RESIZING */
static void                          
draw_xor_hline (GtkSheet * sheet)
{
  GtkWidget *widget;
  
  g_return_if_fail (sheet != NULL);
  
  widget = GTK_WIDGET (sheet);

  gdk_draw_line (widget->window, sheet->xor_gc,  
		 widget->style->klass->xthickness+sheet->row_title_area.width,
                 sheet->y_drag,                                       
                        
	         sheet->sheet_window_width + 1,                      
                 sheet->y_drag);                                             
}

/* SELECTED RANGE */
static void
draw_xor_rectangle(GtkSheet *sheet, GtkSheetRange range)
{
   gint i;
   GdkRectangle clip_area, area;
   GdkGCValues values, oldvalues;

   area.x=COLUMN_LEFT_XPIXEL(sheet, range.col0);
   area.y=ROW_TOP_YPIXEL(sheet, range.row0);
   area.width=COLUMN_LEFT_XPIXEL(sheet, range.coli)-area.x+
                                        sheet->column[range.coli].width;
   area.height=ROW_TOP_YPIXEL(sheet, range.rowi)-area.y+
                                        sheet->row[range.rowi].height;

   clip_area.x=sheet->row_title_area.width;
   clip_area.y=sheet->column_title_area.height;
   clip_area.width=sheet->sheet_window_width;
   clip_area.height=sheet->sheet_window_height;

   if(area.x<0) {
      area.width=area.width+area.x;
      area.x=0;
   }
   if(area.width>clip_area.width) area.width=clip_area.width+10;
   if(area.y<0) {
      area.height=area.height+area.y;
      area.y=0;
   }
   if(area.height>clip_area.height) area.height=clip_area.height+10;

/*   gdk_gc_get_values(sheet->xor_gc, &values);
   gdk_gc_destroy(sheet->xor_gc);

   oldvalues=values;

   values.foreground=GTK_WIDGET(sheet)->style->bg[1];

   sheet->xor_gc = gdk_gc_new_with_values (GTK_WIDGET(sheet)->window,
					  &values,
					  GDK_GC_FOREGROUND |
					  GDK_GC_FUNCTION |
					  GDK_GC_SUBWINDOW);


*/
   gdk_gc_set_foreground(sheet->xor_gc, &GTK_WIDGET(sheet)->style->bg[1]);

   gdk_gc_set_clip_rectangle(sheet->xor_gc, &clip_area);

   for(i=-1;i<=1;i=i++)
     gdk_draw_rectangle(sheet->sheet_window,
                        sheet->xor_gc,
		        FALSE,
		        area.x+i, area.y+i,
                        area.width-2*i, area.height-2*i);


   gdk_gc_set_clip_rectangle(sheet->xor_gc, NULL);

   gdk_gc_set_foreground(sheet->xor_gc, &GTK_WIDGET(sheet)->style->white);

/*   gdk_gc_destroy(sheet->xor_gc);
   sheet->xor_gc = gdk_gc_new_with_values (GTK_WIDGET(sheet)->window,
					  &oldvalues,
					  GDK_GC_FOREGROUND |
					  GDK_GC_FUNCTION |
					  GDK_GC_SUBWINDOW);
*/

}                      

  
/* this function returns the new width of the column being resized given
 * the column and x position of the cursor; the x cursor position is passed
 * in as a pointer and automagicly corrected if it's beyond min/max limits */
static gint
new_column_width (GtkSheet * sheet,
		  gint column,
		  gint * x)
{
  gint cx, width;

  cx = *x;

  /* first translate the x position from widget->window
   * to sheet->sheet_window */
  cx -= GTK_WIDGET (sheet)->style->klass->xthickness;


  /* you can't shrink a column to less than its minimum width */
  if (cx < (COLUMN_LEFT_XPIXEL (sheet, column) + COLUMN_MIN_WIDTH))
    {
      *x = cx = COLUMN_LEFT_XPIXEL (sheet, column) + COLUMN_MIN_WIDTH +
	GTK_WIDGET (sheet)->style->klass->xthickness;
      cx -= GTK_WIDGET (sheet)->style->klass->xthickness;
    }

  /* don't grow past the end of the window */
  /*
  if (cx > sheet->sheet_window_width)
    {
      *x = cx = sheet->sheet_window_width + GTK_WIDGET (sheet)->style->klass->xthickness;
      cx -= GTK_WIDGET (sheet)->style->klass->xthickness;
    }
    */
  /* calculate new column width making sure it doesn't end up
   * less than the minimum width */
  width = (cx - COLUMN_LEFT_XPIXEL (sheet, column));
  if (width < COLUMN_MIN_WIDTH)
    width = COLUMN_MIN_WIDTH;

  sheet->column[column].width = width;
  gtk_sheet_recalc_left_xpixels(sheet, column+1);
  sheet->view.coli=COLUMN_FROM_XPIXEL(sheet, sheet->sheet_window_width);
  size_allocate_column_title_buttons (sheet);

  return width;
}

/* this function returns the new height of the row being resized given
 * the row and y position of the cursor; the y cursor position is passed
 * in as a pointer and automagicly corrected if it's beyond min/max limits */
static gint
new_row_height (GtkSheet * sheet,
		  gint row,
		  gint * y)
{
  gint cy, height;

  cy = *y;

  /* first translate the y position from widget->window
   * to sheet->sheet_window */
  cy -= GTK_WIDGET (sheet)->style->klass->ythickness;

  /* you can't shrink a row to less than its minimum width */
  if (cy < (ROW_TOP_YPIXEL (sheet, row) + 
DEFAULT_ROW_HEIGHT(GTK_WIDGET(sheet))))
    {
      *y = cy = ROW_TOP_YPIXEL (sheet, row) + 
DEFAULT_ROW_HEIGHT(GTK_WIDGET(sheet)) +
	GTK_WIDGET (sheet)->style->klass->ythickness;
      cy -= GTK_WIDGET (sheet)->style->klass->ythickness;
    }

  /* don't grow past the end of the window */
  /*
  if (cy > sheet->sheet_window_height)
    {
      *y = cy = sheet->sheet_window_height + GTK_WIDGET (sheet)->style->klass->ythickness;
      cy -= GTK_WIDGET (sheet)->style->klass->ythickness;
    }
    */
  /* calculate new row height making sure it doesn't end up
   * less than the minimum height */
  height = (cy - ROW_TOP_YPIXEL (sheet, row));
  if (height < DEFAULT_ROW_HEIGHT(GTK_WIDGET(sheet)))
    height = DEFAULT_ROW_HEIGHT(GTK_WIDGET(sheet));

  sheet->row[row].height = height;
  gtk_sheet_recalc_top_ypixels(sheet, row);
  sheet->view.rowi=ROW_FROM_YPIXEL(sheet, sheet->sheet_window_height);
  size_allocate_row_title_buttons (sheet);

  return height;
}

void
gtk_sheet_set_column_width (GtkSheet * sheet,
			    gint column,
			    gint width)
{
  g_return_if_fail (sheet != NULL);
  g_return_if_fail (GTK_IS_SHEET (sheet));

  if (column < 0 || column >= sheet->maxcol)
    return;

  sheet->column[column].width = width;

  gtk_sheet_recalc_left_xpixels(sheet, column+1);

  if(GTK_WIDGET_REALIZED(GTK_WIDGET(sheet)) && !GTK_SHEET_IS_FROZEN(sheet)){
    size_allocate_column_title_buttons (sheet);
    adjust_scrollbars (sheet);
    gtk_sheet_size_allocate_entry(sheet);
    gtk_sheet_range_draw (sheet, NULL);
  }

}

void
gtk_sheet_set_row_height (GtkSheet * sheet,
			    gint row,
			    gint height)
{
  g_return_if_fail (sheet != NULL);
  g_return_if_fail (GTK_IS_SHEET (sheet));

  if (row < 0 || row >= sheet->maxrow)
    return;

  sheet->row[row].height = height;

  gtk_sheet_recalc_top_ypixels(sheet, row+1);

  if(GTK_WIDGET_REALIZED(GTK_WIDGET(sheet)) && !GTK_SHEET_IS_FROZEN(sheet)){
    size_allocate_row_title_buttons (sheet);
    adjust_scrollbars (sheet);
    gtk_sheet_size_allocate_entry(sheet);
    gtk_sheet_range_draw (sheet, NULL);
  }

}


void
gtk_sheet_add_column(GtkSheet *sheet, gint ncols)
{
 GtkSheetRange range;
 gint i;

 g_return_if_fail (sheet != NULL);
 g_return_if_fail (GTK_IS_SHEET (sheet));

 if(GTK_SHEET_IS_LOCKED(sheet)) return;

 AddColumn(sheet, ncols);
 adjust_scrollbars(sheet);

 if(sheet->state==GTK_SHEET_ROW_SELECTED) sheet->range.coli+=ncols;

 if(!GTK_SHEET_IS_FROZEN(sheet)){
  gtk_signal_emit_by_name (GTK_OBJECT (GTK_RANGE(sheet->vscrollbar)->adjustment), "value_changed");
 }
}

void
gtk_sheet_add_row(GtkSheet *sheet, gint nrows)
{
 GtkSheetRange range;
 gint i;

 g_return_if_fail (sheet != NULL);
 g_return_if_fail (GTK_IS_SHEET (sheet));

 if(GTK_SHEET_IS_LOCKED(sheet)) return;

 AddRow(sheet, nrows);
 adjust_scrollbars(sheet);

 if(sheet->state==GTK_SHEET_COLUMN_SELECTED) sheet->range.rowi+=nrows;

 if(!GTK_SHEET_IS_FROZEN(sheet)){
  gtk_signal_emit_by_name (GTK_OBJECT (GTK_RANGE (sheet->hscrollbar)->adjustment), "value_changed");
 }
}

void
gtk_sheet_insert_rows(GtkSheet *sheet, gint row, gint nrows)
{
 GtkSheetRange range;
 gint i;

 g_return_if_fail (sheet != NULL);
 g_return_if_fail (GTK_IS_SHEET (sheet));

 if(GTK_SHEET_IS_LOCKED(sheet)) return;

 gtk_sheet_unselect_range(sheet, NULL);
 for(i=1;i<=nrows;i++){
           InsertRow(sheet, row);
 }
 if(sheet->state != GTK_SHEET_NORMAL) 
     gtk_sheet_select_range(sheet, NULL);

 adjust_scrollbars(sheet);

 if(sheet->state==GTK_SHEET_COLUMN_SELECTED) sheet->range.rowi+=nrows;
/* if(sheet->state==GTK_SHEET_ROW_SELECTED) sheet->state=NORMAL;
*/

 if(!GTK_SHEET_IS_FROZEN(sheet)){
  gtk_signal_emit_by_name (GTK_OBJECT (GTK_RANGE (sheet->vscrollbar)->adjustment), "value_changed");
 }
}

void
gtk_sheet_insert_columns(GtkSheet *sheet, gint col, gint ncols)
{
 GtkSheetRange range;
 gint i;

 g_return_if_fail (sheet != NULL);
 g_return_if_fail (GTK_IS_SHEET (sheet));

 if(GTK_SHEET_IS_LOCKED(sheet)) return;

 gtk_sheet_unselect_range(sheet, NULL);

 for(i=1;i<=ncols;i++){
           InsertColumn(sheet, col);
 }
 if(sheet->state != GTK_SHEET_NORMAL)
     gtk_sheet_select_range(sheet, NULL);

 adjust_scrollbars(sheet);
 if(sheet->state==GTK_SHEET_ROW_SELECTED) sheet->range.coli+=ncols;
/* if(sheet->state==GTK_SHEET_COLUMN_SELECTED) sheet->state=GTK_SHEET_NORMAL;
*/

 if(!GTK_SHEET_IS_FROZEN(sheet)){
  gtk_signal_emit_by_name (GTK_OBJECT (GTK_RANGE (sheet->hscrollbar)->adjustment), "value_changed");
 }
}

void
gtk_sheet_delete_rows(GtkSheet *sheet, gint row, gint nrows)
{
 GtkSheetRange range;
 gint i;

 g_return_if_fail (sheet != NULL);
 g_return_if_fail (GTK_IS_SHEET (sheet));

 if(GTK_SHEET_IS_LOCKED(sheet)) return;

 gtk_sheet_unselect_range(sheet, NULL);

 for(i=1; i<=nrows; i++)
           DeleteRow(sheet, row);

 adjust_scrollbars(sheet);

 sheet->state=GTK_SHEET_NORMAL;

 if(!GTK_SHEET_IS_FROZEN(sheet)){
  gtk_signal_emit_by_name (GTK_OBJECT (GTK_RANGE (sheet->vscrollbar)->adjustment), "value_changed");
 }
}

void
gtk_sheet_delete_columns(GtkSheet *sheet, gint col, gint ncols)
{
 GtkSheetRange range;
 gint i;

 g_return_if_fail (sheet != NULL);
 g_return_if_fail (GTK_IS_SHEET (sheet));

 if(GTK_SHEET_IS_LOCKED(sheet)) return;

 gtk_sheet_unselect_range(sheet, NULL);

 for(i=1; i<=ncols; i++)
    DeleteColumn(sheet, col);

 adjust_scrollbars(sheet);

 sheet->state=GTK_SHEET_NORMAL;

 if(!GTK_SHEET_IS_FROZEN(sheet)){
  gtk_signal_emit_by_name (GTK_OBJECT (GTK_RANGE (sheet->hscrollbar)->adjustment), "value_changed");
 }
}

void
gtk_sheet_range_set_background(GtkSheet *sheet, GtkSheetRange range, GdkColor *color)
{
  GtkSheetAttr attributes;

  g_return_if_fail (sheet != NULL);
  g_return_if_fail (GTK_IS_SHEET (sheet));

  attributes.type = GTK_SHEET_BACKGROUND;

  attributes.range=range;
  attributes.value.background=*color;
 
  gtk_sheet_set_range_attributes(sheet, attributes); 

}

void
gtk_sheet_range_set_foreground(GtkSheet *sheet, GtkSheetRange range, GdkColor *color)
{
  GtkSheetAttr attributes;

  g_return_if_fail (sheet != NULL);
  g_return_if_fail (GTK_IS_SHEET (sheet));

  attributes.type = GTK_SHEET_FOREGROUND;
  
  attributes.range=range;
  attributes.value.foreground=*color;  

  gtk_sheet_set_range_attributes(sheet, attributes); 
  
}

void
gtk_sheet_range_set_justification(GtkSheet *sheet, GtkSheetRange range, gint just)
{
  GtkSheetAttr attributes;

  g_return_if_fail (sheet != NULL);
  g_return_if_fail (GTK_IS_SHEET (sheet));

  attributes.type = GTK_SHEET_JUSTIFICATION;
  
  attributes.range=range;
  attributes.value.justification = just;

  gtk_sheet_set_range_attributes(sheet, attributes); 
    
}

void
gtk_sheet_range_set_editable(GtkSheet *sheet, GtkSheetRange range, gint editable)
{
  GtkSheetAttr attributes;
 
  g_return_if_fail (sheet != NULL);
  g_return_if_fail (GTK_IS_SHEET (sheet));

  attributes.type = GTK_SHEET_IS_EDITABLE;
  
  attributes.range=range;
  attributes.value.is_editable = editable;

  gtk_sheet_set_range_attributes(sheet, attributes); 
    
}

void
gtk_sheet_range_set_visible(GtkSheet *sheet, GtkSheetRange range, gint visible)
{
  GtkSheetAttr attributes;

  g_return_if_fail (sheet != NULL);
  g_return_if_fail (GTK_IS_SHEET (sheet));

  attributes.type = GTK_SHEET_IS_VISIBLE;
  
  attributes.range=range;
  attributes.value.is_visible = visible;

  gtk_sheet_set_range_attributes(sheet, attributes); 
    
}

void
gtk_sheet_range_set_border(GtkSheet *sheet, GtkSheetRange range, gint mask, 
gint width, gint line_style)
{
  GtkSheetAttr attributes;

  g_return_if_fail (sheet != NULL);
  g_return_if_fail (GTK_IS_SHEET (sheet));

  attributes.type = GTK_SHEET_BORDER;

  attributes.range=range;
  attributes.value.border.mask = mask;
  attributes.value.border.width = width;
  attributes.value.border.line_style=line_style;
  attributes.value.border.cap_style=GDK_CAP_NOT_LAST;
  attributes.value.border.join_style=GDK_JOIN_MITER;
  gtk_sheet_set_range_attributes(sheet, attributes);      
  
}

void
gtk_sheet_range_set_border_color(GtkSheet *sheet, GtkSheetRange range, GdkColor *color)
{
  GtkSheetAttr attributes;

  g_return_if_fail (sheet != NULL);
  g_return_if_fail (GTK_IS_SHEET (sheet));

  attributes.type = GTK_SHEET_BORDER_COLOR;

  attributes.range=range;
  attributes.value.border.color = *color;

  gtk_sheet_set_range_attributes(sheet, attributes);  
  
}

static void
gtk_sheet_set_range_attributes(GtkSheet *sheet, GtkSheetAttr attributes)
{
  gint i, j;
  gint nth;

  nth = sheet->maxrange++;
  if(nth==0)
     sheet->attributes=(GtkSheetAttr *)malloc(2*sizeof(GtkSheetAttr));
  else
     sheet->attributes=(GtkSheetAttr *)realloc(sheet->attributes, (nth+1)*sizeof(GtkSheetAttr));
  
  sheet->attributes[nth] = attributes;

  for(i=attributes.range.row0; i<=attributes.range.rowi; i++)
   for(j=attributes.range.col0; j<=attributes.range.coli; j++)
     if(i <= sheet->maxallocrow && j <= sheet->maxalloccol && sheet->data[i][j])
       gtk_sheet_set_cell_attributes(sheet, i, j, attributes);

  if(GTK_WIDGET_REALIZED(GTK_WIDGET(sheet)) && !GTK_SHEET_IS_FROZEN(sheet)){
       gtk_sheet_range_draw(sheet, &attributes.range);    
  }
}

static void
gtk_sheet_set_cell_attributes(GtkSheet *sheet, gint row, gint col, GtkSheetAttr attributes)
{
       GtkSheetCell **cell;

       if(row > sheet->maxallocrow || col >sheet->maxalloccol || !sheet->data[row][col]) return;

       cell = &sheet->data[row][col];

       /* DEFAULT VALUES */    
/*       (*cell)->attributes.foreground = GTK_WIDGET(sheet)->style->black;
       (*cell)->attributes.background = GTK_WIDGET(sheet)->style->white;
*/       /* NOTE: This is NOT initialized in gtk_style_init */
/*       (*cell)->attributes.foreground.pixel = 0;
       (*cell)->attributes.background.pixel = 1;
       (*cell)->attributes.justification = GTK_JUSTIFY_LEFT;
       (*cell)->attributes.border.width = 0;
       (*cell)->attributes.border.line_style = GDK_LINE_SOLID;
       (*cell)->attributes.border.cap_style = GDK_CAP_NOT_LAST;
       (*cell)->attributes.border.join_style = GDK_JOIN_MITER;
       (*cell)->attributes.border.mask = 0;
       (*cell)->attributes.border.color = GTK_WIDGET(sheet)->style->black;
       (*cell)->attributes.font = GTK_WIDGET(sheet)->style->font;
       (*cell)->attributes.is_editable = TRUE;
       (*cell)->attributes.is_visible = TRUE;

*/
       switch(attributes.type){
         case GTK_SHEET_FOREGROUND:
           (*cell)->attributes.foreground = attributes.value.foreground;
           break;
         case GTK_SHEET_BACKGROUND:
           (*cell)->attributes.background = attributes.value.background;
           break;
         case GTK_SHEET_JUSTIFICATION:
           (*cell)->attributes.justification = attributes.value.justification;
           break;
         case GTK_SHEET_BORDER:
           (*cell)->attributes.border.mask = attributes.value.border.mask;
           (*cell)->attributes.border.width = attributes.value.border.width;
           (*cell)->attributes.border.line_style = attributes.value.border.line_style;
           (*cell)->attributes.border.cap_style = attributes.value.border.cap_style;
           (*cell)->attributes.border.join_style = attributes.value.border.join_style;
           break;
         case GTK_SHEET_BORDER_COLOR:
           (*cell)->attributes.border.color = attributes.value.border.color;
           break;
         case GTK_SHEET_FONT:
           (*cell)->attributes.font = attributes.value.font;
	   break;  
         case GTK_SHEET_IS_EDITABLE:
           (*cell)->attributes.is_editable = attributes.value.is_editable;
	   break;
         case GTK_SHEET_IS_VISIBLE:
           (*cell)->attributes.is_visible = attributes.value.is_visible;
	   break;
       }
}

void
gtk_sheet_get_attributes(GtkSheet *sheet, gint row, gint col, GtkSheetCellAttr *attributes)
{
 gint i;
 GtkSheetRange range;
 GtkSheetAttr attr;
 GtkSheetCellAttr cell;

 g_return_if_fail (sheet != NULL);
 g_return_if_fail (GTK_IS_SHEET (sheet));

 if(row <= sheet->maxallocrow && col <= sheet->maxalloccol && 
 sheet->data[row][col])
 {
  cell=sheet->data[row][col]->attributes;
  attributes->foreground = cell.foreground;
  attributes->background = cell.background;
  attributes->justification = cell.justification;
  attributes->border.width = cell.border.width;
  attributes->border.line_style = cell.border.line_style;
  attributes->border.cap_style = cell.border.cap_style;
  attributes->border.join_style = cell.border.join_style;
  attributes->border.mask = cell.border.mask;
  attributes->border.color = cell.border.color;
  attributes->font = cell.font;
  attributes->is_editable = cell.is_editable;
  attributes->is_visible = cell.is_visible;
  return;
 }

 /* DEFAULT VALUES */    
 attributes->foreground = GTK_WIDGET(sheet)->style->black;
 attributes->background = GTK_WIDGET(sheet)->style->white;
 /* NOTE: This is NOT initialized in gtk_style_init */
 attributes->foreground.pixel = 0;
 attributes->background.pixel = 1;
 attributes->justification = GTK_JUSTIFY_LEFT;
 attributes->border.width = 0;
 attributes->border.line_style = GDK_LINE_SOLID;
 attributes->border.cap_style = GDK_CAP_NOT_LAST;
 attributes->border.join_style = GDK_JOIN_MITER;
 attributes->border.mask = 0;
 attributes->border.color = GTK_WIDGET(sheet)->style->black;
 attributes->font = GTK_WIDGET(sheet)->style->font;
 attributes->is_editable = TRUE;
 attributes->is_visible = TRUE;

 for(i=0; i<sheet->maxrange; i++){
     attr = sheet->attributes[i];
     range = attr.range; 
     if(row >= range.row0 && row <= range.rowi)
        if(col >= range.col0 && col <=range.coli){
          switch (sheet->attributes[i].type){
            case GTK_SHEET_FOREGROUND:
              attributes->foreground = attr.value.foreground;
              break;
            case GTK_SHEET_BACKGROUND:
              attributes->background = attr.value.background;
              break;
            case GTK_SHEET_JUSTIFICATION:
              attributes->justification = attr.value.justification;
              break;
            case GTK_SHEET_BORDER:
              attributes->border.mask = attr.value.border.mask;
              attributes->border.width = attr.value.border.width;
              attributes->border.line_style = attr.value.border.line_style;
              break;
            case GTK_SHEET_BORDER_COLOR:
              attributes->border.color = attr.value.border.color;
              break;
            case GTK_SHEET_FONT:
              attributes->font = attr.value.font;
	      break; 
	    case GTK_SHEET_IS_EDITABLE:
              attributes->is_editable = attr.value.is_editable;
	      break; 
	    case GTK_SHEET_IS_VISIBLE:
              attributes->is_visible = attr.value.is_visible;
	      break; 
          }                         
        }         
 }
}       
 
/**********************************************************************
 * Memory allocation routines: 
 * AddRow & AddColumn allocate memory for GtkSheetColumn & GtkSheetRow structs.
 * InsertRow 
 * InsertColumn
 * DeleteRow
 * DeleteColumn
 * GrowSheet allocates memory for the sheet cells contents using an array of 
 * pointers. Alternative to this could be a linked list or a hash table.
 * CheckBounds checks whether the given cell is currently allocated or not. 
 * If not, it calls to GrowSheet.
 **********************************************************************/

int AddColumn(GtkSheet *tbl, int ncols)
{
   char label[MAXLENGTH];
   gint i;

   tbl->maxcol += ncols;

   tbl->column = (GtkSheetColumn *)realloc(tbl->column,(tbl->maxcol+1)*
                                        sizeof(GtkSheetColumn));

   for(i=tbl->maxcol-ncols; i<= tbl->maxcol; i++){
        tbl->column[i].width=DEFAULT_COLUMN_WIDTH;
	tbl->column[i].button.label=NULL;
        tbl->column[i].button.state=GTK_STATE_NORMAL;
        tbl->column[i].name=NULL;
        tbl->column[i].is_visible=TRUE;
        tbl->column[i].is_sensitive=TRUE;
        tbl->column[i].left_text_column=i;
        tbl->column[i].right_text_column=i;
        if(i>0)
        {
           tbl->column[i].left_text_column=tbl->column[i-1].left_text_column;
           tbl->column[i].left_xpixel=tbl->column[i-1].left_xpixel +
                                     tbl->column[i-1].width;
	}
        else
	   tbl->column[i].left_xpixel=tbl->row_title_area.width;
   }
}

int AddRow(GtkSheet *tbl, int nrows)
{
   char label[MAXLENGTH];
   gint i;

   tbl->maxrow += nrows;

   tbl->row = (GtkSheetRow *)realloc(tbl->row,(tbl->maxrow+1)*
                                        sizeof(GtkSheetRow));

   for(i=tbl->maxrow-nrows; i<= tbl->maxrow; i++){
        tbl->row[i].height=DEFAULT_ROW_HEIGHT(GTK_WIDGET(tbl));
	tbl->row[i].button.label=NULL;
        tbl->row[i].button.state=GTK_STATE_NORMAL;
        tbl->row[i].name=NULL;
        tbl->row[i].is_visible=TRUE;
        tbl->row[i].is_sensitive=TRUE;
        if(i>0)
           tbl->row[i].top_ypixel=tbl->row[i-1].top_ypixel+tbl->row[i-1].height;
	else
	   tbl->row[i].top_ypixel=tbl->column_title_area.height;
     
   }
}


int InsertRow(GtkSheet *tbl, int row)
{
  GtkSheetCell **pp;
  int i,j;
  GtkSheetCell **auxdata;
  GtkSheetRow auxrow;
  char label[MAXLENGTH];

  AddRow(tbl,1);
  auxrow = tbl->row[tbl->maxrow];  
  for(i=tbl->maxrow; i>row; i--){
    tbl->row[i]=tbl->row[i-1];    
    tbl->row[i].is_visible=tbl->row[i-1].is_visible;
    tbl->row[i].is_sensitive=tbl->row[i-1].is_sensitive;
    if(auxrow.is_visible) 
               tbl->row[i].top_ypixel+=DEFAULT_ROW_HEIGHT(GTK_WIDGET(tbl));
  }
  tbl->row[row]=auxrow;
  tbl->row[row].top_ypixel=tbl->row[row+1].top_ypixel-auxrow.height;

  if(row <= tbl->maxallocrow){
   
    GrowSheet(tbl,1,0);
    auxdata = tbl->data[tbl->maxallocrow];

    for(i=tbl->maxallocrow; i>row; i--){
      tbl->data[i]=tbl->data[i-1];

      pp= tbl->data[i];
      for(j=0; j<=tbl->maxalloccol; j++,pp++){
        if(*pp!=(GtkSheetCell *)NULL)
                                    (*pp)->row=i;
      
      }
    }
    tbl->data[row]=auxdata;
  }

}  

int InsertColumn(GtkSheet *tbl, int col)
{
  int i,j;
  GtkSheetCell **auxdata;
  GtkSheetColumn auxcol;
  char label[MAXLENGTH];

  AddColumn(tbl,1);

  auxcol = tbl->column[tbl->maxcol];

  for(i=tbl->maxcol; i>col; i--){
    tbl->column[i]=tbl->column[i-1];
    tbl->column[i].is_visible=tbl->column[i-1].is_visible;
    tbl->column[i].is_sensitive=tbl->column[i-1].is_sensitive;
    tbl->column[i].left_text_column=tbl->column[i-1].left_text_column;
    tbl->column[i].right_text_column=tbl->column[i-1].right_text_column;
    if(auxcol.is_visible) tbl->column[i].left_xpixel+=DEFAULT_COLUMN_WIDTH;
  }

  tbl->column[col]=auxcol;
  tbl->column[col].left_xpixel=tbl->column[col+1].left_xpixel-auxcol.width;

  if(col <= tbl->maxalloccol){
   
    GrowSheet(tbl,0,1);

    for(i=0; i<=tbl->maxallocrow; i++){
      for(j=tbl->maxalloccol; j>col; j--){
        tbl->data[i][j]=tbl->data[i][j-1];
        if(tbl->data[i][j]) tbl->data[i][j]->col=j;
      }
      tbl->data[i][col]=NULL;
    }
  }

}

int DeleteRow(GtkSheet *tbl, int row)
{
  GtkSheetCell **pp;
  int i,j;
  GtkSheetCell **auxdata;
  GtkSheetRow auxrow;
  char label[MAXLENGTH];

  auxrow = tbl->row[row];  

  for(i=row; i<tbl->maxrow; i++){
    tbl->row[i]=tbl->row[i+1];
    tbl->row[i].is_visible=tbl->row[i+1].is_visible;
    tbl->row[i].is_sensitive=tbl->row[i+1].is_sensitive;
    if(auxrow.is_visible) tbl->row[i].top_ypixel-=auxrow.height;
  }

  tbl->row[tbl->maxrow]=auxrow;
  tbl->maxrow--;

  if(row <= tbl->maxallocrow){

    auxdata = tbl->data[row];
    for(i=0; i<=tbl->maxalloccol; i++){
              tbl->data[row][i]=NULL;
    }
    for(i=row; i<tbl->maxallocrow; i++){
      tbl->data[i]=tbl->data[i+1];
      pp= tbl->data[i];
      for(j=0; j<=tbl->maxalloccol; j++,pp++){
          if(*pp!=(GtkSheetCell *)NULL)
                                    (*pp)->row=i;
      }
    }
    tbl->data[tbl->maxrow]=auxdata;
    tbl->maxallocrow--;

  }

} 

int DeleteColumn(GtkSheet *tbl, int column)
{
  GtkSheetCell **pp;
  int i,j;
  GtkSheetColumn auxcol;
  char label[MAXLENGTH];


  auxcol=tbl->column[column];

  for(i=column; i<tbl->maxcol; i++){
    tbl->column[i]=tbl->column[i+1];    
    tbl->column[i].is_visible=tbl->column[i+1].is_visible;
    tbl->column[i].is_sensitive=tbl->column[i+1].is_sensitive;
    tbl->column[i].left_text_column=tbl->column[i+1].left_text_column;
    tbl->column[i].right_text_column=tbl->column[i+1].right_text_column;
    if(auxcol.is_visible) tbl->column[i].left_xpixel-=auxcol.width;
  }

  tbl->column[tbl->maxcol]=auxcol;
   
  tbl->maxcol--;


  if(column <= tbl->maxalloccol){
    for(i=0; i<=tbl->maxallocrow; i++){
            tbl->data[i][column]=NULL;
    }

    for(i=column; i<tbl->maxalloccol; i++){
      for(j=0; j<=tbl->maxallocrow; j++){
       tbl->data[j][i]=tbl->data[j][i+1];
       if(tbl->data[j][i]) tbl->data[j][i]->col=i;
      }
    }
  
    tbl->maxalloccol--;
  }

}  


int GrowSheet(GtkSheet *tbl, int newrows, int newcols)
{
  int i,j;
  GdkRectangle area;
  char label[MAXLENGTH];
  int x,y;
  int inirow, inicol;
  
  tbl->maxalloccol = tbl->maxalloccol==0 ? newcols : tbl->maxalloccol+newcols;
  tbl->maxallocrow = tbl->maxallocrow==0 ? newrows : tbl->maxallocrow+newrows;

  inirow = tbl->maxallocrow == newrows ? 0 : tbl->maxallocrow-newrows+1;
  inicol = tbl->maxalloccol == newcols ? 0 : tbl->maxalloccol-newcols+1;

  if(newrows>0){
      tbl->data= (GtkSheetCell***)
                 realloc(tbl->data,(tbl->maxallocrow+1)*sizeof(GtkSheetCell **));

      for(i=inirow; i<= tbl->maxallocrow; i++){
        tbl->data[i]= (GtkSheetCell **) \
                       malloc((tbl->maxcol+1)*sizeof(GtkSheetCell *));
        for(j=0; j<inicol; j++) {
          tbl->data[i][j]=NULL;
        }
      }
          
  }

  if(newcols>0){
      for(i=0; i<= tbl->maxallocrow; i++) {
        tbl->data[i]= (GtkSheetCell **) \
                       realloc(tbl->data[i],(tbl->maxalloccol+1)*sizeof(GtkSheetCell *));
        for(j=inicol; j <= tbl->maxalloccol; j++) {
          tbl->data[i][j]=NULL;
	}
      }
  }

  return(0);
}	   


int CheckBounds(GtkSheet *tbl, int row, int col)
{
  int newrows=0,newcols=0;

  if(col>tbl->maxalloccol) newcols=col-tbl->maxalloccol;
  if(row>tbl->maxallocrow) newrows=row-tbl->maxallocrow;
  if(newrows>0 || newcols>0) GrowSheet(tbl, newrows, newcols);
  return(0);
} 

