/* GTK - The GIMP Toolkit
 * Copyright (C) 1995-1997 Peter Mattis, Spencer Kimball and Josh MacDonald
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "gtktreeitem.h"
#include "gtktree.h"
#include "gtktreefork.h"

#include <gtk/gtkcontainer.h>
#include <gtk/gtkwidget.h>
#include <gtk/gtksignal.h>


enum {
  EXPAND,
  COLLAPSE,
  LAST_SIGNAL
};


static void gtk_tree_item_class_init       (GtkTreeItemClass *klass);
static void gtk_tree_item_init             (GtkTreeItem      *tree_item);
static void gtk_real_tree_item_expand      (GtkTreeItem      *tree_item);
static void gtk_real_tree_item_collapse    (GtkTreeItem      *tree_item);
static void gtk_tree_item_destroy          (GtkObject        *object);
static void gtk_tree_item_map              (GtkWidget        *widget);
static void gtk_tree_item_unmap            (GtkWidget        *widget);
static void gtk_tree_item_realize          (GtkWidget        *widget);
static void gtk_tree_item_draw             (GtkWidget        *widget,
					    GdkRectangle     *area);
static gint gtk_tree_item_expose           (GtkWidget        *widget,
					    GdkEventExpose   *event);
static void gtk_tree_item_foreach          (GtkContainer     *container,
					    GtkCallback      callback,
					    gpointer         callback_data);


static GtkItemClass *parent_class = NULL;
static gint tree_item_signals[LAST_SIGNAL] = { 0 };


guint
gtk_tree_item_get_type ()
{
  static guint tree_item_type = 0;

  if (!tree_item_type)
    {
      GtkTypeInfo tree_item_info =
      {
	"GtkTreeItem",
	sizeof (GtkTreeItem),
	sizeof (GtkTreeItemClass),
	(GtkClassInitFunc) gtk_tree_item_class_init,
	(GtkObjectInitFunc) gtk_tree_item_init,
	(GtkArgFunc) NULL,
      };

      tree_item_type = gtk_type_unique (gtk_item_get_type (), &tree_item_info);
    }

  return tree_item_type;
}

static void
gtk_tree_item_class_init (GtkTreeItemClass *klass)
{
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;
  GtkContainerClass *container_class;

  object_class = (GtkObjectClass*) klass;
  widget_class = (GtkWidgetClass*) klass;
  container_class = (GtkContainerClass*) klass;

  parent_class = gtk_type_class (gtk_item_get_type ());

  tree_item_signals[EXPAND] =
      gtk_signal_new ("expand",
		      GTK_RUN_FIRST,
		      object_class->type,
		      GTK_SIGNAL_OFFSET (GtkTreeItemClass, expand),
		      gtk_signal_default_marshaller,
		      GTK_TYPE_NONE, 0);

  tree_item_signals[COLLAPSE] =
      gtk_signal_new ("collapse",
		      GTK_RUN_FIRST,
		      object_class->type,
		      GTK_SIGNAL_OFFSET (GtkTreeItemClass, collapse),
		      gtk_signal_default_marshaller,
		      GTK_TYPE_NONE, 0);

  gtk_object_class_add_signals (object_class, tree_item_signals, LAST_SIGNAL);

  
  object_class->destroy = gtk_tree_item_destroy;

  widget_class->map = gtk_tree_item_map;
  widget_class->unmap = gtk_tree_item_unmap;
  widget_class->realize = gtk_tree_item_realize;
  widget_class->draw = gtk_tree_item_draw;
  widget_class->expose_event = gtk_tree_item_expose;

  container_class->foreach = gtk_tree_item_foreach;

  klass->expand = gtk_real_tree_item_expand;
  klass->collapse = gtk_real_tree_item_collapse;
}

static void
gtk_tree_item_init (GtkTreeItem *tree_item)
{
#ifndef WITH_TREE_ITEM_OWN_WINDOW
  GTK_WIDGET_SET_FLAGS (tree_item, GTK_NO_WINDOW | GTK_BASIC);
#endif

  tree_item->fork = NULL;
  tree_item->subtree = NULL;

  tree_item->spacing = 0;
  tree_item->child_align = 0.0;
}

void
gtk_tree_item_set_fork (GtkTreeItem *tree_item,
			GtkWidget   *fork)
{
  g_return_if_fail (tree_item != NULL);
  g_return_if_fail (GTK_IS_TREE_ITEM (tree_item));

  if (tree_item->fork)
    {
      gtk_widget_unparent (tree_item->fork);
    }

  gtk_widget_set_parent (fork, GTK_WIDGET (tree_item));
  tree_item->fork = fork;

  if (GTK_WIDGET (tree_item)->parent)
    gtk_widget_queue_resize (GTK_WIDGET (tree_item));
}

void
gtk_tree_item_set_subtree (GtkTreeItem *tree_item,
			   GtkWidget   *subtree)
{
  g_return_if_fail (tree_item != NULL);
  g_return_if_fail (GTK_IS_TREE_ITEM (tree_item));

  if (tree_item->subtree)
    {
      gtk_widget_unparent (tree_item->subtree);
    }

  gtk_widget_set_parent (subtree, GTK_WIDGET (tree_item));
  tree_item->subtree = subtree;

  if (GTK_WIDGET (tree_item)->parent)
    gtk_widget_queue_resize (GTK_WIDGET (tree_item));
}

void
gtk_tree_item_expand (GtkTreeItem *tree_item)
{
  gtk_signal_emit (GTK_OBJECT (tree_item), tree_item_signals[EXPAND]);
}

void
gtk_tree_item_collapse (GtkTreeItem *tree_item)
{
  gtk_signal_emit (GTK_OBJECT (tree_item), tree_item_signals[COLLAPSE]);
}

void
gtk_real_tree_item_expand (GtkTreeItem *tree_item)
{
  g_return_if_fail (tree_item != NULL);
  g_return_if_fail (GTK_IS_TREE_ITEM (tree_item));

  if (tree_item->fork)
    {
      gtk_tree_fork__xxx_expand (GTK_TREE_FORK (tree_item->fork));
    }

  if (tree_item->subtree)
    {
      gtk_tree__xxx_expand (GTK_TREE (tree_item->subtree));
    }
}

void
gtk_real_tree_item_collapse (GtkTreeItem *tree_item)
{
  g_return_if_fail (tree_item != NULL);
  g_return_if_fail (GTK_IS_TREE_ITEM (tree_item));

  if (tree_item->fork)
    {
      gtk_tree_fork__xxx_collapse (GTK_TREE_FORK (tree_item->fork));
    }

  if (tree_item->subtree)
    {
      gtk_tree__xxx_collapse (GTK_TREE (tree_item->subtree));
    }
}


static void
gtk_tree_item_destroy (GtkObject *object)
{
  GtkTreeItem *tree_item;

  g_return_if_fail (object != NULL);
  g_return_if_fail (GTK_IS_TREE_ITEM (object));

  tree_item = GTK_TREE_ITEM (object);

  if (tree_item->fork)
    {
      gtk_widget_unparent (tree_item->fork);
      gtk_widget_destroy (tree_item->fork);
    }

  if (tree_item->subtree)
    {
      gtk_widget_unparent (tree_item->subtree);
      gtk_widget_destroy (tree_item->subtree);
    }

  if (GTK_OBJECT_CLASS (parent_class)->destroy)
    (* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

static void
gtk_tree_item_map (GtkWidget *widget)
{
  GtkTreeItem *tree_item;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_TREE_ITEM (widget));

  tree_item = GTK_TREE_ITEM (widget);

  if (tree_item->fork &&
      GTK_WIDGET_VISIBLE (tree_item->fork) &&
      !GTK_WIDGET_MAPPED (tree_item->fork))
    {
      gtk_widget_map (tree_item->fork);
    }

  if (tree_item->subtree &&
      GTK_WIDGET_VISIBLE (tree_item->subtree) &&
      !GTK_WIDGET_MAPPED (tree_item->subtree))
    {
      gtk_widget_map (tree_item->subtree);
    }

  if (GTK_WIDGET_CLASS (parent_class)->map)
    (* GTK_WIDGET_CLASS (parent_class)->map) (widget);
}

static void
gtk_tree_item_unmap (GtkWidget *widget)
{
  GtkTreeItem *tree_item;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_TREE_ITEM (widget));

  tree_item = GTK_TREE_ITEM (widget);

  if (!GTK_WIDGET_NO_WINDOW (tree_item))
    {
      if (GTK_WIDGET_CLASS (parent_class)->unmap)
	  (* GTK_WIDGET_CLASS (parent_class)->unmap) (widget);
    }
  else
    {
      GTK_WIDGET_UNSET_FLAGS (tree_item, GTK_MAPPED);

      if (GTK_BIN (tree_item)->child &&
	  GTK_WIDGET_VISIBLE (GTK_BIN (tree_item)->child) &&
	  GTK_WIDGET_MAPPED (GTK_BIN (tree_item)->child))
	{
	      gtk_widget_unmap (GTK_BIN (tree_item)->child);
	}
    }

  if (tree_item->fork &&
      GTK_WIDGET_VISIBLE (tree_item->fork) &&
      GTK_WIDGET_MAPPED (tree_item->fork))
    {
      gtk_widget_unmap (tree_item->fork);
    }

  if (tree_item->subtree &&
      GTK_WIDGET_VISIBLE (tree_item->subtree) &&
      GTK_WIDGET_MAPPED (tree_item->subtree))
    {
      gtk_widget_unmap (tree_item->subtree);
    }
}

static void
gtk_tree_item_realize (GtkWidget *widget)
{
  GtkTreeItem *tree_item;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_TREE_ITEM (widget));

  tree_item = GTK_TREE_ITEM (widget);

  if (!GTK_WIDGET_NO_WINDOW (tree_item))
    {
      if (GTK_WIDGET_CLASS (parent_class)->realize)
	  (* GTK_WIDGET_CLASS (parent_class)->realize) (widget);
    }
  else
    {
      GTK_WIDGET_SET_FLAGS (tree_item, GTK_REALIZED);

      widget->window = widget->parent->window;
      widget->style = widget->parent->style;

      if (GTK_BIN (tree_item)->child &&
	  GTK_WIDGET_VISIBLE (GTK_BIN (tree_item)->child) &&
	  !GTK_WIDGET_REALIZED (GTK_BIN (tree_item)->child))
	{
	  gtk_widget_realize (GTK_BIN (tree_item)->child);
	}
    }

  if (tree_item->fork &&
      GTK_WIDGET_VISIBLE (tree_item->fork) &&
      !GTK_WIDGET_REALIZED (tree_item->fork))
    {
      gtk_widget_realize (tree_item->fork);
    }

  if (tree_item->subtree &&
      GTK_WIDGET_VISIBLE (tree_item->subtree) &&
      !GTK_WIDGET_REALIZED (tree_item->subtree))
    {
      gtk_widget_realize (tree_item->subtree);
    }
}


static void
gtk_tree_item_draw (GtkWidget    *widget,
		    GdkRectangle *area)
{
  GtkBin *bin;
  GtkTreeItem *tree_item;
  GdkRectangle child_area;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_TREE_ITEM (widget));
  g_return_if_fail (area != NULL);

  if (GTK_WIDGET_DRAWABLE (widget))
    {
      bin = GTK_BIN (widget);
      tree_item = GTK_TREE_ITEM (widget);

      if (bin->child)
        {
          if (gtk_widget_intersect (bin->child, area, &child_area))
            gtk_widget_draw (bin->child, &child_area);
        }

      if (tree_item->fork)
        {
          if (gtk_widget_intersect (tree_item->fork, area, &child_area))
            gtk_widget_draw (tree_item->fork, &child_area);
        }

      if (tree_item->subtree)
        {
          if (gtk_widget_intersect (tree_item->subtree, area, &child_area))
            gtk_widget_draw (tree_item->subtree, &child_area);
        }
    }
}

static gint
gtk_tree_item_expose (GtkWidget      *widget,
		      GdkEventExpose *event)
{
  GtkBin *bin;
  GtkTreeItem *tree_item;
  GdkEventExpose child_event;

  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_TREE_ITEM (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  if (GTK_WIDGET_DRAWABLE (widget))
    {
      bin = GTK_BIN (widget);
      tree_item = GTK_TREE_ITEM (widget);

      if (bin->child)
        {
          child_event = *event;

          if (GTK_WIDGET_NO_WINDOW (bin->child) &&
              gtk_widget_intersect (bin->child, &event->area,
				    &child_event.area))
            gtk_widget_event (bin->child, (GdkEvent*) &child_event);
        }

      if (tree_item->fork)
        {
          child_event = *event;

          if (GTK_WIDGET_NO_WINDOW (tree_item->fork) &&
              gtk_widget_intersect (tree_item->fork, &event->area,
				    &child_event.area))
	    {
	      gtk_widget_event (tree_item->fork, (GdkEvent*) &child_event);
	    }
        }

      if (tree_item->subtree)
        {
          child_event = *event;

          if (GTK_WIDGET_NO_WINDOW (tree_item->subtree) &&
              gtk_widget_intersect (tree_item->subtree, &event->area,
				    &child_event.area))
	    {
	      gtk_widget_event (tree_item->subtree, (GdkEvent*) &child_event);
	    }
        }
    }

  return FALSE;
}

static void
gtk_tree_item_foreach (GtkContainer *container,
		       GtkCallback  callback,
		       gpointer     callback_data)
{
  GtkTreeItem *tree_item;

  g_return_if_fail (container != NULL);
  g_return_if_fail (GTK_IS_TREE_ITEM (container));
  g_return_if_fail (callback != NULL);

  tree_item = GTK_TREE_ITEM (container);

  if (GTK_BIN (tree_item)->child)
    (* callback) (GTK_BIN (tree_item)->child, callback_data);

  if (tree_item->fork)
    (* callback) (tree_item->fork, callback_data);
  
  if (tree_item->subtree)
    (* callback) (tree_item->subtree, callback_data);
}
