/* GTK - The GIMP Toolkit
 * Copyright (C) 1995-1997 Peter Mattis, Spencer Kimball and Josh MacDonald
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <string.h>
#include "gtktreefactory.h"
#include "gtktree.h"
#include "gtktreeitem.h"
#include "gtktreefork.h"
#include "gtkvtree.h"
#include "gtkhtreeitem.h"
#include "gtkvtreefork.h"

#include <gtk/gtkobject.h>
#include <gtk/gtksignal.h>
#include <gtk/gtkhbox.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkvbox.h>
#include <gtk/gtkbutton.h>


enum
{
  CREATE  = 1 << 0,
  DESTROY = 1 << 1,
};


static void         gtk_tree_factory_create         (GtkTreeFactory *factory,
						     GtkTreeEntry   *entry,
						     GtkWidget      *parent,
						     const char     *path);
static void         gtk_tree_factory_remove         (GtkTreeFactory *factory,
						     GtkWidget      *parent,
						     const char     *path);
static GtkWidget*   gtk_tree_factory_make_widget    (GtkTreeFactory *factory);
static GtkTreePath* gtk_tree_factory_get            (GtkTreeFactory *factory,
						     GtkWidget      *parent,
						     const char     *path,
						     int             flags);
static GtkTreePath* gtk_tree_factory_find_recurse   (GtkTreeFactory *factory,
						     GtkWidget      *parent,
						     const char     *path);
static void gtk_tree_factory_default_create_item    (GtkTreePath  *tree_path,
						      GtkWidget    *parent);
static void gtk_tree_factory_default_create_subtree (GtkTreePath  *tree_path);


GtkTreeFactory*
gtk_tree_factory_new ()
{
  GtkTreeFactory *factory;

  factory = g_new (GtkTreeFactory, 1);

  factory->path = NULL;
  factory->widget = NULL;
  factory->subfactories = NULL;

  factory->create_item = gtk_tree_factory_default_create_item;
  factory->create_subtree = gtk_tree_factory_default_create_subtree;

  return factory;
}

void
gtk_tree_factory_destroy (GtkTreeFactory *factory)
{
  GtkTreeFactory *subfactory;
  GList *tmp_list;

  g_return_if_fail (factory != NULL);

  if (factory->path)
    g_free (factory->path);

  tmp_list = factory->subfactories;
  while (tmp_list)
    {
      subfactory = tmp_list->data;
      tmp_list = tmp_list->next;

      gtk_tree_factory_destroy (subfactory);
    }
}

void
gtk_tree_factory_add_entries (GtkTreeFactory *factory,
			      GtkTreeEntry   *entries,
			      int             nentries)
{
  int i;

  g_return_if_fail (factory != NULL);
  g_return_if_fail (entries != NULL);
  g_return_if_fail (nentries > 0);

  if (!factory->widget)
    factory->widget = gtk_tree_factory_make_widget (factory);

  for (i = 0; i < nentries; i++)
    gtk_tree_factory_create (factory,
			     &entries[i], factory->widget, entries[i].path);
}

void
gtk_tree_factory_add_subfactory (GtkTreeFactory *factory,
				 GtkTreeFactory *subfactory,
				 const char     *path)
{
  g_return_if_fail (factory != NULL);
  g_return_if_fail (subfactory != NULL);
  g_return_if_fail (path != NULL);

  if (subfactory->path)
    g_free (subfactory->path);
  subfactory->path = g_strdup (path);

  factory->subfactories = g_list_append (factory->subfactories, subfactory);
}

void
gtk_tree_factory_remove_paths (GtkTreeFactory  *factory,
			       char           **paths,
			       int              npaths)
{
  int i;

  g_return_if_fail (factory != NULL);
  g_return_if_fail (paths != NULL);
  g_return_if_fail (npaths > 0);

  if (factory->widget)
    {
      for (i = 0; i < npaths; i++)
	gtk_tree_factory_remove (factory, factory->widget, paths[i]);
    }
}

void
gtk_tree_factory_remove_entries (GtkTreeFactory *factory,
				 GtkTreeEntry   *entries,
				 int             nentries)
{
  int i;

  g_return_if_fail (factory != NULL);
  g_return_if_fail (entries != NULL);
  g_return_if_fail (nentries > 0);

  if (factory->widget)
    {
      for (i = 0; i < nentries; i++)
	gtk_tree_factory_remove (factory, factory->widget, entries[i].path);
    }
}

void
gtk_tree_factory_remove_subfactory (GtkTreeFactory *factory,
				    GtkTreeFactory *subfactory,
				    const char     *path)
{
  g_return_if_fail (factory != NULL);
  g_return_if_fail (subfactory != NULL);
  g_return_if_fail (path != NULL);

  g_warning ("FIXME: gtk_tree_factory_remove_subfactory");
}

GtkTreePath*
gtk_tree_factory_find (GtkTreeFactory *factory,
		       const char     *path)
{
  g_return_val_if_fail (factory != NULL, NULL);
  g_return_val_if_fail (path != NULL, NULL);

  return gtk_tree_factory_find_recurse (factory, factory->widget, path);
}


static void
gtk_tree_factory_create (GtkTreeFactory *factory,
			 GtkTreeEntry   *entry,
			 GtkWidget      *parent,
			 const char     *path)
{
  GtkTreeFactory *subfactory;
  GtkTreePath *tree_path;
  GtkWidget *tree;
  GList *tmp_list;
  char tmp_path[256];
  char *p;

  g_return_if_fail (factory != NULL);
  g_return_if_fail (entry != NULL);

  /* If 'path' is empty, then simply return.
   */
  if (!path || path[0] == '\0')
    return;

  /* Strip off the next part of the path.
   */
  p = strchr (path, '/');

  /* If this is the last part of the path ('p' is
   *  NULL), then we create an item.
   */
  if (!p)
    {
      tree_path = gtk_tree_factory_get (factory, parent, path, CREATE);
      entry->widget = tree_path->widget;

      if (strcmp (path, "<nothing>") == 0)
	  gtk_widget_hide (entry->widget);
    }
  else
    {
      strncpy (tmp_path, path, (unsigned int) ((long) p - (long) path));
      tmp_path[(long) p - (long) path] = '\0';

      tree_path = gtk_tree_factory_get (factory, parent, tmp_path, 0);
      if (!tree_path)
	{
	  tmp_list = factory->subfactories;
	  while (tmp_list)
	    {
	      subfactory = tmp_list->data;
	      tmp_list = tmp_list->next;

	      if (subfactory->path &&
		  (strcmp (subfactory->path, tmp_path) == 0))
		{
		  if (!subfactory->widget)
		    subfactory->widget = gtk_tree_factory_make_widget (subfactory);
		  gtk_tree_factory_create (subfactory,
					   entry, subfactory->widget, p + 1);
		  return;
		}
	    }

	  tree_path = gtk_tree_factory_get (factory, parent, tmp_path, CREATE);
	}

      entry->widget = tree_path->widget;
      tree = GTK_TREE_ITEM (tree_path->widget)->subtree;

      if (!tree)
	{
	  if (factory->create_subtree)
	    {
	      (* factory->create_subtree) (tree_path);
	      tree = GTK_TREE_ITEM (tree_path->widget)->subtree;
	    }
	}

      gtk_tree_factory_create (factory, entry, tree, p + 1);
    }
}

static void
gtk_tree_factory_remove (GtkTreeFactory *factory,
			 GtkWidget      *parent,
			 const char     *path)
{
  GtkTreeFactory *subfactory;
  GtkTreePath *tree_path;
  GtkWidget *tree;
  GList *tmp_list;
  char tmp_path[256];
  char *p;

  if (!path || path[0] == '\0')
    return;

  p = strchr (path, '/');

  if (!p)
    {
      if (parent)
	gtk_tree_factory_get (factory, parent, path, DESTROY);
    }
  else
    {
      strncpy (tmp_path, path, (unsigned int) ((long) p - (long) path));
      tmp_path[(long) p - (long) path] = '\0';

      tree_path = gtk_tree_factory_get (factory, parent, tmp_path, 0);
      if (!tree_path)
	{
	  tmp_list = factory->subfactories;
	  while (tmp_list)
	    {
	      subfactory = tmp_list->data;
	      tmp_list = tmp_list->next;

	      if (subfactory->path &&
		  (strcmp (subfactory->path, tmp_path) == 0))
		{
		  if (!subfactory->widget)
		    return;
		  gtk_tree_factory_remove (subfactory, subfactory->widget, p + 1);
		}
	    }
	}
      else
	{
	  tree = GTK_TREE_ITEM (tree_path->widget)->subtree;
	  if (tree)
	    gtk_tree_factory_remove (factory, tree, p + 1);
	}
    }
}

static GtkWidget*
gtk_tree_factory_make_widget (GtkTreeFactory *factory)
{
  GtkWidget *widget;

  g_return_val_if_fail (factory != NULL, NULL);

  widget = gtk_vtree_new ();

  return widget;
}

static GtkTreePath*
gtk_tree_factory_get (GtkTreeFactory *factory,
		      GtkWidget *parent,
		      const char *path,
		      int        flags)
{
  GtkTreePath *tree_path;
  GList *tmp_list;

  tmp_list = gtk_object_get_user_data (GTK_OBJECT (parent));
  while (tmp_list)
    {
      tree_path = tmp_list->data;
      tmp_list = tmp_list->next;

      if (strcmp (tree_path->path, path) == 0)
	{
	  if (flags & DESTROY)
	    {
	      tmp_list = gtk_object_get_user_data (GTK_OBJECT (parent));
	      tmp_list = g_list_remove (tmp_list, tree_path);
	      gtk_object_set_user_data (GTK_OBJECT (parent), tmp_list);

	      gtk_widget_destroy (tree_path->widget);
	      g_free (tree_path->path);
	      g_free (tree_path);

	      return NULL;
	    }
	  else
	    {
	      return tree_path;
	    }
	}
    }

  if (flags & CREATE)
    {
      tree_path = g_new (GtkTreePath, 1);
      tree_path->path = g_strdup (path);

      if (factory->create_item)
	{
	  (* factory->create_item) (tree_path, parent);
	  gtk_object_set_user_data (GTK_OBJECT (tree_path->widget), NULL);
	}

      tmp_list = gtk_object_get_user_data (GTK_OBJECT (parent));
      tmp_list = g_list_prepend (tmp_list, tree_path);
      gtk_object_set_user_data (GTK_OBJECT (parent), tmp_list);

      return tree_path;
    }

  return NULL;
}

static GtkTreePath*
gtk_tree_factory_find_recurse (GtkTreeFactory *factory,
			       GtkWidget      *parent,
			       const char     *path)
{
  GtkTreeFactory *subfactory;
  GtkTreePath *tree_path;
  GtkWidget *tree;
  GList *tmp_list;
  char tmp_path[256];
  char *p;

  if (!path || path[0] == '\0')
    return NULL;

  p = strchr (path, '/');

  if (!p)
    {
      if (parent)
	return gtk_tree_factory_get (factory, parent, path, 0);
    }
  else
    {
      strncpy (tmp_path, path, (unsigned int) ((long) p - (long) path));
      tmp_path[(long) p - (long) path] = '\0';

      tree_path = gtk_tree_factory_get (factory, parent, tmp_path, 0);
      if (!tree_path)
	{
	  tmp_list = factory->subfactories;
	  while (tmp_list)
	    {
	      subfactory = tmp_list->data;
	      tmp_list = tmp_list->next;

	      if (subfactory->path &&
		  (strcmp (subfactory->path, tmp_path) == 0))
		{
		  if (!subfactory->widget)
		    return NULL;
		  return gtk_tree_factory_find_recurse (subfactory, subfactory->widget, p + 1);
		}
	    }

	  return NULL;
	}

      tree = GTK_TREE_ITEM (tree_path->widget)->subtree;
      if (tree)
	return gtk_tree_factory_find_recurse (factory, tree, p + 1);
    }

  return NULL;
}


void
gtk_tree_factory_default_create_item (GtkTreePath *tree_path,
				      GtkWidget *parent)
{
  tree_path->widget = gtk_htree_item_new_with_label (tree_path->path);

  gtk_container_add (GTK_CONTAINER (parent), tree_path->widget);
  gtk_widget_show (tree_path->widget);
}

void
gtk_tree_factory_default_create_subtree (GtkTreePath *tree_path)
{
  GtkWidget *tree;
  GtkWidget *tree_fork;

  tree_fork = gtk_vtree_fork_new (GTK_VTREE_FORK_TYPE_B);
  gtk_tree_item_set_fork (GTK_TREE_ITEM (tree_path->widget), tree_fork);
  gtk_widget_show (tree_fork);

  tree = gtk_vtree_new ();
  gtk_tree_item_set_subtree (GTK_TREE_ITEM (tree_path->widget), tree);
  gtk_widget_show(tree);
}
