/* GTK - The GIMP Toolkit
 * Copyright (C) 1995-1997 Peter Mattis, Spencer Kimball and Josh MacDonald
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#define NO_INSERT

#ifdef DEBUG
#include <stdio.h>
#endif

#include <stdlib.h>

#include <sys/types.h>
#include <regex.h>
#include <string.h>

#include <glib.h>
#include <gdk/gdkkeysyms.h>
#include <gtk/gtkmain.h>
#include <gtk/gtksignal.h>
#include "gtkeditor.h"

/* hilite flags */
#define BASIC         0x00
#define IN_COMMENT    0x01
#define IN_STRING     0x02
#define IN_KEYWORD    0x04

#define TIME_DELAY    300	/* delay in milisec. before hilite */

#ifdef DEBUG_MATCH
#define MATCH(a,b,c) (g_print ("\nMATCH (%d,%d,%d)\n", (a), (b), (c)), \
		      (MAX (MAX ((a),(b)), (c)) > -1))
#define FIRST_LEAST(a,b,c) (g_print ("\nFL (%d,%d,%d)\n", (a), (b), (c)), \
			    (((b)>-1 ? (a) <= (b) : 1) \
			    && ((c)>-1 ? (a) <= (c) : 1) \
			    && ((a)>-1)))
#else
#define MATCH(a,b,c) (MAX (MAX ((a),(b)), (c)) > -1)
#define FIRST_LEAST(a,b,c) (((b)>-1 ? (a) <= (b) : 1) \
			    && ((c)>-1 ? (a) <= (c) : 1) \
			    && ((a)>-1))
#endif

typedef struct _MarkData MarkData;

/* FIXME: I need some config stuff to set this flag. */
#ifdef GNU_REGEXP_LIB
/* GNU REGEXP LIBRARY -- more regexp stuff. */
typedef struct _Regex {
  struct re_pattern_buffer buf;
  struct re_registers reg;
} Regex;
#else  /* use posix */
typedef regex_t Regex;
#endif

typedef struct _Match {
  gint from;
  gint to;
} Match;

struct _GtkEditorHiliteRegexps {
  Regex comment_start_regex;
  Regex comment_end_regex;
  Regex string_start_regex;
  Regex string_end_regex;
  Regex keywords_regex;
};

struct _GtkEditorHiliteColors {
  GdkColormap *cmap;
  GdkColor plain;
  GdkColor comment;
  GdkColor string;
  GdkColor keywords;
};

struct _MarkData {
  guint8 state;
};

/* widget stuff */
static void   gtk_editor_class_init          (GtkEditorClass   *klass);
static void   gtk_editor_init                (GtkEditor        *editor);
static void   gtk_editor_destroy             (GtkObject        *object);

/* text property data */
gboolean      cmp_prop_data                  (MarkData         *d1,
					      MarkData         *d2);

/* search stuff */
static gint   search                         (GtkEditor        *editor,
					      guint             pos,
					      gchar            *string,
					      gboolean          casein);
static gint   search_back                    (GtkEditor        *editor,
					      guint             pos,
					      gchar            *string,
					      gboolean          casein);

/* regex stuff */
static gboolean compile_regex                (char             *pattern,
					      Regex            *regex);
/* I don't use the following 2 functions right now, but I don't dare
 * throw them away yet...who knows when they will be usefull again? */
static gint   regex_search_string            (Regex            *regex,
					      char             *string,
					      Match            *m);
static gint   regex_match_string             (Regex            *regex,
					      char             *string);
static gint   regex_search                   (GtkText          *text,
					      guint             pos,
					      Regex            *regex,
					      gboolean          forward,
					      Match            *m);
static gint   regex_match                    (GtkText          *text,
					      guint             pos,
					      Regex            *regex);

/* hilit */
static void   destroy_regexps                (GtkEditorHiliteRegexps *regexps);
static GtkEditorHiliteColors *
              new_colors                     (GdkColormap      *cmap);
static void   hilite_interval                (GtkEditor        *editor,
					      guint             from,
					      guint             to);
static gboolean hilite_when_idle             (GtkEditor        *editor);

/* events */
static gint   gtk_editor_key_press           (GtkWidget        *widget,
					      GdkEventKey      *event);
static void   gtk_editor_changed_pre         (GtkEditor        *editor);
static void   gtk_editor_changed_post        (GtkEditor        *editor);


/* --<local data>--------------------------------------------------------- */
static GtkWidgetClass *parent_class = NULL;

#ifdef GNU_REGEXP_LIB
static GtkEditorHilitePatterns default_patterns =
{ "/\\*",			/* comment_start */
  "\\*/",			/* comment_end */
  "\"\\|'",			/* string_start */
  "\"\\|'",			/* string_end */
  "\\b\\(if\\|else\\|for\\|while\\)\\b" /* keywords */
};
#else
static GtkEditorHilitePatterns default_patterns =
{ "/\\*",			/* comment_start */
  "\\*/",			/* comment_end */
  "\"|'",			/* string_start */
  "\"|'",			/* string_end */
  "if|else|for|while"		/* keywords */
};
#endif /* GNU_REGEXP_LIB */

static GtkEditorHiliteRegexps *default_regexps = NULL;

/* --<widget initialization, constructor and destructor>------------------ */
guint
gtk_editor_get_type ()
{
  static guint editor_type = 0;

  if (!editor_type)
    {
      GtkTypeInfo editor_info =
      {
	"GtkEditor",
	sizeof (GtkEditor),
	sizeof (GtkEditorClass),
	(GtkClassInitFunc) gtk_editor_class_init,
	(GtkObjectInitFunc) gtk_editor_init,
	(GtkArgSetFunc) NULL,
        (GtkArgGetFunc) NULL,
      };

      editor_type = gtk_type_unique (gtk_text_get_type (), &editor_info);
    }

  return editor_type;
}

static void
gtk_editor_class_init (GtkEditorClass *class)
{
  GtkObjectClass *object_class;

  object_class = (GtkObjectClass*) class;

#ifdef GNU_REGEXP_LIB
  re_set_syntax (RE_SYNTAX_EMACS);
#endif

  /* init local data */
  parent_class = gtk_type_class (gtk_text_get_type ());
  default_regexps = gtk_editor_new_regexps (&default_patterns);
  g_assert (default_regexps);	/* the defaults shouldn't fail */

  /* setup signals */
  object_class->destroy = gtk_editor_destroy;
}

static void
gtk_editor_init (GtkEditor *editor)
{
  GtkText *text = GTK_TEXT (editor);

  text->compare_data = (gboolean(*)(gpointer,gpointer))cmp_prop_data;

  editor->regexps = default_regexps;
  editor->colors = new_colors (gtk_widget_get_colormap (GTK_WIDGET (editor)));

  gtk_signal_connect (GTK_OBJECT (editor), "key_press_event",
		      GTK_SIGNAL_FUNC (gtk_editor_key_press),
		      NULL);

  editor->hilite_when_idle = TRUE;
  editor->hilite_time = 0;
  editor->changed_from = editor->changed_to = 0;

  /* to keep track of changes, for hilite_when_idle */
  gtk_signal_connect (GTK_OBJECT (editor), "insert_text",
		      GTK_SIGNAL_FUNC (gtk_editor_changed_pre),
		      NULL);
  gtk_signal_connect (GTK_OBJECT (editor), "delete_text",
		      GTK_SIGNAL_FUNC (gtk_editor_changed_pre),
		      NULL);
  gtk_signal_connect (GTK_OBJECT (editor), "changed",
		      GTK_SIGNAL_FUNC (gtk_editor_changed_post),
		      NULL);
}

GtkWidget*
gtk_editor_new (GtkAdjustment *hadj,
		GtkAdjustment *vadj)
{
  GtkEditor *editor;

  editor = gtk_type_new (gtk_editor_get_type ());

  gtk_text_set_adjustments (GTK_TEXT (editor), hadj, vadj);

  return GTK_WIDGET (editor);
}

static void
gtk_editor_destroy (GtkObject *object)
{
  g_return_if_fail (object != NULL);
  g_return_if_fail (GTK_IS_EDITOR (object));

  if (GTK_EDITOR (object)->regexps != default_regexps) {
    destroy_regexps (GTK_EDITOR (object)->regexps);
  }
  g_free (GTK_EDITOR (object)->colors);

  GTK_OBJECT_CLASS(parent_class)->destroy (object);
}

/* --<text property data>------------------------------------------------- */
gboolean
cmp_prop_data (MarkData *d1, MarkData *d2)
{
  if (d1 == d2) return TRUE;

  if (d1 && d2) return (d1->state == d2->state);

  return FALSE;
}


/* --<search'n'stuff>----------------------------------------------------- */
/* NB! There is really no reason why all this searching stuff couldn't
 * be moved to the text widget...the only reason I can think of is
 * that, the both gtktext and gtkeditor would have to deal with
 * regexps, but that doesn't seem too discouraging. We will have to
 * make them more 'public' though...the editor _needs_ some of the
 * static functions, so we can't hide them in the text widget. */

/* search -- searches in editor, from pos and forward, for string.  If
 * string is found, returns position, otherwise returns -1.
 * If casein is true, search is case insensitive. */
/* FIXME: This could and should be optimised. There is no need to
 * get_text all the time, is there?  We just have to be carefull with
 * the gap. */
static gint
search (GtkEditor *editor, guint pos, gchar *string, gboolean casein)
{
  gchar *buf = NULL;
  gint match = -1;
  gint text_len = gtk_text_get_length (GTK_TEXT (editor));
  gint len = strlen (string);

  g_return_val_if_fail (pos <= text_len, -1);
  if (pos == text_len) return -1; /* not illegal, but nothing to do */

  for (; pos < (text_len - len) + 1; pos++) {
    buf = gtk_editable_get_chars (GTK_EDITABLE (editor), pos, pos + len);
    if (casein) {
      if (strcasecmp (buf, string) == 0) {
	match = pos;
	break;
      }
    } else {
      if (strcmp (buf, string) == 0) {
	match = pos;
	break;
      }
    }
    g_free (buf);
  }
  g_free (buf);

  return match;
}

/* gtk_editor_search_from_point -- searches for string, if found, goto
 * and select match. If casein is true, search is case insensitive. If
 * string is found it returns TRUE, otherwise FALSE. */
gboolean
gtk_editor_search_from_point (GtkEditor *editor, gchar *string, gboolean casein)
{
  gint new_pos;

  g_return_val_if_fail (editor != NULL, FALSE);
  g_return_val_if_fail (string != NULL, FALSE);

  new_pos = search (editor, gtk_text_get_point (GTK_TEXT (editor)), string, casein);
  if (new_pos > -1) {
    gtk_editable_set_position (GTK_EDITABLE (editor), new_pos);
    gtk_editable_select_region (GTK_EDITABLE (editor), new_pos,
				new_pos + strlen (string));
    return TRUE;
  } else {
    return FALSE;
  }
}

/* search_back -- searches in editor, from pos and backward, for
 * string.  If string is found, returns position, otherwise returns
 * -1.  If casein is true, search is case insensitive. */
/* FIXME: This could and should be optimised. There is no need to
 * get_text all the time, is there?  We just have to be carefull with
 * the gap. */
static gint
search_back (GtkEditor *editor, guint pos, gchar *string, gboolean casein)
{
  gchar *buf = NULL;
  gint match = -1;
  gint text_len = gtk_text_get_length (GTK_TEXT (editor));
  gint len = strlen (string);

  if (pos == 0) return -1; /* not illegal, but nothing to do */
  if (pos > text_len - len) pos = text_len - len; /* we must be reasonable */

  for (; pos >= 0; pos--) {
    buf = gtk_editable_get_chars (GTK_EDITABLE (editor), pos, pos + len);
    if (casein) {
      if (strcasecmp (buf, string) == 0) {
	match = pos;
	break;
      }
    } else {
      if (strcmp (buf, string) == 0) {
	match = pos;
	break;
      }
    }
    g_free (buf);
  }
  g_free (buf);

  return match;
}

/* gtk_editor_search_back_from_point -- searches backward for string,
 * if found, goto and select match. If casein is true, search is case
 * insensitive. If string is found it returns TRUE, otherwise it
 * returns FALSE. */
gboolean
gtk_editor_search_back_from_point (GtkEditor *editor, 
				   gchar *string,
				   gboolean casein)
{
  gint new_pos;

  g_return_val_if_fail (editor != NULL, FALSE);
  g_return_val_if_fail (string != NULL, FALSE);

  new_pos = search_back (editor, gtk_text_get_point (GTK_TEXT (editor)),
			 string, casein);
  if (new_pos > -1) {
    gtk_editable_set_position (GTK_EDITABLE (editor), new_pos);
    gtk_editable_select_region (GTK_EDITABLE (editor), new_pos,
				new_pos + strlen (string));
    return TRUE;
  } else {
    return FALSE;
  }
}

/* compile_regex -- compiles pattern into a regex structure that can
 * be used in regex_search and regex_match.  Returns TRUE if
 * succesfull, FALSE if not.*/
static gboolean
compile_regex (gchar *pattern, Regex *regex)
{
#ifdef GNU_REGEXP_LIB
  regex->buf.translate = NULL;
  regex->buf.fastmap = NULL;
  regex->buf.allocated = 0;
  regex->buf.buffer = NULL;
  if (re_compile_pattern (pattern, strlen (pattern), &regex->buf) == 0) {
    return TRUE;
  } else {
    return FALSE;
  }
#else
  if (regcomp (regex, pattern, REG_EXTENDED) == 0) {
    return TRUE;
  } else {
    return FALSE;
  }
#endif /* GNU_REGEXP_LIB */
}

/* regex_free -- wrapper around regfree, to abstract from the differen
 * regex libs.  This function does *not* free the Regex structure!  If
 * this has to be done call g_free on the structure after calling
 * regex_free. */
static void
regex_free (Regex *regex)
{
#ifdef GNU_REGEXP_LIB
  if (regex->buf.regs_allocated) {
    free (regex->reg.start);
    free (regex->reg.end);
  }
  regfree (&regex->buf);
#else
  regfree (regex);
#endif
}

/* regex_search_string -- searches for regex in string. If found,
 * returns index of beginning of the match, otherwise returns < 0.  If
 * found the match interval is [m.from,m.to[.  m.from is allways equal
 * to the return value, but m.to is undefined if no match. */
static gint
regex_search_string (Regex *regex, char *string, Match *m)
{
#ifdef GNU_REGEXP_LIB
  gint len = strlen (string);
  m->from = re_search (regex->buf, string, len, 0, len, &regex->reg);
  if (m->from > -1) m->to = regex->reg.end[0];
  return m->from;
#else
  regmatch_t pm[1];		/* Why array? So it's easier to extend! */

  if (regexec (regex, string, 1, pm, 0) == 0) {
    m->from = pm[0].rm_so;
    m->to = pm[0].rm_eo;
  } else {
    m->from = -1;
  }
  return m->from;
#endif /* GNU_REGEXP_LIB */
}

/* regex_match_string -- tries to match regex in beginning of
 * string. It returns the number of chars matched, or -1 if no match.
 * Warning!  The number matched can be 0, if the regex matches the
 * empty string. */
static gint
regex_match_string (Regex *regex, char *string)
{
#ifdef GNU_REGEXP_LIB
  gint len = strlen (string);
  return re_match (regex->buf, string, len, 0, NULL);
#else
  /* FIXME: This is *very* inefficient since it searches forward until
   * it finds a match, and *then* checks wether it was in the
   * beginning of the string.  This *has* to be reimplemented! */
  regmatch_t pm;

  if (regexec (regex, string, 1, &pm, 0) == 0) {
    if (pm.rm_so == 0) {
      return pm.rm_eo - pm.rm_so;
    }
  }
  return -1;			/* no match */
#endif /* GNU_REGEXP_LIB */
}

/* regex_search -- searches for regex in text from position
 * 'start'. If 'forward' is TRUE, it searches forward, otherwise
 * backwards.  If found, returns index of beginning of the match,
 * otherwise returns < 0.  If found the match interval is
 * [m.from,m.to[.  m.from is allways equal to the return value, but
 * m.to is undefined if no match.  I search in GtkText, since I don't
 * really need an editor for this function. This will also make it
 * easier to move this function to the text widget, should we want
 * that. */
static gint
regex_search (GtkText *text, guint start, Regex *regex,
	      gboolean forward, Match *m)
{
  g_return_val_if_fail (start <= text->text_len, -1);
  g_return_val_if_fail (regex != NULL, -1);
  g_return_val_if_fail (m != NULL, -1);

#ifdef GNU_REGEXP_LIB
  m->from = re_search_2 (regex->buf,
			 /* text before gap */
			 text->text, text->gap_position,
			 /* text after gap */
			 text->text + text->gap_position + text->gap_size,
			 text->text_len - (text->gap_position + text->gap_size),
			 /* from 'start' and forward to the end */
			 start,
			 (forward ? text->text_end - text->gap_size - start
			  : - start),
			 &regex->reg, text->text_end - text->gap_size);
  if (m->from > -1) m->to = regex->reg.end[0];
  return m->from;    
#else
  g_warning ("%s (%u): regex_search () not implemented yet!", __FILE__, __LINE__);
  return -1;
#endif
}

/* select_regex -- a wrapper around _regex_search_ and
 * _regex_search_back_.  They behave the same besides the search
 * direction.  If forward is TRUE, searches forward, otherwise
 * searches backwards. */
gint
select_regex (GtkEditor *editor, gchar *regex, gboolean forward)
{
  Regex re;
  Match m;

  g_return_val_if_fail (editor != NULL, -1);
  g_return_val_if_fail (regex != NULL, -1);

  if (!compile_regex (regex, &re)) {
    regex_free (&re);
    return -1;
  }
  if (regex_search (GTK_TEXT (editor), GTK_TEXT (editor)->point.index,
		    &re, forward, &m) > -1) {
    gtk_editable_set_position (GTK_EDITABLE (editor), m.from);
    gtk_editable_select_region (GTK_EDITABLE (editor), m.from, m.to);
    regex_free (&re);
    return 1;
  } else {
    regex_free (&re);
    return 0;
  }
}

/* gtk_editor_search_regex_from_point -- searches for string matching
 * regex, if found, goto and select match.  If the regex doesn't
 * compile (or a serious error occurs) it returns -1, if a match is
 * found it returns 1, and if no match is found it returns 0. */
gint
gtk_editor_regex_search_from_point (GtkEditor     *editor,
				    gchar         *regex)
{
  return select_regex (editor, regex, TRUE);
}

/* gtk_editor_search_regex_back_from_point -- searches for string
 * matching regex, if found, goto and select match.  If the regex
 * doesn't compile (or a serious error occurs) it returns -1, if a
 * match is found it returns 1, and if no match is found it returns
 * 0. */
gint
gtk_editor_regex_search_back_from_point (GtkEditor     *editor,
					 gchar         *regex)
{
  return select_regex (editor, regex, FALSE);
}


/* regex_match -- tries to match regex at the 'pos' position in the
 * text. It returns the number of chars matched, or -1 if no match.
 * Warning!  The number matched can be 0, if the regex matches the
 * empty string.  The reason for workin on GtkText is the same as in
 * regex_search. */
static gint
regex_match (GtkText *text, guint pos, Regex *regex)
{
  g_return_val_if_fail (pos <= text->text_len, -1);
  g_return_val_if_fail (regex != NULL, -1);

#ifdef GNU_REGEXP_LIB
  return re_match_2 (regex->buf,
		     /* text before gap */
		     text->text, text->gap_position,
		     /* text after gap */
		     text->text + text->gap_position + text->gap_size,
		     text->text_len - (text->gap_position + text->gap_size),
		     /* from pos and not after the end */
		     pos, NULL, text->text_end - text->gap_size);
#else
  g_warning ("%s (%u): regex_match () not implemented yet!", __FILE__, __LINE__);
  return -1;
#endif
}



/* --<hilite stuff>------------------------------------------------------- */
/* gtk_editor_new_regexps -- returns compiled regexps from patterns, if all
 * compilations are succesfull. If an error occurs, returns NULL. This
 * function is useful if you want to remember different kinds of
 * 'modes', otherwise you can use gtk_editor_set_regexps. */
GtkEditorHiliteRegexps *
gtk_editor_new_regexps (GtkEditorHilitePatterns *patterns)
{
  GtkEditorHiliteRegexps *regexps = g_new (GtkEditorHiliteRegexps, 1);

  if (!compile_regex (patterns->comment_start,
		      &regexps->comment_start_regex)) {
    g_free (regexps);
    return NULL;
  }
  if (!compile_regex (patterns->comment_end,
		      &regexps->comment_end_regex)) {
    regex_free (&regexps->comment_start_regex);
    g_free (regexps);
    return NULL;
  }
  if (!compile_regex (patterns->string_start,
		      &regexps->string_start_regex)) {
    regex_free (&regexps->comment_start_regex);
    regex_free (&regexps->comment_end_regex);
    g_free (regexps);
    return NULL;
  }
  if (!compile_regex (patterns->string_end,
		      &regexps->string_end_regex)) {
    regex_free (&regexps->comment_start_regex);
    regex_free (&regexps->comment_end_regex);
    regex_free (&regexps->string_start_regex);
    g_free (regexps);
    return NULL;
  }
  if (!compile_regex (patterns->keywords,
		      &regexps->keywords_regex)) {
    regex_free (&regexps->comment_start_regex);
    regex_free (&regexps->comment_end_regex);
    regex_free (&regexps->string_start_regex);
    regex_free (&regexps->string_end_regex);
    g_free (regexps);
    return NULL;
  }

  return regexps;
}

/* gtk_editor_set_regexps -- sets all regexps to the new patterns in
 * patterns.  If the patterns cannot compile, it returns FALSE, and
 * the editors regexps remain unchanged.  Otherwise it returns TRUE.
 * This is just a wrapper around gtk_new_regexps, that saves you from
 * having to keep the result in a tmp var. */
gboolean
gtk_editor_set_regexps (GtkEditor *editor,
			GtkEditorHilitePatterns *patterns)
{
  GtkEditorHiliteRegexps *tmp;

  g_return_val_if_fail (editor != NULL, FALSE);
  g_return_val_if_fail (patterns != NULL, FALSE);

  if (editor->regexps != default_regexps) {
    destroy_regexps (editor->regexps);
  }
  if ((tmp = gtk_editor_new_regexps (patterns))) {
    editor->regexps = tmp;
    return TRUE;
  } else {
    return FALSE;
  }
}

/* gtk_editor_upd_regexps -- Changes the regexps corresponding to the
 * non-NULL patterns in patterns.  If successful, it returns TRUE,
 * otherwise FALSE.  If not successful, some of the regexps might be
 * changed, so beware, there is not guaranty of consistency!  If you
 * want guaranty of consistency use gtk_editor_set_regexps. */
gboolean
gtk_editor_upd_regexps (GtkEditor *editor,
			GtkEditorHilitePatterns *patterns)
{
  Regex tmp;
  g_return_val_if_fail (editor != NULL, FALSE);
  g_return_val_if_fail (patterns != NULL, FALSE);

  if (editor->regexps == default_regexps) {
    /* we cannot begin changing the defaults, so we get our own copy. */
    editor->regexps = gtk_editor_new_regexps (&default_patterns);
  }

  if (patterns->comment_start) {
    if (compile_regex (patterns->comment_start, &tmp)) {
      regex_free (&editor->regexps->comment_start_regex);
      editor->regexps->comment_start_regex = tmp;
    } else {
      regex_free (&tmp);
      return FALSE;
    }
  }

  if (patterns->comment_end) {
    if (compile_regex (patterns->comment_end, &tmp)) {
      regex_free (&editor->regexps->comment_end_regex);
      editor->regexps->comment_end_regex = tmp;
    } else {
      regex_free (&tmp);
      return FALSE;
    }
  }

  if (patterns->string_start) {
    if (compile_regex (patterns->string_start, &tmp)) {
      regex_free (&editor->regexps->string_start_regex);
      editor->regexps->string_start_regex = tmp;
    } else {
      regex_free (&tmp);
      return FALSE;
    }
  }

  if (patterns->string_end) {
    if (compile_regex (patterns->string_end, &tmp)) {
      regex_free (&editor->regexps->string_end_regex);
      editor->regexps->string_end_regex = tmp;
    } else {
      regex_free (&tmp);
      return FALSE;
    }
  }

  if (patterns->keywords) {
    if (compile_regex (patterns->keywords, &tmp)) {
      regex_free (&editor->regexps->keywords_regex);
      editor->regexps->keywords_regex = tmp;
    } else {
      regex_free (&tmp);
      return FALSE;
    }
  }
  
  return TRUE;
}

static void
destroy_regexps (GtkEditorHiliteRegexps *regexps)
{
  regex_free (&regexps->comment_start_regex);
  regex_free (&regexps->comment_end_regex);
  regex_free (&regexps->string_start_regex);
  regex_free (&regexps->string_end_regex);
  regex_free (&regexps->keywords_regex);

  g_free (regexps);
}

static GtkEditorHiliteColors *
new_colors (GdkColormap *cmap)
{
  GtkEditorHiliteColors *colors = g_new (GtkEditorHiliteColors, 1);

  colors->cmap = cmap;

  /* FIXME: these colors should of cause be specified/configured in
   * .gtkrc and in the tool using the editor */
  colors->plain.pixel = 0;
  colors->plain.red = 0;
  colors->plain.green = 0;
  colors->plain.blue = 0;

  colors->comment.pixel = 0;
  colors->comment.red = 65535;
  colors->comment.green = 0;
  colors->comment.blue = 0;

  colors->string.pixel = 0;
  colors->string.red = 65535;
  colors->string.green = 43908;
  colors->string.blue = 16383;

  colors->keywords.pixel = 0;
  colors->keywords.red = 0;
  colors->keywords.green = 0;
  colors->keywords.blue = 65535;

  gdk_color_alloc(cmap, &(colors->plain));
  gdk_color_alloc(cmap, &(colors->comment));
  gdk_color_alloc(cmap, &(colors->string));
  gdk_color_alloc(cmap, &(colors->keywords));

  return colors;
}

/* hilite_interval -- yep! That's what it does. It gets the state in
 * position 'from', and hightlights according to the regexps forward
 * to 'to'.  More precise, it hilites from the beginning of 'from's
 * textproperty and forward to 'to', and if need be (to get to a
 * consistent state) a little further. 
 *
 * I use the funny way of deleting text to sneak around the
 * delete_text event. By calling the text widgets delete routines
 * directly the signal will not be emitted.  If the signal *was*
 * emitted during highlighting, we would re-install hilite_when_idle,
 * which in the end would result in Eternal Happy Highlighting. */

/* FIXME: There's a lot of deleting/inserting going on just to set
 * text properties.  It would be a Good Thing (tm) to add functions
 * for setting properties in an inteval to the text widget. We need to
 * do that when we the time.*/

/* FIXME: This function is cursed! */
static void
hilite_interval (GtkEditor *editor, guint from, guint to)
{
  guchar *buf;
  MarkData *mdata;
  gint8 state;
  guint pos;
  guint match_from, insert_from;
  Match next_comment, next_string, next_keyword;

  g_return_if_fail (editor != NULL);
  g_return_if_fail (from <= to);
  g_return_if_fail (to <= gtk_text_get_length (GTK_TEXT (editor)));

  pos = gtk_editable_get_position (GTK_EDITABLE (editor));

  gtk_text_freeze (GTK_TEXT (editor));

  /* Go to beginning of state. If we are on offset 1, then the state
   * should really be the previous state.  It's a wonder to me, why it
   * shouldn't be offset 0...but this works, it doesn't with offset 0.
   * It's probably because this is to fix deleting the last char of a
   * state, and this can only be done with backsspace, which puts us
   * in offset 1, or something. */
  gtk_text_set_point (GTK_TEXT (editor), from);
  if (GTK_TEXT (editor)->point.offset == 1) {
    GList *prev = GTK_TEXT (editor)->point.property->prev;
    if (prev) {
      /* goto previous frame...the 1 is for the offset==1 thingy */
      from -= gtk_text_get_property_length ((GtkTextProperty*)prev->data) + 1;
      if ((gint)from < 0) from = 0;	/* not before beginning of text */
    }
  } else {
    from -= GTK_TEXT (editor)->point.offset; /* go to beginning of property. */
  }
  /* and correct point */
  gtk_text_set_point (GTK_TEXT (editor), from);

  mdata = gtk_text_get_property_data (GTK_TEXT (editor)->point.property->data);
  if (!mdata) {			/* happens first time only */
    /* assume basic */
    mdata = g_new (MarkData, 1);
    mdata->state = BASIC;
    gtk_text_set_property_data (GTK_TEXT (editor)->point.property->data,
				mdata);
  }
  state = mdata->state;

#ifdef DEBUG
  g_print ("hilite_interval [%u,%u](%u)\n", from, to, state);
#endif

  /* fix up start conditions */
  switch (state) {
  case BASIC:
    /* nothing to do really */
    insert_from = match_from = from;
    break;

  case IN_COMMENT:
    /* make sure it's still a comment, and if it is, set match_from to
     * end of 'comment_start' match. */
    match_from = regex_match (GTK_TEXT (editor), from,
			      &editor->regexps->comment_start_regex);
    if ((gint)match_from == -1) {
      /* no longer a comment */
      state = BASIC;
      insert_from = match_from = from;
    } else {
      /* a comment, match from end of start_comment match, which is at
       * from + match_from */
      insert_from = from;
      match_from += from;
    }
    break;

  case IN_STRING:
    /* make sure it's still a string, and if it is, set match_from to
     * end of 'string_start' match. */
    match_from = regex_match (GTK_TEXT (editor), from,
			      &editor->regexps->string_start_regex);
    if ((gint)match_from == -1) {
      /* no longer a string */
      state = BASIC;
      insert_from = match_from = from;
    }else {
      /* a string, match from end of start_string match, which is at
       * from + match_from */
      insert_from = from;
      match_from += from;
    }
    break;

  case IN_KEYWORD:
    /* Make sure it's still a keyword, and if it is jump to end of the
     * keyword. */
    match_from = regex_match (GTK_TEXT (editor), from,
			      &editor->regexps->keywords_regex);
    if ((gint)match_from == -1) {
      /* no longer keyword */
      state = BASIC;
      insert_from = match_from = from;
    } else {
      /* it was a keyword, jump forward to end of property */
      state = BASIC;		/* we jump to the next state, which must be BASIC */
      insert_from = from + match_from;
      match_from = insert_from;
    }
    break;
  }

#ifdef DEBUG
  g_print ("hilite_interval corrected [%u,%u](%u)\n", from, to, state);
#endif

#ifdef DEBUG
  g_print ("insert: ");
#endif    

  /* scan text and insert hilite'ed text */
  while (insert_from < to) {
    switch (state) {
    case BASIC:
      /* FIXME: This can be optimised (I think) by combining these
       * regexps into a single regexp, searching for that, and then
       * find out which regex matched, with regex_match (). */
      regex_search (GTK_TEXT (editor), match_from,
		    &editor->regexps->comment_start_regex,
		    TRUE, &next_comment);
      regex_search (GTK_TEXT (editor), match_from,
		    &editor->regexps->string_start_regex,
		    TRUE, &next_string);
      regex_search (GTK_TEXT (editor), match_from,
		    &editor->regexps->keywords_regex,
		    TRUE, &next_keyword);
      if (MATCH (next_comment.from, next_string.from, next_keyword.from)) {
	if (FIRST_LEAST (next_comment.from, next_string.from,
			 next_keyword.from)) { /* found start of comment */

#ifdef DEBUG
	  g_print ("(b->c)[%u,%u] ",  insert_from, next_comment.from);
#endif

	  buf = gtk_editable_get_chars (GTK_EDITABLE (editor), insert_from,
					next_comment.from);
	  GTK_EDITABLE_CLASS (parent_class)->delete_text (GTK_EDITABLE (editor),
							  insert_from,
							  next_comment.from);

	  mdata = g_new (MarkData, 1);
	  mdata->state = BASIC;
	  gtk_text_insert_with_data (GTK_TEXT (editor), NULL,
				     &editor->colors->plain, NULL,
				     mdata, buf, next_comment.from - insert_from);
	  g_free (buf);

	  insert_from = next_comment.from;
	  match_from = next_comment.to;
	  state = IN_COMMENT;

	} else if (FIRST_LEAST (next_string.from, next_comment.from,
				next_keyword.from)) { /* found start of string */

#ifdef DEBUG
	  g_print ("(b->s)[%u,%u] ", insert_from, next_string.from);
#endif

	  buf = gtk_editable_get_chars (GTK_EDITABLE (editor), insert_from,
					next_string.from);
	  GTK_EDITABLE_CLASS (parent_class)->delete_text (GTK_EDITABLE (editor),
							  insert_from,
							  next_string.from);

	  mdata = g_new (MarkData, 1);
	  mdata->state = BASIC;
	  gtk_text_insert_with_data (GTK_TEXT (editor), NULL,
				     &editor->colors->plain, NULL,
				     mdata, buf, next_string.from - insert_from);
	  g_free (buf);

	  insert_from = next_string.from;
	  match_from = next_string.to;
	  state = IN_STRING;

	} else {		/* found keyword */

#ifdef DEBUG
	  g_print ("(b->k)[%u,%u] ", insert_from, next_keyword.from);
#endif

	  /* insert plain */
	  buf = gtk_editable_get_chars (GTK_EDITABLE (editor), insert_from,
					next_keyword.from);
	  GTK_EDITABLE_CLASS (parent_class)->delete_text (GTK_EDITABLE (editor),
							  insert_from,
							  next_keyword.from);
	  mdata = g_new (MarkData, 1);
	  mdata->state = BASIC;
	  gtk_text_insert_with_data (GTK_TEXT (editor), NULL,
				     &editor->colors->plain, NULL,
				     mdata, buf, next_keyword.from - insert_from);
	  g_free (buf);

#ifdef DEBUG
	  g_print ("(k)[%u,%u] ", next_keyword.from, next_keyword.to);
#endif

	  /* then keyword */

	  buf = gtk_editable_get_chars (GTK_EDITABLE (editor), next_keyword.from,
					next_keyword.to);
	  GTK_EDITABLE_CLASS (parent_class)->delete_text (GTK_EDITABLE (editor),
							  next_keyword.from,
							  next_keyword.to);

	  mdata = g_new (MarkData, 1);
	  mdata->state = IN_KEYWORD;
	  gtk_text_insert_with_data (GTK_TEXT (editor), NULL,
				     &editor->colors->keywords, NULL,
				     mdata, buf, next_keyword.to - next_keyword.from);
	  g_free (buf);

	  match_from = insert_from = next_keyword.to;
	}

      } else {
	/* no match */

	/* Since we match in entire widget, no match means *no*
	 * match...so we must insert this state in the rest of the
	 * buffer. */
	to = gtk_text_get_length (GTK_TEXT (editor));

#ifdef DEBUG
	g_print ("(rb)[%u,%u] ", insert_from, to);
#endif

	buf = gtk_editable_get_chars (GTK_EDITABLE (editor), insert_from, to);
	GTK_EDITABLE_CLASS (parent_class)->delete_text (GTK_EDITABLE (editor),
							insert_from, to);

	mdata = g_new (MarkData, 1);
	mdata->state = BASIC;
	gtk_text_insert_with_data (GTK_TEXT (editor), NULL,
				   &editor->colors->plain, NULL,
				   mdata, buf, to - insert_from);
	g_free (buf);

	insert_from = to;	/* we're done */
      }
      break;

    case IN_COMMENT:
      if (regex_search (GTK_TEXT (editor), match_from,
			&editor->regexps->comment_end_regex,
			TRUE, &next_comment) > -1) {

#ifdef DEBUG
	g_print ("(c)[%u,%u] ", insert_from, next_comment.to);
#endif

	buf = gtk_editable_get_chars (GTK_EDITABLE (editor), insert_from,
				      next_comment.to);
	GTK_EDITABLE_CLASS (parent_class)->delete_text (GTK_EDITABLE (editor),
							insert_from,
							next_comment.to);

	mdata = g_new (MarkData, 1);
	mdata->state = IN_COMMENT;
	gtk_text_insert_with_data (GTK_TEXT (editor), NULL,
				   &editor->colors->comment, NULL,
				   mdata, buf, next_comment.to - insert_from);
	g_free (buf);

	match_from = insert_from = next_comment.to;
	state = BASIC;

      } else {
	/* no match */

	/* Since we match in entire widget, no match means *no*
	 * match...so we must insert this state in the rest of the
	 * buffer. */
	to = gtk_text_get_length (GTK_TEXT (editor));

#ifdef DEBUG
	g_print ("(rc)[%u,%u] ", insert_from, to);
#endif

	buf = gtk_editable_get_chars (GTK_EDITABLE (editor), insert_from, to);
	GTK_EDITABLE_CLASS (parent_class)->delete_text (GTK_EDITABLE (editor),
							insert_from, to);

	mdata = g_new (MarkData, 1);
	mdata->state = IN_COMMENT;
	gtk_text_insert_with_data (GTK_TEXT (editor), NULL,
				   &editor->colors->comment, NULL,
				   mdata, buf, to - insert_from);
	g_free (buf);

	insert_from = to;	/* we're done */
      }
      break;

    case IN_STRING:
      if (regex_search (GTK_TEXT (editor), match_from,
			&editor->regexps->string_end_regex,
			TRUE, &next_string) > -1) {

#ifdef DEBUG
	g_print ("(s)[%u,%u] ", insert_from, next_string.to);
#endif

	buf = gtk_editable_get_chars (GTK_EDITABLE (editor), insert_from,
				      next_string.to);
	GTK_EDITABLE_CLASS (parent_class)->delete_text (GTK_EDITABLE (editor),
							insert_from,
							next_string.to);

	mdata = g_new (MarkData, 1);
	mdata->state = IN_STRING;
	gtk_text_insert_with_data (GTK_TEXT (editor), NULL,
				   &editor->colors->string, NULL,
				   mdata, buf, next_string.to - insert_from);
	g_free (buf);

	match_from = insert_from = next_string.to;
	state = BASIC;

      } else {
	/* no match */

	/* Since we match in entire widget, no match means *no*
	 * match...so we must insert this state in the rest of the
	 * buffer. */
	to = gtk_text_get_length (GTK_TEXT (editor));

#ifdef DEBUG
	g_print ("(rs)[%u,%u] ", insert_from, to);
#endif

	buf = gtk_editable_get_chars (GTK_EDITABLE (editor), insert_from, to);
	GTK_EDITABLE_CLASS (parent_class)->delete_text (GTK_EDITABLE (editor),
							insert_from, to);

	mdata = g_new (MarkData, 1);
	mdata->state = IN_STRING;
	gtk_text_insert_with_data (GTK_TEXT (editor), NULL,
				   &editor->colors->string, NULL,
				   mdata, buf, to - insert_from);
	g_free (buf);

	insert_from = to;	/* we're done */

      }
      break;

    case IN_KEYWORD:
      /* Can't happen! We can start hiliting in a keyword, (we *have*
       * taken care of that), but we cannot enter a keyword
       * during. (See match in BASIC state). */
      break;
    }
  }

#ifdef DEBUG
  g_print ("\n");
  /* DELETEME: using a functions hacked in gtktext */
  gtk_text_show_props (GTK_TEXT (editor));
#endif

  gtk_editable_set_position (GTK_EDITABLE (editor), pos);
  gtk_text_thaw (GTK_TEXT (editor));
}

void
gtk_editor_hilite_buffer (GtkEditor *editor)
{
  hilite_interval (editor, 0,
		   GTK_TEXT (editor)->text_end - GTK_TEXT (editor)->gap_size);
  /* and just a little extra service */
  editor->changed_from = editor->changed_to = GTK_TEXT (editor)->point.index;
}

static gboolean
hilite_when_idle (GtkEditor *editor)
{
#ifdef DEBUG
  g_print ("hilite_when_idle [%u,%u]\n", 
	   editor->changed_from, editor->changed_to);
#endif

  hilite_interval (editor, editor->changed_from, editor->changed_to);
  editor->changed_from = editor->changed_to = GTK_TEXT (editor)->point.index;
  editor->hilite_time = 0;

  return FALSE;
}

/* --<event handles>------------------------------------------------------ */
static gint
gtk_editor_key_press (GtkWidget *widget, GdkEventKey *event)
{
  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (event != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_EDITOR (widget), FALSE);

  if (event->keyval == GDK_Tab) {
    gtk_signal_emit_stop_by_name (GTK_OBJECT (widget), "key_press_event");

    /* FIXME: call one-line indent */
    return TRUE;
  }

  return FALSE;
}

/* To keep track of changes for hilite_when_idle */
static void
gtk_editor_changed_pre (GtkEditor *editor)
{
  if (editor->hilite_when_idle) {
    /* the first MIN looks silly, but think about deleting the last
     * char in a text, and you will understand. */
    editor->changed_from = MIN (MIN (editor->changed_from,
				     GTK_TEXT (editor)->point.index),
				gtk_text_get_length (GTK_TEXT (editor)) - 1);
  }
}

/* Keeps track of changes for hilite_when_idle, and installs hilite
 * function with a time delay. */
static void
gtk_editor_changed_post (GtkEditor *editor)
{
  if (editor->hilite_when_idle) {
    editor->changed_to = MIN (MAX (editor->changed_to,
				   GTK_TEXT (editor)->point.index),
			      gtk_text_get_length (GTK_TEXT (editor)));

    /* something changed...re-highlight when idle */
    if (editor->hilite_time) gtk_timeout_remove (editor->hilite_time);
    editor->hilite_time = gtk_timeout_add (TIME_DELAY, (GtkFunction) hilite_when_idle,
					   (gpointer) editor);
  }
}
