/*
 * GtkMyToolbar by Kemal 'Disq' Hadimli
 * Modified from GtkToolbar by Federico Mena
 *
 * Coded for GTK - The GIMP Toolkit
 * Copyright (C) 1995-1997 Peter Mattis, Spencer Kimball and Josh MacDonald
 * GtkToolbar copyright (C) Federico Mena
 *
 *
 * This is the second edition, released to the public just for comments
 * If this code is OK, I'll submit it as a patch for GtkToolbar
 *
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gtk/gtkbutton.h>
#include <gtk/gtktogglebutton.h>
#include <gtk/gtkradiobutton.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkvbox.h>
#include "gtkmytoolbar.h"


#define DEFAULT_SPACE_SIZE  5
#define DEFAULT_SPACE_STYLE GTK_MYTOOLBAR_SPACE_EMPTY

#define SPACE_LINE_DIVISION 10
#define SPACE_LINE_START    3
#define SPACE_LINE_END      7

enum {
  ORIENTATION_CHANGED,
  STYLE_CHANGED,
  LAST_SIGNAL
};

typedef struct _GtkMyToolbarChildSpace GtkMyToolbarChildSpace;
struct _GtkMyToolbarChildSpace
{
  GtkMyToolbarChild child;

  gint alloc_x, alloc_y;
};

static void gtk_mytoolbar_class_init               (GtkMyToolbarClass *class);
static void gtk_mytoolbar_init                     (GtkMyToolbar      *mytoolbar);
static void gtk_mytoolbar_destroy                  (GtkObject       *object);
static void gtk_mytoolbar_map                      (GtkWidget       *widget);
static void gtk_mytoolbar_unmap                    (GtkWidget       *widget);
static void gtk_mytoolbar_draw                     (GtkWidget       *widget,
				                  GdkRectangle    *area);
static gint gtk_mytoolbar_expose                   (GtkWidget       *widget,
						  GdkEventExpose  *event);
static void gtk_mytoolbar_size_request             (GtkWidget       *widget,
				                  GtkRequisition  *requisition);
static void gtk_mytoolbar_size_allocate            (GtkWidget       *widget,
				                  GtkAllocation   *allocation);
static void gtk_mytoolbar_add                      (GtkContainer    *container,
				                  GtkWidget       *widget);
static void gtk_mytoolbar_remove                   (GtkContainer    *container,
						  GtkWidget       *widget);
static void gtk_mytoolbar_forall                   (GtkContainer    *container,
						  gboolean	   include_internals,
				                  GtkCallback      callback,
				                  gpointer         callback_data);
static void gtk_real_mytoolbar_orientation_changed (GtkMyToolbar      *mytoolbar,
						  GtkOrientation   orientation);
static void gtk_real_mytoolbar_style_changed       (GtkMyToolbar      *mytoolbar,
						  GtkMyToolbarStyle  style);

static void gtk_mytoolbar_enter_event (GtkWidget *widget, GdkEvent *event, GtkMyToolbarChild *child);
static void gtk_mytoolbar_leave_event (GtkWidget *widget, GdkEvent *event, GtkMyToolbarChild *child);


static GtkContainerClass *parent_class;

static guint mytoolbar_signals[LAST_SIGNAL] = { 0 };


guint
gtk_mytoolbar_get_type (void)
{
  static guint mytoolbar_type = 0;

  if (!mytoolbar_type)
    {
      static const GtkTypeInfo mytoolbar_info =
      {
	"GtkMyToolbar",
	sizeof (GtkMyToolbar),
	sizeof (GtkMyToolbarClass),
	(GtkClassInitFunc) gtk_mytoolbar_class_init,
	(GtkObjectInitFunc) gtk_mytoolbar_init,
	/* reserved_1 */ NULL,
	/* reserved_2 */ NULL,
        (GtkClassInitFunc) NULL,
      };

      mytoolbar_type = gtk_type_unique (gtk_container_get_type (), &mytoolbar_info);
    }

  return mytoolbar_type;
}

static void
gtk_mytoolbar_class_init (GtkMyToolbarClass *class)
{
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;
  GtkContainerClass *container_class;

  object_class = (GtkObjectClass *) class;
  widget_class = (GtkWidgetClass *) class;
  container_class = (GtkContainerClass *) class;

  parent_class = gtk_type_class (gtk_container_get_type ());

  mytoolbar_signals[ORIENTATION_CHANGED] =
    gtk_signal_new ("orientation_changed",
		    GTK_RUN_FIRST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GtkMyToolbarClass, orientation_changed),
		    gtk_marshal_NONE__INT,
		    GTK_TYPE_NONE, 1,
		    GTK_TYPE_INT);
  mytoolbar_signals[STYLE_CHANGED] =
    gtk_signal_new ("style_changed",
		    GTK_RUN_FIRST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GtkMyToolbarClass, style_changed),
		    gtk_marshal_NONE__INT,
		    GTK_TYPE_NONE, 1,
		    GTK_TYPE_INT);

  gtk_object_class_add_signals (object_class, mytoolbar_signals, LAST_SIGNAL);

  object_class->destroy = gtk_mytoolbar_destroy;

  widget_class->map = gtk_mytoolbar_map;
  widget_class->unmap = gtk_mytoolbar_unmap;
  widget_class->draw = gtk_mytoolbar_draw;
  widget_class->expose_event = gtk_mytoolbar_expose;
  widget_class->size_request = gtk_mytoolbar_size_request;
  widget_class->size_allocate = gtk_mytoolbar_size_allocate;

  container_class->add = gtk_mytoolbar_add;
  container_class->remove = gtk_mytoolbar_remove;
  container_class->forall = gtk_mytoolbar_forall;
  container_class->focus = NULL;

  class->orientation_changed = gtk_real_mytoolbar_orientation_changed;
  class->style_changed = gtk_real_mytoolbar_style_changed;
}

static void
gtk_mytoolbar_init (GtkMyToolbar *mytoolbar)
{
  GTK_WIDGET_SET_FLAGS (mytoolbar, GTK_NO_WINDOW);
  GTK_WIDGET_UNSET_FLAGS (mytoolbar, GTK_CAN_FOCUS);

  mytoolbar->num_children = 0;
  mytoolbar->children     = NULL;
  mytoolbar->orientation  = GTK_ORIENTATION_HORIZONTAL;
  mytoolbar->style        = GTK_MYTOOLBAR_ICONS;
  mytoolbar->relief       = GTK_RELIEF_NORMAL;
  mytoolbar->space_size   = DEFAULT_SPACE_SIZE;
  mytoolbar->space_style  = DEFAULT_SPACE_STYLE;
  mytoolbar->tooltips     = gtk_tooltips_new ();
  mytoolbar->button_maxw  = 0;
  mytoolbar->button_maxh  = 0;
}

GtkWidget *
gtk_mytoolbar_new (GtkOrientation  orientation,
		 GtkMyToolbarStyle style)
{
  GtkMyToolbar *mytoolbar;

  mytoolbar = gtk_type_new (gtk_mytoolbar_get_type ());

  mytoolbar->orientation = orientation;
  mytoolbar->style = style;

  return GTK_WIDGET (mytoolbar);
}

static void
gtk_mytoolbar_destroy (GtkObject *object)
{
  GtkMyToolbar *mytoolbar;
  GList *children;

  g_return_if_fail (object != NULL);
  g_return_if_fail (GTK_IS_MYTOOLBAR (object));

  mytoolbar = GTK_MYTOOLBAR (object);

  gtk_object_unref (GTK_OBJECT (mytoolbar->tooltips));
  mytoolbar->tooltips = NULL;

  for (children = mytoolbar->children; children; children = children->next)
    {
      GtkMyToolbarChild *child;

      child = children->data;

      if (child->type != GTK_MYTOOLBAR_CHILD_SPACE)
	{
	  gtk_widget_ref (child->widget);
	  gtk_widget_unparent (child->widget);
	  gtk_widget_destroy (child->widget);
	  gtk_widget_unref (child->widget);
	}

      g_free (child);
    }

  g_list_free (mytoolbar->children);
  mytoolbar->children = NULL;
  
  if (GTK_OBJECT_CLASS (parent_class)->destroy)
    (* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

static void
gtk_mytoolbar_map (GtkWidget *widget)
{
  GtkMyToolbar *mytoolbar;
  GList *children;
  GtkMyToolbarChild *child;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_MYTOOLBAR (widget));

  mytoolbar = GTK_MYTOOLBAR (widget);
  GTK_WIDGET_SET_FLAGS (mytoolbar, GTK_MAPPED);

  for (children = mytoolbar->children; children; children = children->next)
    {
      child = children->data;

      if ((child->type != GTK_MYTOOLBAR_CHILD_SPACE)
	  && GTK_WIDGET_VISIBLE (child->widget) && !GTK_WIDGET_MAPPED (child->widget))
	gtk_widget_map (child->widget);
    }
}

static void
gtk_mytoolbar_unmap (GtkWidget *widget)
{
  GtkMyToolbar *mytoolbar;
  GList *children;
  GtkMyToolbarChild *child;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_MYTOOLBAR (widget));

  mytoolbar = GTK_MYTOOLBAR (widget);
  GTK_WIDGET_UNSET_FLAGS (mytoolbar, GTK_MAPPED);

  for (children = mytoolbar->children; children; children = children->next)
    {
      child = children->data;

      if ((child->type != GTK_MYTOOLBAR_CHILD_SPACE)
	  && GTK_WIDGET_VISIBLE (child->widget) && GTK_WIDGET_MAPPED (child->widget))
	gtk_widget_unmap (child->widget);
    }
}

static void
gtk_mytoolbar_paint_space_line (GtkWidget       *widget,
			      GdkRectangle    *area,
			      GtkMyToolbarChild *child)
{
  GtkMyToolbar *mytoolbar;
  GtkMyToolbarChildSpace *child_space;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_MYTOOLBAR (widget));
  g_return_if_fail (child != NULL);
  g_return_if_fail (child->type == GTK_MYTOOLBAR_CHILD_SPACE);

  mytoolbar = GTK_MYTOOLBAR (widget);

  child_space = (GtkMyToolbarChildSpace *) child;

  if (mytoolbar->orientation == GTK_ORIENTATION_HORIZONTAL)
    gtk_paint_vline (widget->style, widget->window,
		     GTK_WIDGET_STATE (widget), area, widget,
		     "mytoolbar",
		     child_space->alloc_y + mytoolbar->button_maxh *
		     SPACE_LINE_START / SPACE_LINE_DIVISION,
		     child_space->alloc_y + mytoolbar->button_maxh *
		     SPACE_LINE_END / SPACE_LINE_DIVISION,
		     child_space->alloc_x +
		     (mytoolbar->space_size -
		      widget->style->klass->xthickness) / 2);
  else
    gtk_paint_hline (widget->style, widget->window,
		     GTK_WIDGET_STATE (widget), area, widget,
		     "mytoolbar",
		     child_space->alloc_x + mytoolbar->button_maxw *
		     SPACE_LINE_START / SPACE_LINE_DIVISION,
		     child_space->alloc_x + mytoolbar->button_maxw *
		     SPACE_LINE_END / SPACE_LINE_DIVISION,
		     child_space->alloc_y +
		     (mytoolbar->space_size -
		      widget->style->klass->ythickness) / 2);
}

static void
gtk_mytoolbar_draw (GtkWidget    *widget,
		  GdkRectangle *area)
{
  GtkMyToolbar *mytoolbar;
  GList *children;
  GtkMyToolbarChild *child;
  GdkRectangle child_area;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_MYTOOLBAR (widget));

  if (GTK_WIDGET_DRAWABLE (widget))
    {
      mytoolbar = GTK_MYTOOLBAR (widget);

      for (children = mytoolbar->children; children; children = children->next)
	{
	  child = children->data;

	  if (child->type == GTK_MYTOOLBAR_CHILD_SPACE)
	    {
	      if (mytoolbar->space_style == GTK_MYTOOLBAR_SPACE_LINE)
		gtk_mytoolbar_paint_space_line (widget, area, child);
	    }
	  else if (gtk_widget_intersect (child->widget, area, &child_area))
	    gtk_widget_draw (child->widget, &child_area);
	}
    }
}

static gint
gtk_mytoolbar_expose (GtkWidget      *widget,
		    GdkEventExpose *event)
{
  GtkMyToolbar *mytoolbar;
  GList *children;
  GtkMyToolbarChild *child;
  GdkEventExpose child_event;

  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_MYTOOLBAR (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  if (GTK_WIDGET_DRAWABLE (widget))
    {
      mytoolbar = GTK_MYTOOLBAR (widget);

      child_event = *event;

      for (children = mytoolbar->children; children; children = children->next)
	{
	  child = children->data;

	  if (child->type == GTK_MYTOOLBAR_CHILD_SPACE)
	    {
	      if (mytoolbar->space_style == GTK_MYTOOLBAR_SPACE_LINE)
		gtk_mytoolbar_paint_space_line (widget, &event->area, child);
	    }
	  else if (GTK_WIDGET_NO_WINDOW (child->widget)
		   && gtk_widget_intersect (child->widget, &event->area, &child_event.area))
	    gtk_widget_event (child->widget, (GdkEvent *) &child_event);
	}
    }

  return FALSE;
}

static void
gtk_mytoolbar_size_request (GtkWidget      *widget,
			  GtkRequisition *requisition)
{
  GtkMyToolbar *mytoolbar;
  GList *children;
  GtkMyToolbarChild *child;
  gint nbuttons;
  gint button_maxw, button_maxh;
  gint widget_maxw, widget_maxh;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_MYTOOLBAR (widget));
  g_return_if_fail (requisition != NULL);

  mytoolbar = GTK_MYTOOLBAR (widget);

  requisition->width = GTK_CONTAINER (mytoolbar)->border_width * 2;
  requisition->height = GTK_CONTAINER (mytoolbar)->border_width * 2;
  nbuttons = 0;
  button_maxw = 0;
  button_maxh = 0;
  widget_maxw = 0;
  widget_maxh = 0;

  for (children = mytoolbar->children; children; children = children->next)
    {
      child = children->data;

      switch (child->type)
	{
	case GTK_MYTOOLBAR_CHILD_SPACE:
	  if (mytoolbar->orientation == GTK_ORIENTATION_HORIZONTAL)
	    requisition->width += mytoolbar->space_size;
	  else
	    requisition->height += mytoolbar->space_size;

	  break;

	case GTK_MYTOOLBAR_CHILD_BUTTON:
	case GTK_MYTOOLBAR_CHILD_RADIOBUTTON:
	case GTK_MYTOOLBAR_CHILD_TOGGLEBUTTON:
	  if (GTK_WIDGET_VISIBLE (child->widget))
	    {
	      gtk_widget_size_request (child->widget, &child->widget->requisition);

	      nbuttons++;
	      button_maxw = MAX (button_maxw, child->widget->requisition.width);
	      button_maxh = MAX (button_maxh, child->widget->requisition.height);
	    }

	  break;

	case GTK_MYTOOLBAR_CHILD_WIDGET:
	  if (GTK_WIDGET_VISIBLE (child->widget))
	    {
	      gtk_widget_size_request (child->widget, &child->widget->requisition);

	      widget_maxw = MAX (widget_maxw, child->widget->requisition.width);
	      widget_maxh = MAX (widget_maxh, child->widget->requisition.height);

	      if (mytoolbar->orientation == GTK_ORIENTATION_HORIZONTAL)
		requisition->width += child->widget->requisition.width;
	      else
		requisition->height += child->widget->requisition.height;
	    }

	  break;

	default:
	  g_assert_not_reached ();
	}
    }

  if (mytoolbar->orientation == GTK_ORIENTATION_HORIZONTAL)
    {
      requisition->width += nbuttons * button_maxw;
      requisition->height += MAX (button_maxh, widget_maxh);
    }
  else
    {
      requisition->width += MAX (button_maxw, widget_maxw);
      requisition->height += nbuttons * button_maxh;
    }

  mytoolbar->button_maxw = button_maxw;
  mytoolbar->button_maxh = button_maxh;
}

static void
gtk_mytoolbar_size_allocate (GtkWidget     *widget,
			   GtkAllocation *allocation)
{
  GtkMyToolbar *mytoolbar;
  GList *children;
  GtkMyToolbarChild *child;
  GtkMyToolbarChildSpace *child_space;
  GtkAllocation alloc;
  gint border_width;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_MYTOOLBAR (widget));
  g_return_if_fail (allocation != NULL);

  mytoolbar = GTK_MYTOOLBAR (widget);
  widget->allocation = *allocation;

  border_width = GTK_CONTAINER (mytoolbar)->border_width;

  if (mytoolbar->orientation == GTK_ORIENTATION_HORIZONTAL)
    alloc.x = allocation->x + border_width;
  else
    alloc.y = allocation->y + border_width;

  for (children = mytoolbar->children; children; children = children->next)
    {
      child = children->data;

      switch (child->type)
	{
	case GTK_MYTOOLBAR_CHILD_SPACE:

	  child_space = (GtkMyToolbarChildSpace *) child;

	  if (mytoolbar->orientation == GTK_ORIENTATION_HORIZONTAL)
	    {
	      child_space->alloc_x = alloc.x;
	      child_space->alloc_y = allocation->y + (allocation->height - mytoolbar->button_maxh) / 2;
	      alloc.x += mytoolbar->space_size;
	    }
	  else
	    {
	      child_space->alloc_x = allocation->x + (allocation->width - mytoolbar->button_maxw) / 2;
	      child_space->alloc_y = alloc.y;
	      alloc.y += mytoolbar->space_size;
	    }

	  break;

	case GTK_MYTOOLBAR_CHILD_BUTTON:
	case GTK_MYTOOLBAR_CHILD_RADIOBUTTON:
	case GTK_MYTOOLBAR_CHILD_TOGGLEBUTTON:
	  if (!GTK_WIDGET_VISIBLE (child->widget))
	    break;

	  alloc.width = mytoolbar->button_maxw;
	  alloc.height = mytoolbar->button_maxh;

	  if (mytoolbar->orientation == GTK_ORIENTATION_HORIZONTAL)
	    alloc.y = allocation->y + (allocation->height - mytoolbar->button_maxh) / 2;
	  else
	    alloc.x = allocation->x + (allocation->width - mytoolbar->button_maxw) / 2;

	  gtk_widget_size_allocate (child->widget, &alloc);

	  if (mytoolbar->orientation == GTK_ORIENTATION_HORIZONTAL)
	    alloc.x += mytoolbar->button_maxw;
	  else
	    alloc.y += mytoolbar->button_maxh;

	  break;

	case GTK_MYTOOLBAR_CHILD_WIDGET:
	  if (!GTK_WIDGET_VISIBLE (child->widget))
	    break;

	  alloc.width = child->widget->requisition.width;
	  alloc.height = child->widget->requisition.height;

	  if (mytoolbar->orientation == GTK_ORIENTATION_HORIZONTAL)
	    alloc.y = allocation->y + (allocation->height - child->widget->requisition.height) / 2;
	  else
	    alloc.x = allocation->x + (allocation->width - child->widget->requisition.width) / 2;

	  gtk_widget_size_allocate (child->widget, &alloc);

	  if (mytoolbar->orientation == GTK_ORIENTATION_HORIZONTAL)
	    alloc.x += child->widget->requisition.width;
	  else
	    alloc.y += child->widget->requisition.height;

	  break;

	default:
	  g_assert_not_reached ();
	}
    }
}

static void
gtk_mytoolbar_add (GtkContainer *container,
		 GtkWidget    *widget)
{
  g_return_if_fail (container != NULL);
  g_return_if_fail (GTK_IS_MYTOOLBAR (container));
  g_return_if_fail (widget != NULL);

  gtk_mytoolbar_append_widget (GTK_MYTOOLBAR (container), widget, NULL, NULL);
}

static void
gtk_mytoolbar_remove (GtkContainer *container,
		    GtkWidget    *widget)
{
  GtkMyToolbar *mytoolbar;
  GList *children;
  GtkMyToolbarChild *child;

  g_return_if_fail (container != NULL);
  g_return_if_fail (GTK_IS_MYTOOLBAR (container));
  g_return_if_fail (widget != NULL);

  mytoolbar = GTK_MYTOOLBAR (container);

  for (children = mytoolbar->children; children; children = children->next)
    {
      child = children->data;

      if ((child->type != GTK_MYTOOLBAR_CHILD_SPACE) && (child->widget == widget))
	{
	  gboolean was_visible;

	  was_visible = GTK_WIDGET_VISIBLE (widget);
	  gtk_widget_unparent (widget);

	  mytoolbar->children = g_list_remove_link (mytoolbar->children, children);
	  g_free (child);
	  g_list_free (children);
	  mytoolbar->num_children--;

	  if (was_visible && GTK_WIDGET_VISIBLE (container))
	    gtk_widget_queue_resize (GTK_WIDGET (container));

	  break;
	}
    }
}

static void
gtk_mytoolbar_forall (GtkContainer *container,
		    gboolean	  include_internals,
		    GtkCallback   callback,
		    gpointer      callback_data)
{
  GtkMyToolbar *mytoolbar;
  GList *children;
  GtkMyToolbarChild *child;

  g_return_if_fail (container != NULL);
  g_return_if_fail (GTK_IS_MYTOOLBAR (container));
  g_return_if_fail (callback != NULL);

  mytoolbar = GTK_MYTOOLBAR (container);

  for (children = mytoolbar->children; children; children = children->next)
    {
      child = children->data;

      if (child->type != GTK_MYTOOLBAR_CHILD_SPACE)
	(*callback) (child->widget, callback_data);
    }
}

GtkWidget *
gtk_mytoolbar_append_item (GtkMyToolbar    *mytoolbar,
			 const char    *text,
			 const char    *tooltip_text,
			 const char    *tooltip_private_text,
			 GtkWidget     *icon,
			 GtkSignalFunc  callback,
			 gpointer       user_data)
{
  return gtk_mytoolbar_insert_element (mytoolbar, GTK_MYTOOLBAR_CHILD_BUTTON,
				     NULL, text,
				     tooltip_text, tooltip_private_text,
				     icon, callback, user_data,
				     mytoolbar->num_children);
}

GtkWidget *
gtk_mytoolbar_prepend_item (GtkMyToolbar    *mytoolbar,
			  const char    *text,
			  const char    *tooltip_text,
			  const char    *tooltip_private_text,
			  GtkWidget     *icon,
			  GtkSignalFunc  callback,
			  gpointer       user_data)
{
  return gtk_mytoolbar_insert_element (mytoolbar, GTK_MYTOOLBAR_CHILD_BUTTON,
				     NULL, text,
				     tooltip_text, tooltip_private_text,
				     icon, callback, user_data,
				     0);
}

GtkWidget *
gtk_mytoolbar_insert_item (GtkMyToolbar    *mytoolbar,
			 const char    *text,
			 const char    *tooltip_text,
			 const char    *tooltip_private_text,
			 GtkWidget     *icon,
			 GtkSignalFunc  callback,
			 gpointer       user_data,
			 gint           position)
{
  return gtk_mytoolbar_insert_element (mytoolbar, GTK_MYTOOLBAR_CHILD_BUTTON,
				     NULL, text,
				     tooltip_text, tooltip_private_text,
				     icon, callback, user_data,
					 position);
}


GtkWidget *
gtk_mytoolbar_append_item_2 (GtkMyToolbar    *mytoolbar,
			 const char    *text,
			 const char    *tooltip_text,
			 const char    *tooltip_private_text,
			 GtkWidget     *icon,
			 GtkWidget     *icon_2,
			 GtkSignalFunc  callback,
			 gpointer       user_data)
{
  return gtk_mytoolbar_insert_element_2 (mytoolbar, GTK_MYTOOLBAR_CHILD_BUTTON,
				     NULL, text,
				     tooltip_text, tooltip_private_text,
				     icon, icon_2, callback, user_data,
				     mytoolbar->num_children);
}

GtkWidget *
gtk_mytoolbar_prepend_item_2 (GtkMyToolbar    *mytoolbar,
			  const char    *text,
			  const char    *tooltip_text,
			  const char    *tooltip_private_text,
			  GtkWidget     *icon,
			  GtkWidget     *icon_2,
			  GtkSignalFunc  callback,
			  gpointer       user_data)
{
  return gtk_mytoolbar_insert_element_2 (mytoolbar, GTK_MYTOOLBAR_CHILD_BUTTON,
				     NULL, text,
				     tooltip_text, tooltip_private_text,
				     icon, icon_2, callback, user_data,
				     0);
}

GtkWidget *
gtk_mytoolbar_insert_item_2 (GtkMyToolbar    *mytoolbar,
			 const char    *text,
			 const char    *tooltip_text,
			 const char    *tooltip_private_text,
			 GtkWidget     *icon,
			 GtkWidget     *icon_2,
			 GtkSignalFunc  callback,
			 gpointer       user_data,
			 gint           position)
{
  return gtk_mytoolbar_insert_element_2 (mytoolbar, GTK_MYTOOLBAR_CHILD_BUTTON,
				     NULL, text,
				     tooltip_text, tooltip_private_text,
				     icon, icon_2, callback, user_data,
				     position);
}


void
gtk_mytoolbar_append_space (GtkMyToolbar *mytoolbar)
{
  gtk_mytoolbar_insert_element (mytoolbar, GTK_MYTOOLBAR_CHILD_SPACE,
			      NULL, NULL,
			      NULL, NULL,
			      NULL, NULL, NULL,
			      mytoolbar->num_children);
}

void
gtk_mytoolbar_prepend_space (GtkMyToolbar *mytoolbar)
{
  gtk_mytoolbar_insert_element (mytoolbar, GTK_MYTOOLBAR_CHILD_SPACE,
			      NULL, NULL,
			      NULL, NULL,
			      NULL, NULL, NULL,
			      0);
}

void
gtk_mytoolbar_insert_space (GtkMyToolbar *mytoolbar,
			  gint        position)
{
  gtk_mytoolbar_insert_element (mytoolbar, GTK_MYTOOLBAR_CHILD_SPACE,
			      NULL, NULL,
			      NULL, NULL,
			      NULL, NULL, NULL,
			      position);
}

void
gtk_mytoolbar_append_widget (GtkMyToolbar  *mytoolbar,
			   GtkWidget   *widget,
			   const gchar *tooltip_text,
			   const gchar *tooltip_private_text)
{
  gtk_mytoolbar_insert_element (mytoolbar, GTK_MYTOOLBAR_CHILD_WIDGET,
			      widget, NULL,
			      tooltip_text, tooltip_private_text,
			      NULL, NULL, NULL,
			      mytoolbar->num_children);
}

void
gtk_mytoolbar_prepend_widget (GtkMyToolbar  *mytoolbar,
			    GtkWidget   *widget,
			    const gchar *tooltip_text,
			    const gchar *tooltip_private_text)
{
  gtk_mytoolbar_insert_element (mytoolbar, GTK_MYTOOLBAR_CHILD_WIDGET,
			      widget, NULL,
			      tooltip_text, tooltip_private_text,
			      NULL, NULL, NULL,
			      mytoolbar->num_children);
}

void
gtk_mytoolbar_insert_widget (GtkMyToolbar *mytoolbar,
			   GtkWidget  *widget,
			   const char *tooltip_text,
			   const char *tooltip_private_text,
			   gint        position)
{
  gtk_mytoolbar_insert_element (mytoolbar, GTK_MYTOOLBAR_CHILD_WIDGET,
			      widget, NULL,
			      tooltip_text, tooltip_private_text,
			      NULL, NULL, NULL,
			      position);
}


GtkWidget *
gtk_mytoolbar_append_element (GtkMyToolbar          *mytoolbar,
			    GtkMyToolbarChildType  type,
			    GtkWidget           *widget,
			    const char          *text,
			    const char          *tooltip_text,
			    const char          *tooltip_private_text,
			    GtkWidget           *icon,
			    GtkSignalFunc        callback,
			    gpointer             user_data)
{
  return gtk_mytoolbar_insert_element (mytoolbar, type, widget, text,
				     tooltip_text, tooltip_private_text,
				     icon, callback, user_data,
				     mytoolbar->num_children);
}

GtkWidget *
gtk_mytoolbar_prepend_element (GtkMyToolbar          *mytoolbar,
			     GtkMyToolbarChildType  type,
			     GtkWidget           *widget,
			     const char          *text,
			     const char          *tooltip_text,
			     const char          *tooltip_private_text,
			     GtkWidget           *icon,
			     GtkSignalFunc        callback,
			     gpointer             user_data)
{
  return gtk_mytoolbar_insert_element (mytoolbar, type, widget, text,
				     tooltip_text, tooltip_private_text,
				     icon, callback, user_data, 0);
}

GtkWidget *
gtk_mytoolbar_insert_element (GtkMyToolbar          *mytoolbar,
			    GtkMyToolbarChildType  type,
			    GtkWidget           *widget,
			    const char          *text,
			    const char          *tooltip_text,
			    const char          *tooltip_private_text,
			    GtkWidget           *icon,
			    GtkSignalFunc        callback,
			    gpointer             user_data,
			    gint                 position)
{
  return gtk_mytoolbar_insert_element_2 (mytoolbar, type, widget, text,
				     tooltip_text, tooltip_private_text,
				     icon, NULL, callback, user_data, position);
}

GtkWidget *
gtk_mytoolbar_append_element_2 (GtkMyToolbar          *mytoolbar,
			    GtkMyToolbarChildType  type,
			    GtkWidget           *widget,
			    const char          *text,
			    const char          *tooltip_text,
			    const char          *tooltip_private_text,
			    GtkWidget           *icon,
			    GtkWidget           *icon_2,
			    GtkSignalFunc        callback,
			    gpointer             user_data)
{
  return gtk_mytoolbar_insert_element_2 (mytoolbar, type, widget, text,
				     tooltip_text, tooltip_private_text,
				     icon, icon_2, callback, user_data,
				     mytoolbar->num_children);
}

GtkWidget *
gtk_mytoolbar_prepend_element_2 (GtkMyToolbar          *mytoolbar,
			     GtkMyToolbarChildType  type,
			     GtkWidget           *widget,
			     const char          *text,
			     const char          *tooltip_text,
			     const char          *tooltip_private_text,
			     GtkWidget           *icon,
			     GtkWidget           *icon_2,
			     GtkSignalFunc        callback,
			     gpointer             user_data)
{
  return gtk_mytoolbar_insert_element_2 (mytoolbar, type, widget, text,
				     tooltip_text, tooltip_private_text,
				     icon, icon_2, callback, user_data, 0);
}

GtkWidget *
gtk_mytoolbar_insert_element_2 (GtkMyToolbar          *mytoolbar,
			    GtkMyToolbarChildType  type,
			    GtkWidget           *widget,
			    const char          *text,
			    const char          *tooltip_text,
			    const char          *tooltip_private_text,
			    GtkWidget           *icon,
			    GtkWidget           *icon_2,
			    GtkSignalFunc        callback,
			    gpointer             user_data,
			    gint                 position)
{
  GtkWidget *vbox;
  GtkMyToolbarChild *child;

  g_return_val_if_fail (mytoolbar != NULL, NULL);
  g_return_val_if_fail (GTK_IS_MYTOOLBAR (mytoolbar), NULL);
  if (type == GTK_MYTOOLBAR_CHILD_WIDGET)
    {
      g_return_val_if_fail (widget != NULL, NULL);
      g_return_val_if_fail (GTK_IS_WIDGET (widget), NULL);
    }
  else if (type != GTK_MYTOOLBAR_CHILD_RADIOBUTTON)
    g_return_val_if_fail (widget == NULL, NULL);

  if (type == GTK_MYTOOLBAR_CHILD_SPACE)
    child = (GtkMyToolbarChild *) g_new (GtkMyToolbarChildSpace, 1);
  else
    child = g_new (GtkMyToolbarChild, 1);

  child->type = type;
  child->icon = NULL;
  child->icon_2 = NULL;
  child->label = NULL;
  child->parent = (gpointer) mytoolbar;
  child->hasfocus = FALSE;

  switch (type)
    {
    case GTK_MYTOOLBAR_CHILD_SPACE:
      child->widget = NULL;
      ((GtkMyToolbarChildSpace *) child)->alloc_x =
	((GtkMyToolbarChildSpace *) child)->alloc_y = 0;
      break;

    case GTK_MYTOOLBAR_CHILD_WIDGET:
      child->widget = widget;
      break;

    case GTK_MYTOOLBAR_CHILD_BUTTON:
    case GTK_MYTOOLBAR_CHILD_TOGGLEBUTTON:
    case GTK_MYTOOLBAR_CHILD_RADIOBUTTON:
      if (type == GTK_MYTOOLBAR_CHILD_BUTTON)
	{
	  child->widget = gtk_button_new ();
	  gtk_button_set_relief (GTK_BUTTON (child->widget), mytoolbar->relief);
	}
      else if (type == GTK_MYTOOLBAR_CHILD_TOGGLEBUTTON)
	{
	  child->widget = gtk_toggle_button_new ();
	  gtk_button_set_relief (GTK_BUTTON (child->widget), mytoolbar->relief);
	  gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON (child->widget),
				      FALSE);
	}
      else
	{
	  child->widget = gtk_radio_button_new (widget
						? gtk_radio_button_group (GTK_RADIO_BUTTON (widget))
						: NULL);
	  gtk_button_set_relief (GTK_BUTTON (child->widget), mytoolbar->relief);
	  gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON (child->widget), FALSE);
	}

      GTK_WIDGET_UNSET_FLAGS (child->widget, GTK_CAN_FOCUS);

      if (callback)
	gtk_signal_connect (GTK_OBJECT (child->widget), "clicked",
			    callback, user_data);

      vbox = gtk_vbox_new (FALSE, 0);
      gtk_container_add (GTK_CONTAINER (child->widget), vbox);
      gtk_widget_show (vbox);

      if (text)
	{
	  child->label = gtk_label_new (text);
	  gtk_box_pack_end (GTK_BOX (vbox), child->label, FALSE, FALSE, 0);
	  if (mytoolbar->style != GTK_MYTOOLBAR_ICONS)
	    gtk_widget_show (child->label);
	}

      if (icon)
	{
	  child->icon = GTK_WIDGET (icon);
	  gtk_box_pack_end (GTK_BOX (vbox), child->icon, FALSE, FALSE, 0);
	  if (mytoolbar->style != GTK_MYTOOLBAR_TEXT)
	    gtk_widget_show (child->icon);
	}

      if (icon_2)
	{
	  child->icon_2 = GTK_WIDGET (icon_2);
	  gtk_box_pack_end (GTK_BOX (vbox), child->icon_2, FALSE, FALSE, 0);
      gtk_widget_hide (child->icon_2);
	}

      gtk_widget_show (child->widget);
      break;

    default:
      g_assert_not_reached ();
    }

  gtk_signal_connect(GTK_OBJECT(child->widget), "enter_notify_event",
		     (GtkSignalFunc)gtk_mytoolbar_enter_event, (gpointer)child);

  gtk_signal_connect(GTK_OBJECT(child->widget), "leave_notify_event",
		     (GtkSignalFunc)gtk_mytoolbar_leave_event, (gpointer)child);


  if ((type != GTK_MYTOOLBAR_CHILD_SPACE) && tooltip_text)
    gtk_tooltips_set_tip (mytoolbar->tooltips, child->widget,
			  tooltip_text, tooltip_private_text);

  mytoolbar->children = g_list_insert (mytoolbar->children, child, position);
  mytoolbar->num_children++;

  if (type != GTK_MYTOOLBAR_CHILD_SPACE)
    gtk_widget_set_parent (child->widget, GTK_WIDGET (mytoolbar));

  if ((type != GTK_MYTOOLBAR_CHILD_SPACE) && GTK_WIDGET_VISIBLE (mytoolbar))
    {
      if (GTK_WIDGET_REALIZED (mytoolbar)
	  && !GTK_WIDGET_REALIZED (child->widget))
	gtk_widget_realize (child->widget);
	
      if (GTK_WIDGET_MAPPED (mytoolbar)
	  && !GTK_WIDGET_MAPPED (child->widget))
	gtk_widget_map (child->widget);
    }

  if (GTK_WIDGET_VISIBLE (mytoolbar) &&
      ((type == GTK_MYTOOLBAR_CHILD_SPACE) ||
       GTK_WIDGET_VISIBLE (child->widget)))
    gtk_widget_queue_resize (GTK_WIDGET (mytoolbar));

  return child->widget;
}

void
gtk_mytoolbar_set_orientation (GtkMyToolbar     *mytoolbar,
			     GtkOrientation  orientation)
{
  g_return_if_fail (mytoolbar != NULL);
  g_return_if_fail (GTK_IS_MYTOOLBAR (mytoolbar));

  gtk_signal_emit (GTK_OBJECT (mytoolbar), mytoolbar_signals[ORIENTATION_CHANGED], orientation);
}

void
gtk_mytoolbar_set_style (GtkMyToolbar      *mytoolbar,
		       GtkMyToolbarStyle  style)
{
  g_return_if_fail (mytoolbar != NULL);
  g_return_if_fail (GTK_IS_MYTOOLBAR (mytoolbar));

  gtk_signal_emit (GTK_OBJECT (mytoolbar), mytoolbar_signals[STYLE_CHANGED], style);
}

void
gtk_mytoolbar_set_space_size (GtkMyToolbar *mytoolbar,
			    gint        space_size)
{
  g_return_if_fail (mytoolbar != NULL);
  g_return_if_fail (GTK_IS_MYTOOLBAR (mytoolbar));

  if (mytoolbar->space_size != space_size)
    {
      mytoolbar->space_size = space_size;
      gtk_widget_queue_resize (GTK_WIDGET (mytoolbar));
    }
}

void
gtk_mytoolbar_set_space_style (GtkMyToolbar           *mytoolbar,
			     GtkMyToolbarSpaceStyle  space_style)
{
  g_return_if_fail (mytoolbar != NULL);
  g_return_if_fail (GTK_IS_MYTOOLBAR (mytoolbar));

  if (mytoolbar->space_style != space_style)
    {
      mytoolbar->space_style = space_style;
      gtk_widget_queue_resize (GTK_WIDGET (mytoolbar));
    }
}

void
gtk_mytoolbar_set_tooltips (GtkMyToolbar *mytoolbar,
			  gint        enable)
{
  g_return_if_fail (mytoolbar != NULL);
  g_return_if_fail (GTK_IS_MYTOOLBAR (mytoolbar));

  if (enable)
    gtk_tooltips_enable (mytoolbar->tooltips);
  else
    gtk_tooltips_disable (mytoolbar->tooltips);
}

void
gtk_mytoolbar_set_button_relief (GtkMyToolbar *mytoolbar,
			       GtkReliefStyle relief)
{
  GList *children;
  GtkMyToolbarChild *child;
  
  g_return_if_fail (mytoolbar != NULL);
  g_return_if_fail (GTK_IS_MYTOOLBAR (mytoolbar));

  if (mytoolbar->relief != relief)
    {
      mytoolbar->relief = relief;
      
      for (children = mytoolbar->children; children; children = children->next)
	{
	  child = children->data;
	  if (child->type == GTK_MYTOOLBAR_CHILD_BUTTON ||
	      child->type == GTK_MYTOOLBAR_CHILD_RADIOBUTTON ||
	      child->type == GTK_MYTOOLBAR_CHILD_TOGGLEBUTTON)
	    gtk_button_set_relief (GTK_BUTTON (child->widget), relief);
	}
      
      gtk_widget_queue_resize (GTK_WIDGET (mytoolbar));
    }
}

GtkReliefStyle
gtk_mytoolbar_get_button_relief (GtkMyToolbar *mytoolbar)
{
  g_return_val_if_fail (mytoolbar != NULL, GTK_RELIEF_NORMAL);
  g_return_val_if_fail (GTK_IS_MYTOOLBAR (mytoolbar), GTK_RELIEF_NORMAL);

  return mytoolbar->relief;
}

static void
gtk_real_mytoolbar_orientation_changed (GtkMyToolbar     *mytoolbar,
				      GtkOrientation  orientation)
{
  g_return_if_fail (mytoolbar != NULL);
  g_return_if_fail (GTK_IS_MYTOOLBAR (mytoolbar));

  if (mytoolbar->orientation != orientation)
    {
      mytoolbar->orientation = orientation;
      gtk_widget_queue_resize (GTK_WIDGET (mytoolbar));
    }
}

static void
gtk_real_mytoolbar_style_changed (GtkMyToolbar      *mytoolbar,
				GtkMyToolbarStyle  style)
{
  GList *children;
  GtkMyToolbarChild *child;

  g_return_if_fail (mytoolbar != NULL);
  g_return_if_fail (GTK_IS_MYTOOLBAR (mytoolbar));

  if (mytoolbar->style != style)
    {
      mytoolbar->style = style;

      for (children = mytoolbar->children; children; children = children->next)
	{
	  child = children->data;

	  if (child->type == GTK_MYTOOLBAR_CHILD_BUTTON ||
	      child->type == GTK_MYTOOLBAR_CHILD_RADIOBUTTON ||
	      child->type == GTK_MYTOOLBAR_CHILD_TOGGLEBUTTON)
	    switch (style)
	      {
	      case GTK_MYTOOLBAR_ICONS:
		if (child->icon)
		  {
			if (!child->hasfocus)  /* GTK_WIDGET_HAS_FOCUS() does not work properly here */
			  gtk_widget_show (child->icon);
			else
		  	  gtk_widget_hide (child->icon);
		  }

		if (child->icon_2)
		  {
			if (!child->hasfocus)
			  gtk_widget_hide (child->icon_2);
			else
		  	  gtk_widget_show (child->icon_2);
		  }

		if (child->label && GTK_WIDGET_VISIBLE (child->label))
		  gtk_widget_hide (child->label);

		break;

	      case GTK_MYTOOLBAR_TEXT:
		if (child->icon && GTK_WIDGET_VISIBLE (child->icon))
		  gtk_widget_hide (child->icon);

		if (child->icon_2 && GTK_WIDGET_VISIBLE (child->icon_2))
		  gtk_widget_hide (child->icon_2);

		if (child->label && !GTK_WIDGET_VISIBLE (child->label))
		  gtk_widget_show (child->label);

		break;

	      case GTK_MYTOOLBAR_BOTH:
		if (child->icon)
		  {
			if (!child->hasfocus)
			  gtk_widget_show (child->icon);
			else
	    	  gtk_widget_hide (child->icon);
		  }

		if (child->icon_2)
		  {
			if (!child->hasfocus)
			  gtk_widget_hide (child->icon_2);
			else
		  	  gtk_widget_show (child->icon_2);
		  }

		if (child->label && !GTK_WIDGET_VISIBLE (child->label))
		  gtk_widget_show (child->label);

		break;

	      default:
		g_assert_not_reached ();
	      }
	}
		
      gtk_widget_queue_resize (GTK_WIDGET (mytoolbar));
    }
}



static void gtk_mytoolbar_enter_event (GtkWidget *widget, GdkEvent *event, GtkMyToolbarChild *child)
{
  child->hasfocus = TRUE;

    if ((child->icon) && (child->icon_2) && (GTK_MYTOOLBAR(child->parent)->style != GTK_MYTOOLBAR_TEXT))
	  {
	  gtk_widget_hide(child->icon);
	  gtk_widget_show(child->icon_2);
	  }
}


static void gtk_mytoolbar_leave_event (GtkWidget *widget, GdkEvent *event, GtkMyToolbarChild *child)
{
  child->hasfocus = FALSE;

    if ((child->icon) && (child->icon_2) && (GTK_MYTOOLBAR(child->parent)->style != GTK_MYTOOLBAR_TEXT))
	  {  
	  gtk_widget_hide(child->icon_2);
	  gtk_widget_show(child->icon);
	  }
}
