static const char* op_c_source =
"/* This file is an image processing operation for GEGL                        \n"
" *                                                                            \n"
" * GEGL is free software; you can redistribute it and/or                      \n"
" * modify it under the terms of the GNU Lesser General Public                 \n"
" * License as published by the Free Software Foundation; either               \n"
" * version 3 of the License, or (at your option) any later version.           \n"
" *                                                                            \n"
" * GEGL is distributed in the hope that it will be useful,                    \n"
" * but WITHOUT ANY WARRANTY; without even the implied warranty of             \n"
" * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU          \n"
" * Lesser General Public License for more details.                            \n"
" *                                                                            \n"
" * You should have received a copy of the GNU Lesser General Public           \n"
" * License along with GEGL; if not, see <http://www.gnu.org/licenses/>.       \n"
" *                                                                            \n"
" * SLIC (Simple Linear Iterative Clustering)                                  \n"
" * Superpixels based on k-means clustering                                    \n"
" *                                                                            \n"
" * Copyright 2017 Thomas Manni <thomas.manni@free.fr>                         \n"
" *                                                                            \n"
" */                                                                           \n"
"                                                                              \n"
"#include \"config.h\"                                                         \n"
"#include <glib/gi18n-lib.h>                                                   \n"
"                                                                              \n"
"#ifdef GEGL_PROPERTIES                                                        \n"
"                                                                              \n"
"property_int (cluster_size, _(\"Regions size\"), 32)                          \n"
" description (_(\"Size of a region side\"))                                   \n"
" value_range (2, G_MAXINT)                                                    \n"
"    ui_range (2, 1024)                                                        \n"
"                                                                              \n"
"property_int (compactness, _(\"Compactness\"), 20)                            \n"
" description (_(\"Cluster size\"))                                            \n"
" value_range (1, 40)                                                          \n"
"    ui_range (1, 40)                                                          \n"
"                                                                              \n"
"property_int (iterations, _(\"Iterations\"), 1)                               \n"
" description (_(\"Number of iterations\"))                                    \n"
" value_range (1, 30)                                                          \n"
"    ui_range (1, 15)                                                          \n"
"                                                                              \n"
"#else                                                                         \n"
"                                                                              \n"
"#define GEGL_OP_FILTER                                                        \n"
"#define GEGL_OP_NAME      slic                                                \n"
"#define GEGL_OP_C_SOURCE  slic.c                                              \n"
"                                                                              \n"
"#include \"gegl-op.h\"                                                        \n"
"#include <math.h>                                                             \n"
"                                                                              \n"
"#define POW2(x) ((x)*(x))                                                     \n"
"                                                                              \n"
"typedef struct                                                                \n"
"{                                                                             \n"
"  gfloat        center[5];                                                    \n"
"  gfloat        sum[5];                                                       \n"
"  glong         n_pixels;                                                     \n"
"  GeglRectangle search_window;                                                \n"
"} Cluster;                                                                    \n"
"                                                                              \n"
"                                                                              \n"
"static inline gfloat                                                          \n"
"get_distance (gfloat         *c1,                                             \n"
"              gfloat         *c2,                                             \n"
"              GeglProperties *o)                                              \n"
"{                                                                             \n"
"  gfloat color_dist = sqrtf (POW2(c2[0] - c1[0]) +                            \n"
"                             POW2(c2[1] - c1[1]) +                            \n"
"                             POW2(c2[2] - c1[2]));                            \n"
"                                                                              \n"
"  gfloat spacial_dist = sqrtf (POW2(c2[3] - c1[3]) +                          \n"
"                               POW2(c2[4] - c1[4]));                          \n"
"                                                                              \n"
"  return sqrtf (POW2(color_dist) +                                            \n"
"                POW2(o->compactness) * POW2(spacial_dist / o->cluster_size)); \n"
"}                                                                             \n"
"                                                                              \n"
"static GArray *                                                               \n"
"init_clusters (GeglBuffer     *input,                                         \n"
"               GeglProperties *o,                                             \n"
"               gint            level)                                         \n"
"{                                                                             \n"
"  GeglSampler *sampler;                                                       \n"
"  GArray      *clusters;                                                      \n"
"  gint         n_clusters;                                                    \n"
"  gint i, x, y;                                                               \n"
"  gint cx, cy;                                                                \n"
"  gint h_offset, v_offset;                                                    \n"
"  gint width  = gegl_buffer_get_width (input);                                \n"
"  gint height = gegl_buffer_get_height (input);                               \n"
"                                                                              \n"
"  gint n_h_clusters = width / o->cluster_size;                                \n"
"  gint n_v_clusters = height / o->cluster_size;                               \n"
"                                                                              \n"
"  if (width % o->cluster_size)                                                \n"
"   n_h_clusters++;                                                            \n"
"                                                                              \n"
"  if (height % o->cluster_size)                                               \n"
"    n_v_clusters++;                                                           \n"
"                                                                              \n"
"  h_offset = (width % o->cluster_size) ? (width % o->cluster_size) / 2 : o->cluster_size / 2;\n"
"  v_offset = (height % o->cluster_size) ? (height % o->cluster_size) / 2 : o->cluster_size / 2;\n"
"                                                                              \n"
"  n_clusters = n_h_clusters * n_v_clusters;                                   \n"
"                                                                              \n"
"  clusters = g_array_sized_new (FALSE, TRUE, sizeof (Cluster), n_clusters);   \n"
"                                                                              \n"
"  sampler = gegl_buffer_sampler_new_at_level (input,                          \n"
"                                              babl_format (\"CIE Lab float\"),\n"
"                                              GEGL_SAMPLER_NEAREST, level);   \n"
"  x = y = 0;                                                                  \n"
"                                                                              \n"
"  for (i = 0; i < n_clusters; i++)                                            \n"
"    {                                                                         \n"
"      gfloat pixel[3];                                                        \n"
"      Cluster c;                                                              \n"
"                                                                              \n"
"      cx = x * o->cluster_size + h_offset;                                    \n"
"      cy = y * o->cluster_size + v_offset;                                    \n"
"                                                                              \n"
"      gegl_sampler_get (sampler, cx, cy, NULL,                                \n"
"                        pixel, GEGL_ABYSS_CLAMP);                             \n"
"                                                                              \n"
"      c.center[0] = pixel[0];                                                 \n"
"      c.center[1] = pixel[1];                                                 \n"
"      c.center[2] = pixel[2];                                                 \n"
"      c.center[3] = (gfloat) cx;                                              \n"
"      c.center[4] = (gfloat) cy;                                              \n"
"                                                                              \n"
"      c.sum[0] = 0.0;                                                         \n"
"      c.sum[1] = 0.0;                                                         \n"
"      c.sum[2] = 0.0;                                                         \n"
"      c.sum[3] = 0.0;                                                         \n"
"      c.sum[4] = 0.0;                                                         \n"
"                                                                              \n"
"      c.n_pixels = 0;                                                         \n"
"                                                                              \n"
"      c.search_window.x = cx - o->cluster_size;                               \n"
"      c.search_window.y = cy - o->cluster_size;                               \n"
"      c.search_window.width  =                                                \n"
"      c.search_window.height = o->cluster_size * 2 + 1;                       \n"
"                                                                              \n"
"      g_array_append_val (clusters, c);                                       \n"
"                                                                              \n"
"      x++;                                                                    \n"
"      if (x >= n_h_clusters)                                                  \n"
"        {                                                                     \n"
"          x = 0;                                                              \n"
"          y++;                                                                \n"
"        }                                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  g_object_unref (sampler);                                                   \n"
"                                                                              \n"
"  return clusters;                                                            \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"assign_labels (GeglBuffer     *labels,                                        \n"
"               GeglBuffer     *input,                                         \n"
"               GArray         *clusters,                                      \n"
"               GeglProperties *o)                                             \n"
"{                                                                             \n"
"  GeglBufferIterator *iter;                                                   \n"
"  GArray  *clusters_index;                                                    \n"
"                                                                              \n"
"  clusters_index = g_array_sized_new (FALSE, FALSE, sizeof (gint), 9);        \n"
"                                                                              \n"
"  iter = gegl_buffer_iterator_new (input, NULL, 0,                            \n"
"                                   babl_format (\"CIE Lab float\"),           \n"
"                                   GEGL_ACCESS_READ, GEGL_ABYSS_NONE);        \n"
"                                                                              \n"
"  gegl_buffer_iterator_add (iter, labels, NULL, 0,                            \n"
"                            babl_format_n (babl_type (\"u32\"), 1),           \n"
"                            GEGL_ACCESS_WRITE, GEGL_ABYSS_NONE);              \n"
"                                                                              \n"
"  while (gegl_buffer_iterator_next (iter))                                    \n"
"   {                                                                          \n"
"      gfloat  *pixel = iter->data[0];                                         \n"
"      guint32 *label = iter->data[1];                                         \n"
"      glong    n_pixels = iter->length;                                       \n"
"      gint     x, y, i;                                                       \n"
"                                                                              \n"
"      x = iter->roi->x;                                                       \n"
"      y = iter->roi->y;                                                       \n"
"                                                                              \n"
"      /* construct an array of clusters index for which search_window         \n"
"       * intersect with the current roi                                       \n"
"       */                                                                     \n"
"                                                                              \n"
"      for (i = 0; i < clusters->len ; i++)                                    \n"
"        {                                                                     \n"
"          Cluster *c = &g_array_index (clusters, Cluster, i);                 \n"
"                                                                              \n"
"          if (gegl_rectangle_intersect (NULL, &c->search_window, iter->roi))  \n"
"            g_array_append_val (clusters_index, i);                           \n"
"        }                                                                     \n"
"                                                                              \n"
"      if (!clusters_index->len)                                               \n"
"        {                                                                     \n"
"          g_printerr (\"no clusters for roi %d,%d,%d,%d\\n\", iter->roi->x, iter->roi->y, iter->roi->width, iter->roi->height);\n"
"          continue;                                                           \n"
"        }                                                                     \n"
"                                                                              \n"
"      while (n_pixels--)                                                      \n"
"        {                                                                     \n"
"          Cluster *c;                                                         \n"
"          gfloat feature[5] = {pixel[0], pixel[1], pixel[2],                  \n"
"                               (gfloat) x, (gfloat) y};                       \n"
"                                                                              \n"
"          /* find the nearest cluster */                                      \n"
"                                                                              \n"
"          gfloat  min_distance = G_MAXFLOAT;                                  \n"
"          gint    best_cluster = *label;                                      \n"
"                                                                              \n"
"          for (i = 0; i < clusters_index->len ; i++)                          \n"
"            {                                                                 \n"
"              gfloat distance;                                                \n"
"              gint index = g_array_index (clusters_index, gint, i);           \n"
"              Cluster *tmp = &g_array_index (clusters, Cluster, index);       \n"
"                                                                              \n"
"              if (x < tmp->search_window.x ||                                 \n"
"                  y < tmp->search_window.y ||                                 \n"
"                  x >= tmp->search_window.x + tmp->search_window.width ||     \n"
"                  y >= tmp->search_window.y + tmp->search_window.height)      \n"
"                continue;                                                     \n"
"                                                                              \n"
"              distance = get_distance (tmp->center, feature, o);              \n"
"                                                                              \n"
"              if (distance < min_distance)                                    \n"
"                {                                                             \n"
"                  min_distance = distance;                                    \n"
"                  best_cluster = index;                                       \n"
"                }                                                             \n"
"            }                                                                 \n"
"                                                                              \n"
"          c = &g_array_index (clusters, Cluster, best_cluster);               \n"
"          c->sum[0] += pixel[0];                                              \n"
"          c->sum[1] += pixel[1];                                              \n"
"          c->sum[2] += pixel[2];                                              \n"
"          c->sum[3] += (gfloat) x;                                            \n"
"          c->sum[4] += (gfloat) y;                                            \n"
"          c->n_pixels++;                                                      \n"
"                                                                              \n"
"          g_assert (best_cluster != -1);                                      \n"
"                                                                              \n"
"          *label = best_cluster;                                              \n"
"                                                                              \n"
"          pixel += 3;                                                         \n"
"          label++;                                                            \n"
"                                                                              \n"
"          x++;                                                                \n"
"          if (x >= iter->roi->x + iter->roi->width)                           \n"
"            {                                                                 \n"
"              y++;                                                            \n"
"              x = iter->roi->x;                                               \n"
"            }                                                                 \n"
"        }                                                                     \n"
"                                                                              \n"
"      clusters_index->len = 0;                                                \n"
"   }                                                                          \n"
"                                                                              \n"
"  g_array_free (clusters_index, TRUE);                                        \n"
"}                                                                             \n"
"                                                                              \n"
"static gboolean                                                               \n"
"update_clusters (GArray         *clusters,                                    \n"
"                 GeglProperties *o)                                           \n"
"{                                                                             \n"
"  gint i;                                                                     \n"
"                                                                              \n"
"  for (i = 0; i < clusters->len; i++)                                         \n"
"    {                                                                         \n"
"      Cluster *c = &g_array_index (clusters, Cluster, i);                     \n"
"                                                                              \n"
"      c->center[0] = c->sum[0] / c->n_pixels;                                 \n"
"      c->center[1] = c->sum[1] / c->n_pixels;                                 \n"
"      c->center[2] = c->sum[2] / c->n_pixels;                                 \n"
"      c->center[3] = c->sum[3] / c->n_pixels;                                 \n"
"      c->center[4] = c->sum[4] / c->n_pixels;                                 \n"
"                                                                              \n"
"      c->sum[0] = 0.0f;                                                       \n"
"      c->sum[1] = 0.0f;                                                       \n"
"      c->sum[2] = 0.0f;                                                       \n"
"      c->sum[3] = 0.0f;                                                       \n"
"      c->sum[4] = 0.0f;                                                       \n"
"                                                                              \n"
"      c->n_pixels = 0;                                                        \n"
"                                                                              \n"
"      c->search_window.x = (gint) c->center[3] - o->cluster_size;             \n"
"      c->search_window.y = (gint) c->center[4] - o->cluster_size;             \n"
"    }                                                                         \n"
"                                                                              \n"
"  return TRUE;                                                                \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"set_output (GeglBuffer *output,                                               \n"
"            GeglBuffer *labels,                                               \n"
"            GArray     *clusters)                                             \n"
"{                                                                             \n"
"  GeglBufferIterator *iter;                                                   \n"
"                                                                              \n"
"  iter = gegl_buffer_iterator_new (output, NULL, 0,                           \n"
"                                   babl_format (\"CIE Lab float\"),           \n"
"                                   GEGL_ACCESS_WRITE, GEGL_ABYSS_NONE);       \n"
"                                                                              \n"
"  gegl_buffer_iterator_add (iter, labels, NULL, 0,                            \n"
"                            babl_format_n (babl_type (\"u32\"), 1),           \n"
"                            GEGL_ACCESS_READ, GEGL_ABYSS_NONE);               \n"
"                                                                              \n"
"  while (gegl_buffer_iterator_next (iter))                                    \n"
"    {                                                                         \n"
"      gfloat  *pixel    = iter->data[0];                                      \n"
"      guint32 *label    = iter->data[1];                                      \n"
"      glong    n_pixels = iter->length;                                       \n"
"                                                                              \n"
"      while (n_pixels--)                                                      \n"
"        {                                                                     \n"
"          Cluster *c = &g_array_index (clusters, Cluster, *label);            \n"
"                                                                              \n"
"          pixel[0] = c->center[0];                                            \n"
"          pixel[1] = c->center[1];                                            \n"
"          pixel[2] = c->center[2];                                            \n"
"                                                                              \n"
"          pixel += 3;                                                         \n"
"          label++;                                                            \n"
"        }                                                                     \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"prepare (GeglOperation *operation)                                            \n"
"{                                                                             \n"
"  const Babl *format = babl_format (\"CIE Lab float\");                       \n"
"                                                                              \n"
"  gegl_operation_set_format (operation, \"input\",  format);                  \n"
"  gegl_operation_set_format (operation, \"output\", format);                  \n"
"}                                                                             \n"
"                                                                              \n"
"static GeglRectangle                                                          \n"
"get_required_for_output (GeglOperation       *operation,                      \n"
"                         const gchar         *input_pad,                      \n"
"                         const GeglRectangle *roi)                            \n"
"{                                                                             \n"
"  GeglRectangle result = *gegl_operation_source_get_bounding_box (operation, \"input\");\n"
"                                                                              \n"
"  /* Don't request an infinite plane */                                       \n"
"  if (gegl_rectangle_is_infinite_plane (&result))                             \n"
"    return *roi;                                                              \n"
"                                                                              \n"
"  return result;                                                              \n"
"}                                                                             \n"
"                                                                              \n"
"static GeglRectangle                                                          \n"
"get_cached_region (GeglOperation       *operation,                            \n"
"                   const GeglRectangle *roi)                                  \n"
"{                                                                             \n"
"  GeglRectangle result = *gegl_operation_source_get_bounding_box (operation, \"input\");\n"
"                                                                              \n"
"  if (gegl_rectangle_is_infinite_plane (&result))                             \n"
"    return *roi;                                                              \n"
"                                                                              \n"
"  return result;                                                              \n"
"}                                                                             \n"
"                                                                              \n"
"static gboolean                                                               \n"
"process (GeglOperation       *operation,                                      \n"
"         GeglBuffer          *input,                                          \n"
"         GeglBuffer          *output,                                         \n"
"         const GeglRectangle *result,                                         \n"
"         gint                 level)                                          \n"
"{                                                                             \n"
"  GeglProperties *o = GEGL_PROPERTIES (operation);                            \n"
"                                                                              \n"
"  const GeglRectangle *src_region = gegl_buffer_get_extent (input);           \n"
"  GeglBuffer *labels;                                                         \n"
"  GArray     *clusters;                                                       \n"
"  gint        i;                                                              \n"
"                                                                              \n"
"  labels = gegl_buffer_new (src_region, babl_format_n (babl_type (\"u32\"), 1));\n"
"                                                                              \n"
"  /* clusters initialization */                                               \n"
"                                                                              \n"
"  clusters = init_clusters (input, o, level);                                 \n"
"                                                                              \n"
"  /* perform segmentation */                                                  \n"
"                                                                              \n"
"  for (i = 0; i < o->iterations; i++)                                         \n"
"    {                                                                         \n"
"      assign_labels (labels, input, clusters, o);                             \n"
"                                                                              \n"
"      update_clusters (clusters, o);                                          \n"
"    }                                                                         \n"
"                                                                              \n"
"  /* apply clusters colors to output */                                       \n"
"                                                                              \n"
"  set_output (output, labels, clusters);                                      \n"
"                                                                              \n"
"  g_object_unref (labels);                                                    \n"
"  g_array_free (clusters, TRUE);                                              \n"
"                                                                              \n"
"  return TRUE;                                                                \n"
"}                                                                             \n"
"                                                                              \n"
"static gboolean                                                               \n"
"operation_process (GeglOperation        *operation,                           \n"
"                   GeglOperationContext *context,                             \n"
"                   const gchar          *output_prop,                         \n"
"                   const GeglRectangle  *result,                              \n"
"                   gint                  level)                               \n"
"{                                                                             \n"
"  GeglOperationClass  *operation_class;                                       \n"
"                                                                              \n"
"  const GeglRectangle *in_rect =                                              \n"
"    gegl_operation_source_get_bounding_box (operation, \"input\");            \n"
"                                                                              \n"
"  operation_class = GEGL_OPERATION_CLASS (gegl_op_parent_class);              \n"
"                                                                              \n"
"  if (in_rect && gegl_rectangle_is_infinite_plane (in_rect))                  \n"
"    {                                                                         \n"
"      gpointer in = gegl_operation_context_get_object (context, \"input\");   \n"
"      gegl_operation_context_take_object (context, \"output\",                \n"
"                                          g_object_ref (G_OBJECT (in)));      \n"
"      return TRUE;                                                            \n"
"    }                                                                         \n"
"                                                                              \n"
"  return operation_class->process (operation, context, output_prop, result,   \n"
"                                   gegl_operation_context_get_level (context));\n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"gegl_op_class_init (GeglOpClass *klass)                                       \n"
"{                                                                             \n"
"  GeglOperationClass       *operation_class;                                  \n"
"  GeglOperationFilterClass *filter_class;                                     \n"
"                                                                              \n"
"  operation_class = GEGL_OPERATION_CLASS (klass);                             \n"
"  filter_class    = GEGL_OPERATION_FILTER_CLASS (klass);                      \n"
"                                                                              \n"
"  filter_class->process                    = process;                         \n"
"  operation_class->prepare                 = prepare;                         \n"
"  operation_class->process                 = operation_process;               \n"
"  operation_class->get_required_for_output = get_required_for_output;         \n"
"  operation_class->get_cached_region       = get_cached_region;               \n"
"  operation_class->opencl_support          = FALSE;                           \n"
"  operation_class->threaded                = FALSE;                           \n"
"  operation_class->want_in_place           = FALSE;                           \n"
"                                                                              \n"
"  gegl_operation_class_set_keys (operation_class,                             \n"
"      \"name\",        \"gegl:slic\",                                         \n"
"      \"title\",       _(\"Simple Linear Iterative Clustering\"),             \n"
"      \"categories\",  \"color:segmentation\",                                \n"
"      \"reference-hash\", \"9fa3122f5fcc436bbd0750150290f9d7\",               \n"
"      \"description\", _(\"Superpixels based on k-means clustering\"),        \n"
"      NULL);                                                                  \n"
"}                                                                             \n"
"                                                                              \n"
"#endif                                                                        \n"
;
