/* CSL - Common Sound Layer
 * Copyright (C) 2000-2001 Stefan Westerfeld and Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General
 * Public License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 */
#include	"cslprivate.h"

#include	"cslutils.h"
#include	<string.h>
#include	<fcntl.h>
#include	<errno.h>
#include	<unistd.h>
#include	<stdlib.h>
#include	<sys/ioctl.h>
#include	<linux/soundcard.h>


#define	FMT_ERROR (0xffffffff)

typedef struct {
  CslPcmStream stream;
  int   fd;
  char *device_name;
} OSSPcmStream;
#define	OSS_PCM_STREAM(pcmstream)	((OSSPcmStream*) (pcmstream))


/* --- prototypes --- */
static CslErrorType  oss_pcm_driver_init		(CslDriver      *driver);
static void	     oss_pcm_driver_shutdown		(CslDriver      *driver);
static CslErrorType  oss_pcm_stream_init		(CslDriver       *driver,
							 const char      *role,
							 unsigned int     rate,
							 unsigned int     n_channels,
							 CslPcmFormatType format,
							 CslBool	  readable,
							 CslBool	  writable,
							 CslPcmStream	**stream_p);
static void	     oss_pcm_stream_destroy            	(CslPcmStream    *stream);
static int	     oss_pcm_read               	(CslPcmStream    *stream,
							 unsigned int     n_bytes,
							 char            *bytes);
static int	     oss_pcm_write              	(CslPcmStream    *stream,
							 unsigned int     n_bytes,
							 char            *bytes);
static CslErrorType  oss_pcm_update_status         	(CslPcmStream    *stream);
static CslErrorType  oss_pcm_sync               	(CslPcmStream    *stream);
static CslErrorType  oss_pcm_flush              	(CslPcmStream    *stream);
static CslErrorType  oss_pcm_activate           	(CslPcmStream    *stream);
static CslErrorType  oss_pcm_suspend            	(CslPcmStream    *stream);
static CslErrorType  oss_pcm_set_title          	(CslPcmStream   *stream,
							 const char     *title);
static CslErrorType  oss_pcm_set_stream_mode    	(CslPcmStream   *stream,
							 unsigned int    buffer_size);
static CslErrorType  oss_pcm_set_stream_watermark	(CslPcmStream   *stream,
							 unsigned int    n_bytes);
static CslErrorType  oss_pcm_set_packet_mode		(CslPcmStream   *stream,
							 unsigned int    n_packets,
							 unsigned int    packet_size);
static CslErrorType  oss_pcm_set_packet_watermark	(CslPcmStream   *stream,
							 unsigned int    n_packets);
static CslErrorType  oss_pcm_set_channel_mapping        (CslPcmStream   *stream,
							 unsigned int    channel,
							 const char     *mapping);
static CslErrorType  pcm_device_oss_setup		(OSSPcmStream	*stream);


/* --- varables --- */
CslPcmDriverVTable _csl_oss_pcm_vtable = {
  oss_pcm_driver_init,
  oss_pcm_driver_shutdown,
  oss_pcm_stream_init,
  oss_pcm_stream_destroy,
  oss_pcm_read,
  oss_pcm_write,
  oss_pcm_update_status,
  oss_pcm_sync,
  oss_pcm_flush,
  oss_pcm_activate,
  oss_pcm_suspend,
  oss_pcm_set_title,
  oss_pcm_set_stream_mode,
  oss_pcm_set_stream_watermark,
  oss_pcm_set_packet_mode,
  oss_pcm_set_packet_watermark,
  oss_pcm_set_channel_mapping,
};


/* --- functions --- */
static CslErrorType
oss_pcm_driver_init (CslDriver *driver)
{
  CslErrorType error;
  char *device_name;
  int fd;

  device_name = getenv ("OSS_DEVICE");
  device_name = device_name ? device_name : "/dev/dsp";
  fd = open (device_name, O_WRONLY | O_NONBLOCK, 0); /* always open non-blocking to avoid open() hangs */
  if (fd >= 0)
    {
      static const char *pcm_mappings[] = {
	CSL_PCM_CHANNEL_CENTER,
	CSL_PCM_CHANNEL_FRONT_LEFT,
	CSL_PCM_CHANNEL_FRONT_RIGHT,
      };
      
      close (fd);

      /* initialize _all_ of driver's PCM portion */
      driver->pcm_vtable = &_csl_oss_pcm_vtable;
      driver->pcm_data = CSL_UINT_TO_POINTER (0);
      driver->n_pcm_mappings = sizeof (pcm_mappings) / sizeof (pcm_mappings[0]);
      driver->pcm_mappings = pcm_mappings;

      error = CSL_ENONE;
    }
  else
    error = CSL_ENODRIVER;

  if (csl_debug (PCM))
    csl_message ("+OSS-STREAM-INIT(%s): %s%s%s",
		 device_name,
		 error == CSL_ENONE ? "succeeded" : "failed",
		 error == CSL_ENONE ? "" : ": ",
		 error == CSL_ENONE ? "" : csl_strerror (error));

  return error;
}

static void
oss_pcm_driver_shutdown (CslDriver *driver)
{
  if (CSL_POINTER_TO_UINT (driver->pcm_data))
    csl_warning ("shutting down PCM driver while %d streams still alive",
		 CSL_POINTER_TO_INT (driver->pcm_data));
  if (csl_debug (PCM))
    csl_message ("-OSS-STREAM-SHUTDOWN");
}

static unsigned int
oss_format_translate (unsigned int csl_fmt)
{
  unsigned int oss_fmt = 0;

  switch (csl_fmt & (CSL_PCM_FORMAT_SIZE_MASK | CSL_PCM_FORMAT_ENDIAN_MASK | CSL_PCM_FORMAT_ENCODING_MASK))
    {
    case CSL_PCM_FORMAT_SIZE_8 | CSL_PCM_FORMAT_ENCODING_LINEAR_UNSIGNED | CSL_PCM_FORMAT_ENDIAN_BE:
    case CSL_PCM_FORMAT_SIZE_8 | CSL_PCM_FORMAT_ENCODING_LINEAR_UNSIGNED | CSL_PCM_FORMAT_ENDIAN_LE:
    case CSL_PCM_FORMAT_SIZE_8 | CSL_PCM_FORMAT_ENCODING_LINEAR_UNSIGNED:
      oss_fmt |= AFMT_U8;
      break;
    case CSL_PCM_FORMAT_SIZE_16 | CSL_PCM_FORMAT_ENCODING_LINEAR_UNSIGNED | CSL_PCM_FORMAT_ENDIAN_BE:
      oss_fmt |= AFMT_U16_BE;
      break;
    case CSL_PCM_FORMAT_SIZE_16 | CSL_PCM_FORMAT_ENCODING_LINEAR_UNSIGNED | CSL_PCM_FORMAT_ENDIAN_LE:
      oss_fmt |= AFMT_U16_LE;
      break;
    case CSL_PCM_FORMAT_SIZE_16 | CSL_PCM_FORMAT_ENCODING_LINEAR_SIGNED | CSL_PCM_FORMAT_ENDIAN_BE:
      oss_fmt |= AFMT_S16_BE;
      break;
    case CSL_PCM_FORMAT_SIZE_16 | CSL_PCM_FORMAT_ENCODING_LINEAR_SIGNED | CSL_PCM_FORMAT_ENDIAN_LE:
      oss_fmt |= AFMT_S16_LE;
      break;
    default:
      oss_fmt |= FMT_ERROR;
      break;
    }
  return oss_fmt;
}

static unsigned int
csl_bit_storage (unsigned int number)
{
  register unsigned int n_bits = 0;

  do
    {
      n_bits++;
      number >>= 1;
    }
  while (number);
  return n_bits;
}

static CslErrorType
oss_pcm_stream_init (CslDriver       *driver,
		     const char      *role,
		     unsigned int     rate,
		     unsigned int     n_channels,
		     CslPcmFormatType format,
		     CslBool	      readable,
		     CslBool	      writable,
		     CslPcmStream   **stream_p)
{
  OSSPcmStream *stream;
  CslErrorType error;
  char *device_name;
  int omode = 0;
  int fd;

  if (writable && readable)
    omode |= O_RDWR;
  else if (writable)
    omode |= O_WRONLY;
  else if (readable)
    omode |= O_RDONLY;
  else
    csl_return_val_if_fail (readable || writable, CSL_EINTERN);

  device_name = getenv ("OSS_DEVICE");
  device_name = device_name ? device_name : "/dev/dsp"; /* syn this with oss_pcm_driver_init() */
  fd = open (device_name, omode | O_NONBLOCK, 0); /* always open non-blocking to avoid open() hangs */
  if (fd < 0)
    {
      if (errno == EBUSY)
	return CSL_EBUSY;
      else if (errno == EISDIR || errno == EACCES || errno == EROFS)
	return CSL_EPERMS;
      else
	return CSL_EIO;
    }
  stream = csl_new0 (OSSPcmStream, 1);
  stream->stream.driver = driver;
  stream->stream.role = csl_strdup (role);
  stream->stream.title = csl_strdup (role);
  stream->stream.blocking = TRUE;
  stream->stream.readable = readable;
  stream->stream.writable = writable;
  stream->stream.packet_mode = TRUE;
  stream->stream.stream_mode = !stream->stream.packet_mode;
  stream->stream.active = FALSE;
  stream->stream.n_channels = n_channels;
  stream->stream.channel_mappings = NULL;
  stream->stream.format = format;
  stream->stream.rate = rate;

  /* pick buffering defaults for OSS fragments (packet_mode = TRUE) */
  stream->stream.packet.n_total_packets = 128;
  stream->stream.packet.packet_size = 1024;
  stream->stream.packet.n_packets_available = 0; /* unused in _setup() */
  stream->stream.packet.packet_watermark = 1; /* OSS supports just one packet */

  /* setup buffering values based on default packet setup */
  stream->stream.buffer_size = (stream->stream.packet.n_total_packets *
				stream->stream.packet.packet_size);
  stream->stream.n_bytes_available = 0; /* unused in _setup() */
  stream->stream.buffer_watermark = (stream->stream.packet.packet_size *
				     stream->stream.packet.packet_watermark);
  stream->fd = fd;
  stream->device_name = csl_strdup (device_name);

  error = pcm_device_oss_setup (stream);
  if (error)
    {
      close (stream->fd);
      csl_free (stream->stream.channel_mappings);
      csl_free (stream->stream.role);
      csl_free (stream->stream.title);
      csl_free (stream->device_name);
      csl_free (stream);
    }
  else
    {
      /* _setup() didn't fill in buffer settings */
      oss_pcm_update_status (&stream->stream);
      /* done */
      *stream_p = &stream->stream;
      driver->pcm_data = CSL_UINT_TO_POINTER (CSL_POINTER_TO_UINT (driver->pcm_data) + 1);
      stream->stream.active = TRUE;
    }

  return error;

  
  *stream_p = NULL;
  return CSL_ENODRIVER;
}

static void
oss_pcm_stream_destroy (CslPcmStream *_stream)
{
  OSSPcmStream *stream = OSS_PCM_STREAM (_stream);

  close (stream->fd);
  csl_free (stream->stream.channel_mappings);
  csl_free (stream->stream.role);
  csl_free (stream->stream.title);
  csl_free (stream->device_name);
  csl_free (stream);
  stream->stream.driver->pcm_data = CSL_UINT_TO_POINTER (CSL_POINTER_TO_UINT (stream->stream.driver->pcm_data) - 1);
}

static int
oss_pcm_read (CslPcmStream *_stream,
	      unsigned int  n_bytes,
	      char         *bytes)
{
  OSSPcmStream *stream = OSS_PCM_STREAM (_stream);
  int n;

  n = read (stream->fd, bytes, n_bytes);

  return MAX (n, 0);	/* -1 shouldn't occour in OSS */
}

static int
oss_pcm_write (CslPcmStream *_stream,
	       unsigned int  n_bytes,
	       char         *bytes)
{
  OSSPcmStream *stream = OSS_PCM_STREAM (_stream);
  int n;

  n = write (stream->fd, bytes, n_bytes);

  return MAX (n, 0);	/* -1 shouldn't occour in OSS */
}

static CslErrorType
oss_pcm_update_status (CslPcmStream *_stream)
{
  OSSPcmStream *stream = OSS_PCM_STREAM (_stream);
  audio_buf_info info;

  memset (&info, 0, sizeof (info));
  if (stream->stream.readable)
    (void) ioctl (stream->fd, SNDCTL_DSP_GETISPACE, &info);
  else /* stream->stream.writable */
    (void) ioctl (stream->fd, SNDCTL_DSP_GETOSPACE, &info);

  /* always fill package info, deduce stream buffer sizes from there */
  stream->stream.packet.n_total_packets = info.fragstotal;
  stream->stream.packet.packet_size = info.fragsize;
  stream->stream.packet.n_packets_available = info.fragments;
  stream->stream.packet.packet_watermark = 1; /* OSS supports just one packet */

  /* setup buffering values from packet information */
  stream->stream.buffer_size = (stream->stream.packet.n_total_packets *
				stream->stream.packet.packet_size);
  stream->stream.n_bytes_available = info.bytes; /* info.fragments * info.fragsize */
  stream->stream.buffer_watermark = (stream->stream.packet.packet_size *
				     stream->stream.packet.packet_watermark);
  /* kludge around buggy OSS drivers (e.g. es1371 in 2.3.34) */
  stream->stream.n_bytes_available = MIN (stream->stream.n_bytes_available,
					  stream->stream.buffer_size);
  if (csl_debug (PCM))
    csl_message ("OSS-STATUS(%s): left=%d/%d frags: total=%d size=%d count=%d bytes=%d",
		 stream->device_name,
		 stream->stream.n_bytes_available,
		 stream->stream.buffer_size,
		 info.fragstotal,
		 info.fragsize,
		 info.fragments,
		 info.bytes);

  return CSL_ENONE;
}

static CslErrorType
oss_pcm_flush (CslPcmStream *_stream)
{
  OSSPcmStream *stream = OSS_PCM_STREAM (_stream);

  /* cross fingers that format, fragment settings etc. remain
   * untouched across OSS driver RESET implementations
   */
  (void) ioctl (stream->fd, SNDCTL_DSP_RESET);

  return CSL_ENONE;
}

static CslErrorType
oss_pcm_sync (CslPcmStream *_stream)
{
  OSSPcmStream *stream = OSS_PCM_STREAM (_stream);

  (void) ioctl (stream->fd, SNDCTL_DSP_SYNC);

  return CSL_ENONE;
}

static CslErrorType
oss_pcm_activate (CslPcmStream *_stream)
{
  OSSPcmStream *stream = OSS_PCM_STREAM (_stream);

  stream->stream.active = TRUE;

  /* dum di du, we don't support suspend on OSS yet,
   * because someone could steal our sound-device etc.
   * yes, life-sucks, but yours sucks worse than mine ;)
   */
  
  return CSL_ENONE;
}

static CslErrorType
oss_pcm_suspend (CslPcmStream *_stream)
{
  OSSPcmStream *stream = OSS_PCM_STREAM (_stream);

  oss_pcm_sync (_stream);

  stream->stream.active = FALSE;

  /* read up in oss_pcm_activate() to learn about an imperfect world */
  
  return CSL_ENONE;
}

static CslErrorType
oss_pcm_set_title (CslPcmStream *_stream,
		   const char   *title)
{
  OSSPcmStream *stream = OSS_PCM_STREAM (_stream);
  
  csl_free (stream->stream.title);
  stream->stream.title = csl_strdup (title);
  
  return CSL_ENONE;
}

static CslErrorType
oss_pcm_set_stream_mode (CslPcmStream   *stream,
			 unsigned int    buffer_size)
{
  /* FIXME: someone implement streaming->fragment aproximation for OSS */
  return CSL_ECAPSUPPORT;
}

static CslErrorType
oss_pcm_set_stream_watermark (CslPcmStream   *stream,
			      unsigned int    n_bytes)
{
  /* FIXME: someone implement streaming->fragment aproximation for OSS */
  return CSL_ECAPSUPPORT;
}

static CslErrorType
oss_pcm_set_packet_mode	(CslPcmStream   *_stream,
			 unsigned int    n_packets,
			 unsigned int    packet_size)
{
  OSSPcmStream *stream = OSS_PCM_STREAM (_stream);
  CslErrorType error;
  
  stream->stream.packet_mode = TRUE;
  stream->stream.stream_mode = FALSE;

  (void) ioctl (stream->fd, SNDCTL_DSP_RESET);

  /* setup OSS fragments */
  stream->stream.packet.n_total_packets = n_packets;
  stream->stream.packet.packet_size = packet_size;
  stream->stream.packet.n_packets_available = 0; /* unused in _setup() */
  stream->stream.packet.packet_watermark = 1; /* OSS supports just one packet */

  /* setup buffering values based on default packet setup */
  stream->stream.buffer_size = (stream->stream.packet.n_total_packets *
				stream->stream.packet.packet_size);
  stream->stream.n_bytes_available = 0; /* unused in _setup() */
  stream->stream.buffer_watermark = (stream->stream.packet.packet_size *
				     stream->stream.packet.packet_watermark);

  /* close eyes, cross fingers */
  csl_free (stream->stream.channel_mappings);
  stream->stream.channel_mappings = NULL;
  error = pcm_device_oss_setup (stream);

  /* _setup() didn't fill in buffer settings */
  oss_pcm_update_status (&stream->stream);
  
  return error;
}

static CslErrorType
oss_pcm_set_packet_watermark (CslPcmStream   *stream,
			      unsigned int    n_packets)
{
  /* user is supposed to read-out real watermark after invoking this
   * function, for OSS it's constant 1, yes, suckage
   */
  return CSL_ENONE;
}

static CslErrorType
oss_pcm_set_channel_mapping (CslPcmStream   *stream,
			     unsigned int    channel,
			     const char     *mapping)
{
  /* FIXME: channel mapping for OSS? HAHA <evil-grin> */
  return CSL_ECAPSUPPORT;
}


static CslErrorType
pcm_device_oss_setup (OSSPcmStream *stream)
{
  unsigned int oss_format;
  int fd = stream->fd;
  long d_long;
  int d_int;

  /* control blocking */
  d_long = fcntl (fd, F_GETFL, 0);
  if (stream->stream.blocking)
    d_long &= ~O_NONBLOCK;
  else
    d_long |= O_NONBLOCK;
  if (fcntl (fd, F_SETFL, d_long))
    return CSL_EIO;

  /* sample format */
  oss_format = oss_format_translate (stream->stream.format);
  if (oss_format == FMT_ERROR)
    return CSL_EFMTINVAL;
  d_int = oss_format;
  if (ioctl (fd, SNDCTL_DSP_GETFMTS, &d_int) < 0)
    return CSL_EGETCAPS;
  if ((d_int & oss_format) != oss_format)
    return CSL_ECAPSUPPORT;
  d_int = oss_format;
  if (ioctl (fd, SNDCTL_DSP_SETFMT, &d_int) < 0 ||
      d_int != oss_format)
    return CSL_ESETCAPS;

  /* number of channels (STEREO || MONO) */
  if (stream->stream.n_channels < 1 || stream->stream.n_channels > 2)
    return CSL_ECAPSUPPORT;
  d_int = stream->stream.n_channels - 1;
  if (ioctl (fd, SNDCTL_DSP_STEREO, &d_int) < 0)
    return CSL_ESETCAPS;
  stream->stream.n_channels = d_int + 1;

  /* channel mappings */
  if (stream->stream.n_channels == 1)
    {
      stream->stream.channel_mappings = csl_new (char*, 1);
      stream->stream.channel_mappings[0] = CSL_PCM_CHANNEL_CENTER;
    }
  else /* stereo */
    {
      stream->stream.channel_mappings = csl_new (char*, 2);
      stream->stream.channel_mappings[0] = CSL_PCM_CHANNEL_FRONT_LEFT;
      stream->stream.channel_mappings[1] = CSL_PCM_CHANNEL_FRONT_RIGHT;
    }

  /* frequency or sample rate */
  d_int = stream->stream.rate;
  if (ioctl (fd, SNDCTL_DSP_SPEED, &d_int) < 0)
    return CSL_ESETCAPS;
  stream->stream.rate = d_int;

  /* buffer settings */
  if (stream->stream.stream_mode)
    {
      return CSL_ECAPSUPPORT; // FIXME
    }
  else
    {
      unsigned int fragment_size, n_fragments;
      
      /* Note: fragment = n_fragments << 16;
       *       fragment |= g_bit_storage (fragment_size - 1);
       */
      fragment_size = CLAMP (stream->stream.packet.packet_size, 128, 32768);
      n_fragments = CLAMP (stream->stream.packet.n_total_packets, 2, 128);
      d_int = (n_fragments << 16) | csl_bit_storage (fragment_size - 1);
      if (ioctl (fd, SNDCTL_DSP_SETFRAGMENT, &d_int) < 0)
	return CSL_ESETCAPS;
    }

  /* check buffer settings retrieval */
  if (stream->stream.writable)
    {
      audio_buf_info info = { 0, };

      if (ioctl (fd, SNDCTL_DSP_GETOSPACE, &info) < 0)
	return CSL_EGETCAPS;
    }
  if (stream->stream.readable)
    {
      audio_buf_info info = { 0, };

      if (ioctl (fd, SNDCTL_DSP_GETISPACE, &info) < 0)
	return CSL_EGETCAPS;
    }

  if (csl_debug (PCM))
    csl_message ("OSS-SETUP(%s): w=%d r=%d n_channels=%d rate=%d bufsz=%d bufavail=%d",
		 stream->device_name,
		 stream->stream.writable,
		 stream->stream.readable,
		 stream->stream.n_channels,
		 stream->stream.rate,
		 stream->stream.buffer_size,
		 stream->stream.n_bytes_available);

  return CSL_ENONE;
}
