/* CSL - Common Sound Layer
 * Copyright (C) 2000-2001 Stefan Westerfeld and Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General
 * Public License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "artscsladapter.h"
#include "artsmcopmethods.h"
#include "cslutils.h"
#include "artsbinbuffer.h"
#include "artsmcopdispatcher.h"

#include <stdio.h>

struct _ArtsCslAdapter
{
  ArtsMcopObject object;
  
  int method_open;
  int method_set_params;
  int method_set_title;
  int method_write;
  int method_read;
  int method_activate;
  int method_wait_buffer_level;
  int method_close;
};
/* HACK! */
#define	REQUEST_RESULT(dispatcher, request_id)	(&(dispatcher)->requests[(request_id)].result)

ArtsCslAdapter *_arts_csl_adapter_from_reference (ArtsMcopDispatcher	*dispatcher,
						  ArtsMcopObjectReference *reference,
						  CslBool		 need_copy)
{
  ArtsCslAdapter *result
    = (ArtsCslAdapter *)csl_malloc0 (sizeof (ArtsCslAdapter));
  
  result->object.dispatcher = dispatcher;
  result->object.object_id = reference->object_id;
  
  if (need_copy)
    _arts_mcop_object_copy_remote(&result->object);
  _arts_mcop_object_use_remote(&result->object);
  return result;
}

int
_arts_csl_adapter_open (ArtsCslAdapter *adapter,
			const char *role,
			ArtsCslAdapterDirection dir)
{
  ArtsBinBuffer buffer;
  unsigned int request_id;
  int result;
  
  csl_return_val_if_fail (adapter != NULL, 0);
  csl_return_val_if_fail (role != NULL, 0);
  
  if (!adapter->method_open)
    {
      adapter->method_open = _arts_mcop_dispatcher_lookup_hex_method (adapter->object.dispatcher,
								      adapter->object.object_id, CSL_CSL_Adapter_open);
    }
  
  /* method header and return type */
  request_id = _arts_mcop_dispatcher_setup_invocation (adapter->object.dispatcher, &buffer,
						       adapter->object.object_id,
						       adapter->method_open,
						       ARTS_MCOP_RESULT_LONG);
  /* method args */
  _arts_bin_buffer_put_string (&buffer, role);
  _arts_bin_buffer_put_int (&buffer, (int)dir);
  /* do invocation, fetch args */
  if (_arts_mcop_dispatcher_invoke (adapter->object.dispatcher, &buffer, request_id))
    result = REQUEST_RESULT (adapter->object.dispatcher, request_id)->v_long;
  else
    result = 0;
  /* free request slot */
  _arts_mcop_dispatcher_free_request (adapter->object.dispatcher, request_id);
  
  return result;
}

static void
_arts_csl_adapter_stream_params_marshal (ArtsBinBuffer           *buffer,
					 ArtsCslAdapterStreamParams *params)
{
  _arts_bin_buffer_put_int(buffer, params->rate);
  _arts_bin_buffer_put_int(buffer, params->channels);
  _arts_bin_buffer_put_int(buffer, (int)params->format);
}

CslBool
_arts_csl_adapter_set_params	(ArtsCslAdapter *adapter,
				 int streamID, 
				 ArtsCslAdapterStreamParams *params)
{
  ArtsBinBuffer buffer;
  unsigned int request_id;
  CslBool result;
  
  csl_return_val_if_fail (adapter != NULL, FALSE);
  csl_return_val_if_fail (params != NULL, FALSE);
  
  if (!adapter->method_set_params)
    {
      adapter->method_set_params = _arts_mcop_dispatcher_lookup_hex_method (adapter->object.dispatcher,
									    adapter->object.object_id, CSL_CSL_Adapter_setParams);
    }
  
  /* method header and return type */
  request_id = _arts_mcop_dispatcher_setup_invocation (adapter->object.dispatcher, &buffer,
						       adapter->object.object_id,
						       adapter->method_set_params,
						       ARTS_MCOP_RESULT_BOOLEAN);
  /* method args */
  _arts_bin_buffer_put_int (&buffer, streamID);
  _arts_csl_adapter_stream_params_marshal (&buffer, params);
  
  /* do invocation, fetch args */
  if (_arts_mcop_dispatcher_invoke (adapter->object.dispatcher, &buffer, request_id))
    result = REQUEST_RESULT (adapter->object.dispatcher, request_id)->v_boolean;
  else
    result = FALSE;
  /* free request slot */
  _arts_mcop_dispatcher_free_request (adapter->object.dispatcher, request_id);
  
  return result;
}

void
_arts_csl_adapter_set_title (ArtsCslAdapter *adapter,
			     int             streamID,
			     const char     *title)
{
  ArtsBinBuffer buffer;
  unsigned int request_id;
  
  csl_return_if_fail (adapter != NULL);
  csl_return_if_fail (title != NULL);
  
  if (!adapter->method_set_title)
    {
      adapter->method_set_title = _arts_mcop_dispatcher_lookup_hex_method (adapter->object.dispatcher,
									   adapter->object.object_id,
									   CSL_CSL_Adapter_setTitle);
    }
  
  /* method header and return type */
  request_id = _arts_mcop_dispatcher_setup_invocation (adapter->object.dispatcher,
						       &buffer,
						       adapter->object.object_id,
						       adapter->method_set_title,
						       ARTS_MCOP_RESULT_VOID);
  /* method args */
  _arts_bin_buffer_put_int (&buffer, streamID);
  _arts_bin_buffer_put_string (&buffer, title);
  
  /* do invocation, fetch args */
  _arts_mcop_dispatcher_invoke (adapter->object.dispatcher, &buffer, request_id);
  /* free request slot */
  _arts_mcop_dispatcher_free_request (adapter->object.dispatcher, request_id);
}

void
_arts_csl_adapter_write (ArtsCslAdapter *adapter,
			 int streamID,
			 unsigned char *bytes,
			 unsigned int n_bytes)
{
  ArtsBinBuffer buffer;
  
  csl_return_if_fail (adapter != NULL);
  csl_return_if_fail (bytes != NULL);
  
  if (!adapter->method_write)
    {
      adapter->method_write = _arts_mcop_dispatcher_lookup_hex_method (adapter->object.dispatcher,
								       adapter->object.object_id, CSL_CSL_Adapter_write);
    }
  
  /* method header and return type */
  _arts_mcop_dispatcher_setup_oneway_invocation (adapter->object.dispatcher, &buffer,
						 adapter->object.object_id,
						 adapter->method_write);
  /* method args */
  _arts_bin_buffer_put_int (&buffer, streamID);
  _arts_bin_buffer_put_bytes (&buffer, n_bytes, bytes);
  
  /* do invocation, fetch args */
  _arts_mcop_dispatcher_oneway_invoke (adapter->object.dispatcher, &buffer);
}

/* this function has a bit strange "language binding" (i.e. no IDL compiler
 * could possibly generate it like that) but since we're handwriting all that,
 * it's okay
 */
unsigned int
_arts_csl_adapter_read (ArtsCslAdapter *adapter,
		    	int streamID,
		    	unsigned int n_bytes,
		    	unsigned char *bytes)
{
  ArtsBinBuffer buffer;
  unsigned int request_id;
  unsigned int result;
  
  csl_return_val_if_fail (adapter != NULL, 0);
  csl_return_val_if_fail (bytes != NULL, 0);
  
  if (!adapter->method_read)
    {
      adapter->method_read = _arts_mcop_dispatcher_lookup_hex_method (adapter->object.dispatcher,
			      					      adapter->object.object_id,
								      CSL_CSL_Adapter_read);
    }
  
  /* method header and return type */
  request_id = _arts_mcop_dispatcher_setup_invocation (adapter->object.dispatcher, &buffer,
						       adapter->object.object_id,
						       adapter->method_read,
						       ARTS_MCOP_RESULT_BYTE_SEQ);

  /* method args */
  _arts_bin_buffer_put_int (&buffer, streamID);
  _arts_bin_buffer_put_int (&buffer, n_bytes);
  
  /* do invocation, fetch args */
  if (_arts_mcop_dispatcher_invoke (adapter->object.dispatcher, &buffer, request_id))
    {
      unsigned char *read_bytes
	= REQUEST_RESULT (adapter->object.dispatcher, request_id)->v_byte_seq.bytes;
      result = REQUEST_RESULT (adapter->object.dispatcher, request_id)->v_byte_seq.n_bytes;

      memcpy (bytes, read_bytes, result);
    }
  else
    {
      result = 0;
    }

  /* free request slot */
  _arts_mcop_dispatcher_free_request (adapter->object.dispatcher, request_id);

  return result;
}

void
_arts_csl_adapter_activate (ArtsCslAdapter *adapter,
			    int streamID)
{
  ArtsBinBuffer buffer;
  unsigned int request_id;
  
  csl_return_if_fail (adapter != NULL);
  
  if (!adapter->method_activate)
    {
      adapter->method_activate = _arts_mcop_dispatcher_lookup_hex_method (adapter->object.dispatcher,
									  adapter->object.object_id, CSL_CSL_Adapter_activate);
    }
  
  /* method header and return type */
  request_id = _arts_mcop_dispatcher_setup_invocation (adapter->object.dispatcher, &buffer,
						       adapter->object.object_id,
						       adapter->method_activate,
						       ARTS_MCOP_RESULT_VOID);
  /* method args */
  _arts_bin_buffer_put_int (&buffer, streamID);
  
  /* do invocation, fetch args */
  _arts_mcop_dispatcher_invoke (adapter->object.dispatcher, &buffer, request_id);
  
  /* free request slot -- necessary? */
  _arts_mcop_dispatcher_free_request (adapter->object.dispatcher, request_id);
}

int _arts_csl_adapter_wait_buffer_level (ArtsCslAdapter *adapter,
					 int streamID,
					 int bufferUsed)
{
  ArtsBinBuffer buffer;
  unsigned int request_id;
  int result;
  
  csl_return_val_if_fail (adapter != NULL, 0);
  
  if (!adapter->method_wait_buffer_level)
    {
      adapter->method_wait_buffer_level = _arts_mcop_dispatcher_lookup_hex_method (adapter->object.dispatcher,
										   adapter->object.object_id, CSL_CSL_Adapter_waitBufferLevel);
    }
  
  /* method header and return type */
  request_id = _arts_mcop_dispatcher_setup_invocation (adapter->object.dispatcher, &buffer,
						       adapter->object.object_id,
						       adapter->method_wait_buffer_level,
						       ARTS_MCOP_RESULT_LONG);
  /* method args */
  _arts_bin_buffer_put_int (&buffer, streamID);
  _arts_bin_buffer_put_int (&buffer, bufferUsed);
  
  /* do invocation, fetch args */
  if (_arts_mcop_dispatcher_invoke (adapter->object.dispatcher, &buffer, request_id))
    result = REQUEST_RESULT (adapter->object.dispatcher, request_id)->v_long;
  else
    result = 0;
  /* free request slot */
  _arts_mcop_dispatcher_free_request (adapter->object.dispatcher, request_id);
  
  return result;
}

void
_arts_csl_adapter_close (ArtsCslAdapter *adapter,
			 int streamID)
{
  ArtsBinBuffer buffer;
  unsigned int request_id;
  
  csl_return_if_fail (adapter != NULL);
  
  if (!adapter->method_close)
    {
      adapter->method_close = _arts_mcop_dispatcher_lookup_hex_method (adapter->object.dispatcher,
								       adapter->object.object_id, CSL_CSL_Adapter_close);
    }
  
  /* method header and return type */
  request_id = _arts_mcop_dispatcher_setup_invocation (adapter->object.dispatcher, &buffer,
						       adapter->object.object_id,
						       adapter->method_close,
						       ARTS_MCOP_RESULT_VOID);
  /* method args */
  _arts_bin_buffer_put_int (&buffer, streamID);
  
  /* do invocation, fetch args */
  _arts_mcop_dispatcher_invoke (adapter->object.dispatcher, &buffer, request_id);
  
  /* free request slot -- necessary? */
  _arts_mcop_dispatcher_free_request (adapter->object.dispatcher, request_id);
}

/* vim:ts=8:sw=2:sts=2
 */
