/* BSE - Bedevilled Sound Engine
 * Copyright (C) 1998 Olaf Hoehmann and Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */
#include	"loadwav.h"
#include	"bsevalueblock.h"
#include	<sys/types.h>
#include	<sys/stat.h>
#include	<fcntl.h>
#include	<unistd.h>




/* load routine for Microsofts WAV format sample files
 * ref.: C't 01/1993 pp. 213
 */


/* --- structures --- */
typedef	struct
{
  gchar	main_chunk[5];
  guint	length;
  gchar	chunk_type[5];
} WavHeader;

typedef	struct
{
  gchar sub_chunk[5];
  guint length;
  guint is_pcm;
  guint n_channels;
  guint sample_freq;
  guint byte_per_second;
  guint byte_per_sample;
  guint bit_per_sample;
} FmtHeader;

typedef	struct
{
  gchar data_chunk[5];
  guint length;
} DataHeader;


/* --- prototypes --- */
static	guint		read_header	(int		fd,
					 WavHeader	*header);
static	guint		read_fmt_header	(int		fd,
					 FmtHeader	*header);
static	guint		read_data_header(int		fd,
					 DataHeader	*header,
					 guint		byte_width);
static	guint		read_data	(int		fd,
					 FmtHeader	*header,
					 guint		n_values,
					 BseSampleValue	*values,
					 guint		byte_length);


/* --- variables --- */


/* --- functions --- */
guint
bse_load_wav (const gchar	*file_name,
	      BseSample		**sample_p)
{
  register guint error = 0;
  register int fd;
  static WavHeader wav_header;
  static FmtHeader fmt_header;
  register guint data_width;
  static DataHeader data_header;
  register BseValueBlock *block;
  

  g_return_val_if_fail (file_name != NULL, BSE_ERROR_INTERNAL);
  g_return_val_if_fail (sample_p != NULL, BSE_ERROR_INTERNAL);

  fd = open (file_name, O_RDONLY);
  if (fd < 0)
    return BSE_ERROR_FILE_IO;

  error = read_header (fd, &wav_header);
  if (error)
  {
    close (fd);
    return error;
  }

  error = read_fmt_header (fd, &fmt_header);
  if (error)
  {
    close (fd);
    return error;
  }
  if (fmt_header.bit_per_sample != 8 &&
      fmt_header.bit_per_sample != 12 &&
      fmt_header.bit_per_sample != 16)
  {
    close (fd);
    return BSE_ERROR_FORMAT_UNSUPPORTED;
  }
  printf ("wav: n_channels: %d sample_freq: %d\n",
   fmt_header.n_channels,
   fmt_header.sample_freq);

  data_width = (fmt_header.bit_per_sample + 7) / 8;
  error = read_data_header (fd, &data_header, data_width * fmt_header.n_channels);
  if (error)
  {
    close (fd);
    return error;
  }
  block = bse_value_block_alloc (data_header.length / data_width);
  error = read_data (fd, &fmt_header, block->n_values, block->values, data_header.length);
  close (fd);
  if (error)
  {
    bse_value_block_free (block);
    return error;
  }

  block = bse_value_block_setup (block, file_name, 0, 0);
  if (!block)
    return BSE_ERROR_INTERNAL;

  *sample_p = bse_sample_new (BSE_SAMPLE_NOTE_MUNKS, file_name, file_name);
  bse_sample_set_params (*sample_p, fmt_header.n_channels, fmt_header.sample_freq);
  bse_sample_set_munk (*sample_p,
		       0, BSE_KAMMER_NOTE /* FIXME */,
		       block, 0, block->n_values / fmt_header.n_channels,
		       0, 0);
  bse_sample_fillup_munks (*sample_p);
  
  return BSE_ERROR_NONE;
}

static guint
read_header (int    fd,
	     WavHeader *header)
{
  static guchar bytes[8];

  header->main_chunk[4] = 0;
  if (read (fd, header->main_chunk, 4) != 4)
    return BSE_ERROR_FILE_TOO_SHORT;
  if (read (fd, bytes, 4) != 4)
    return BSE_ERROR_FILE_TOO_SHORT;
  header->length = (bytes[3] << 24 ) + (bytes[2] << 16 ) + (bytes[1] << 8 ) + bytes[0];
  header->chunk_type[4] = 0;
  if (read (fd, header->chunk_type, 4) != 4)
    return BSE_ERROR_FILE_TOO_SHORT;

  if (g_str_equal (header->main_chunk, "RIFF") &&
      header->length >= 40 &&
      g_str_equal (header->chunk_type, "WAVE"))
    return BSE_ERROR_NONE;

  return BSE_ERROR_HEADER_CORRUPT;
}

static guint
read_fmt_header	(int		fd,
		 FmtHeader	*header)
{
  static guchar bytes[8];

  header->sub_chunk[4] = 0;
  if (read (fd, header->sub_chunk, 4) != 4)
    return BSE_ERROR_FILE_TOO_SHORT;
  if (read (fd, bytes, 4) != 4)
    return BSE_ERROR_FILE_TOO_SHORT;
  header->length = (bytes[3] << 24 ) + (bytes[2] << 16 ) + (bytes[1] << 8 ) + bytes[0];
  if (read (fd, bytes, 2) != 2)
    return BSE_ERROR_FILE_TOO_SHORT;
  header->is_pcm = (bytes[1] << 8 ) + bytes[0];
  if (read (fd, bytes, 2) != 2)
    return BSE_ERROR_FILE_TOO_SHORT;
  header->n_channels = (bytes[1] << 8 ) + bytes[0];
  if (read (fd, bytes, 4) != 4)
    return BSE_ERROR_FILE_TOO_SHORT;
  header->sample_freq = (bytes[3] << 24 ) + (bytes[2] << 16 ) + (bytes[1] << 8 ) + bytes[0];
  if (read (fd, bytes, 4) != 4)
    return BSE_ERROR_FILE_TOO_SHORT;
  header->byte_per_second = (bytes[3] << 24 ) + (bytes[2] << 16 ) + (bytes[1] << 8 ) + bytes[0];
  if (read (fd, bytes, 2) != 2)
    return BSE_ERROR_FILE_TOO_SHORT;
  header->byte_per_sample = (bytes[1] << 8 ) + bytes[0];
  if (read (fd, bytes, 2) != 2)
    return BSE_ERROR_FILE_TOO_SHORT;
  header->bit_per_sample = (bytes[1] << 8 ) + bytes[0];

  if (header->is_pcm != 1 ||
      header->n_channels > BSE_MAX_SAMPLE_CHANNELS ||
      header->sample_freq < BSE_MIN_MIX_FREQ ||
      header->sample_freq > BSE_MAX_MIX_FREQ)
    return BSE_ERROR_FORMAT_UNSUPPORTED;

  if (g_str_equal (header->sub_chunk, "fmt ") &&
      header->length >= 16 &&
      header->n_channels > 0 &&
      header->byte_per_second == header->sample_freq * header->byte_per_sample &&
      header->byte_per_sample == (header->bit_per_sample + 7) / 8 * header->n_channels)
    return BSE_ERROR_NONE;

  return BSE_ERROR_SUB_HEADER_CORRUPT;
}

static guint
read_data_header (int		fd,
		  DataHeader	*header,
		  guint		byte_width)
{
  static guchar bytes[8];

  header->data_chunk[4] = 0;
  if (read (fd, header->data_chunk, 4) != 4)
    return BSE_ERROR_FILE_TOO_SHORT;
  if (read (fd, bytes, 4) != 4)
    return BSE_ERROR_FILE_TOO_SHORT;
  header->length = (bytes[3] << 24 ) + (bytes[2] << 16 ) + (bytes[1] << 8 ) + bytes[0];

  if (g_str_equal (header->data_chunk, "data") &&
      header->length % byte_width == 0)
    return BSE_ERROR_NONE;

  return BSE_ERROR_SUB_HEADER_CORRUPT;
}

static guint
read_data (int		  fd,
	   FmtHeader	  *header,
	   guint	  n_values,
	   BseSampleValue *values,
	   guint	  byte_length)
{
  register gchar *bytes;
  register gchar *int_8p;
  register BseSampleValue *int_16p;
  register guint i;

  g_assert (byte_length == (header->bit_per_sample + 7) / 8 * n_values);

  bytes = g_new0 (gchar, byte_length);

  if (read (fd, bytes, byte_length) != byte_length)
  {
    g_free (bytes);
    return BSE_ERROR_FILE_TOO_SHORT;
  }
  /*
    if (read (fd, &tbyte, 1) != 0)
    {
    g_free (bytes);
    return BSE_ERROR_FILE_TOO_LONG;
    }
    */

  int_8p = bytes;
  int_16p = (gpointer) bytes;
  if (header->bit_per_sample == 8)
    for (i = 0; i < n_values; i++)
    {
      values[i] = *int_8p << 8;
      int_8p++;
    }
  else if (header->bit_per_sample == 12)
    for (i = 0; i < n_values; i++)
    {
      values[i] = *int_16p << 4;
      int_16p++;
    }
  else if (header->bit_per_sample == 16)
    for (i = 0; i < n_values; i++)
    {
      values[i] = *int_16p;
      int_16p++;
    }
  else
  {
    g_free (bytes);
    return BSE_ERROR_FORMAT_UNSUPPORTED;
  }

  g_free (bytes);

  return BSE_ERROR_NONE;
}
