/* BSE - Bedevilled Sound Engine
 * Copyright (C) 1998 Olaf Hoehmann and Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */
#include	"bseutils.h"
#include	"bsebase.h"
#include	"bsemem.h"
#include	<stdlib.h>
#include	<stdio.h>


/* --- variables --- */
static	struct { gchar *s; guint v; } bse_note_table[] =
{
  { "ces",	BSE_KAMMER_NOTE - 10 - BSE_KAMMER_OCTAVE * 12 },
  { "cis",	BSE_KAMMER_NOTE -  8 - BSE_KAMMER_OCTAVE * 12 },
  { "c",	BSE_KAMMER_NOTE -  9 - BSE_KAMMER_OCTAVE * 12 },
  { "des",	BSE_KAMMER_NOTE -  8 - BSE_KAMMER_OCTAVE * 12 },
  { "dis",	BSE_KAMMER_NOTE -  6 - BSE_KAMMER_OCTAVE * 12 },
  { "d",	BSE_KAMMER_NOTE -  7 - BSE_KAMMER_OCTAVE * 12 },
  { "es",	BSE_KAMMER_NOTE -  6 - BSE_KAMMER_OCTAVE * 12 },
  { "eis",	BSE_KAMMER_NOTE -  4 - BSE_KAMMER_OCTAVE * 12 },
  { "e",	BSE_KAMMER_NOTE -  5 - BSE_KAMMER_OCTAVE * 12 },
  { "fes",	BSE_KAMMER_NOTE -  5 - BSE_KAMMER_OCTAVE * 12 },
  { "fis",	BSE_KAMMER_NOTE -  3 - BSE_KAMMER_OCTAVE * 12 },
  { "f",	BSE_KAMMER_NOTE -  4 - BSE_KAMMER_OCTAVE * 12 },
  { "ges",	BSE_KAMMER_NOTE -  3 - BSE_KAMMER_OCTAVE * 12 },
  { "gis",	BSE_KAMMER_NOTE -  1 - BSE_KAMMER_OCTAVE * 12 },
  { "g",	BSE_KAMMER_NOTE -  2 - BSE_KAMMER_OCTAVE * 12 },
  { "as",	BSE_KAMMER_NOTE -  1 - BSE_KAMMER_OCTAVE * 12 },
  { "ais",	BSE_KAMMER_NOTE +  1 - BSE_KAMMER_OCTAVE * 12 },
  { "a",	BSE_KAMMER_NOTE	     - BSE_KAMMER_OCTAVE * 12 },
  { "bes",	BSE_KAMMER_NOTE +  1 - BSE_KAMMER_OCTAVE * 12 },
  { "bis",	BSE_KAMMER_NOTE +  3 - BSE_KAMMER_OCTAVE * 12 },
  { "b",	BSE_KAMMER_NOTE +  2 - BSE_KAMMER_OCTAVE * 12 },
};
static	guint	bse_note_table_length = sizeof (bse_note_table) / sizeof (bse_note_table[0]);
static	gchar	*bse_note_name_table[12] =
{
  "c", "cis", "d", "dis", "e", "f",
  "fis", "g", "gis", "a", "ais", "b",
};
static guint		 key_blurb_id = 0;


/* --- functions --- */
gchar*
bse_sample_name_make_valid (gchar *string)
{
  return string;
}

gchar*
bse_song_name_make_valid (gchar *string)
{
  return string;
}

guint
bse_string_2_note (const gchar	  *note_string)
{
  register gchar *string;
  register guint note;
  register gboolean fit;
  register guint i;
  
  g_return_val_if_fail (note_string != NULL, BSE_NOTE_VOID);
  
  string = g_strdup (note_string);
  g_strdown (string);
  
  note = BSE_NOTE_VOID;
  
  if (string[0] == 'v' &&
      string[1] == 'o' &&
      string[2] == 'i' &&
      string[3] == 'd' &&
      string[4] == 0)
    return note;
  
  note = BSE_NOTE_UNPARSABLE;
  
  fit = FALSE;
  for (i = 0; i < bse_note_table_length && !fit; i++)
    {
      register guint p;
      
      p = 0;
      do
	fit = bse_note_table[i].s[p] == string[p];
      while (bse_note_table[i].s[++p] && fit);
    }
  g_assert (i > 0);
  i--;
  
  if (fit)
    {
      gchar *s;
      register gint o;
      
      note = bse_note_table[i].v;
      if ( *(string + strlen (bse_note_table[i].s)))
	{
	  o = strtol (string + strlen (bse_note_table[i].s), &s, 10);
	  if (s && *s)
	    note = BSE_NOTE_VOID;
	}
      else
	o = 0;
      
      if (note != BSE_NOTE_VOID)
	note = CLAMP ((gint) bse_note_table[i].v + o * 12, BSE_MIN_NOTE, BSE_MAX_NOTE);
    }
  
  g_free (string);
  
  return note;
}

gchar*
bse_note_2_string (guint note)
{
  guint ht;
  gint o;
  static gchar string[64];
  
  g_return_val_if_fail (note >= BSE_MIN_NOTE && note <= BSE_MAX_NOTE, NULL);
  
  bse_note_examine (note, &o, &ht, NULL, NULL);
  
  if (o)
    sprintf (string, "%s%+d", bse_note_name_table[ht], o);
  else
    strcpy (string, bse_note_name_table[ht]);
  
  return g_strdup (string);
}

void
bse_note_examine (guint note,
		  gint *octave_p,
		  guint *half_tone_p,
		  gboolean *ht_up_p,
		  gchar	*letter_p)
{
  register guint half_tone;
  register gint octave;
  gboolean ht_flags[12] = { 0, 1, 0, 1, 0, 0, 1, 0, 1, 0, 1, 0 };
  
  g_return_if_fail (note >= BSE_MIN_NOTE && note <= BSE_MAX_NOTE);
  
  half_tone = note % 12 + (9 - (BSE_KAMMER_NOTE % 12));
  
  note -= half_tone;
  octave = (gint) note - (BSE_KAMMER_NOTE - 9);
  octave = octave / 12 + BSE_KAMMER_OCTAVE;
  
  if (octave_p)
    *octave_p = octave;
  if (half_tone_p)
    *half_tone_p = half_tone;
  if (ht_up_p)
    *ht_up_p = ht_flags[half_tone];
  if (letter_p)
    *letter_p = bse_note_name_table[half_tone][0];
}

void
_bse_struct_set_string (gpointer     bse_struct,
			const gchar *key,
			const gchar *string)
{
  g_return_if_fail (bse_struct != NULL);
  g_return_if_fail (key != NULL);
  
  g_dataset_set_data_full (bse_struct, key, g_strdup (string), g_free);
}

gchar*
_bse_struct_get_string (gpointer    *bse_struct,
			const gchar *key)
{
  g_return_val_if_fail (bse_struct != NULL, NULL);
  g_return_val_if_fail (key != NULL, NULL);
  
  return g_dataset_get_data (bse_struct, key);
}

void
_bse_struct_set_long (gpointer	   bse_struct,
		      const gchar *key,
		      glong	   v_long)
{
  glong *l;
  
  g_return_if_fail (bse_struct != NULL);
  g_return_if_fail (key != NULL);
  
  l = g_new (glong, 1);
  *l = v_long;
  
  g_dataset_set_data_full (bse_struct, key, l, g_free);
}

glong
_bse_struct_get_long (gpointer	  *bse_struct,
		      const gchar *key)
{
  glong *l;
  
  g_return_val_if_fail (bse_struct != NULL, 0);
  g_return_val_if_fail (key != NULL, 0);
  
  l = g_dataset_get_data (bse_struct, key);
  if (l)
    return *l;
  else
    return 0;
}

void
_bse_struct_set_blurb (gpointer	     bse_struct,
		       const gchar *blurb)
{
  g_return_if_fail (bse_struct != NULL);
  
  if (!key_blurb_id)
    key_blurb_id = g_dataset_force_id ("bse-blurb");
  
  g_dataset_id_set_data_full (bse_struct, key_blurb_id, g_strdup (blurb), g_free);
}

gchar*
_bse_struct_get_blurb (gpointer *bse_struct)
{
  g_return_val_if_fail (bse_struct != NULL, NULL);
  
  return g_dataset_id_get_data (bse_struct, key_blurb_id);
}

void
bse_struct_delete_cbs (BseCallbackNode *nodes)
{
  g_return_if_fail (nodes != NULL);
  
  while (nodes)
    {
      BseCallbackNode *to_free;
      
      to_free = nodes;
      nodes = nodes->next;
      
      bse_callback_node_chunk_free (to_free);
    }
}

void
bse_struct_add_cb (gpointer	  bse_struct,
		   guint	  key_id,
		   BseGenericCB	  callback,
		   gpointer	  callback_data)
{
  BseCallbackNode *new_node, *node;
  
  g_return_if_fail (bse_struct != NULL);
  g_return_if_fail (callback != NULL);
  
  new_node = bse_callback_node_chunk_new0 ();
  new_node->callback = callback;
  new_node->callback_data = callback_data;
  new_node->next = NULL;
  
  node = g_dataset_id_get_data (bse_struct, key_id);
  if (node)
    {
      while (node->next)
	node = node->next;
      node->next = new_node;
    }
  else
    g_dataset_id_set_data_full (bse_struct,
				key_id,
				new_node,
				(GDestroyNotify) bse_struct_delete_cbs);
}

void
bse_struct_remove_cb (gpointer	     bse_struct,
		      guint	     key_id,
		      BseGenericCB   callback,
		      gpointer	     callback_data)
{
  BseCallbackNode *node;
  BseCallbackNode *prev;
  
  g_return_if_fail (bse_struct != NULL);
  g_return_if_fail (callback != NULL);
  
  node = g_dataset_id_get_data (bse_struct, key_id);
  prev = NULL;
  while (node)
    {
      if (node->callback == callback &&
	  node->callback_data == callback_data)
	{
	  if (prev)
	    prev->next = node->next;
	  else
	    {
	      g_dataset_id_set_destroy (bse_struct, key_id, NULL);
	      g_dataset_id_set_data_full (bse_struct,
					  key_id,
					  node->next,
					  (GDestroyNotify) bse_struct_delete_cbs);
	    }
	  
	  bse_callback_node_chunk_free (node);
	  break;
	}
      prev = node;
      node = prev->next;
    }
}

BseCallbackNode*
bse_struct_detach_cbs (gpointer	      bse_struct,
		       guint	      key_id)
{
  BseCallbackNode *nodes;
  
  g_return_val_if_fail (bse_struct != NULL, NULL);
  
  nodes = g_dataset_id_get_data (bse_struct, key_id);
  if (nodes)
    {
      g_dataset_id_set_destroy (bse_struct, key_id, NULL);
      g_dataset_id_remove_data (bse_struct, key_id);
    }
  
  return nodes;
}

void
bse_struct_attach_cbs (gpointer		bse_struct,
		       guint		key_id,
		       BseCallbackNode *nodes)
{
  g_return_if_fail (bse_struct != NULL);
  
  if (nodes)
    {
      BseCallbackNode *node;
      
      node = g_dataset_id_get_data (bse_struct, key_id);
      
      if (node)
	{
	  while (node->next)
	    node = node->next;
	  node->next = nodes;
	}
      else
	g_dataset_id_set_data_full (bse_struct,
				    key_id,
				    nodes,
				    (GDestroyNotify) bse_struct_delete_cbs);
    }
}

void
bse_struct_remove_cbs (gpointer	      bse_struct,
		       guint	      key_id)
{
  g_return_if_fail (bse_struct != NULL);
  
  g_dataset_id_remove_data (bse_struct, key_id);
}

/* our special glib additions */
#include	"glib_extra.c"
