/* BSE - Bedevilled Sound Engine
 * Copyright (C) 1998 Olaf Hoehmann and Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */
#include	"bsepattern.h"
#include	"bsesong.h"
#include	"bsemem.h"
#include	"bseprivate.h"



/* --- functions --- */
BsePattern*
bse_pattern_new (BseSong	*song)
{
  BsePattern *pattern;
  guint i;
  
  g_return_val_if_fail (song != NULL, NULL);
  
  pattern = bse_pattern_chunk_new0 ();
  
  pattern->song = song;
  pattern->name = g_strdup ("");
  pattern->blurb = g_strdup ("");
  pattern->n_channels = song->n_channels;
  pattern->n_rows = song->pattern_length;
  
  pattern->notes = g_new (BseNote, pattern->n_channels * pattern->n_rows);
  for (i = 0; i < pattern->n_channels * pattern->n_rows; i++)
    {
      pattern->notes[i].note = BSE_NOTE_VOID;
      pattern->notes[i].instrument = NULL;
      pattern->notes[i].next_effect = NULL;
    }
  
  bse_song_pattern_changed (pattern->song, pattern);
  
  return pattern;
}

void
bse_pattern_delete (BsePattern	   *pattern)
{
  g_return_if_fail (pattern != NULL);
  
  g_dataset_destroy (pattern);
  
  g_free (pattern->name);
  g_free (pattern->blurb);
  
  g_free (pattern->notes);
  bse_pattern_chunk_free (pattern);
}

guint
bse_pattern_get_guid (BsePattern     *pattern)
{
  gint i;
  
  g_return_val_if_fail (pattern != NULL, 0);
  
  i = g_list_index (pattern->song->patterns, pattern);
  if (i < 0)
    return 0;
  else
    return i + 1;
}

void
bse_pattern_set_note (BsePattern *pattern,
		      guint	 channel,
		      guint	 row,
		      guint	 note,
		      BseInstrument *instrument)
{
  BseNote *bnote;
  
  g_return_if_fail (pattern != NULL);
  g_return_if_fail (channel < pattern->n_channels);
  g_return_if_fail (row < pattern->n_rows);
  
  bnote = bse_pattern_get_note (pattern, channel, row);
  if (bnote->note != note ||
      bnote->instrument != instrument)
    {
      bnote->note = note;
      bnote->instrument = instrument;
      
      bse_song_note_changed (pattern->song,
			     pattern,
			     channel,
			     row);
    }
}

BseNote*
bse_pattern_get_note (BsePattern     *pattern,
		      guint	     channel,
		      guint	     row)
{
  BseNote *note;
  
  g_return_val_if_fail (pattern != NULL, NULL);
  g_return_val_if_fail (channel < pattern->n_channels, NULL);
  g_return_val_if_fail (row < pattern->n_rows, NULL);
  
  note = pattern->notes + row * pattern->n_channels + channel;
  
  return note;
}

void
bse_pattern_set_name (BsePattern	 *pattern,
		      const gchar	 *name)
{
  g_return_if_fail (pattern != NULL);
  g_return_if_fail (name != NULL);
  
  if (!name)
    name = "";

  g_free (pattern->name);
  pattern->name = g_strdup (name);
  
  bse_song_pattern_changed (pattern->song, pattern);
}

gchar*
bse_pattern_get_name (BsePattern	 *pattern)
{
  g_return_val_if_fail (pattern != NULL, NULL);
  
  return pattern->name;
}

void
bse_pattern_set_blurb (BsePattern	  *pattern,
		       const gchar	  *blurb)
{
  g_return_if_fail (pattern != NULL);
  g_return_if_fail (blurb != NULL);
  
  if (!blurb)
    blurb = "";

  g_free (pattern->blurb);
  pattern->blurb = g_strdup (blurb);
  
  bse_song_pattern_changed (pattern->song, pattern);
}

gchar*
bse_pattern_get_blurb (BsePattern	  *pattern)
{
  g_return_val_if_fail (pattern != NULL, NULL);
  
  return pattern->blurb;
}
