/* BSE - Bedevilled Sound Engine
 * Copyright (C) 1998 Olaf Hoehmann and Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */

/* we just get included from bseparser.c
 */


/* --- parser tables --- */
static	BseParserSymbol		song_song_symbols[] =
{
  { "blurb",			parse_string,		JOB_ID_SONG_BLURB },
  { "author",			parse_string,		JOB_ID_SONG_AUTHOR },
  { "copyright",		parse_string,		JOB_ID_SONG_COPYRIGHT },
  { "bse-version",		parse_string,		JOB_ID_SONG_BSE_VER },
  { "created",			parse_date,		JOB_ID_SONG_CREATED },
  { "modified",			parse_date,		JOB_ID_SONG_MODIFIED },
  { "bpm",			parse_number,		JOB_ID_SONG_BPM },
  { "volume",			parse_number,		JOB_ID_SONG_VOLUME },
  { "sample-instrument",	parse_instrument,	JOB_ID_NONE },
  { "pattern",			parse_pattern,		JOB_ID_NONE },
};
static	BseParserSymbol		song_instrument_symbols[] =
{
  { "blurb",			parse_string,		JOB_ID_INSTRUMENT_BLURB },
  { "sample-path",		parse_string,		JOB_ID_INSTRUMENT_DSAMPLE_PATH },
  { "name",			parse_string,		JOB_ID_INSTRUMENT_NAME },
  { "with-interpolation",	parse_flag,		JOB_ID_INSTRUMENT_INTERPOLATION_ON },
  { "without-interpolation",	parse_flag,		JOB_ID_INSTRUMENT_INTERPOLATION_OFF },
  { "with-polyphony",		parse_flag,		JOB_ID_INSTRUMENT_POLYPHONY_ON },
  { "without-polyphony",	parse_flag,		JOB_ID_INSTRUMENT_POLYPHONY_OFF },
  { "volume",			parse_number,		JOB_ID_INSTRUMENT_VOLUME },
  { "balance",			parse_number,		JOB_ID_INSTRUMENT_BALANCE },
  { "transpose",		parse_number,		JOB_ID_INSTRUMENT_TRANSPOSE },
  { "fine-tune",		parse_number,		JOB_ID_INSTRUMENT_FINE_TUNE },
  { "delay-time",		parse_number,		JOB_ID_INSTRUMENT_DELAY_TIME },
  { "attack-time",		parse_number,		JOB_ID_INSTRUMENT_ATTACK_TIME },
  { "attack-level",		parse_number,		JOB_ID_INSTRUMENT_ATTACK_LEVEL },
  { "decay-time",		parse_number,		JOB_ID_INSTRUMENT_DECAY_TIME },
  { "sustain-level",		parse_number,		JOB_ID_INSTRUMENT_SUSTAIN_LEVEL },
  { "sustain-time",		parse_number,		JOB_ID_INSTRUMENT_SUSTAIN_TIME },
  { "release-level",		parse_number,		JOB_ID_INSTRUMENT_RELEASE_LEVEL },
  { "release-time",		parse_number,		JOB_ID_INSTRUMENT_RELEASE_TIME },
};
static	BseParserSymbol		song_pattern_symbols[] =
{
  { "name",			parse_string,		JOB_ID_PATTERN_NAME },
  { "blurb",			parse_string,		JOB_ID_PATTERN_BLURB },
  { "skip",			parse_pattern_cntr,	JOB_ID_PATTERN_STEP_ROW },
  { "set-row",			parse_pattern_cntr,	JOB_ID_PATTERN_SET_ROW },
  { "set-channel",		parse_pattern_cntr,	JOB_ID_PATTERN_SET_CHANNEL },
  { "note",			parse_pattern_note,	JOB_ID_NONE },
};


/* --- functions --- */
static  guint
parse_pattern_cntr (GScanner           *scanner,
		    BseParserSymbol    *symbol,
		    guint               job_id,
		    gpointer            struct_data)
{
  BseSong *song;
  BsePattern *pattern;
  gint number = 0;
  
  g_return_val_if_fail (struct_data != NULL, G_TOKEN_ERROR);
  pattern = struct_data;
  song = pattern->song;

  if (job_id != JOB_ID_PATTERN_STEP_ROW)
    {
      g_scanner_get_next_token (scanner);
      if (scanner->token == G_TOKEN_INT)
	number = scanner->value.v_int;
      else
	return G_TOKEN_INT;
    }
  
  switch (job_id)
    {
    case  JOB_ID_PATTERN_STEP_ROW:
      PDATA (scanner)->pattern_cntr_row++;
      if (PDATA (scanner)->pattern_cntr_row >= pattern->n_rows)
	{
	  PDATA (scanner)->pattern_cntr_row = 0;
	  PDATA (scanner)->pattern_cntr_channel++;
	  if (PDATA (scanner)->pattern_cntr_channel >= pattern->n_channels)
	    {
	      PDATA (scanner)->pattern_cntr_channel--;
	      g_scanner_warn (scanner, "row stepping exceeds pattern bounds");
	    }
	}
      break;

    case  JOB_ID_PATTERN_SET_ROW:
      if (number < 1 || number > pattern->n_rows)
	{
	  g_scanner_warn (scanner, "row index exceeds pattern bounds");
	  number = number < 1 ? 1 : pattern->n_rows;
	}
      PDATA (scanner)->pattern_cntr_row = number - 1;
      break;

    case  JOB_ID_PATTERN_SET_CHANNEL:
      if (number < 1 || number > pattern->n_channels)
	{
	  g_scanner_warn (scanner, "channel index exceeds pattern bounds");
	  number = number < 1 ? 1 : pattern->n_channels;
	}
      PDATA (scanner)->pattern_cntr_channel = number - 1;
      break;

    default:
      g_assert_not_reached ();
    }

  g_scanner_get_next_token (scanner);
  if (scanner->token != ')')
    return ')';

  return G_TOKEN_NONE;
}

static	guint
parse_song (GScanner		*scanner,
	    BseParserSymbol	*symbol,
	    guint		job_id,
	    gpointer		struct_data)
{
  register BseSong	*song;
  register guint	expected_token;
  register gchar	*song_name;
  register guint	n_channels;
  register guint	pattern_length;
  
  g_scanner_get_next_token (scanner);
  if (scanner->token != G_TOKEN_STRING)
    return G_TOKEN_STRING;
  
  song_name = scanner->value.v_string;
  bse_song_name_make_valid (song_name);
  song_name = parser_check_song_name (scanner, song_name);

  if (!song_name)
    {
      parser_skip_statement (scanner, 1);
      
      return G_TOKEN_NONE;
    }
  
  g_scanner_get_next_token (scanner);
  if (scanner->token != G_TOKEN_INT)
    {
      g_free (song_name);

      return G_TOKEN_INT;
    }
  n_channels = scanner->value.v_int;

  g_scanner_get_next_token (scanner);
  if (scanner->token != G_TOKEN_INT)
    {
      g_free (song_name);

      return G_TOKEN_INT;
    }
  pattern_length = scanner->value.v_int;

  if (n_channels < BSE_MIN_CHANNELS || n_channels > BSE_MAX_CHANNELS)
    {
      g_scanner_warn (scanner, "number of channels out of bounds, defaulting to %u", BSE_DFL_N_CHANNELS);
      n_channels = BSE_DFL_N_CHANNELS;
    }

  if (pattern_length < BSE_MIN_PATTERN_LENGTH || pattern_length > BSE_MAX_PATTERN_LENGTH)
    {
      g_scanner_warn (scanner, "pattern length out of bounds, defaulting to %u", BSE_DFL_PATTERN_LENGTH);
      pattern_length = BSE_DFL_PATTERN_LENGTH;
    }

  song = bse_song_new (song_name, NULL, NULL, n_channels);
  g_free (song_name);
  song_name = NULL;
  bse_song_set_pattern_length (song, pattern_length);
  
  expected_token = parser_parse_rest (scanner, song_song_symbols, song);

  if (expected_token != G_TOKEN_ERROR)
    PDATA (scanner)->new_songs = g_slist_prepend (PDATA (scanner)->new_songs, song);
  else
    bse_song_unref (song);
  
  return expected_token;
}

static  guint
parse_instrument (GScanner           *scanner,
		  BseParserSymbol    *symbol,
		  guint               job_id,
		  gpointer            struct_data)
{
  guint          expected_token;
  BseInstrument *instrument;
  BseSong *song;
  BseSample *sample;

  g_return_val_if_fail (struct_data != NULL, G_TOKEN_ERROR);
  song = struct_data;
  
  g_scanner_get_next_token (scanner);
  if (scanner->token != G_TOKEN_STRING)
    return G_TOKEN_STRING;

  sample = bse_get_zero_sample ();
  instrument = bse_song_sample_instrument_new (song, sample);
  bse_sample_unref (sample);
  g_free (instrument->deferred_sample_name);
  instrument->deferred_sample_name = g_strdup (scanner->value.v_string);

  expected_token = parser_parse_rest (scanner, song_instrument_symbols, instrument);

  return expected_token;
}

static  guint
parse_pattern (GScanner           *scanner,
	       BseParserSymbol    *symbol,
	       guint               job_id,
	       gpointer            struct_data)
{
  register guint           expected_token;
  BsePattern *pattern;
  BseSong *song;

  g_return_val_if_fail (struct_data != NULL, G_TOKEN_ERROR);
  song = struct_data;

  PDATA (scanner)->pattern_cntr_channel = 0;
  PDATA (scanner)->pattern_cntr_row = 0;

  pattern = bse_song_add_pattern (song);

  expected_token = parser_parse_rest (scanner, song_pattern_symbols, pattern);

  return expected_token;
}

static  guint
parse_pattern_note (GScanner           *scanner,
		    BseParserSymbol    *symbol,
		    guint               job_id,
		    gpointer            struct_data)
{
  guint expected_token;
  BsePattern *pattern;
  guint note;
  BseInstrument *instrument;

  g_return_val_if_fail (struct_data != NULL, G_TOKEN_ERROR);

  pattern = struct_data;

  expected_token = parser_scan_note (scanner, &note, NULL);
  if (expected_token != G_TOKEN_NONE)
    return expected_token;

  /* optional instrument id
   */
  g_scanner_peek_next_token (scanner);
  if (scanner->next_token == G_TOKEN_INT)
    {
      g_scanner_get_next_token (scanner);

      if (scanner->value.v_int > 0)
	instrument = bse_song_get_instrument (pattern->song, scanner->value.v_int);
      else
	instrument = NULL;
    }
  else
    instrument = NULL;

  bse_pattern_set_note (pattern,
			PDATA (scanner)->pattern_cntr_channel,
			PDATA (scanner)->pattern_cntr_row,
			note,
			instrument);

  PDATA (scanner)->pattern_cntr_row += 1;
  if (PDATA (scanner)->pattern_cntr_row >= pattern->n_rows)
    {
      PDATA (scanner)->pattern_cntr_row = 0;
      PDATA (scanner)->pattern_cntr_channel += 1;

      if (PDATA (scanner)->pattern_cntr_channel >= pattern->n_channels)
	{
	  PDATA (scanner)->pattern_cntr_row = pattern->n_rows - 1;
	  PDATA (scanner)->pattern_cntr_channel = pattern->n_channels - 1;
	}
    }

  g_scanner_get_next_token (scanner);
  if (scanner->token != ')')
    return ')';

  return G_TOKEN_NONE;
}
