/* BEAST - Bedevilled Audio System
 * Copyright (C) 1998 Olaf Hoehmann and Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */
#include "bstshell.h"
#include "bstsongshell.h"
#include "gtkitemfactory.h"
#include "bstmessagebox.h"
#include "bstrcargs.h"
#include <gtk/gtksignal.h>
#include <gdk/gdkkeysyms.h>
#include <string.h>
#include <sys/stat.h>



/* --- signals --- */
enum
{
  LAST_SIGNAL
};


/* --- CList columns --- */
enum
{
  CLIST_COL_NAME,
  CLIST_COL_TYPE,
  CLIST_COL_N_CHANNELS,
  CLIST_COL_BLURB,
  CLIST_N_COLS
};
static gchar *clist_titles[CLIST_N_COLS] =
{
  "Name",
  "Type",
  "Ch.",
  "Blurb",
};
static guint clist_width[CLIST_N_COLS] =
{
  80,
  70,
  20,
  100,
};


/* --- dialogs data --- */
typedef struct
{
  BstShell  *shell;
  GtkWidget *window;
  GtkWidget *name_entry;
  GtkWidget *author_entry;
  GtkWidget *copyright_entry;
  GtkWidget *channels_spin;
} BstSongNewData;


/* --- prototypes --- */
static void	bst_shell_class_init		(BstShellClass	*klass);
static void	bst_shell_init			(BstShell		*pe);
static void	bst_shell_shutdown		(GtkObject		*object);
static void	bst_shell_destroy		(GtkObject		*object);
static gint	bst_shell_configure_event       (GtkWidget              *widget,
						 GdkEventConfigure      *event);
static gint	bst_shell_delete_event		(GtkWidget		*widget,
						 GdkEventAny		*event);


/* --- static variables --- */
static GtkWindowClass	*parent_class = NULL;
static guint		 shell_signals[LAST_SIGNAL] = { 0 };
static const gchar	 *read_songs_key = "bst-read-songs";
static guint		  read_songs_key_id = 0;
static const gchar	 *read_samples_key = "bst-read-samples";
static guint		  read_samples_key_id = 0;
static const gchar	 *read_binapp_key = "bst-read-binary-appendices";
static guint		  read_binapp_key_id = 0;
static const gchar	 *post_load_key = "bst-post-load-sample-cache-lookup";
static guint		  post_load_key_id = 0;
static const gchar	*class_rc_string =
"#style'bst-BstShellClass-style'"
"{"
"bg[NORMAL]={1.,1.,1.}"
"base[NORMAL]={1.,0.,0.}"
"}"
"widget_class'*BstShell'style'bst-BstShellClass-style'"
"\n"
;


/* --- menus --- */
static gchar	*bst_shell_factories_path = "<BstShell>";
static GtkItemFactoryEntry menubar_entries[] =
{
#define	SHELL_OP(shell_op)	((GtkItemFactoryCallback) bst_shell_operation), (BST_SHELL_OP_ ## shell_op)
  { "/File/New",	"<ctrl>N",	SHELL_OP (SONG_NEW),	"<Item>" },
  { "/File/Open",	"<ctrl>O",	SHELL_OP (OPEN_ANY),	"<Item>" },
  { "/File/-----",	"",		NULL, 0,		"<Separator>" },
  { "/File/Preferences...", "",		SHELL_OP (PREFERENCES),	"<Item>" },
  { "/File/Update (Test)", "",		SHELL_OP (UPDATE_ALL),	"<Item>" },
  { "/File/-----",	"",		NULL, 0,		"<Separator>" },
  { "/File/Quit",	"<ctrl>Q",	SHELL_OP (DELETE),	"<Item>" },
  { "/Tests/A",		"<ShIft>A",	NULL, 0,		"<RadioItem>" },
  { "/Tests/B",		"<Control>B",	NULL, 0,		"/Tests/A" },
  { "/Tests/C",		"<CTL>C",	NULL, 0,		"/Tests/A" },
  { "/Tests/D",		"<aLt>D",	NULL, 0,		"/Tests/A" },
  { "/Tests/E",		"<mOd1>E",	NULL, 0,		"/Tests/A" },
  { "/Tests/F",		"<ctrL>F",	NULL, 0,		"/Tests/A" },
  { "/Tests/G",		"<ALT>G",	NULL, 0,		"/Tests/A" },
  { "/Help",		"",		NULL,	0,		"<LastBranch>" },
  { "/Help/About...",	"",		SHELL_OP (ABOUT),	"<Item>" },
#undef	SHELL_OP
};
static guint n_menubar_entries = sizeof (menubar_entries) / sizeof (menubar_entries[0]);


/* --- functions --- */
GtkType
bst_shell_get_type (void)
{
  static GtkType shell_type = 0;
  
  if (!shell_type)
    {
      GtkTypeInfo shell_info =
      {
	"BstShell",
	sizeof (BstShell),
	sizeof (BstShellClass),
	(GtkClassInitFunc) bst_shell_class_init,
	(GtkObjectInitFunc) bst_shell_init,
	(GtkArgSetFunc) NULL,
	(GtkArgGetFunc) NULL,
      };
      
      shell_type = gtk_type_unique (gtk_window_get_type (), &shell_info);
    }
  
  return shell_type;
}

static void
bst_shell_class_init (BstShellClass *class)
{
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;
  GtkContainerClass *container_class;
  GtkWindowClass *window_class;

  object_class = (GtkObjectClass*) class;
  widget_class = (GtkWidgetClass*) class;
  container_class = (GtkContainerClass*) class;
  window_class = (GtkWindowClass*) class;

  parent_class = gtk_type_class (gtk_window_get_type ());

  read_songs_key_id = gtk_object_data_force_id (read_songs_key);
  read_samples_key_id = gtk_object_data_force_id (read_samples_key);
  read_binapp_key_id = gtk_object_data_force_id (read_binapp_key);
  post_load_key_id = gtk_object_data_force_id (post_load_key);

  // gtk_object_class_add_signals (object_class, song_shell_signals, LAST_SIGNAL);

  widget_class->configure_event = bst_shell_configure_event;
  widget_class->delete_event = bst_shell_delete_event;
  object_class->shutdown = bst_shell_shutdown;
  object_class->destroy = bst_shell_destroy;

  class->factories_path = bst_shell_factories_path;

  gtk_rc_parse_string (class_rc_string);
}

static void
bst_shell_shutdown (GtkObject		*object)
{
  BstShell *shell;
  GList *list, *free_list;

  g_return_if_fail (object != NULL);

  shell = BST_SHELL (object);

  gtk_clist_clear (GTK_CLIST (shell->clist));

  free_list = bse_song_list_all ();
  for (list = free_list; list; list = list->next)
    {
      BseSong *song;
      GtkWidget *song_shell;

      song = list->data;

      song_shell = (GtkWidget*) bst_song_shell_from_song (song);

      if (song_shell)
	gtk_widget_destroy (song_shell);
    }
  g_list_free (free_list);

  /* kill orphans!
   */
  for (list = shell->orphan_samples; list; list = list->next)
    bse_sample_unref (list->data);
  g_list_free (shell->orphan_samples);
  shell->orphan_samples = NULL;

  GTK_OBJECT_CLASS (parent_class)->shutdown (object);
}

static void
bst_shell_destroy (GtkObject		*object)
{
  BstShell *shell;

  g_return_if_fail (object != NULL);

  shell = BST_SHELL (object);

  gtk_object_unref (shell->menubar_factory);
  shell->menubar_factory = NULL;

  GTK_OBJECT_CLASS (parent_class)->destroy (object);
}

static void
bst_shell_init (BstShell *shell)
{
  GtkWidget *main_vbox;
  GtkWidget *vbox;
  GtkWidget *frame;
  guint i;

  shell->orphan_samples = NULL;

  gtk_widget_set (GTK_WIDGET (shell),
		  "GtkWindow::type", GTK_WINDOW_TOPLEVEL,
		  "GtkWindow::title", "The BEAST",
		  "GtkWindow::window_position", GTK_WIN_POS_NONE,
		  "GtkWidget::x", BST_RCVAL_SHELL_INIT_X,
		  "GtkWidget::y", BST_RCVAL_SHELL_INIT_Y,
		  "GtkWindow::allow_shrink", FALSE,
		  "GtkWindow::allow_grow", TRUE,
		  "GtkWindow::auto_shrink", FALSE,
		  "GtkObject::signal::map", bst_shell_update_samples, shell,
		  NULL);
  main_vbox =
    gtk_widget_new (gtk_vbox_get_type (),
		    "GtkBox::homogeneous", FALSE,
		    "GtkBox::spacing", 0,
		    "GtkContainer::border_width", 0,
		    "GtkWidget::parent", shell,
		    "GtkWidget::visible", TRUE,
		    NULL);
  /* MenuBar
   */
  shell->menubar_factory = gtk_item_factory_new (gtk_menu_bar_get_type (),
						 bst_shell_factories_path,
						 NULL);
  gtk_window_add_accelerator_table (GTK_WINDOW (shell),
				    GTK_ITEM_FACTORY (shell->menubar_factory)->table);
  gtk_item_factory_create_items (GTK_ITEM_FACTORY (shell->menubar_factory),
				 n_menubar_entries,
				 menubar_entries,
				 shell);
  gtk_box_pack_start (GTK_BOX (main_vbox),
		      GTK_ITEM_FACTORY (shell->menubar_factory)->widget,
		      FALSE,
		      TRUE,
		      0);
  gtk_widget_show (GTK_ITEM_FACTORY (shell->menubar_factory)->widget);


  /* main container
   */
  vbox =
    gtk_widget_new (gtk_vbox_get_type (),
		    "GtkBox::homogeneous", FALSE,
		    "GtkBox::spacing", 5,
		    "GtkContainer::border_width", 5,
		    "GtkWidget::parent", main_vbox,
		    "GtkWidget::visible", TRUE,
		    NULL);
  
  /* Sample List
   */
  frame =
    gtk_widget_new (gtk_frame_get_type (),
		    "GtkFrame::label", "Samples",
		    "GtkFrame::label_xalign", 0.5,
		    "GtkWidget::visible", TRUE,
		    "GtkWidget::parent", vbox,
		    NULL);
  shell->clist = gtk_clist_new_with_titles (CLIST_N_COLS, clist_titles);
  gtk_widget_set (shell->clist,
		  "GtkWidget::width", 250,
		  "GtkWidget::height", 200,
		  // "GtkObject::signal::select_row", gle_shell_row_selected, shell,
		  "GtkWidget::visible", TRUE,
		  "GtkWidget::parent", frame,
		  NULL);
  gtk_clist_set_policy (GTK_CLIST (shell->clist), GTK_POLICY_ALWAYS, GTK_POLICY_AUTOMATIC);
  gtk_clist_set_selection_mode (GTK_CLIST (shell->clist), GTK_SELECTION_BROWSE);
  gtk_clist_column_titles_passive (GTK_CLIST (shell->clist));
  for (i = 0; i < CLIST_N_COLS; i++)
    gtk_clist_set_column_width (GTK_CLIST (shell->clist), i, clist_width[i]);

  /* variables
   */
  shell->dialog_song_new = NULL;
  shell->dialog_file_open = NULL;
}

BstShell*
bst_shell_get (void)
{
  static BstShell *the_shell = NULL;

  if (!the_shell)
    {
      the_shell = (BstShell*)
	gtk_widget_new (bst_shell_get_type (),
			"GtkObject::signal::destroy", gtk_widget_destroyed, &the_shell,
			NULL);
    }

  return the_shell;
}

static gint
bst_shell_configure_event (GtkWidget	       *widget,
			   GdkEventConfigure   *event)
{
  if (BST_RCVAL_TRACK_SHELL_MOVEMENTS)
    {
      BST_RCVAL_SHELL_INIT_X = event->x;
      BST_RCVAL_SHELL_INIT_Y = event->y;
    }

  if (GTK_WIDGET_CLASS (parent_class)->configure_event)
    return GTK_WIDGET_CLASS (parent_class)->configure_event (widget, event);
  else
    return FALSE;
}

static gint
bst_shell_delete_event (GtkWidget	       *widget,
			GdkEventAny	       *event)
{
  bst_shell_operation (BST_SHELL (widget), BST_SHELL_OP_DELETE);

  return TRUE;
}

static void
bst_shell_create_song (GtkWidget      *button,
		       BstSongNewData *data)
{
  gchar *name;
  guint n_channels;
  BseSong *song;
  BstShell *shell;

  shell = data->shell;

  /* check the song name
   */
  name = gtk_entry_get_text (GTK_ENTRY (data->name_entry));
  if (!name || name[0] == 0)
    {
      bst_message_box_popup (GTK_OBJECT (button),
			     "Invalid empty song name \"\".");
      return;
    }
  else if (bse_song_lookup (name))
    {
      bst_message_box_popup (GTK_OBJECT (button),
			     "Song \"%s\"\n is already in existance.",
			     name);
      return;
    }

  /* check the channel count
   */
  n_channels = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (data->channels_spin));
  if (n_channels < BSE_MIN_CHANNELS)
    {
      bst_message_box_popup (GTK_OBJECT (button),
			     "Number of channels (%d) is too low\n"
			     "minimum is %d channels.",
			     n_channels,
			     BSE_MIN_CHANNELS);
      return;
    }
  else if (n_channels > BSE_MAX_CHANNELS)
    {
      bst_message_box_popup (GTK_OBJECT (button),
			     "Number of channels (%d) is too hight\n"
			     "maximum is %d channels.",
			     n_channels,
			     BSE_MAX_CHANNELS);
      return;
    }

  /* create a new song, update song shell list, and forget
   * about it.
   */
  song = bse_song_new (name,
		       gtk_entry_get_text (GTK_ENTRY (data->author_entry)),
		       gtk_entry_get_text (GTK_ENTRY (data->copyright_entry)),
		       n_channels);
  bse_song_add_pattern (song);

  /* FIXME: each song gets a zero instrument
   */
  {
    BseSample *sample;
    BseInstrument *instrument;

    sample = bse_get_zero_sample ();
    instrument = bse_song_sample_instrument_new (song, sample);
    bse_sample_unref (sample);
  }

  gtk_widget_destroy (data->window);
  gdk_flush ();

  bst_shell_operation (shell, BST_SHELL_OP_UPDATE_SONGS);
  bst_shell_operation (shell, BST_SHELL_OP_UPDATE_SAMPLES);

  bse_song_unref (song);
}

void
bst_shell_update_songs (BstShell       *shell)
{
  GList *list, *free_list;

  g_return_if_fail (shell != NULL);
  g_return_if_fail (BST_IS_SHELL (shell));

  free_list = bse_song_list_all ();
  for (list = free_list; list; list = list->next)
    {
      BseSong *song;
      GtkWidget *song_shell;

      song = list->data;

      song_shell = (GtkWidget*) bst_song_shell_from_song (song);

      if (!song_shell)
	{
	  song_shell = bst_song_shell_new (song);
	  gtk_widget_show (song_shell);
	}
    }
  g_list_free (free_list);
}

void
bst_shell_update_samples (BstShell	  *shell)
{
  GList *list, *free_list;

  g_return_if_fail (shell != NULL);
  g_return_if_fail (BST_IS_SHELL (shell));

  gtk_clist_freeze (GTK_CLIST (shell->clist));
  gtk_clist_clear (GTK_CLIST (shell->clist));

  free_list = bse_sample_list_all ();
  for (list = free_list; list; list = list->next)
    {
      BseSample *sample;
      gchar *text[16] = { NULL, };
      gchar buffer[64];
      gint row;

      sample = list->data;

      text[0] = sample->name;
      text[1] = (gchar*) bse_sample_type_get_nick (sample->type);
      sprintf (buffer, "%u %c%c%c%c%c%c%c ref_count: %u",
	       ' ', ' ', ' ', ' ', ' ', ' ', ' ',
	       sample->n_channels,
	       GTK_OBJECT (sample)->ref_count);
      text[2] = buffer;
      text[3] = (gchar*) bse_sample_get_blurb (sample);

      row = gtk_clist_append (GTK_CLIST (shell->clist), text);
      gtk_clist_set_row_data (GTK_CLIST (shell->clist), row, sample);
    }
  g_list_free (free_list);

  gtk_clist_thaw (GTK_CLIST (shell->clist));
}

static void
bst_shell_dialog_song_new (BstShell	  *shell)
{
  BstSongNewData *data;
  GtkWidget *main_vbox;
  GtkWidget *vbox;
  GtkWidget *hbox;
  GtkWidget *label;
  GtkObject *obj;
  GtkWidget *any;
  GtkWidget *button;

  g_return_if_fail (shell != NULL);
  g_return_if_fail (BST_IS_SHELL (shell));
  g_return_if_fail (shell->dialog_song_new == NULL);

  data = g_new0 (BstSongNewData, 1);

  data->shell = shell;
  data->window =
    gtk_widget_new (gtk_window_get_type (),
		    "GtkWindow::type", GTK_WINDOW_TOPLEVEL,
		    "GtkWindow::title", "New Song",
		    "GtkWindow::window_position", GTK_WIN_POS_MOUSE,
		    "GtkWindow::allow_shrink", FALSE,
		    "GtkWindow::allow_grow", FALSE,
		    "GtkWindow::auto_shrink", TRUE,
		    NULL);
  gtk_window_add_accelerator_table (GTK_WINDOW (data->window),
				    GTK_ITEM_FACTORY (shell->menubar_factory)->table);
  shell->dialog_song_new = data->window;
  gtk_signal_connect (GTK_OBJECT (data->window),
		      "destroy",
		      GTK_SIGNAL_FUNC (gtk_widget_destroyed),
		      &shell->dialog_song_new);
  gtk_signal_connect_object_while_alive (GTK_OBJECT (shell),
					 "destroy",
					 GTK_SIGNAL_FUNC (gtk_widget_destroy),
					 GTK_OBJECT (shell->dialog_song_new));
  gtk_object_set_data_full (GTK_OBJECT (data->window), "bst-BstSongNewData", data, g_free);
  main_vbox =
    gtk_widget_new (gtk_vbox_get_type (),
		    "GtkBox::homogeneous", FALSE,
		    "GtkBox::spacing", 0,
		    "GtkContainer::border_width", 0,
		    "GtkWidget::parent", data->window,
		    "GtkWidget::visible", TRUE,
		    NULL);
  hbox =
    gtk_widget_new (gtk_hbox_get_type (),
		    "GtkBox::homogeneous", FALSE,
		    "GtkBox::spacing", 0,
		    "GtkContainer::border_width", 5,
		    "GtkWidget::parent", main_vbox,
		    "GtkWidget::visible", TRUE,
		    NULL);
  vbox =
    gtk_widget_new (gtk_vbox_get_type (),
		    "GtkBox::homogeneous", TRUE,
		    "GtkBox::spacing", 0,
		    "GtkContainer::border_width", 0,
		    "GtkWidget::parent", hbox,
		    "GtkWidget::visible", TRUE,
		    NULL);
  label =
    gtk_widget_new (gtk_label_get_type (),
		    "GtkLabel::label", "Name:",
		    "GtkMisc::xalign", 0.0,
		    "GtkWidget::parent", vbox,
		    "GtkWidget::visible", TRUE,
		    NULL);
  label =
    gtk_widget_new (gtk_label_get_type (),
		    "GtkLabel::label", "Author:",
		    "GtkMisc::xalign", 0.0,
		    "GtkWidget::parent", vbox,
		    "GtkWidget::visible", TRUE,
		    NULL);
  label =
    gtk_widget_new (gtk_label_get_type (),
		    "GtkLabel::label", "Copyrigth:",
		    "GtkMisc::xalign", 0.0,
		    "GtkWidget::parent", vbox,
		    "GtkWidget::visible", TRUE,
		    NULL);
  label =
    gtk_widget_new (gtk_label_get_type (),
		    "GtkLabel::label", "Channels:",
		    "GtkMisc::xalign", 0.0,
		    "GtkWidget::parent", vbox,
		    "GtkWidget::visible", TRUE,
		    NULL);
  vbox =
    gtk_widget_new (gtk_vbox_get_type (),
		    "GtkBox::homogeneous", TRUE,
		    "GtkBox::spacing", 0,
		    "GtkContainer::border_width", 0,
		    "GtkWidget::parent", hbox,
		    "GtkWidget::visible", TRUE,
		    NULL);
  data->name_entry =
    gtk_widget_new (gtk_entry_get_type (),
		    "GtkWidget::parent", vbox,
		    "GtkWidget::visible", TRUE,
		    "GtkObject::object_signal::activate", gtk_window_activate_default, data->window,
		    NULL);
  gtk_widget_grab_focus (data->name_entry);
  data->author_entry =
    gtk_widget_new (gtk_entry_get_type (),
		    "GtkWidget::parent", vbox,
		    "GtkWidget::visible", TRUE,
		    "GtkObject::object_signal::activate", gtk_window_activate_default, data->window,
		    NULL);
  data->copyright_entry =
    gtk_widget_new (gtk_entry_get_type (),
		    "GtkWidget::parent", vbox,
		    "GtkWidget::visible", TRUE,
		    "GtkObject::object_signal::activate", gtk_window_activate_default, data->window,
		    NULL);
  data->channels_spin =
    gtk_widget_new (gtk_spin_button_get_type (),
		    "GtkWidget::parent", vbox,
		    "GtkWidget::visible", TRUE,
		    NULL);
  obj = gtk_adjustment_new (BSE_DFL_N_CHANNELS,
			    BSE_MIN_CHANNELS, BSE_MAX_CHANNELS,
			    1,
			    10, 0);
  gtk_spin_button_construct (GTK_SPIN_BUTTON (data->channels_spin),
			     GTK_ADJUSTMENT (obj),
			     0, 0);
  any =
    gtk_widget_new (gtk_hseparator_get_type (),
		    "GtkWidget::parent", main_vbox,
		    "GtkWidget::visible", TRUE,
		    NULL);
  hbox =
    gtk_widget_new (gtk_hbox_get_type (),
		    "GtkBox::homogeneous", TRUE,
		    "GtkBox::spacing", 0,
		    "GtkContainer::border_width", 5,
		    "GtkWidget::parent", main_vbox,
		    "GtkWidget::visible", TRUE,
		    NULL);
  button =
    gtk_widget_new (gtk_button_get_type (),
		    "GtkButton::label", "Ok",
		    "GtkWidget::parent", hbox,
		    "GtkWidget::visible", TRUE,
		    "GtkWidget::can_default", TRUE,
		    "GtkWidget::has_default", TRUE,
		    "GtkObject::signal::clicked", bst_shell_create_song, data,
		    NULL);
  button =
    gtk_widget_new (gtk_button_get_type (),
		    "GtkButton::label", "Cancel",
		    "GtkWidget::parent", hbox,
		    "GtkWidget::visible", TRUE,
		    "GtkWidget::can_default", TRUE,
		    "GtkObject::object_signal::clicked", gtk_widget_destroy, data->window,
		    NULL);
}

static BseIoData*
post_load_samles (gpointer        callback_data,
		  const gchar    *sample_name,
		  const gchar    *sample_path)
{
  BseIoDataFlags read_flags;
  BseIoData *io_data;

  read_flags = 0;
  read_flags |= BSE_IO_DATA_SAMPLES;
  read_flags |= BSE_IO_DATA_BINARY_APPENDIX;

  io_data = bse_io_load_auto (sample_path, read_flags);

  return io_data;
}

static void
bst_shell_fs_load (BstShell *shell)
{
  GtkFileSelection *fs;
  GtkToggleButton *toggle_songs;
  GtkToggleButton *toggle_samples;
  GtkToggleButton *toggle_binapp;
  GtkToggleButton *toggle_postload;
  BseIoDataFlags read_flags;
  gchar	 *file_name;
  gboolean post_load;

  g_return_if_fail (shell != NULL);
  g_return_if_fail (BST_IS_SHELL (shell));

  fs = shell->dialog_file_open;

  file_name = g_strdup (gtk_file_selection_get_filename (fs));

  if (!S_ISREG (g_scanner_stat_mode (file_name)))
    {
      gtk_file_selection_set_filename (fs, file_name);
      g_free (file_name);

      return;
    }

  toggle_songs = gtk_object_get_data_by_id (GTK_OBJECT (fs), read_songs_key_id);
  toggle_samples = gtk_object_get_data_by_id (GTK_OBJECT (fs), read_samples_key_id);
  toggle_binapp = gtk_object_get_data_by_id (GTK_OBJECT (fs), read_binapp_key_id);
  toggle_postload = gtk_object_get_data_by_id (GTK_OBJECT (fs), post_load_key_id);

  read_flags = 0;
  read_flags |= toggle_songs->active ? BSE_IO_DATA_SONGS : 0;
  read_flags |= toggle_samples->active ? BSE_IO_DATA_SAMPLES : 0;
  read_flags |= toggle_binapp->active && toggle_samples->active ? BSE_IO_DATA_BINARY_APPENDIX : 0;
  post_load = toggle_postload->active;

  gtk_widget_destroy (GTK_WIDGET (fs));
  fs = NULL;
  gdk_flush ();

  if (read_flags)
    {
      BseIoData *io_data;

      io_data = bse_io_load_auto (file_name, read_flags);

      if (io_data->error)
	bst_message_box_popup (GTK_OBJECT (shell),
			       "Error encountered while loading\n`%s':\n%s",
			       file_name,
			       bse_error_type_get_description (io_data->error));
      else
	{
	  GList *list;
	  
	  /* collect newly loaded samples
	   */
	  for (list = io_data->samples; list; list = list->next)
	    {
	      BseSample *sample;

	      sample = list->data;

	      if (GTK_OBJECT (sample)->ref_count == 1)
		bst_shell_adapt_sample (shell, sample);
	    }

	  /* collect songs
	   */
	  bst_shell_operation (shell, BST_SHELL_OP_UPDATE_SONGS);

	  /* resolve sample references from instruments
	   */
	  for (list = io_data->songs; list; list = list->next)
	    bse_song_reload_instrument_samples (list->data,
						post_load ? post_load_samles : NULL,
						NULL);

	  bse_io_data_destroy (io_data);

	  bst_shell_operation (shell, BST_SHELL_OP_UPDATE_SAMPLES);
	}
    }

  g_free (file_name);
}

void
bst_shell_adapt_sample (BstShell       *shell,
			BseSample      *sample)
{
  g_return_if_fail (shell != NULL);
  g_return_if_fail (BST_IS_SHELL (shell));
  g_return_if_fail (sample != NULL);
  g_return_if_fail (GTK_OBJECT (sample)->ref_count == 1);
  g_return_if_fail (g_list_find (shell->orphan_samples, sample) == NULL);

  shell->orphan_samples = g_list_append (shell->orphan_samples, sample);
  bse_sample_ref (sample);
}

static void
bst_shell_dialog_file_open (BstShell	   *shell)
{
  GtkFileSelection *fs;
  GtkWidget *main_vbox;
  GtkWidget *hbox;
  GtkWidget *any;
  GtkWidget *frame;
  GtkWidget *vbox;
  GtkWidget *toggle;

  g_return_if_fail (shell != NULL);
  g_return_if_fail (BST_IS_SHELL (shell));
  g_return_if_fail (shell->dialog_file_open == NULL);

  shell->dialog_file_open = (GtkFileSelection*)
    gtk_widget_new (gtk_file_selection_get_type (),
		    "GtkWindow::title", "Load File",
		    "GtkWindow::window_position", GTK_WIN_POS_MOUSE,
		    "GtkWindow::allow_shrink", FALSE,
		    "GtkWindow::allow_grow", TRUE,
		    "GtkWindow::auto_shrink", FALSE,
		    "GtkObject::signal::delete_event", gtk_widget_destroy, NULL,
		    "GtkObject::signal::delete_event", gtk_true, NULL,
		    NULL);
  gtk_window_add_accelerator_table (GTK_WINDOW (shell->dialog_file_open),
				    GTK_ITEM_FACTORY (shell->menubar_factory)->table);
  fs = shell->dialog_file_open;
  gtk_signal_connect (GTK_OBJECT (fs),
		      "destroy",
		      GTK_SIGNAL_FUNC (gtk_widget_destroyed),
		      &shell->dialog_file_open);
  gtk_signal_connect_object_while_alive (GTK_OBJECT (shell),
					 "destroy",
					 GTK_SIGNAL_FUNC (gtk_widget_destroy),
					 GTK_OBJECT (shell->dialog_file_open));
  gtk_widget_set (fs->cancel_button,
		  "GtkObject::object_signal::clicked", gtk_widget_destroy, fs,
		  NULL);
  gtk_widget_set (fs->ok_button,
		  "GtkObject::object_signal::clicked", bst_shell_fs_load, shell,
		  NULL);

  /* button placement
   */
  gtk_container_border_width (GTK_CONTAINER (fs), 0);
  gtk_file_selection_hide_fileop_buttons (fs);
  gtk_widget_ref (fs->main_vbox);
  gtk_container_remove (GTK_CONTAINER (fs), fs->main_vbox);
  gtk_box_set_spacing (GTK_BOX (fs->main_vbox), 0);
  gtk_container_border_width (GTK_CONTAINER (fs->main_vbox), 5);
  main_vbox =
    gtk_widget_new (gtk_vbox_get_type (),
		    "GtkBox::homogeneous", FALSE,
		    "GtkBox::spacing", 0,
		    "GtkContainer::border_width", 0,
		    "GtkWidget::parent", fs,
		    "GtkWidget::visible", TRUE,
		    NULL);
  gtk_box_pack_start (GTK_BOX (main_vbox), fs->main_vbox, TRUE, TRUE, 0);
  gtk_widget_unref (fs->main_vbox);
  gtk_widget_hide (fs->ok_button->parent);
  hbox =
    gtk_widget_new (gtk_hbox_get_type (),
		    "GtkBox::homogeneous", TRUE,
		    "GtkBox::spacing", 0,
		    "GtkContainer::border_width", 5,
		    "GtkWidget::visible", TRUE,
		    NULL);
  gtk_box_pack_end (GTK_BOX (main_vbox), hbox, FALSE, TRUE, 0);
  gtk_widget_reparent (fs->ok_button, hbox);
  gtk_widget_reparent (fs->cancel_button, hbox);
  gtk_widget_grab_default (fs->ok_button);
  gtk_label_set (GTK_LABEL (GTK_BUTTON (fs->ok_button)->child), "Ok");
  gtk_label_set (GTK_LABEL (GTK_BUTTON (fs->cancel_button)->child), "Cancel");

  /* toggle buttons to configure load behaviour
   */
  gtk_box_set_child_packing (GTK_BOX (fs->action_area->parent),
			     fs->action_area,
			     FALSE, TRUE,
			     5, GTK_PACK_START);
  frame =
    gtk_widget_new (gtk_frame_get_type (),
		    "GtkFrame::label", "Load Options",
		    "GtkFrame::label_xalign", 0.5,
		    "GtkWidget::visible", TRUE,
		    "GtkWidget::parent", fs->action_area,
		    NULL);
  vbox =
    gtk_widget_new (gtk_vbox_get_type (),
		    "GtkBox::homogeneous", FALSE,
		    "GtkBox::spacing", 0,
		    "GtkContainer::border_width", 0,
		    "GtkWidget::parent", frame,
		    "GtkWidget::visible", TRUE,
		    NULL);
  toggle =
    gtk_widget_new (gtk_check_button_get_type (),
		    "GtkButton::label", "Load Songs",
		    "GtkWidget::parent", vbox,
		    NULL);
  gtk_misc_set_alignment (GTK_MISC (GTK_BUTTON (toggle)->child), 0, 0.5);
  gtk_object_set_data_by_id (GTK_OBJECT (shell->dialog_file_open), read_songs_key_id, toggle);
  toggle =
    gtk_widget_new (gtk_check_button_get_type (),
		    "GtkButton::label", "Load Samples",
		    "GtkWidget::parent", vbox,
		    NULL);
  gtk_misc_set_alignment (GTK_MISC (GTK_BUTTON (toggle)->child), 0, 0.5);
  gtk_object_set_data_by_id (GTK_OBJECT (shell->dialog_file_open), read_samples_key_id, toggle);
  toggle =
    gtk_widget_new (gtk_check_button_get_type (),
		    "GtkButton::label", "Load Binary Appendix Samples",
		    "GtkWidget::parent", vbox,
		    NULL);
  gtk_misc_set_alignment (GTK_MISC (GTK_BUTTON (toggle)->child), 0, 0.5);
  gtk_object_set_data_by_id (GTK_OBJECT (shell->dialog_file_open), read_binapp_key_id, toggle);
  toggle =
    gtk_widget_new (gtk_check_button_get_type (),
		    "GtkButton::label", "Automatically post-load Samples",
		    "GtkWidget::parent", vbox,
		    NULL);
  gtk_misc_set_alignment (GTK_MISC (GTK_BUTTON (toggle)->child), 0, 0.5);
  gtk_object_set_data_by_id (GTK_OBJECT (shell->dialog_file_open), post_load_key_id, toggle);

  /* add a separator
   */
  any =
    gtk_widget_new (gtk_hseparator_get_type (),
		    "GtkWidget::visible", TRUE,
		    NULL);
  gtk_box_pack_end (GTK_BOX (main_vbox), any, FALSE, TRUE, 0);
}

static void
bst_shell_dialog_file_open_adjust (BstShell	  *shell,
				   BstShellOps	   sop)
{
  GtkWidget *toggle_songs;
  GtkWidget *toggle_samples;
  GtkWidget *toggle_binapp;
  GtkWidget *toggle_postload;

  g_return_if_fail (shell != NULL);
  g_return_if_fail (BST_IS_SHELL (shell));
  g_return_if_fail (shell->dialog_file_open != NULL);

  toggle_songs = gtk_object_get_data_by_id (GTK_OBJECT (shell->dialog_file_open), read_songs_key_id);
  toggle_samples = gtk_object_get_data_by_id (GTK_OBJECT (shell->dialog_file_open), read_samples_key_id);
  toggle_binapp = gtk_object_get_data_by_id (GTK_OBJECT (shell->dialog_file_open), read_binapp_key_id);
  toggle_postload = gtk_object_get_data_by_id (GTK_OBJECT (shell->dialog_file_open), post_load_key_id);

  switch (sop)
    {
    case  BST_SHELL_OP_OPEN_SONG:
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (toggle_songs), TRUE);
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (toggle_samples), TRUE);
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (toggle_binapp), TRUE);
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (toggle_postload), TRUE);
      gtk_widget_show (toggle_songs);
      gtk_widget_hide (toggle_samples);
      gtk_widget_hide (toggle_binapp);
      gtk_widget_show (toggle_postload);
      break;
    case  BST_SHELL_OP_OPEN_SAMPLE:
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (toggle_songs), FALSE);
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (toggle_samples), TRUE);
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (toggle_binapp), TRUE);
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (toggle_postload), FALSE);
      gtk_widget_hide (toggle_songs);
      gtk_widget_show (toggle_samples);
      gtk_widget_show (toggle_binapp);
      gtk_widget_hide (toggle_postload);
      break;
    case  BST_SHELL_OP_OPEN_ANY:
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (toggle_songs), TRUE);
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (toggle_samples), TRUE);
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (toggle_binapp), TRUE);
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (toggle_postload), TRUE);
      gtk_widget_show (toggle_songs);
      gtk_widget_show (toggle_samples);
      gtk_widget_show (toggle_binapp);
      gtk_widget_show (toggle_postload);
      break;
    default:
      g_assert_not_reached ();
      break;
    }
  gtk_widget_queue_resize (shell->dialog_file_open->main_vbox);
  if (!GTK_WIDGET_VISIBLE (shell->dialog_file_open))
    gtk_widget_grab_focus (shell->dialog_file_open->selection_entry);
}

void
bst_shell_operation (BstShell	    *shell,
		     BstShellOps     shell_op)
{
  g_return_if_fail (shell != NULL);
  g_return_if_fail (BST_IS_SHELL (shell));
  g_return_if_fail (shell_op < BST_SHELL_OP_LAST);

  gtk_widget_ref (GTK_WIDGET (shell));

  switch (shell_op)
    {
    case  BST_SHELL_OP_SONG_NEW:
      if (!shell->dialog_song_new)
	bst_shell_dialog_song_new (shell);
      gtk_widget_show (shell->dialog_song_new);
      gdk_window_raise (shell->dialog_song_new->window);
      break;
    case  BST_SHELL_OP_OPEN_SONG:
    case  BST_SHELL_OP_OPEN_SAMPLE:
    case  BST_SHELL_OP_OPEN_ANY:
      if (!shell->dialog_file_open)
	bst_shell_dialog_file_open (shell);
      bst_shell_dialog_file_open_adjust (shell, shell_op);
      gtk_widget_show (GTK_WIDGET (shell->dialog_file_open));
      gdk_window_raise (GTK_WIDGET (shell->dialog_file_open)->window);
      break;
    case  BST_SHELL_OP_DELETE:
      gtk_widget_destroy (GTK_WIDGET (shell));
      break;
    case  BST_SHELL_OP_UPDATE_SONGS:
      bst_shell_update_songs (shell);
      break;
    case  BST_SHELL_OP_UPDATE_SAMPLES:
      bst_shell_update_samples (shell);
      break;
    case  BST_SHELL_OP_UPDATE_ALL:
      bst_shell_operation (shell, BST_SHELL_OP_UPDATE_SONGS);
      bst_shell_operation (shell, BST_SHELL_OP_UPDATE_SAMPLES);
      break;
    default:
      printf ("BstShellOps: %d\n", shell_op);
      break;
    }

  gtk_widget_unref (GTK_WIDGET (shell));
}
