/* BEAST - Bedevilled Audio System
 * Copyright (C) 1998 Olaf Hoehmann and Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */
#include "bstpatterneditor.h"
#include <gtk/gtksignal.h>
#include <gdk/gdkkeysyms.h>
#include <string.h>



/* --- defines --- */
#define	MAX_WIDTH			(600)
#define	MAX_HEIGHT			(460)
#define	CHAR_SPACE			(3)
#define	CHANNEL_SPACE			(4)
#define	LINE_SPACE			(1)
#define	FONT_HEIGHT(font)		((font)->ascent + (font)->descent)
#define	X_THICK(w)			(((GtkWidget*) (w))->style->klass->xthickness)
#define	Y_THICK(w)			(((GtkWidget*) (w))->style->klass->ythickness)
#define	ALLOCATION(w)			(&((GtkWidget*) (w))->allocation)
#define	CHAR_WIDTH(pe)			(((BstPatternEditor*) (pe))->char_width)
#define	CHAR_HEIGHT(pe)			(((BstPatternEditor*) (pe))->char_height)
#define	INDEX_BORDER			(CHANNEL_SPACE / 4)
#define	HEADLINE_HEIGHT(pe)		(((BstPatternEditor*) (pe))->char_height + 1)
#define OUTER_X_BORDER(pe)		(((GtkContainer*) (pe))->border_width + 10)
#define OUTER_Y_BORDER(pe)		(((GtkContainer*) (pe))->border_width + 10)
#define INNER_X_BORDER(w)		(X_THICK(w) + 1)
#define INNER_Y_BORDER(w)		(Y_THICK(w) + 1)
#define	RIGHT_BAR(pe)			(((BstPatternEditor*)(pe))->vscrollbar->requisition.width)
#define	BOTTOM_BAR(pe)			(((BstPatternEditor*)(pe))->hscrollbar->requisition.height)
#define X_BORDER(w)			(INNER_X_BORDER (w) + OUTER_X_BORDER (w))
#define Y_BORDER(w)			(INNER_Y_BORDER (w) + OUTER_Y_BORDER (w))
#define	INDEX_SA_X(pe)			(X_BORDER (pe))
#define	INDEX_SA_Y(pe)			(HEADLINE_SA_Y (pe) + HEADLINE_SA_HEIGHT (pe))
#define	INDEX_SA_WIDTH(pe)		(INDEX_BORDER + CHAR_WIDTH (pe) * INDEX_DIGITS + INDEX_BORDER)
#define	INDEX_SA_HEIGHT(pe)		(ALLOCATION (pe)->height - HEADLINE_SA_Y (pe) - HEADLINE_SA_HEIGHT (pe) - BOTTOM_BAR (pe) - Y_BORDER (pe))
#define	HEADLINE_SA_X(pe)		(INDEX_SA_X (pe) + INDEX_SA_WIDTH (pe))
#define	HEADLINE_SA_Y(pe)		(Y_BORDER (pe))
#define	HEADLINE_SA_WIDTH(pe)		(ALLOCATION (pe)->width - INDEX_SA_X (pe) - INDEX_SA_WIDTH (pe) - RIGHT_BAR (pe) - X_BORDER (pe))
#define	HEADLINE_SA_HEIGHT(pe)		(CHAR_HEIGHT (pe))
#define	PANEL_SA_X(pe)			(INDEX_SA_X (pe) + INDEX_SA_WIDTH (pe))
#define	PANEL_SA_Y(pe)			(HEADLINE_SA_Y (pe) + HEADLINE_SA_HEIGHT (pe))
#define	PANEL_SA_WIDTH(pe)		(HEADLINE_SA_WIDTH (pe))
#define	PANEL_SA_HEIGHT(pe)		(INDEX_SA_HEIGHT (pe))
#define	INDEX_DIGITS			(3)
#define	INSTRUMENT_DIGITS		(4)
#define	NOTE_DIGITS			(5)
#define	EFFECT_WIDTH(pe)		(1 * CHAR_WIDTH (pe))

/* --- signals --- */
enum
{
  SIGNAL_PATTERN_STEP,
  LAST_SIGNAL
};
typedef	void	(*SignalPatternStep)	(GtkObject	*object,
					 guint		 current_guid,
					 gint		 difference,
					 gpointer	 func_data);


/* --- prototypes --- */
static void	bst_pattern_editor_class_init	(BstPatternEditorClass	*klass);
static void	bst_pattern_editor_init		(BstPatternEditor	*pe);
static void	bst_pattern_editor_shutdown	(GtkObject		*object);
static void	bst_pattern_editor_size_request	(GtkWidget		*widget,
						 GtkRequisition		*requisition);
static void	bst_pattern_editor_size_allocate(GtkWidget		*widget,
						 GtkAllocation		*allocation);
static void	bst_pattern_editor_style_set	(GtkWidget		*widget,
						 GtkStyle		*previous_style);
static void	bst_pattern_editor_state_changed(GtkWidget		*widget,
						 guint			 previous_state);
static void	bst_pattern_editor_realize	(GtkWidget		*widget);
static void	bst_pattern_editor_unrealize	(GtkWidget		*widget);
static void	bst_pattern_editor_map		(GtkWidget		*widget);
static void	bst_pattern_editor_unmap	(GtkWidget		*widget);
static gint	bst_pattern_editor_focus_in	(GtkWidget		*widget,
						 GdkEventFocus		*event);
static gint	bst_pattern_editor_focus_out	(GtkWidget		*widget,
						 GdkEventFocus		*event);
static gint	bst_pattern_editor_expose	(GtkWidget		*widget,
						 GdkEventExpose		*event);
static void	bst_pattern_editor_draw_focus	(GtkWidget		*widget);
static void	bst_pattern_editor_draw_cell_focus (GtkWidget		*widget);
static void	bst_pattern_editor_draw_main	(BstPatternEditor	*pe);
static void	bst_pattern_editor_draw_panel	(BstPatternEditor	*pe,
						 guint			 channel,
						 guint			 row,
						 gboolean		 clear);
static void	bst_pattern_editor_draw_index	(BstPatternEditor	*pe,
						 guint			 row,
						 gboolean		 clear);
static void	bst_pattern_editor_draw_head	(BstPatternEditor	*pe,
						 guint			 channel,
						 gboolean		 clear);
static gint	bst_pattern_editor_key_press	(GtkWidget		*widget,
						 GdkEventKey		*event);
static gint	bst_pattern_editor_key_release	(GtkWidget		*widget,
						 GdkEventKey		*event);
static gint	bst_pattern_editor_button_press	(GtkWidget		*widget,
						 GdkEventButton		*event);
static gint	bst_pattern_editor_button_release(GtkWidget		*widget,
						  GdkEventButton	*event);
static void	bst_pattern_editor_draw		(GtkWidget		*widget,
						 GdkRectangle		*area);
static void	bst_pattern_editor_foreach	 (GtkContainer		*container,
						  GtkCallback		 callback,
						  gpointer		 callback_data);
static void	bst_pattern_editor_channel_popup(BstPatternEditor	*pe,
						 guint			 channel,
						 guint			 mouse_button,
						 guint32		 time);
static void	bst_pattern_editor_release_pattern (BstPatternEditor	*pe);
static void	adjustments_value_changed	(GtkAdjustment		*adjustment,
						 BstPatternEditor	*pe);
static void	note_changed			(BstPatternEditor	*pe,
						 BsePattern		*pattern,
						 guint			 channel,
						 guint			 row);


/* --- static variables --- */
static GtkContainerClass *parent_class = NULL;
static guint		 pe_signals[LAST_SIGNAL] = { 0 };
static const gchar	*class_rc_string =
"style'bst-BstPatternEditorClass-style'"
"{"
"font='-misc-fixed-*-*-*-*-*-130-*-*-*-*-*-*'\n"
"bg[NORMAL]={1.,1.,1.}"
"base[NORMAL]={1.,0.,0.}"
"base[INSENSITIVE]={.5,.5,.5}"
"}"
"widget_class'*BstPatternEditor'style'bst-BstPatternEditorClass-style'"
"\n"
;


/* --- functions --- */
GtkType
bst_pattern_editor_get_type (void)
{
  static GtkType pe_type = 0;
  
  if (!pe_type)
    {
      GtkTypeInfo pe_info =
      {
	"BstPatternEditor",
	sizeof (BstPatternEditor),
	sizeof (BstPatternEditorClass),
	(GtkClassInitFunc) bst_pattern_editor_class_init,
	(GtkObjectInitFunc) bst_pattern_editor_init,
	(GtkArgSetFunc) NULL,
	(GtkArgGetFunc) NULL,
      };
      
      pe_type = gtk_type_unique (gtk_container_get_type (), &pe_info);
    }
  
  return pe_type;
}

static void
bst_pattern_editor_marshal_pattern_step (GtkObject	*object,
					 GtkSignalFunc	func,
					 gpointer	func_data,
					 GtkArg		*args)
{
  SignalPatternStep sfunc = (SignalPatternStep) func;
  
  (* sfunc) (object,
	     GTK_VALUE_UINT (args[0]),
	     GTK_VALUE_INT (args[1]),
	     func_data);
}

static void
bst_pattern_editor_class_init (BstPatternEditorClass *class)
{
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;
  GtkContainerClass *container_class;
  
  object_class = (GtkObjectClass*) class;
  widget_class = (GtkWidgetClass*) class;
  container_class = (GtkContainerClass*) class;
  
  parent_class = gtk_type_class (gtk_container_get_type ());
  
  pe_signals[SIGNAL_PATTERN_STEP] =
    gtk_signal_new ("pattern_step",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (BstPatternEditorClass, pattern_step),
		    bst_pattern_editor_marshal_pattern_step,
		    GTK_TYPE_NONE, 2,
		    GTK_TYPE_UINT,
		    GTK_TYPE_INT);
  gtk_object_class_add_signals (object_class, pe_signals, LAST_SIGNAL);

  object_class->shutdown = bst_pattern_editor_shutdown;
  
  widget_class->draw_focus = bst_pattern_editor_draw_focus;
  widget_class->size_request = bst_pattern_editor_size_request;
  widget_class->size_allocate = bst_pattern_editor_size_allocate;
  widget_class->realize = bst_pattern_editor_realize;
  widget_class->unrealize = bst_pattern_editor_unrealize;
  widget_class->style_set = bst_pattern_editor_style_set;
  widget_class->state_changed = bst_pattern_editor_state_changed;
  widget_class->draw = bst_pattern_editor_draw;
  widget_class->expose_event = bst_pattern_editor_expose;
  widget_class->focus_in_event = bst_pattern_editor_focus_in;
  widget_class->focus_out_event = bst_pattern_editor_focus_out;
  widget_class->map = bst_pattern_editor_map;
  widget_class->unmap = bst_pattern_editor_unmap;
  widget_class->key_press_event = bst_pattern_editor_key_press;
  widget_class->key_release_event = bst_pattern_editor_key_release;
  widget_class->button_press_event = bst_pattern_editor_button_press;
  widget_class->button_release_event = bst_pattern_editor_button_release;
  
  container_class->foreach = bst_pattern_editor_foreach;
  
  class->pea_ktab = g_hash_table_new (g_direct_hash, NULL);
  
  gtk_rc_parse_string (class_rc_string);
}

static void
bst_pattern_editor_init (BstPatternEditor *pe)
{
  GtkWidget *widget;
  GtkAdjustment *adjustment;
  
  widget = GTK_WIDGET (pe);
  
  GTK_WIDGET_UNSET_FLAGS (pe, GTK_NO_WINDOW);
  GTK_WIDGET_SET_FLAGS (pe, GTK_CAN_FOCUS);
  
  pe->plain_width = 0;
  pe->plain_height = 0;
  pe->panel_width = 0;
  pe->panel_height = 0;
  
  pe->char_width = 5;
  pe->char_height = 5;
  pe->char_descent = 2;
  
  pe->index_sa = NULL;
  pe->index = NULL;
  pe->headline_sa = NULL;
  pe->headline = NULL;
  pe->panel_sa = NULL;
  pe->panel = NULL;
  
  pe->popup_tag = 0;
  pe->channel_popup = NULL;
  
  pe->pattern = NULL;
  pe->instruments = NULL;
  pe->wrap_type = 0;
  pe->channel_page = 2;
  pe->row_page = 4;
  
  pe->cell_focus = TRUE;
  pe->next_moves_left = FALSE;
  pe->next_moves_right = FALSE;
  pe->next_moves_up = FALSE;
  pe->next_moves_down = TRUE;
  
  pe->focus_channel = 0;
  pe->focus_row = 0;
  pe->last_row = -1;
  pe->base_octave = 0;
  
  pe->vscrollbar = gtk_vscrollbar_new (NULL);
  adjustment = gtk_range_get_adjustment (GTK_RANGE (pe->vscrollbar));
  gtk_signal_connect (GTK_OBJECT (adjustment),
		      "value_changed",
		      GTK_SIGNAL_FUNC (adjustments_value_changed),
		      pe);
  gtk_widget_set_parent (pe->vscrollbar, widget);
  gtk_widget_show (pe->vscrollbar);
  pe->hscrollbar = gtk_hscrollbar_new (NULL);
  adjustment = gtk_range_get_adjustment (GTK_RANGE (pe->hscrollbar));
  gtk_signal_connect (GTK_OBJECT (adjustment),
		      "value_changed",
		      GTK_SIGNAL_FUNC (adjustments_value_changed),
		      pe);
  gtk_widget_set_parent (pe->hscrollbar, widget);
  gtk_widget_show (pe->hscrollbar);
}

static void
bst_pattern_editor_shutdown (GtkObject		    *object)
{
  BstPatternEditor *pe;

  g_return_if_fail (object != NULL);
  g_return_if_fail (BST_IS_PATTERN_EDITOR (object));

  pe = BST_PATTERN_EDITOR (object);

  if (pe->channel_popup)
    {
      gtk_widget_unref (pe->channel_popup);
      pe->channel_popup = NULL;
    }

  gtk_widget_unparent (pe->vscrollbar);
  pe->vscrollbar = NULL;
  gtk_widget_unparent (pe->hscrollbar);
  pe->hscrollbar = NULL;

  bst_pattern_editor_release_pattern (pe);

  GTK_OBJECT_CLASS (parent_class)->shutdown (object);
}

static void
bst_pattern_editor_style_set (GtkWidget	     *widget,
			      GtkStyle	     *previous_style)
{
  BstPatternEditor *pe;
  guint i;
  
  g_return_if_fail (widget != NULL);
  g_return_if_fail (BST_IS_PATTERN_EDITOR (widget));
  
  pe = BST_PATTERN_EDITOR (widget);
  
  pe->char_width = 0;
  for (i = 0; i < 256; i++)
    {
      register guint width;
      
      width = gdk_char_width (widget->style->font, i);
      pe->char_width = MAX (pe->char_width, width);
    }
  pe->char_height = FONT_HEIGHT (widget->style->font);
  pe->char_descent = widget->style->font->descent;
  gtk_widget_queue_resize (widget);
}

static void
bst_pattern_editor_state_changed (GtkWidget	 *widget,
				  guint		  previous_state)
{
  BstPatternEditor *pe;
  
  g_return_if_fail (widget != NULL);
  g_return_if_fail (BST_IS_PATTERN_EDITOR (widget));
  
  pe = BST_PATTERN_EDITOR (widget);
  
  if (GTK_WIDGET_REALIZED (pe))
    {
      gtk_style_set_background (widget->style, widget->window, GTK_STATE_INSENSITIVE);
      gdk_window_clear (widget->window);
      gtk_style_set_background (widget->style, pe->index, GTK_WIDGET_STATE (pe));
      gdk_window_clear (pe->index);
      gtk_style_set_background (widget->style, pe->headline, GTK_WIDGET_STATE (pe));
      gdk_window_clear (pe->headline);
      gtk_style_set_background (widget->style, pe->panel, GTK_WIDGET_STATE (pe));
      gdk_window_clear (pe->panel);
    }
}

GtkWidget*
bst_pattern_editor_new (BsePattern  *pattern)
{
  GtkWidget *widget;
  
  widget = gtk_type_new (bst_pattern_editor_get_type ());
  
  if (pattern)
    bst_pattern_editor_set_pattern (BST_PATTERN_EDITOR (widget), pattern);
  
  return widget;
}

static void
bst_pattern_editor_release_pattern (BstPatternEditor	   *pe)
{
  g_return_if_fail (pe != NULL);
  g_return_if_fail (BST_IS_PATTERN_EDITOR (pe));
  g_return_if_fail (pe->pattern != NULL);

  gtk_signal_disconnect_by_func (GTK_OBJECT (pe->pattern->song),
				 GTK_SIGNAL_FUNC (note_changed),
				 pe);
  pe->pattern = NULL;
  g_free (pe->instruments);
  pe->instruments = NULL;
}

void
bst_pattern_editor_set_pattern (BstPatternEditor       *pe,
				BsePattern	       *pattern)
{
  g_return_if_fail (pe != NULL);
  g_return_if_fail (BST_IS_PATTERN_EDITOR (pe));
  g_return_if_fail (pattern != NULL);
  
  if (pe->pattern != pattern)
    {
      guint i;
      GList *list;

      if (pe->pattern)
	bst_pattern_editor_release_pattern (pe);

      /* FIXME: we need some kind of destroy notification, and then??? */
      pe->pattern = pattern;
      gtk_signal_connect_object (GTK_OBJECT (pe->pattern->song),
				 "note-changed",
				 GTK_SIGNAL_FUNC (note_changed),
				 GTK_OBJECT (pe));
      
      pe->instruments = g_new (BseInstrument*, pe->pattern->n_channels);
      for (i = 0, list = pe->pattern->song->instruments; list; list = list->next, i++)
	pe->instruments[i] = list->data;
      while (i < pe->pattern->n_channels)
	pe->instruments[i++] = NULL;
      
      pe->focus_channel = MIN (pe->focus_channel, pe->pattern->n_channels - 1);
      pe->focus_row = MIN (pe->focus_row, pe->pattern->n_rows - 1);
      gtk_widget_queue_resize (GTK_WIDGET (pe));
    }
}

void
note_changed (BstPatternEditor	*pe,
	      BsePattern	*pattern,
	      guint		 channel,
	      guint		 row)
{
  g_return_if_fail (pe != NULL);
  g_return_if_fail (BST_IS_PATTERN_EDITOR (pe));
  g_return_if_fail (pattern != NULL);
  
  bst_pattern_editor_draw_panel (pe, channel, row, TRUE);
  if (channel == pe->focus_channel &&
      row == pe->focus_row)
    bst_pattern_editor_draw_cell_focus (GTK_WIDGET (pe));
}

static void
bst_pattern_editor_size_request	(GtkWidget		*widget,
				 GtkRequisition		*requisition)
{
  BstPatternEditor *pe;
  guint x;
  guint y;
  guint width;
  guint height;
  
  g_return_if_fail (widget != NULL);
  g_return_if_fail (BST_IS_PATTERN_EDITOR (widget));
  g_return_if_fail (requisition != NULL);
  
  pe = BST_PATTERN_EDITOR (widget);
  
  gtk_widget_size_request (pe->vscrollbar, &pe->vscrollbar->requisition);
  gtk_widget_size_request (pe->hscrollbar, &pe->hscrollbar->requisition);
  
  bst_pattern_editor_allocate_tone (pe,
				    pe->pattern->n_channels - 1,
				    pe->pattern->n_rows - 1,
				    &x, &y,
				    &width, &height);
  pe->plain_width = INDEX_SA_X (pe) + INDEX_SA_WIDTH (pe) + RIGHT_BAR (pe) + X_BORDER (pe);
  pe->plain_height = HEADLINE_SA_Y (pe) + HEADLINE_SA_HEIGHT (pe) + BOTTOM_BAR (pe) + Y_BORDER (pe);
  pe->panel_width = x + width;
  pe->panel_height = y + height;
  requisition->width = MIN (pe->plain_width + pe->panel_width / 2, MAX_WIDTH);
  requisition->height = MIN (pe->plain_height + pe->panel_height / 2, MAX_HEIGHT);
}


void
bst_pattern_editor_allocate_tone (BstPatternEditor	*pe,
				  guint			 channel,
				  guint			 row,
				  guint			*x_p,
				  guint			*y_p,
				  guint			*width_p,
				  guint			*height_p)
{
  guint instrument_width;
  guint note_width;
  guint effect_width;
  guint channel_width;
  
  g_return_if_fail (pe != NULL);
  g_return_if_fail (BST_IS_PATTERN_EDITOR (pe));
  
  instrument_width = pe->char_width * INSTRUMENT_DIGITS;
  note_width = pe->char_width * NOTE_DIGITS;
  effect_width = EFFECT_WIDTH (pe);
  channel_width = (instrument_width + CHAR_SPACE +
		   note_width + CHAR_SPACE +
		   effect_width);
  if (x_p)
    *x_p = channel * (channel_width + CHANNEL_SPACE);
  if (y_p)
    *y_p = row * (pe->char_height + LINE_SPACE);
  if (height_p)
    *height_p = pe->char_height + LINE_SPACE;
  if (width_p)
    *width_p = channel_width + CHANNEL_SPACE;
}

void
bst_pattern_editor_offset_cell (BstPatternEditor *pe,
				BstCellType	  cell_type,
				guint		 *x_p,
				guint		 *y_p,
				guint		 *width_p,
				guint		 *height_p)
{
  guint note_width;
  guint instrument_width;
  guint effect_width;
  
  g_return_if_fail (pe != NULL);
  g_return_if_fail (BST_IS_PATTERN_EDITOR (pe));
  
  note_width = pe->char_width * NOTE_DIGITS;
  instrument_width = pe->char_width * INSTRUMENT_DIGITS;
  effect_width = EFFECT_WIDTH (pe);
  
  if (y_p)
    *y_p += LINE_SPACE / 2 + pe->char_height - pe->char_descent;
  if (height_p)
    *height_p = pe->char_height;
  
  switch (cell_type)
    {
    case  BST_CELL_NOTE:
      if (width_p)
	*width_p = note_width;
      if (x_p)
	*x_p += CHANNEL_SPACE / 2;
      break;
    case  BST_CELL_INSTRUMENT:
      if (width_p)
	*width_p = instrument_width;
      if (x_p)
	*x_p += CHANNEL_SPACE / 2 + note_width + CHAR_SPACE;
      break;
    case  BST_CELL_EFFECT:
      if (width_p)
	*width_p = effect_width;
      if (x_p)
	*x_p += CHANNEL_SPACE / 2 + note_width + CHAR_SPACE + instrument_width + CHAR_SPACE;
      break;
    default:
      if (x_p) *x_p = ~0;
      if (width_p) *width_p = ~0;
      break;
    }
}

gint
bst_pattern_editor_get_cell (BstPatternEditor	*pe,
			     gint		 loc_x,
			     gint		 loc_y,
			     BstCellType	 *cell_type_p,
			     guint		 *channel_p,
			     guint		 *row_p)
{
  BsePattern *pattern;
  guint x;
  guint y;
  guint instrument_width;
  guint note_width;
  guint effect_width;
  guint channel_width;
  guint total_width;
  guint total_height;
  guint row;
  guint channel;
  BstCellType cell_type;
  gboolean check_cell;
  gboolean within;
  
  if (cell_type_p) *cell_type_p = ~0;
  if (channel_p) *channel_p = ~0;
  if (row_p) *row_p = ~0;
  
  g_return_val_if_fail (pe != NULL, FALSE);
  g_return_val_if_fail (BST_IS_PATTERN_EDITOR (pe), FALSE);
  
  pattern = pe->pattern;
  
  x = 0;
  y = 0;
  instrument_width = pe->char_width * INSTRUMENT_DIGITS;
  note_width = pe->char_width * NOTE_DIGITS;
  effect_width = EFFECT_WIDTH (pe);
  channel_width = (instrument_width + CHAR_SPACE +
		   note_width + CHAR_SPACE +
		   effect_width);
  total_width = pattern->n_channels * (channel_width + CHANNEL_SPACE);
  total_height = pattern->n_rows * (pe->char_height + LINE_SPACE);
  
  /* let's go!
   */
  cell_type = BST_CELL_INVALID;
  check_cell = TRUE;
  within = TRUE;
  
  /* figure row and strip to begin of first row
   */
  if (loc_y > y + total_height)
    {
      row = pattern->n_rows - 1;
      check_cell = FALSE;
      within = FALSE;
    }
  else if (loc_y > y)
    {
      loc_y -= y;
      row = loc_y / (pe->char_height + LINE_SPACE);
      
      if (row >= pe->pattern->n_rows)
	{
	  row = pe->pattern->n_rows - 1;
	  check_cell = FALSE;
	  within = FALSE;
	}
      
      /* check for row spaces
       */
      if (check_cell)
	{
	  guint offset;
	  
	  offset = loc_y % (pe->char_height + LINE_SPACE);
	  if (offset <= LINE_SPACE / 2 ||
	      offset > LINE_SPACE / 2 + pe->char_height)
	    check_cell = FALSE;
	}
    }
  else
    {
      row = 0;
      check_cell = FALSE;
      within = FALSE;
    }
  
  /* figure channel and strip to begin of first channel
   */
  if (loc_x > x + total_width)
    {
      channel = pattern->n_channels - 1;
      check_cell = FALSE;
      within = FALSE;
    }
  else if (loc_x > x)
    {
      loc_x -= x;
      channel = loc_x / (channel_width + CHANNEL_SPACE);
      
      if (channel >= pe->pattern->n_channels)
	{
	  channel = pe->pattern->n_channels - 1;
	  check_cell = FALSE;
	  within = FALSE;
	}
      
      /* check for channel spaces
       */
      if (check_cell)
	{
	  guint offset;
	  
	  offset = loc_x % (channel_width + CHANNEL_SPACE);
	  if (offset <= CHANNEL_SPACE / 2 ||
	      offset > CHANNEL_SPACE / 2 + channel_width)
	    check_cell = FALSE;
	}
    }
  else
    {
      channel = 0;
      check_cell = FALSE;
      within = FALSE;
    }
  
  /* remove channel offset
   */
  if (check_cell)
    {
      loc_x -= channel * (channel_width + CHANNEL_SPACE);
      x = CHANNEL_SPACE / 2;
    }
  
  /* check for note cell
   */
  if (check_cell &&
      loc_x >= x &&
      loc_x <= x + note_width)
    {
      cell_type = BST_CELL_NOTE;
      check_cell = FALSE;
    }
  else
    x += note_width + CHAR_SPACE;
  
  /* check for instrument cell
   */
  if (check_cell &&
      loc_x >= x &&
      loc_x <= x + instrument_width)
    {
      cell_type = BST_CELL_INSTRUMENT;
      check_cell = FALSE;
    }
  else
    x += instrument_width + CHAR_SPACE;
  
  /* check for effect cell
   */
  if (check_cell &&
      loc_x >= x &&
      loc_x <= x + effect_width)
    {
      cell_type = BST_CELL_EFFECT;
      check_cell = FALSE;
    }
  else
    x += effect_width + CHAR_SPACE;
  
  /* return values
   */
  if (cell_type_p)
    *cell_type_p = cell_type;
  if (channel_p)
    *channel_p = channel;
  if (row_p)
    *row_p = row;
  
  /* printf ("CELL-GET: within=%d channel=%d row=%d\n", within, channel, row); */
  
  return within;
}

static void
bst_pattern_editor_size_allocate (GtkWidget		*widget,
				  GtkAllocation		*allocation)
{
  BstPatternEditor *pe;
  guint min_panel_width;
  guint min_panel_height;
  GtkAllocation bar_allocation;
  
  g_return_if_fail (widget != NULL);
  g_return_if_fail (BST_IS_PATTERN_EDITOR (widget));
  g_return_if_fail (allocation != NULL);
  
  pe = BST_PATTERN_EDITOR (widget);
  
  bst_pattern_editor_allocate_tone (pe, 0, 0, NULL, NULL, &min_panel_width, &min_panel_height);
  widget->allocation.x = allocation->x;
  widget->allocation.y = allocation->y;
  widget->allocation.width = MIN (pe->plain_width + pe->panel_width,
				  MAX (pe->plain_width + min_panel_width * 1.5,
				       allocation->width));
  widget->allocation.height = MIN (pe->plain_height + pe->panel_height,
				   MAX (pe->plain_height + min_panel_height * 1.5,
					allocation->height));
  
  GTK_RANGE (pe->vscrollbar)->adjustment->page_size = PANEL_SA_HEIGHT (pe);
  GTK_RANGE (pe->vscrollbar)->adjustment->page_increment = PANEL_SA_HEIGHT (pe) / 2;
  GTK_RANGE (pe->vscrollbar)->adjustment->step_increment = min_panel_height * 4;
  GTK_RANGE (pe->vscrollbar)->adjustment->lower = 0;
  GTK_RANGE (pe->vscrollbar)->adjustment->upper = pe->panel_height;
  bar_allocation.x = widget->allocation.width - RIGHT_BAR (pe) - X_BORDER (pe);
  bar_allocation.y = PANEL_SA_Y (pe);
  bar_allocation.width = RIGHT_BAR (pe);
  bar_allocation.height = PANEL_SA_HEIGHT (pe);
  gtk_widget_size_allocate (pe->vscrollbar, &bar_allocation);
  
  GTK_RANGE (pe->hscrollbar)->adjustment->page_size = PANEL_SA_WIDTH (pe);
  GTK_RANGE (pe->hscrollbar)->adjustment->page_increment = PANEL_SA_WIDTH (pe) / 2;
  GTK_RANGE (pe->hscrollbar)->adjustment->step_increment = min_panel_width;
  GTK_RANGE (pe->hscrollbar)->adjustment->lower = 0;
  GTK_RANGE (pe->hscrollbar)->adjustment->upper = pe->panel_width;
  bar_allocation.x = PANEL_SA_X (pe);
  bar_allocation.y = widget->allocation.height - BOTTOM_BAR (pe) - Y_BORDER (pe);
  bar_allocation.width = PANEL_SA_WIDTH (pe);
  bar_allocation.height = BOTTOM_BAR (pe);
  gtk_widget_size_allocate (pe->hscrollbar, &bar_allocation);
  
  if (GTK_WIDGET_REALIZED (widget))
    {
      gdk_window_move_resize (widget->window,
			      widget->allocation.x,
			      widget->allocation.y,
			      widget->allocation.width,
			      widget->allocation.height);
      gdk_window_move_resize (pe->index_sa,
			      INDEX_SA_X (pe),
			      INDEX_SA_Y (pe),
			      INDEX_SA_WIDTH (pe),
			      INDEX_SA_HEIGHT (pe));
      gdk_window_move_resize (pe->headline_sa,
			      HEADLINE_SA_X (pe),
			      HEADLINE_SA_Y (pe),
			      HEADLINE_SA_WIDTH (pe),
			      HEADLINE_SA_HEIGHT (pe));
      gdk_window_move_resize (pe->panel_sa,
			      PANEL_SA_X (pe),
			      PANEL_SA_Y (pe),
			      PANEL_SA_WIDTH (pe),
			      PANEL_SA_HEIGHT (pe));
      gdk_window_resize (pe->index,
			 INDEX_SA_WIDTH (pe),
			 pe->panel_height);
      gdk_window_resize (pe->headline,
			 pe->panel_width,
			 HEADLINE_SA_HEIGHT (pe));
      gdk_window_resize (pe->panel,
			 pe->panel_width,
			 pe->panel_height);
    }
  
  bst_pattern_editor_adjust_sas (pe, TRUE);
}

static void
adjustments_value_changed (GtkAdjustment    *adjustment,
			   BstPatternEditor *pe)
{
  if (GTK_WIDGET_REALIZED (pe))
    {
      gdk_window_move (pe->index,
		       0,
		       - GTK_RANGE (pe->vscrollbar)->adjustment->value);
      gdk_window_move (pe->headline,
		       - GTK_RANGE (pe->hscrollbar)->adjustment->value,
		       0);
      gdk_window_move (pe->panel,
		       - GTK_RANGE (pe->hscrollbar)->adjustment->value,
		       - GTK_RANGE (pe->vscrollbar)->adjustment->value);
    }
}

void
bst_pattern_editor_adjust_sas (BstPatternEditor *pe,
			       gboolean		 check_bounds)
{
  GtkWidget *widget;
  gfloat oh_value;
  gfloat ov_value;
  
  g_return_if_fail (pe != NULL);
  g_return_if_fail (BST_IS_PATTERN_EDITOR (pe));
  
  check_bounds = check_bounds != 0;
  
  widget = GTK_WIDGET (pe);
  
  ov_value = GTK_RANGE (pe->vscrollbar)->adjustment->value;
  oh_value = GTK_RANGE (pe->hscrollbar)->adjustment->value;
  do
    {
      gfloat h_value;
      gfloat v_value;
      guint h_pos;
      guint v_pos;
      guint width;
      guint height;
      
      bst_pattern_editor_allocate_tone (pe, 0, 0, NULL, NULL, &width, &height);
      if (check_bounds)
	{
	  h_pos = 0;
	  v_pos = 0;
	}
      else
	{
	  h_pos = pe->focus_channel * width;
	  v_pos = pe->focus_row * height;
	}
      v_value = GTK_RANGE (pe->vscrollbar)->adjustment->value;
      h_value = GTK_RANGE (pe->hscrollbar)->adjustment->value;
      
      if (v_value > v_pos)
	GTK_RANGE (pe->vscrollbar)->adjustment->value = v_pos;
      if (v_value + PANEL_SA_HEIGHT (pe) < v_pos + height)
	GTK_RANGE (pe->vscrollbar)->adjustment->value = v_pos + height - PANEL_SA_HEIGHT (pe);
      if (h_value > h_pos)
	GTK_RANGE (pe->hscrollbar)->adjustment->value = h_pos;
      if (h_value + PANEL_SA_WIDTH (pe) < h_pos + width)
	GTK_RANGE (pe->hscrollbar)->adjustment->value = h_pos + width - PANEL_SA_WIDTH (pe);
    }
  while (check_bounds-- > 0);
  
  if (ov_value != GTK_RANGE (pe->vscrollbar)->adjustment->value)
    gtk_signal_emit_by_name (GTK_OBJECT (GTK_RANGE (pe->vscrollbar)->adjustment), "value_changed");
  if (oh_value != GTK_RANGE (pe->hscrollbar)->adjustment->value)
    gtk_signal_emit_by_name (GTK_OBJECT (GTK_RANGE (pe->hscrollbar)->adjustment), "value_changed");
}

static void
bst_pattern_editor_realize (GtkWidget		*widget)
{
  BstPatternEditor *pe;
  GdkWindowAttr attributes;
  gint attributes_mask;
  
  g_return_if_fail (widget != NULL);
  g_return_if_fail (BST_IS_PATTERN_EDITOR (widget));
  
  pe = BST_PATTERN_EDITOR (widget);
  
  GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);
  
  attributes.window_type = GDK_WINDOW_CHILD;
  
  attributes.wclass = GDK_INPUT_OUTPUT;
  attributes.visual = gtk_widget_get_visual (widget);
  attributes.colormap = gtk_widget_get_colormap (widget);
  attributes.event_mask = gtk_widget_get_events (widget);
  
  attributes.x = widget->allocation.x;
  attributes.y = widget->allocation.y;
  attributes.width = widget->allocation.width;
  attributes.height = widget->allocation.height;
  attributes.event_mask |= (GDK_EXPOSURE_MASK |
			    GDK_ENTER_NOTIFY_MASK |
			    GDK_LEAVE_NOTIFY_MASK);
  attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;
  widget->window = gdk_window_new (gtk_widget_get_parent_window (widget), &attributes, attributes_mask);
  gdk_window_set_user_data (widget->window, pe);
  
  attributes.x = INDEX_SA_X (pe);
  attributes.y = INDEX_SA_Y (pe);
  attributes.width = INDEX_SA_WIDTH (pe);
  attributes.height = INDEX_SA_HEIGHT (pe);
  attributes.wclass = GDK_INPUT_OUTPUT;
  attributes.event_mask = 0;
  attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;
  pe->index_sa = gdk_window_new (widget->window, &attributes, attributes_mask);
  gdk_window_set_user_data (pe->index_sa, pe);
  
  attributes.x = 0;
  attributes.y = 0;
  attributes.width = INDEX_SA_WIDTH (pe);
  attributes.height = pe->panel_height;
  attributes.wclass = GDK_INPUT_OUTPUT;
  attributes.event_mask = gtk_widget_get_events (widget);
  attributes.event_mask |= (GDK_EXPOSURE_MASK |
			    GDK_BUTTON_PRESS_MASK |
			    GDK_BUTTON_RELEASE_MASK);
  attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;
  pe->index = gdk_window_new (pe->index_sa, &attributes, attributes_mask);
  gdk_window_set_user_data (pe->index, pe);
  
  attributes.x = HEADLINE_SA_X (pe);
  attributes.y = HEADLINE_SA_Y (pe);
  attributes.width = HEADLINE_SA_WIDTH (pe);
  attributes.height = HEADLINE_SA_HEIGHT (pe);
  attributes.wclass = GDK_INPUT_OUTPUT;
  attributes.event_mask = 0;
  attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;
  pe->headline_sa = gdk_window_new (widget->window, &attributes, attributes_mask);
  gdk_window_set_user_data (pe->headline_sa, pe);
  
  attributes.x = 0;
  attributes.y = 0;
  attributes.width = pe->panel_width;
  attributes.height = HEADLINE_SA_HEIGHT (pe);
  attributes.wclass = GDK_INPUT_OUTPUT;
  attributes.event_mask = gtk_widget_get_events (widget);
  attributes.event_mask |= (GDK_EXPOSURE_MASK |
			    GDK_BUTTON_PRESS_MASK |
			    GDK_BUTTON_RELEASE_MASK);
  attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;
  pe->headline = gdk_window_new (pe->headline_sa, &attributes, attributes_mask);
  gdk_window_set_user_data (pe->headline, pe);
  
  attributes.x = PANEL_SA_X (pe);
  attributes.y = PANEL_SA_Y (pe);
  attributes.width = PANEL_SA_WIDTH (pe);
  attributes.height = PANEL_SA_HEIGHT (pe);
  attributes.wclass = GDK_INPUT_OUTPUT;
  attributes.event_mask = 0;
  attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;
  pe->panel_sa = gdk_window_new (widget->window, &attributes, attributes_mask);
  gdk_window_set_user_data (pe->panel_sa, pe);
  
  attributes.x = 0;
  attributes.y = 0;
  attributes.width = pe->panel_width;
  attributes.height = pe->panel_height;
  attributes.wclass = GDK_INPUT_OUTPUT;
  attributes.event_mask = gtk_widget_get_events (widget);
  attributes.event_mask |= (GDK_EXPOSURE_MASK |
			    GDK_BUTTON_PRESS_MASK |
			    GDK_BUTTON_RELEASE_MASK |
			    GDK_KEY_PRESS_MASK |
			    GDK_KEY_RELEASE_MASK);
  attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;
  pe->panel = gdk_window_new (pe->panel_sa, &attributes, attributes_mask);
  gdk_window_set_user_data (pe->panel, pe);
  
  widget->style = gtk_style_attach (widget->style, widget->window);
  gtk_style_set_background (widget->style, widget->window, GTK_STATE_INSENSITIVE);
  gtk_style_set_background (widget->style, pe->index_sa, GTK_STATE_INSENSITIVE);
  gtk_style_set_background (widget->style, pe->index, GTK_WIDGET_STATE (pe));
  gtk_style_set_background (widget->style, pe->headline_sa, GTK_STATE_INSENSITIVE);
  gtk_style_set_background (widget->style, pe->headline, GTK_WIDGET_STATE (pe));
  gtk_style_set_background (widget->style, pe->panel_sa, GTK_STATE_INSENSITIVE);
  gtk_style_set_background (widget->style, pe->panel, GTK_WIDGET_STATE (pe));
  
  bst_pattern_editor_adjust_sas (pe, FALSE);
}

static void
bst_pattern_editor_unrealize (GtkWidget		*widget)
{
  BstPatternEditor *pe;
  
  g_return_if_fail (widget != NULL);
  g_return_if_fail (BST_IS_PATTERN_EDITOR (widget));
  
  pe = BST_PATTERN_EDITOR (widget);
  
  GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);
  
  gdk_window_set_user_data (pe->index_sa, NULL);
  gdk_window_destroy (pe->index_sa);
  pe->index_sa = NULL;
  gdk_window_set_user_data (pe->index, NULL);
  gdk_window_destroy (pe->index);
  pe->index = NULL;
  gdk_window_set_user_data (pe->headline_sa, NULL);
  gdk_window_destroy (pe->headline_sa);
  pe->headline_sa = NULL;
  gdk_window_set_user_data (pe->headline, NULL);
  gdk_window_destroy (pe->headline);
  pe->headline = NULL;
  gdk_window_set_user_data (pe->panel_sa, NULL);
  gdk_window_destroy (pe->panel_sa);
  pe->panel_sa = NULL;
  gdk_window_set_user_data (pe->panel, NULL);
  gdk_window_destroy (pe->panel);
  pe->panel = NULL;
  
  if (GTK_WIDGET_CLASS (parent_class)->unrealize)
    (* GTK_WIDGET_CLASS (parent_class)->unrealize) (widget);
}

static void
bst_pattern_editor_map (GtkWidget *widget)
{
  BstPatternEditor *pe;
  
  g_return_if_fail (widget != NULL);
  g_return_if_fail (BST_IS_PATTERN_EDITOR (widget));
  
  pe = BST_PATTERN_EDITOR (widget);
  
  GTK_WIDGET_SET_FLAGS (widget, GTK_MAPPED);
  
  gdk_window_show (widget->window);
  gdk_window_show (pe->headline_sa);
  gdk_window_show (pe->headline);
  gdk_window_show (pe->index_sa);
  gdk_window_show (pe->index);
  gdk_window_show (pe->panel_sa);
  gdk_window_show (pe->panel);
  
  if (GTK_WIDGET_VISIBLE (pe->vscrollbar) &&
      !GTK_WIDGET_MAPPED (pe->vscrollbar))
    gtk_widget_map (pe->vscrollbar);
  if (GTK_WIDGET_VISIBLE (pe->hscrollbar) &&
      !GTK_WIDGET_MAPPED (pe->hscrollbar))
    gtk_widget_map (pe->hscrollbar);
}

static void
bst_pattern_editor_unmap (GtkWidget *widget)
{
  BstPatternEditor *pe;
  
  g_return_if_fail (widget != NULL);
  g_return_if_fail (BST_IS_PATTERN_EDITOR (widget));
  
  pe = BST_PATTERN_EDITOR (widget);
  
  GTK_WIDGET_UNSET_FLAGS (widget, GTK_MAPPED);
  
  gdk_window_hide (widget->window);
  
  if (GTK_WIDGET_MAPPED (pe->vscrollbar))
    gtk_widget_unmap (pe->vscrollbar);
  if (GTK_WIDGET_MAPPED (pe->hscrollbar))
    gtk_widget_unmap (pe->hscrollbar);
}

static gint
bst_pattern_editor_expose (GtkWidget	  *widget,
			   GdkEventExpose *event)
{
  BstPatternEditor *pe;
  
  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (BST_IS_PATTERN_EDITOR (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);
  
  pe = BST_PATTERN_EDITOR (widget);
  
  if (GTK_WIDGET_DRAWABLE (pe))
    {
      guint b_c;
      guint e_c;
      guint b_r;
      guint e_r;
      guint c;
      guint r;
      
      bst_pattern_editor_get_cell (pe,
				   event->area.x,
				   event->area.y,
				   NULL,
				   &b_c,
				   &b_r);
      bst_pattern_editor_get_cell (pe,
				   event->area.x + event->area.width,
				   event->area.y + event->area.height,
				   NULL,
				   &e_c,
				   &e_r);
      
      if (event->window == pe->panel)
	{
	  for (c = b_c; c <= e_c; c++)
	    for (r = b_r; r <= e_r; r++)
	      bst_pattern_editor_draw_panel (pe, c, r, FALSE);
	  gtk_widget_draw_focus (widget);
	}
      else if (event->window == pe->index)
	{
	  for (r = b_r; r <= e_r; r++)
	    bst_pattern_editor_draw_index (pe, r, FALSE);
	}
      else if (event->window == pe->headline)
	{
	  for (c = b_c; c <= e_c; c++)
	    bst_pattern_editor_draw_head (pe, c, FALSE);
	}
      else if (event->window == widget->window)
	{
	  /* FIXME
	     g_print ("this damn expose event:\n  send=%d\n  count=%d\n	 x=%d y=%d\n  width=%d height=%d\n",
	     event->send_event, event->count, event->area.x, event->area.y, event->area.width, event->area.height);
	  */
	  bst_pattern_editor_draw_main (pe);
	  gtk_widget_draw_focus (widget);
	}
      else
	{
	  g_warning ("BstPatternEditor: unknown expose event, window=%p, widget=%p (%s)",
		     event->window,
		     gtk_get_event_widget ((GdkEvent*) event),
		     gtk_widget_get_name (gtk_get_event_widget ((GdkEvent*) event)));
	}
    }
  
  return TRUE;
}

static void
bst_pattern_editor_draw (GtkWidget		*widget,
			 GdkRectangle		*area)
{
  BstPatternEditor *pe;
  
  g_return_if_fail (widget != NULL);
  g_return_if_fail (BST_IS_PATTERN_EDITOR (widget));
  g_return_if_fail (area != NULL);
  
  pe = BST_PATTERN_EDITOR (widget);
  
  if (GTK_WIDGET_DRAWABLE (widget))
    {
      gdk_window_clear_area_e (widget->window,
			       0, 0,
			       widget->allocation.width,
			       widget->allocation.height);
      gdk_window_clear_area_e (pe->index,
			       0, 0,
			       INDEX_SA_WIDTH (pe),
			       pe->panel_height);
      gdk_window_clear_area_e (pe->headline,
			       0, 0,
			       pe->panel_width,
			       HEADLINE_SA_HEIGHT (pe));
      gdk_window_clear_area_e (pe->panel,
			       0, 0,
			       pe->panel_width,
			       pe->panel_height);
      
      gtk_widget_draw (pe->vscrollbar, NULL);
      gtk_widget_draw (pe->hscrollbar, NULL);
    }
}

static gint
bst_pattern_editor_focus_in (GtkWidget		    *widget,
			     GdkEventFocus	    *event)
{
  BstPatternEditor *pe;
  
  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (BST_IS_PATTERN_EDITOR (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);
  
  pe = BST_PATTERN_EDITOR (widget);
  GTK_WIDGET_SET_FLAGS (pe, GTK_HAS_FOCUS);
  
  gtk_widget_draw_focus (widget);
  
  return TRUE;
}

static gint
bst_pattern_editor_focus_out (GtkWidget		     *widget,
			      GdkEventFocus	     *event)
{
  BstPatternEditor *pe;
  
  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (BST_IS_PATTERN_EDITOR (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);
  
  pe = BST_PATTERN_EDITOR (widget);
  GTK_WIDGET_UNSET_FLAGS (pe, GTK_HAS_FOCUS);
  
  gtk_widget_draw_focus (widget);
  
  return TRUE;
}

static void
bst_pattern_editor_draw_focus (GtkWidget *widget)
{
  BstPatternEditor *pe;
  
  g_return_if_fail (widget != NULL);
  g_return_if_fail (BST_IS_PATTERN_EDITOR (widget));
  
  pe = BST_PATTERN_EDITOR (widget);
  
  if (GTK_WIDGET_DRAWABLE (pe))
    {
      bst_pattern_editor_draw_main (pe);
      bst_pattern_editor_draw_cell_focus (widget);
    }
}

static void
bst_pattern_editor_draw_cell_focus (GtkWidget *widget)
{
  BstPatternEditor *pe;
  
  g_return_if_fail (widget != NULL);
  g_return_if_fail (BST_IS_PATTERN_EDITOR (widget));
  
  pe = BST_PATTERN_EDITOR (widget);
  
  if (GTK_WIDGET_DRAWABLE (pe))
    {
      guint x;
      guint y;
      guint width;
      guint height;
      
      /* draw focused tone
       */
      bst_pattern_editor_allocate_tone (pe,
					pe->focus_channel,
					pe->focus_row,
					&x, &y,
					&width, &height);
      if (GTK_WIDGET_HAS_FOCUS (pe) && pe->cell_focus)
	{
	  gdk_draw_rectangle (pe->panel,
			      widget->style->base_gc[GTK_STATE_NORMAL],
			      FALSE,
			      x + 1, y,
			      width - 2, height - 2);
	}
      else
	{
	  gdk_draw_rectangle (pe->panel,
			      widget->style->bg_gc[GTK_WIDGET_STATE (pe)],
			      FALSE,
			      x + 1, y,
			      width - 2, height - 2);
	}
      /*
       * gdk_window_clear_area (pe->panel, x, y, width, height);
       * bst_pattern_editor_draw_panel (pe, pe->focus_channel, pe->focus_row);
       */
    }
}

static void
bst_pattern_editor_draw_main (BstPatternEditor	*pe)
{
  GtkWidget *widget;
  guint x;
  guint y;
  guint width;
  guint height;
  guint channel_width;
  
  widget = GTK_WIDGET (pe);
  
  channel_width = (pe->char_width * INSTRUMENT_DIGITS + CHAR_SPACE +
		   pe->char_width * NOTE_DIGITS + CHAR_SPACE +
		   EFFECT_WIDTH (pe));
  
  /* draw shadow
   */
  gtk_draw_shadow (widget->style,
		   widget->window,
		   GTK_STATE_NORMAL,
		   GTK_SHADOW_IN,
		   OUTER_X_BORDER (pe),
		   OUTER_Y_BORDER (pe),
		   widget->allocation.width - 2 * OUTER_X_BORDER (pe),
		   widget->allocation.height - 2 * OUTER_Y_BORDER (pe));
  
  /* draw widget focus
   */
  if (GTK_WIDGET_HAS_FOCUS (pe))
    {
      gdk_draw_rectangle (widget->window,
			  widget->style->base_gc[GTK_STATE_NORMAL],
			  FALSE,
			  X_BORDER (pe) - 1,
			  Y_BORDER (pe) - 1,
			  widget->allocation.width - 2 * X_BORDER (pe) + 2 * 1,
			  widget->allocation.height - 2 * Y_BORDER (pe) + 2 * 1);
    }
  else
    {
      gdk_draw_rectangle (widget->window,
			  widget->style->bg_gc[GTK_STATE_INSENSITIVE],
			  FALSE,
			  X_BORDER (pe) - 1,
			  Y_BORDER (pe) - 1,
			  widget->allocation.width - 2 * X_BORDER (pe) + 2 * 1,
			  widget->allocation.height - 2 * Y_BORDER (pe) + 2 * 1);
    }
  
  /* draw Guid
   */
  x = INDEX_SA_X (pe);
  y = HEADLINE_SA_Y (pe);
  width = INDEX_SA_WIDTH (pe) - 1;
  height = HEADLINE_SA_HEIGHT (pe);
  
  {
    gchar buffer[16];
    guint offset;
    
    if (GTK_WIDGET_IS_SENSITIVE (pe))
      gdk_draw_rectangle (widget->window,
			  widget->style->base_gc[GTK_STATE_NORMAL],
			  TRUE,
			  x,
			  y,
			  width,
			  height - 1);
    
    sprintf (buffer, "%u", bse_pattern_get_guid (pe->pattern));
    offset = MAX (0, (width - gdk_string_measure (widget->style->font, buffer))) / 2;
    gdk_draw_string (widget->window,
		     widget->style->font,
		     widget->style->fg_gc[GTK_WIDGET_IS_SENSITIVE (pe) ? GTK_STATE_SELECTED : GTK_STATE_INSENSITIVE],
		     x + offset,
		     y + pe->char_height - pe->char_descent,
		     buffer);
  }
}

static void
bst_pattern_editor_draw_index (BstPatternEditor	*pe,
			       guint		 row,
			       gboolean		 clear)
{
  GtkWidget *widget;
  guint x;
  guint y;
  guint width;
  guint height;
  gchar buffer[INDEX_DIGITS + 1];
  
  widget = GTK_WIDGET (pe);
  
  x = 0;
  y = 0;
  width = INDEX_SA_WIDTH (pe) - 1;
  height = pe->char_height + LINE_SPACE / 2;
  
  y = (pe->char_height + LINE_SPACE) * row;
  
  if (GTK_WIDGET_IS_SENSITIVE (pe))
    gdk_draw_rectangle (pe->index,
			widget->style->bg_gc[GTK_STATE_SELECTED],
			TRUE,
			x,
			y,
			width,
			height);
  else if (clear)
    gdk_draw_rectangle (pe->index,
			widget->style->bg_gc[GTK_STATE_INSENSITIVE],
			TRUE,
			x,
			y,
			width,
			height);
  
  sprintf (buffer, "%03u", row + 1);
  gdk_draw_string (pe->index,
		   widget->style->font,
		   widget->style->fg_gc[GTK_WIDGET_IS_SENSITIVE (pe) ? GTK_STATE_SELECTED : GTK_STATE_INSENSITIVE],
		   x + INDEX_BORDER,
		   y + pe->char_height - pe->char_descent,
		   buffer);
}

static void
bst_pattern_editor_draw_head (BstPatternEditor	*pe,
			      guint		 channel,
			      gboolean		 clear)
{
  GtkWidget *widget;
  guint x;
  guint y;
  guint width;
  guint height;
  gchar buffer[64];
  guint channel_width;
  guint offset;
  
  widget = GTK_WIDGET (pe);
  
  channel_width = (pe->char_width * INSTRUMENT_DIGITS + CHAR_SPACE +
		   pe->char_width * NOTE_DIGITS + CHAR_SPACE +
		   EFFECT_WIDTH (pe));
  
  y = 0;
  width = channel_width + CHANNEL_SPACE;
  x = width * channel;
  height = pe->char_height;
  
  
  if (GTK_WIDGET_IS_SENSITIVE (pe))
    gdk_draw_rectangle (pe->headline,
			widget->style->bg_gc[GTK_STATE_SELECTED],
			TRUE,
			x + 1,
			y,
			width - (channel < pe->pattern->n_channels - 1 ? 2 : 1),
			height - 1);
  else if (clear)
    gdk_draw_rectangle (pe->headline,
			widget->style->bg_gc[GTK_STATE_INSENSITIVE],
			TRUE,
			x + 1,
			y,
			width - (channel < pe->pattern->n_channels - 1 ? 2 : 1),
			height - 1);
  
  sprintf (buffer, "Channel %u", channel + 1);
  offset = MAX (0, (channel_width + CHANNEL_SPACE -
		    gdk_string_measure (widget->style->font, buffer))) / 2;
  gdk_draw_string (pe->headline,
		   widget->style->font,
		   widget->style->fg_gc[GTK_WIDGET_IS_SENSITIVE (pe) ? GTK_STATE_SELECTED : GTK_STATE_INSENSITIVE],
		   x + offset,
		   y + pe->char_height - pe->char_descent,
		   buffer);
}

static void
bst_pattern_editor_draw_panel (BstPatternEditor	*pe,
			       guint		 channel,
			       guint		 row,
			       gboolean		 clear)
{
  GtkWidget *widget;
  guint x;
  guint y;
  guint width;
  guint height;
  guint c_x;
  guint c_y;
  GtkStateType state;
  gchar buffer[64];
  gchar *p;
  BseNote *note;
  BseInstrument *instrument;
  
  g_return_if_fail (channel < pe->pattern->n_channels);
  g_return_if_fail (row < pe->pattern->n_rows);
  
  widget = GTK_WIDGET (pe);
  
  state = GTK_WIDGET_STATE (pe);
  
  /* fetch allocation
   */
  bst_pattern_editor_allocate_tone (pe,
				    channel,
				    row,
				    &x, &y,
				    &width, &height);
  if (clear)
    {
      if (channel == 0 &&
	  row == pe->last_row)
	gdk_draw_rectangle (pe->panel,
			    widget->style->base_gc[GTK_STATE_NORMAL],
			    TRUE,
			    x,
			    y,
			    width,
			    height);
      else
	gdk_window_clear_area (pe->panel, x, y, width, height);
    }
  
  /* draw channel side bar
   * dark_gc[state]?
   */
  gdk_draw_line (pe->panel,
		 widget->style->base_gc[GTK_STATE_INSENSITIVE],
		 x,
		 y,
		 x,
		 y + height - 1);
  
  /* draw channel underline
   */
  if (row % 4 == 3 &&
      row < pe->pattern->n_rows - 1)
    gdk_draw_line (pe->panel,
		   widget->style->base_gc[GTK_STATE_INSENSITIVE],
		   x,
		   y + height - 1,
		   x + width,
		   y + height - 1);
  
  /* fetch note
   */
  note = bse_pattern_get_note (pe->pattern, channel, row);
  
  /* draw note
   */
  c_x = x;
  c_y = y;
  bst_pattern_editor_offset_cell (pe,
				  BST_CELL_NOTE,
				  &c_x, &c_y,
				  NULL, NULL);
  if (note->note != BSE_NOTE_VOID)
    {
      gchar letter;
      gboolean ht_flag;
      gint octave;
      
      bse_note_examine (note->note, &octave, NULL, &ht_flag, &letter);
      
      sprintf (buffer,
	       "%c%c% d",
	       letter - ('a' - 'A'),
	       ht_flag ? '#' : ' ',
	       octave);
    }
  else
    sprintf (buffer, "----");
  
  gdk_draw_string (pe->panel,
		   widget->style->font,
		   widget->style->fg_gc[state],
		   c_x, c_y,
		   buffer);
  
  /* draw instrument
   */
  c_x = x;
  c_y = y;
  bst_pattern_editor_offset_cell (pe,
				  BST_CELL_INSTRUMENT,
				  &c_x, &c_y,
				  NULL, NULL);
  instrument = note->instrument;
  if (instrument)
    sprintf (buffer,
	     "%04d",
	     bse_instrument_get_guid (instrument));
  else
    sprintf (buffer, "----");
  buffer[4] = 0;
  p = buffer;
  while (*p == '0')
    *(p++) = '-';
  
  gdk_draw_string (pe->panel,
		   widget->style->font,
		   widget->style->fg_gc[state],
		   c_x, c_y,
		   buffer);
  
  /* draw effect
   */
  c_x = x;
  c_y = y;
  bst_pattern_editor_offset_cell (pe,
				  BST_CELL_EFFECT,
				  &c_x, &c_y,
				  NULL, NULL);
  gdk_draw_string (pe->panel,
		   widget->style->font,
		   widget->style->fg_gc[state],
		   c_x, c_y,
		   "+");
}

void
bst_pattern_editor_set_octave (BstPatternEditor	      *pe,
			       gint		       octave)
{
  gint min_octave;
  gint max_octave;
  
  g_return_if_fail (pe != NULL);
  g_return_if_fail (BST_IS_PATTERN_EDITOR (pe));
  
  bse_note_examine (BSE_MIN_NOTE, &min_octave, NULL, NULL, NULL);
  bse_note_examine (BSE_MAX_NOTE, &max_octave, NULL, NULL, NULL);
  
  pe->base_octave = CLAMP (octave, min_octave, max_octave);
  /* printf ("[%d / %d / %d] = %d\n", min_octave, octave, max_octave, pe->base_octave); */
}

void
bst_pattern_editor_set_focus (BstPatternEditor	*pe,
			      guint		 channel,
			      guint		 row)
{
  GtkWidget *widget;
  
  g_return_if_fail (pe != NULL);
  g_return_if_fail (BST_IS_PATTERN_EDITOR (pe));
  
  widget = GTK_WIDGET (pe);
  
  if (channel >= pe->pattern->n_channels)
    channel = pe->pattern->n_channels - 1;
  if (row >= pe->pattern->n_rows)
    row = pe->pattern->n_rows - 1;
  
  if (pe->cell_focus)
    {
      pe->cell_focus = FALSE;
      bst_pattern_editor_draw_cell_focus (widget);
    }
  
  pe->cell_focus = TRUE;
  pe->focus_channel = channel;
  pe->focus_row = row;
  bst_pattern_editor_draw_cell_focus (widget);
  
  bst_pattern_editor_adjust_sas (pe, FALSE);
}

static gint
bst_pattern_editor_button_press (GtkWidget	     *widget,
				 GdkEventButton	     *event)
{
  BstPatternEditor *pe;
  guint focus_channel;
  guint focus_row;
  gboolean handled;
  
  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (BST_IS_PATTERN_EDITOR (widget), FALSE);
  
  pe = BST_PATTERN_EDITOR (widget);
  
  if (!GTK_WIDGET_HAS_FOCUS (widget))
    gtk_widget_grab_focus (widget);
  
  if (event->window == pe->panel)
    {
      if (bst_pattern_editor_get_cell (pe, event->x, event->y, NULL, &focus_channel, &focus_row))
	bst_pattern_editor_set_focus (pe, focus_channel, focus_row);
      handled = TRUE;
    }
  else if (event->window == pe->headline)
    {
      guint channel;

      bst_pattern_editor_get_cell (pe, event->x, 0, NULL, &channel, NULL);
      if (event->button == 3 && channel < pe->pattern->n_channels)
	bst_pattern_editor_channel_popup (pe, channel, event->button, event->time);
      handled = TRUE;
    }
  
  handled = FALSE;
  
  return handled;
}

static gint
bst_pattern_editor_button_release (GtkWidget	       *widget,
				   GdkEventButton      *event)
{
  BstPatternEditor *pe;
  gboolean handled;
  
  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (BST_IS_PATTERN_EDITOR (widget), FALSE);
  
  pe = BST_PATTERN_EDITOR (widget);
  
  handled = FALSE;
  
  return handled;
}

static void
bst_pattern_editor_foreach (GtkContainer *container,
			    GtkCallback	  callback,
			    gpointer	  callback_data)
{
  BstPatternEditor *pe;
  
  g_return_if_fail (container != NULL);
  g_return_if_fail (BST_IS_PATTERN_EDITOR (container));
  
  pe = BST_PATTERN_EDITOR (container);
  
  if (pe->vscrollbar)
    (*callback) (pe->vscrollbar, callback_data);
  if (pe->hscrollbar)
    (*callback) (pe->hscrollbar, callback_data);
}

void
bst_pattern_editor_class_set_key (BstPatternEditorClass	 *pe_class,
				  guint16		  keyval,
				  guint16		  modifier,
				  BstPEActionType	  pe_action)
{
  g_return_if_fail (pe_class != NULL);
  g_return_if_fail (BST_PATTERN_EDITOR_CLASS (pe_class) != NULL);
  
  g_hash_table_insert (pe_class->pea_ktab,
		       (gpointer) (keyval | ((modifier & BST_MOD_SCA) << 16)),
		       (gpointer) (pe_action | BST_PEA_TAG));
}

static gint
bst_pattern_editor_key_press (GtkWidget		     *widget,
			      GdkEventKey	     *event)
{
  BstPatternEditor *pe;
  BstPEActionType pea;
  guint16 modifier;
  guint16 masks[] = {
    BST_MOD_SCA, BST_MOD_SC0, BST_MOD_S0A, BST_MOD_S00,
    BST_MOD_0CA, BST_MOD_0C0, BST_MOD_00A, BST_MOD_000,
  };
  guint n_masks = sizeof (masks) / sizeof (masks[0]);
  gpointer p;
  guint i;
  BseNote *bnote;
  BseInstrument *instrument;
  gint focus_channel;
  gint focus_row;
  guint note;
  gint difference;
  guint new_focus_channel;
  guint new_focus_row;
  
  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (BST_IS_PATTERN_EDITOR (widget), FALSE);
  
  pe = BST_PATTERN_EDITOR (widget);
  
  modifier = event->state & BST_MOD_SCA;
  p = 0;
  for (i = 0; !(((guint32) p) & BST_PEA_TAG) && i < n_masks; i++)
    if (modifier == masks[i])
      p = g_hash_table_lookup (BST_PATTERN_EDITOR_CLASS (((GtkObject*) pe)->klass)->pea_ktab,
			       (gpointer) (event->keyval | ((modifier & masks[i]) << 16)));
  
  focus_channel = pe->focus_channel;
  focus_row = pe->focus_row;
  bnote = bse_pattern_get_note (pe->pattern, focus_channel, focus_row);
  note = bnote->note;
  instrument = bnote->instrument;
  
  difference = 0;
  new_focus_channel = 0;
  new_focus_row = 0;
  
  pea = (BstPEActionType) p;
  if (pea & BST_PEA_TAG)
    {
      BstPEActionType wrap;
      
      if (pea & BST_PEA_NOTE_MASK)
	note = BSE_NOTE_GENERIC (pea & BST_PEA_NOTE_MASK, pe->base_octave);
      
      if (pea & BST_PEA_INSTRUMENT_MASK)
	{
	  if ((pea & BST_PEA_INSTRUMENT_MASK) == BST_PEA_INSTRUMENT_0F)
	    instrument = pe->instruments[focus_channel];
	  else
	    instrument = bse_song_get_instrument (pe->pattern->song,
						  (pea & BST_PEA_INSTRUMENT_MASK) >> 8);
	}

      if (pea & BST_PEA_SET_INSTRUMENT_0F)
	{
	  pe->instruments[focus_channel] = bse_song_get_instrument (pe->pattern->song,
								    (pea & BST_PEA_INSTRUMENT_MASK) >> 8);
	}
      
      switch (pea & BST_PEA_MOVE_MASK)
	{
	case  BST_PEA_MOVE_NEXT:
	  focus_channel -= (pe->next_moves_left != 0);
	  focus_channel += (pe->next_moves_right != 0);
	  focus_row -= (pe->next_moves_up != 0);
	  focus_row += (pe->next_moves_down != 0);
	  break;
	case  BST_PEA_MOVE_LEFT:
	  focus_channel--;
	  break;
	case  BST_PEA_MOVE_RIGHT:
	  focus_channel++;
	  break;
	case  BST_PEA_MOVE_UP:
	  focus_row--;
	  break;
	case  BST_PEA_MOVE_DOWN:
	  focus_row++;
	  break;
	case  BST_PEA_MOVE_PAGE_LEFT:
	  focus_channel -= pe->channel_page;
	  break;
	case  BST_PEA_MOVE_PAGE_RIGHT:
	  focus_channel += pe->channel_page;
	  break;
	case  BST_PEA_MOVE_PAGE_UP:
	  focus_row -= pe->row_page;
	  break;
	case  BST_PEA_MOVE_PAGE_DOWN:
	  focus_row += pe->row_page;
	  break;
	case  BST_PEA_MOVE_JUMP_LEFT:
	  focus_channel = 0;
	  break;
	case  BST_PEA_MOVE_JUMP_RIGHT:
	  focus_channel = pe->pattern->n_channels - 1;
	  break;
	case  BST_PEA_MOVE_JUMP_TOP:
	  focus_row = 0;
	  break;
	case  BST_PEA_MOVE_JUMP_BOTTOM:
	  focus_row = pe->pattern->n_rows - 1;
	  break;
	case  BST_PEA_MOVE_PREV_PATTERN:
	  difference = -1;
	  new_focus_channel = focus_channel;
	  new_focus_row = focus_row;
	  break;
	case  BST_PEA_MOVE_NEXT_PATTERN:
	  difference = +1;
	  new_focus_channel = focus_channel;
	  new_focus_row = focus_row;
	  break;
	}
      
      switch (pea & BST_PEA_OCTAVE_SHIFT_MASK)
	{
	case  BST_PEA_OCTAVE_SHIFT_UP:
	  if (pea & BST_PEA_AFFECT_BASE_OCTAVE)
	    bst_pattern_editor_set_octave (pe, pe->base_octave + 1);
	  else
	    note = BSE_NOTE_OCTAVE_UP (note);
	  break;
	case  BST_PEA_OCTAVE_SHIFT_DOWN:
	  if (pea & BST_PEA_AFFECT_BASE_OCTAVE)
	    bst_pattern_editor_set_octave (pe, pe->base_octave - 1);
	  else
	    note = BSE_NOTE_OCTAVE_DOWN (note);
	  break;
	case  BST_PEA_OCTAVE_SHIFT_UP2:
	  if (pea & BST_PEA_AFFECT_BASE_OCTAVE)
	    bst_pattern_editor_set_octave (pe, pe->base_octave + 2);
	  else
	    {
	      note = BSE_NOTE_OCTAVE_UP (note);
	      note = BSE_NOTE_OCTAVE_UP (note);
	    }
	  break;
	case  BST_PEA_OCTAVE_SHIFT_DOWN2:
	  if (pea & BST_PEA_AFFECT_BASE_OCTAVE)
	    bst_pattern_editor_set_octave (pe, pe->base_octave - 2);
	  else
	    {
	      note = BSE_NOTE_OCTAVE_DOWN (note);
	      note = BSE_NOTE_OCTAVE_DOWN (note);
	    }
	  break;
	default:
	  if (pea & BST_PEA_AFFECT_BASE_OCTAVE)
	    bst_pattern_editor_set_octave (pe, 0);
	}
      
      if (pea & BST_PEA_NOTE_RESET)
	note = BSE_NOTE_VOID;
      
      if (pea & BST_PEA_INSTRUMENT_RESET)
	instrument = NULL;
      
      if (pea & BST_PEA_WRAP_AS_CONFIG)
	wrap = pe->wrap_type;
      else
	wrap = pea;
      if (wrap & BST_PEA_WRAP_TO_PATTERN)
	wrap &= BST_PEA_WRAP_TO_PATTERN;
      else
	wrap &= BST_PEA_WRAP_TO_NOTE;
      
      if (focus_channel < 0)
	{
	  if (wrap == BST_PEA_WRAP_TO_NOTE)
	    focus_channel = pe->pattern->n_channels - 1;
	  else
	    focus_channel = 0;
	}
      if (focus_channel >= pe->pattern->n_channels)
	{
	  if (wrap == BST_PEA_WRAP_TO_NOTE)
	    focus_channel = 0;
	  else
	    focus_channel = pe->pattern->n_channels - 1;
	}
      if (focus_row < 0)
	{
	  if (wrap == BST_PEA_WRAP_TO_NOTE)
	    focus_row = pe->pattern->n_rows - 1;
	  else if (wrap == BST_PEA_WRAP_TO_PATTERN)
	    {
	      difference = -1;
	      new_focus_row = pe->pattern->n_rows + focus_row;
	      new_focus_channel = focus_channel;
	      focus_row = 0;
	    }
	  else
	    focus_row = 0;
	}
      if (focus_row >= pe->pattern->n_rows)
	{
	  if (wrap == BST_PEA_WRAP_TO_NOTE)
	    focus_row = 0;
	  else if (wrap == BST_PEA_WRAP_TO_PATTERN)
	    {
	      difference = +1;
	      new_focus_row = focus_row - pe->pattern->n_rows;
	      new_focus_channel = focus_channel;
	      focus_row = pe->pattern->n_rows - 1;
	    }
	  else
	    focus_row = pe->pattern->n_rows - 1;
	}
    }
  
  if (note != bnote->note ||
      instrument != bnote->instrument)
    bse_pattern_set_note (pe->pattern,
			  pe->focus_channel,
			  pe->focus_row,
			  note,
			  instrument);
  
  if (focus_channel != pe->focus_channel ||
      focus_row != pe->focus_row)
    bst_pattern_editor_set_focus (pe, focus_channel, focus_row);
  
  if (difference != 0)
    {
      BsePattern *pattern;
      
      pattern = pe->pattern;
      gtk_signal_emit (GTK_OBJECT (pe),
		       pe_signals[SIGNAL_PATTERN_STEP],
		       bse_pattern_get_guid (pe->pattern),
		       difference);
      if (pattern != pe->pattern)
	{
	  if (new_focus_channel != pe->focus_channel ||
	      new_focus_row != pe->focus_row)
	    bst_pattern_editor_set_focus (pe, new_focus_channel, new_focus_row);
	}
    }
  
  return pea & BST_PEA_TAG;
}

static gint
bst_pattern_editor_key_release (GtkWidget	       *widget,
				GdkEventKey	       *event)
{
  BstPatternEditor *pe;
  gboolean handled;
  
  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (BST_IS_PATTERN_EDITOR (widget), FALSE);
  
  pe = BST_PATTERN_EDITOR (widget);
  
  handled = FALSE;
  
  return handled;
}

static void
bst_pattern_editor_set_instrument (GtkWidget *item,
				   BstPatternEditor *pe)
{
  BseInstrument *instrument;
  
  g_return_if_fail (item != NULL);
  g_return_if_fail (pe != NULL);
  g_return_if_fail (BST_IS_PATTERN_EDITOR (pe));
  
  instrument = gtk_object_get_user_data (GTK_OBJECT (item));
  pe->instruments[pe->popup_tag] = instrument;
}

static gint
widget_install_accelerator_stop (GtkWidget *widget)
{
  gtk_signal_emit_stop_by_name (GTK_OBJECT (widget), "install_accelerator");

  return FALSE;
}

static void
bst_pattern_editor_channel_popup (BstPatternEditor *pe,
				  guint		    channel,
				  guint		    mouse_button,
				  guint32	    time)
{
  GtkWidget *item;
  GList *list;
  
  g_return_if_fail (pe != NULL);
  g_return_if_fail (BST_IS_PATTERN_EDITOR (pe));
  
  if (pe->channel_popup)
    gtk_widget_unref (pe->channel_popup);
  
  pe->popup_tag = channel;
  pe->channel_popup =
    gtk_widget_new (gtk_menu_get_type (),
		    NULL);
  gtk_widget_ref (pe->channel_popup);
  gtk_object_sink (GTK_OBJECT (pe->channel_popup));
  item = gtk_menu_item_new_with_label ("Instruments");
  gtk_widget_set (item,
		  "GtkObject::user_data", NULL,
		  "GtkWidget::sensitive", FALSE,
		  "GtkWidget::visible", TRUE,
		  "GtkWidget::parent", pe->channel_popup,
		  "GtkObject::signal::install_accelerator", widget_install_accelerator_stop, NULL,
		  NULL);
  item = gtk_menu_item_new ();
  gtk_widget_set (item,
		  "GtkObject::user_data", NULL,
		  "GtkWidget::sensitive", FALSE,
		  "GtkWidget::visible", TRUE,
		  "GtkWidget::parent", pe->channel_popup,
		  "GtkObject::signal::install_accelerator", widget_install_accelerator_stop, NULL,
		  NULL);
  for (list = pe->pattern->song->instruments; list; list = list->next)
    {
      BseInstrument *instrument;
      gchar *string;
      gchar buffer[32];
      
      instrument = list->data;
      
      sprintf (buffer, "%04d", bse_instrument_get_guid (instrument));
      string = (gchar*) bse_instrument_get_name (instrument);
      if (!string || *string == 0)
	string = instrument->sample->name;
      string = g_strconcat (buffer, ") ", string, NULL);
      item = gtk_menu_item_new_with_label (string);
      gtk_widget_set (item,
		      "GtkObject::user_data", instrument,
		      "GtkObject::signal::activate", bst_pattern_editor_set_instrument, pe,
		      "GtkWidget::sensitive", TRUE,
		      "GtkWidget::visible", TRUE,
		      "GtkWidget::parent", pe->channel_popup,
		      "GtkObject::signal::install_accelerator", widget_install_accelerator_stop, NULL,
		      NULL);
      g_free (string);
    }
  item = gtk_menu_item_new_with_label ("<NONE>");
  gtk_widget_set (item,
		  "GtkObject::user_data", NULL,
		  "GtkObject::signal::activate", bst_pattern_editor_set_instrument, pe,
		  "GtkWidget::sensitive", TRUE,
		  "GtkWidget::visible", TRUE,
		  "GtkWidget::parent", pe->channel_popup,
		  "GtkObject::signal::install_accelerator", widget_install_accelerator_stop, NULL,
		  NULL);
  gtk_menu_popup (GTK_MENU (pe->channel_popup),
		  NULL, NULL,
		  NULL, NULL,
		  mouse_button, time);
}

void
bst_pattern_editor_mark_row (BstPatternEditor	    *pe,
			     gint		     row)
{
  g_return_if_fail (pe != NULL);
  g_return_if_fail (BST_IS_PATTERN_EDITOR (pe));
  g_return_if_fail (row >= -1);
  g_return_if_fail (row < pe->pattern->n_rows);
  
  if (pe->last_row >= 0)
    {
      bst_pattern_editor_draw_panel (pe, 0, pe->last_row, TRUE);
    }
  pe->last_row = row;
  if (pe->last_row >= 0)
    {
      bst_pattern_editor_draw_panel (pe, 0, pe->last_row, TRUE);
    }
}

void
bst_pattern_editor_dfl_stepper (BstPatternEditor       *pe,
				guint			current_guid,
				gint			difference)
{
  GList *list;

  g_return_if_fail (pe != NULL);
  g_return_if_fail (BST_IS_PATTERN_EDITOR (pe));
  g_return_if_fail (bse_pattern_get_guid (pe->pattern) == current_guid);
    
  list = g_list_nth (pe->pattern->song->patterns, difference - 1 + current_guid);

  if (list && list->data != (gpointer) pe->pattern)
    bst_pattern_editor_set_pattern (pe, list->data);
  else
    gdk_beep ();
}
