/* BEAST - Bedevilled Audio System
 * Copyright (C) 1998 Olaf Hoehmann and Tim Janik
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */
#include        <gtk/gtk.h>
#include	"gtkitemfactory.h"
#include        <bse/bse.h>
#include	"../bseconfig.h"
#include	"bstshell.h"
#include	"bstpatterneditor.h"
#include	"bstkeytables.c"
#include	"bstrcargs.h"


/* --- prototypes --- */
static BseStream*	bst_default_ostream	(void);


/* --- functions --- */
static BseStream*
bst_default_ostream (void)
{
  BseStream *stream;

  /* Setup output stream, try the soundcard first (PCM).
   */
  stream = bse_pcm_stream_new ();
  if (!stream)
    g_warning ("beast: No PCM Streams available on this platform");
  else
    {
      BseErrorType error;
      gchar *pcm_device;
      
      pcm_device = bse_pcm_stream_get_default_device ();
      error = bse_stream_open (stream, pcm_device, FALSE, TRUE);
      if (!error)
	{
	  BseStreamAttribs attributes;
	  BseStreamAttribMask attribute_mask = BSE_SA_NONE;
	  
	  attributes.n_channels = 1;
	  attributes.play_frequency = 48000;
	  attributes.fragment_size = 400;
	  attribute_mask |= BSE_SA_N_CHANNELS | BSE_SA_PLAY_FREQUENCY | BSE_SA_FRAGMENT_SIZE;
	  
	  error = bse_stream_set_attribs (stream, attribute_mask, &attributes);
	}
      if (error)
	{
	  g_warning ("beast: Opening PCM stream `%s' failed: %s\n",
		     pcm_device,
		     bse_error_type_get_description (error));
	  bse_stream_unref (stream);
	  stream = NULL;
	}
    }
  if (!stream)
    {
      stream = bse_null_stream_new ();
      bse_stream_open (stream, "NULL", FALSE, TRUE);
    }

  g_print ("beast: Using %s stream \"%s\" output buffer=%d\n",
	   stream->type_name,
	   stream->name,
	   bse_get_obuffer_size ());

  return stream;
}

static gint
bst_iteration (gpointer data)
{
  if (bse_main_can_block ())
    while (bse_main_need_iteration ())
      bse_main_iteration ();
  else
    bse_main_iteration ();

  if (!gtk_events_pending ())
    {
      bse_main_trigger_ticks ();
      bse_main_iteration ();
    }

  /* let gtk do the blocking if we aren't on a time critical mission ;)
   */
  gtk_timeout_add (bse_main_can_block () ? 0 : 100, bst_iteration, NULL);
  
  return FALSE;
}

int
main (int argc, char **argv)
{
  static gchar *beast_rc_file = NULL;
  guint i;
  GtkWidget *shell;
  BseStream *ostream;
  gchar *fname;
  gchar *sample_names[] =
  {
    "bass/acidtb.bse",
    // "bass/hugobass.bse", "bass/mougue.bse",
    "bass/synthbass2.bse",
    "clap/909clap.bse",
    "bassdrum/909kick1.bse",
    "string/junostring.bse",
    //    "bass/bass4dafunk.bse", "bass/moraleebass.bse", "bass/progresso.bse",
    //    "string/3dflanged.bse", "string/delight.bse", "string/translucence.bse",
    //    "string/aahchoir_2080.bse", "string/junostring.bse", "string/wonderouspad.bse",
    //    "hihat/909crash.bse", "hihat/closedhh1.bse", "hihat/openhihat.bse",
    //    "hihat/909ride1.bse", "hihat/closedhh2.bse", "hihat/openshorthihat.bse",
  };
  guint n_sample_names = sizeof (sample_names) / sizeof (sample_names[0]) * 0;


  /* initialize bse
   */
  bse_init (&argc, &argv);

  /* parse rc files
   */
  bst_rc_args_init ();
  beast_rc_file = g_strconcat (g_get_home_dir (), "/.beastrc", NULL);
  bst_rc_parse (beast_rc_file);

  /* initialize gtk
   */
  gtk_init (&argc, &argv);
  gtk_accelerator_table_set_mod_mask (NULL,
				      GDK_SHIFT_MASK |
				      GDK_CONTROL_MASK |
				      GDK_MOD1_MASK);

  /* feature gle
   */
#ifdef	HAVE_LIBGLE
  gle_init (&argc, &argv);
#endif	/* HAVE_LIBGLE */


  /* setup the output stream
   */
  ostream = bst_default_ostream ();
  bse_main_add_output_stream (ostream);

  /* setup default keytable for pattern editor class
   */
  for (i = 0; i < bst_pea_ktab_de_n_entries; i++)
    bst_pattern_editor_class_set_key (gtk_type_class (bst_pattern_editor_get_type ()),
				      bst_pea_ktab_de[i].keyval,
				      bst_pea_ktab_de[i].modifier,
				      bst_pea_ktab_de[i].pe_action);

  /* start up the main BEAST shell
   * preload samples and pass them to the shell
   */
  shell = GTK_WIDGET (bst_shell_get ());
  gtk_window_set_wmclass (GTK_WINDOW (shell), "BstShell", "Beast");
  gtk_widget_set (shell,
		  "GtkObject::signal::destroy", gtk_main_quit, NULL,
		  NULL);
  for (i = 0; i < n_sample_names; i++)
    {
      BseIoData *io_data;
      gchar *string;

      string = g_strconcat ("/samples/olaf/", sample_names[i], NULL);
      io_data = bse_io_load_auto (string, BSE_IO_DATA_SAMPLES | BSE_IO_DATA_BINARY_APPENDIX);
      if (io_data->error)
	g_print ("beast: failed to load sample \"%s\": %s\n",
		 string,
		 bse_error_type_get_description (io_data->error));
      else
	bst_shell_adapt_sample (BST_SHELL (shell), io_data->samples->data);
      bse_io_data_destroy (io_data);
    }
  gtk_widget_show (shell);

  /* hook directly into gtk's main loop, ready, go! ;)
   */
  gtk_timeout_add (0, bst_iteration, NULL);
  gtk_main ();

  /* dump rc file
   */
  bst_rc_dump ("BEAST", beast_rc_file);
  // gtk_item_factory_dump_rc ("<BstSongShell>", FALSE, (GtkPrintFunc) printf, "<BstSongShell> %s\n");

  return 0;
}
