/*
 * Copyright (c) 2005 Versant Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Versant Corporation - initial API and implementation
 */

package org.eclipse.jsr220orm.core.util;

import java.math.BigInteger;
import java.sql.Types;
import java.util.Iterator;
import java.util.Map;
import java.util.TreeMap;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.jsr220orm.core.IEntityModelManager;
import org.eclipse.jsr220orm.metadata.BasicAttribute;
import org.eclipse.jsr220orm.metadata.OrmColumn;
import org.eclipse.jsr220orm.metadata.OrmTable;
import org.eclipse.wst.rdb.internal.core.RDBCorePlugin;
import org.eclipse.wst.rdb.internal.core.connection.ConnectionInfo;
import org.eclipse.wst.rdb.internal.core.definition.DataModelElementFactory;
import org.eclipse.wst.rdb.internal.core.definition.DatabaseDefinition;
import org.eclipse.wst.rdb.internal.core.definition.DatabaseDefinitionRegistry;
import org.eclipse.wst.rdb.internal.core.rte.DDLGenerator;
import org.eclipse.wst.rdb.internal.models.dbdefinition.PredefinedDataTypeDefinition;
import org.eclipse.wst.rdb.internal.models.sql.constraints.PrimaryKey;
import org.eclipse.wst.rdb.internal.models.sql.constraints.SQLConstraintsFactory;
import org.eclipse.wst.rdb.internal.models.sql.constraints.SQLConstraintsPackage;
import org.eclipse.wst.rdb.internal.models.sql.datatypes.DataType;
import org.eclipse.wst.rdb.internal.models.sql.datatypes.PredefinedDataType;
import org.eclipse.wst.rdb.internal.models.sql.expressions.ValueExpressionDefault;
import org.eclipse.wst.rdb.internal.models.sql.expressions.impl.SQLExpressionsFactoryImpl;
import org.eclipse.wst.rdb.internal.models.sql.schema.IdentitySpecifier;
import org.eclipse.wst.rdb.internal.models.sql.schema.SQLObject;
import org.eclipse.wst.rdb.internal.models.sql.schema.SQLSchemaPackage;
import org.eclipse.wst.rdb.internal.models.sql.tables.BaseTable;
import org.eclipse.wst.rdb.internal.models.sql.tables.Column;
import org.eclipse.wst.rdb.internal.models.sql.tables.SQLTablesFactory;
import org.eclipse.wst.rdb.internal.models.sql.tables.Table;

/**
 * A utils class to help in dealing with rdb models
 */
public class RdbUtils {

    /**
     * Returns the list of all to database defenitions to display to the user.
     * Override this method to provide a filtered list of databases.
     * 
     * @return A array of DatabaseDefinition objects that should be displayed in
     *         the Database list
     */
    public static DatabaseDefinitionRegistry getDatabaseDefinitionRegistry() {
        return RDBCorePlugin.getDefault().getDatabaseDefinitionRegistry();
    }

    /**
     * Returns the list of existing connections to display to the user. Override
     * this method to provide a filtered list of connections.
     * 
     * @return A array of ConnectionInfo objects that should be displayed in the
     *         existing connections list
     */
    public static ConnectionInfo[] getRdbConnections() {
        return RDBCorePlugin.getDefault().getConnectionManager()
                .getAllNamedConnectionInfo();
    }

    /**
     * Get DDL for the column (i.e. a columnDefinition for it). This will try to
     * use RDB but fallback to just creating a generic String itself if this
     * fails. Note that this ignores the columnDefinition attribute of the
     * column.
     */
    public static String getColumnDefinition(DatabaseDefinition dbdef,
            PredefinedDataTypeDefinition typeDef, String dataTypeName, 
            OrmColumn c) {
        int jdbcType = c.getJdbcType();
        DDLGenerator gen = dbdef.getDDLGenerator();
        if (gen != null && typeDef != null) {
            Column col = SQLTablesFactory.eINSTANCE.createColumn();
            col.setName(c.getName());
            col.setNullable(c.isNullable());
            PredefinedDataType type = dbdef.getPredefinedDataType(typeDef);
            if (typeDef.isLengthSupported()) {
                EStructuralFeature feature = type.eClass()
                        .getEStructuralFeature("length");
                type.eSet(feature, new Integer(c.getLength()));
            } else if (typeDef.isPrecisionSupported()) {
                EStructuralFeature feature = type.eClass()
                        .getEStructuralFeature("precision");
                type.eSet(feature, new Integer(c.getLength()));
            }
            if (typeDef.isScaleSupported()) {
                EStructuralFeature feature = type.eClass()
                        .getEStructuralFeature("scale");
                type.eSet(feature, new Integer(c.getScale()));
            }
            if(c.isIdentity()){
                final DataModelElementFactory factory = dbdef.getDataModelElementFactory();
                IdentitySpecifier identitySpecifier = (IdentitySpecifier) 
                    factory.create(SQLSchemaPackage.eINSTANCE .getIdentitySpecifier());
                col.setIdentitySpecifier(identitySpecifier);
            }
            col.setContainedType(type);
            String[] a = gen.createSQLObjects(new SQLObject[] { col }, false,
                    false, null);
            if (a.length > 0) {
                return a[0];
            }
        }
        // fallback to generating a String ourselves
        String name = c.getName();
        int length = c.getLength();
        int scale = c.getScale();
        boolean nullable = c.isNullable();
        return getColumnDefinition(typeDef, jdbcType, dataTypeName, name, 
        		length, scale, nullable);
    }

    /**
     * Get DDL for the column. This will try to use RDB but fallback to just
     * creating a generic String itself if this fails.
     */
    public static String getColumnDefinition(Column rdbColumn) {
        DatabaseDefinitionRegistry ddr = getDatabaseDefinitionRegistry();
        DatabaseDefinition dbdef = ddr.getDefinition(rdbColumn.getTable()
                .getSchema().getDatabase());
        DDLGenerator gen = dbdef.getDDLGenerator();
        if (gen != null) {
            String[] a = gen.createSQLObjects(new SQLObject[] { rdbColumn },
                    false, false, null);
            if (a.length > 0) {
                return a[0];
            }
        }
        // fallback to generating a String ourselves
        DataType type = rdbColumn.getDataType();
        PredefinedDataTypeDefinition typeDef = dbdef
                .getPredefinedDataTypeDefinition(type.getName());
        int jdbcType = typeDef.getJdbcEnumType();
        String dataTypeName = rdbColumn.getDataType().getName();
        String name = rdbColumn.getName();
        int length = getLength(typeDef, rdbColumn);
        int scale = getScale(typeDef, rdbColumn);
        boolean nullable = rdbColumn.isNullable();
        return getColumnDefinition(typeDef, jdbcType, dataTypeName, name, length, scale, nullable);
    }

    public static int getScale(PredefinedDataTypeDefinition typeDef,
            Column rdbColumn) {
        if (typeDef.isScaleSupported()) {
            EStructuralFeature feature = rdbColumn.eClass()
                    .getEStructuralFeature("scale");
            Integer scale = (Integer) rdbColumn.eGet(feature);
            if (scale != null) {
                return scale.intValue();
            }
        }
        return 0;
    }

    public static int getLength(PredefinedDataTypeDefinition typeDef,
            Column rdbColumn) {
        DataType type = rdbColumn.getDataType();
        if (typeDef.isLengthSupported()) {
            EStructuralFeature feature = type.eClass()
                    .getEStructuralFeature("length");
            Integer length = (Integer) type.eGet(feature);
            if (length != null) {
                return length.intValue();
            }
        } else if (typeDef.isPrecisionSupported()) {
            EStructuralFeature feature = type.eClass()
                    .getEStructuralFeature("precision");
            Integer precision = (Integer) type.eGet(feature);
            if (precision != null) {
                return precision.intValue();
            }
        }
        return 0;
    }

    /**
     * Get generic DDL for a column column
     */
    public static String getColumnDefinition(
            PredefinedDataTypeDefinition typeDef, int jdbcType, String dataTypeName,
            String name, int length, int scale, boolean nullable) {
        StringBuffer s = new StringBuffer();
        s.append(name);
        s.append(' ');
        if (dataTypeName == null) {
            s.append(JdbcUtils.getJdbcTypeName(jdbcType));
        } else {
            s.append(dataTypeName);
        }
        if (length > 0 && (typeDef == null || isLengthSupported(typeDef))) {
            s.append('(');
            s.append(length);
            if (typeDef == null || isScaleSupported(typeDef, jdbcType)) {
                if (scale != 0) {
                    s.append(',');
                    s.append(scale);
                }
            }
            s.append(')');
        }
        if ((typeDef == null || typeDef.isNullableSupported())
                && !nullable) {
            s.append(" NOT NULL");
        }
        return s.toString();
    }

    /**
     * Set the columnDefinition attribute of the ORM column to the DDL for the
     * RDB column. This will try to use RDB but fallback to just creating a
     * generic String itself if this fails.
     */
    public static void setColumnDefinition(OrmColumn ormColumn, Column rdbColumn) {
        DatabaseDefinitionRegistry ddr = getDatabaseDefinitionRegistry();
        DatabaseDefinition dbdef = ddr.getDefinition(rdbColumn.getTable()
                .getSchema().getDatabase());
        DataType type = rdbColumn.getDataType();
        PredefinedDataTypeDefinition typeDef = dbdef
                .getPredefinedDataTypeDefinition(type.getName());
        ormColumn.setName(rdbColumn.getName());
        ormColumn.setComment(rdbColumn.getDescription());
        ormColumn.setDatabaseType(rdbColumn.getDataType().getName());
        ormColumn.setJdbcType(typeDef.getJdbcEnumType());
        ormColumn.setNullable(rdbColumn.isNullable());
        ormColumn.setLength(getLength(typeDef, rdbColumn));
        ormColumn.setScale(getScale(typeDef, rdbColumn));
        ormColumn.setColumnDefinition(getColumnDefinition(rdbColumn));
    }

    public static boolean isLengthSupported(PredefinedDataTypeDefinition typeDef) {
        if (typeDef != null) {
            return typeDef.isLengthSupported()
                    || typeDef.isPrecisionSupported();
        }
        return true;
    }

    public static boolean isScaleSupported(
            PredefinedDataTypeDefinition typeDef, int jdbcType) {
        if (typeDef != null) {
            return typeDef.isScaleSupported();
        }
        switch (jdbcType) {
        case Types.DECIMAL:
        case Types.NUMERIC:
            return true;
        }
        return false;
    }

    public static String getColumnTypeDescr(Column c) {
        DatabaseDefinitionRegistry ddr = getDatabaseDefinitionRegistry();
        DatabaseDefinition dbdef = ddr.getDefinition(c.getTable()
                .getSchema().getDatabase());
        DataType type = c.getDataType();
        PredefinedDataTypeDefinition typeDef = dbdef
                .getPredefinedDataTypeDefinition(type.getName());
        StringBuffer s = new StringBuffer();
        s.append(type.getName());
        if (isLengthSupported(typeDef)) {
            int length = getLength(typeDef, c);
            s.append('(');
            s.append(length);
            if (isScaleSupported(typeDef, typeDef.getJdbcEnumType())) {
                int scale = getScale(typeDef, c);
                s.append(',');
                s.append(scale);
            }
            s.append(')');
        }
        return s.toString();
    }

    public static void copyToOrm(final OrmColumn ormColumn,
            final Column rdbColumn) {
        IEntityModelManager manager = (IEntityModelManager) ormColumn
                .adapt(IEntityModelManager.class);
        if (manager != null) {
            manager.executeModelChanges(new Runnable() {
                public void run() {
                    copyToOrmImpl(ormColumn, rdbColumn);
                }
            });
        }else{
            copyToOrmImpl(ormColumn, rdbColumn);
        }
    }

    private static void copyToOrmImpl(final OrmColumn ormColumn, final Column rdbColumn) {
        DatabaseDefinitionRegistry ddr = RdbUtils
                .getDatabaseDefinitionRegistry();
        DatabaseDefinition dbdef = ddr.getDefinition(rdbColumn
                .getTable().getSchema().getDatabase());
        DataType type = rdbColumn.getDataType();
        PredefinedDataTypeDefinition typeDef = dbdef
                .getPredefinedDataTypeDefinition(type.getName());
        ormColumn.setName(rdbColumn.getName());
        ormColumn.setComment(rdbColumn.getDescription());
        ormColumn.setDatabaseType(rdbColumn.getDataType().getName());
        ormColumn.setJdbcType(typeDef.getJdbcEnumType());
        ormColumn.setNullable(rdbColumn.isNullable());
        ormColumn.setLength(RdbUtils.getLength(typeDef, rdbColumn));
        ormColumn.setScale(RdbUtils.getScale(typeDef, rdbColumn));
        ormColumn.setColumnDefinition(RdbUtils
                .getColumnDefinition(rdbColumn));
    }
    
    public static void copyToRdb(OrmColumn ormColumn, Column rdbColumn, DatabaseDefinition dbdef) {
        int jdbcType = ormColumn.getJdbcType();
        rdbColumn.setName(ormColumn.getName());
        rdbColumn.setDescription(ormColumn.getComment());
        ValueExpressionDefault expression = SQLExpressionsFactoryImpl.eINSTANCE.createValueExpressionDefault();
        expression.setSQL(ormColumn.getColumnDefinition());
//        rdbColumn.setGenerateExpression(expression);
        rdbColumn.setNullable(ormColumn.isNullable());
     
        IEntityModelManager emm = (IEntityModelManager) 
            ormColumn.adapt(IEntityModelManager.class); 
        PredefinedDataTypeDefinition typeDef = emm.getColumnDataTypeDefinition(ormColumn);
        if (typeDef != null && dbdef != null) {
            PredefinedDataType type = dbdef.getPredefinedDataType(typeDef);
            if (typeDef.isLengthSupported()) {
                EStructuralFeature feature = type.eClass()
                        .getEStructuralFeature("length");
                type.eSet(feature, new Integer(ormColumn.getLength()));
            } else if (typeDef.isPrecisionSupported()) {
                EStructuralFeature feature = type.eClass()
                        .getEStructuralFeature("precision");
                type.eSet(feature, new Integer(ormColumn.getLength()));
            }
            if (typeDef.isScaleSupported()) {
                EStructuralFeature feature = type.eClass()
                        .getEStructuralFeature("scale");
                type.eSet(feature, new Integer(ormColumn.getScale()));
            }
            rdbColumn.setContainedType(type);
        }
        if(ormColumn.isIdentity()){
            final DataModelElementFactory factory = dbdef.getDataModelElementFactory();
            IdentitySpecifier identitySpecifier = (IdentitySpecifier) 
                factory.create(SQLSchemaPackage.eINSTANCE .getIdentitySpecifier());
            identitySpecifier.setStartValue(new BigInteger("0"));
            identitySpecifier.setIncrement(new BigInteger("1"));
            rdbColumn.setIdentitySpecifier(identitySpecifier);
        }
    }

    public static void copyToRdb(OrmTable ormTable, BaseTable table, DatabaseDefinition dbDef) {
        EList pkList = ormTable.getPrimaryKeyList();
        if(!pkList.isEmpty()){
            DataModelElementFactory factory = dbDef.getDataModelElementFactory();
            EClass primaryKeyClass = SQLConstraintsPackage.eINSTANCE.getPrimaryKey();
            PrimaryKey pk = (PrimaryKey) factory.create(primaryKeyClass);
            pk.setName(ormTable.getName()+"_pk");
            for(Iterator atts = pkList.iterator(); atts.hasNext();){
                OrmColumn ormColumn = (OrmColumn) atts.next();
                // TODO Please remove this hack as soon as MySql is fixed in RDB
                if(ormColumn.isIdentity() && dbDef.getProduct().toUpperCase().startsWith("MYSQL")){
                    return;
                }
                // TODO Please remove this hack as soon as MySql is fixed in RDB
                String colName = ormColumn.getName();
                Iterator cols = table.getColumns().iterator();
                while (cols.hasNext()) {
                    Column column = (Column) cols.next();
                    if (column.getName().equals(colName)){
                        pk.getMembers().add(column);
                        break;
                    }
                }
            }
            table.getConstraints().add(pk);
        }
    }
}
