/*
 * Copyright (c) 2005 Versant Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Versant Corporation - initial API and implementation
 */

package org.eclipse.jsr220orm.generic.io;

import java.lang.annotation.Annotation;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.persistence.CascadeType;
import javax.persistence.FetchType;
import javax.persistence.Transient;

import org.eclipse.core.runtime.IAdapterFactory;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.jsr220orm.core.internal.options.IntOption;
import org.eclipse.jsr220orm.generic.GenericEntityModelManager;
import org.eclipse.jsr220orm.generic.Utils;
import org.eclipse.jsr220orm.generic.reflect.RAnnotatedElement;
import org.eclipse.jsr220orm.generic.reflect.RClass;
import org.eclipse.jsr220orm.metadata.AttributeMetaData;
import org.eclipse.jsr220orm.metadata.EntityMetaData;
import org.eclipse.jsr220orm.metadata.OrmColumn;
import org.eclipse.jsr220orm.metadata.TypeMetaData;

/**
 * Maps between an Attribute and its meta data (source code annotations and
 * XML). When the model changes the meta data is updated and visa versa.
 * One of these is created to manage each persistent field or property in the 
 * model. Different subclasses handle different attribute mappings.
 */
public abstract class AttributeIO implements IAdapterFactory {

	protected final EntityIO entityIO;
	
    public static final IntOption MAPPING_NOT_PERSISTENT = 
    	new IntOption(0, "Not persistent", 
    			"Attribute is not stored in the database",
    			Utils.getImage("NotPersistent16"));
	
	protected static final CascadeType[] CASCADE_TYPE_ALL = new CascadeType[]{
		CascadeType.ALL
	};
	
	public static final int RELATIVE_POSITION_FACTOR = 100;
	
	/**
	 * Create new looking up or creating appropriate AttributeMetaData and
	 * adding it the model.
	 */
	public AttributeIO(EntityIO entityIO) {
		this.entityIO = entityIO;
	}
	
	/**
	 * Get the AttributeIO for the AttributeMetaData or null if none.
	 */
	public static AttributeIO get(AttributeMetaData amd) {
		return (AttributeIO)amd.adapt(AttributeIO.class);
	}

	/**
	 * Cleanup any resources associated with us (e.g. markers).
	 */
	public void dispose() {
	}
	
	/**
	 * Update the attribute from its meta data (annotations and XML). This
	 * method must add the AttributeMetaData returned to the model. Do not
	 * keep a reference to the attibute parameter. Return true if the update
	 * has been completed (maybe even with errors) or false if we are waiting
	 * for some other part of the model to update. 
	 */
	public abstract boolean updateModelFromMetaData(RClass cls, 
			RAnnotatedElement attribute, boolean metaDataChanged);

	/**
	 * Update the annotations and/or XML from our attribute. If accessChanged
	 * is true then the access type (FIELD or PROPERTY) has been changed.
	 * If this method returns false then the attribute no longer exists and
	 * should be removed from the model. This might happen for a change in
	 * access type when there is no matching alternate field or method to
	 * use instead of the old method/field.
	 */
	public boolean updateMetaDataFromModel(RClass cls, boolean accessChanged) {
		AttributeMetaData amd = getAttributeMetaData();
		RAnnotatedElement attribute;
		if (amd.isField()) {
			attribute = cls.getDeclaredField(amd.getName());						
		} else {
			attribute = cls.getDeclaredMethod(amd.getGetterMethodName());						
		}
		RAnnotatedElement newAttr = null;
		if (accessChanged) {
			// attribute is the old field or method - try to find a new field
			// or method and write the annotations to that instead
			if (amd.isField()) {
				String s = amd.getName();
				String methodName = "get" + 
					Character.toUpperCase(s.charAt(0)) + s.substring(1);
				newAttr = cls.getDeclaredMethod(methodName);
				if (newAttr != null) {
					amd.setGetterMethodName(methodName);
				}
			} else {
				newAttr = cls.getDeclaredField(amd.getName());
				if (newAttr != null) {
					amd.setGetterMethodName(null);
				}
			}
			if (newAttr != null) {
				Annotation[] all = newAttr.getAnnotations();
				if (all.length > 1 || all.length == 1 && 
						all[0].annotationType() != Transient.class) {
					// New attribute has at least one persistence annotation
					// so remove our attribute from the model. It will be
					// recreated when the model is reloaded from the
					// annotations on the new attribute.
					return false;
				}
			} else {
				// remove the attribute from the model on return as we found 
				// nothing
				return false;
			}
		}
		updateMetaDataFromModel(cls, newAttr == null ? attribute : newAttr);		
		if (newAttr != null) {
			// delete all the persistence annotations on the old attribute
			// except those with errors as they will have be written to the
			// new attribute from the model
			for (Annotation ann : attribute.getAnnotations()) {
				AnnotationEx a = (AnnotationEx)ann;
				if (a.getErrorMap() == null) {
					a.delete();
				}
			}
		}
		return true;
	}

	/**
	 * Update the annotations and/or XML from our attribute. 
	 */
	protected abstract void updateMetaDataFromModel(
			RClass cls, RAnnotatedElement attribute);
	
	/**
	 * Get the attribute we are managing.
	 */
	public abstract AttributeMetaData getAttributeMetaData();
	
	public Object getAdapter(Object adaptableObject, Class adapterType) {
		return this;
	}

	public Class[] getAdapterList() {
		return new Class[]{AttributeIO.class};
	}

	/**
	 * Get the mapping options that are valid for this attribute.
	 */
	public abstract void getPossibleMappings(List ans);

	/**
	 * Get the currently selected mapping option for this attribute.
	 */
	public abstract IntOption getMapping();

	/**
	 * Set the mapping option for this attribute.
	 */
	public abstract void setMapping(IntOption mapping);
	
	/**
	 * Set the fetchType on amd converting the javax.persistence Enum values
	 * to model codes. 
	 */
	protected void setFetchType(AttributeMetaData amd, FetchType ft) {
		switch (ft) {
		case LAZY:
			amd.setFetchType(AttributeMetaData.FETCH_TYPE_LAZY);
			break;
		default:
			amd.setFetchType(AttributeMetaData.FETCH_TYPE_EAGER);
		}		
	}
		
	/**
	 * Get the fetchType of amd as the javax.persistence Enum or null if
	 * invalid. 
	 */
	protected FetchType getFetchType(AttributeMetaData amd) {
		switch (amd.getFetchType()) {
		case AttributeMetaData.FETCH_TYPE_EAGER:
			return FetchType.EAGER;
		case AttributeMetaData.FETCH_TYPE_LAZY:
			return FetchType.LAZY;
		};
		return null;
	}
	
	public GenericEntityModelManager getModelManager() {
		return entityIO.getModelManager();
	}
	
	/**
	 * If amd is null then create and initialize an AttributeMetaData instance
	 * using eClass, add our entityIO to the adapters list so it receives
	 * model events and register us as an AdapterFactory so we are associated
	 * with the attribute. The javaType of the attribute is also updated.
	 * This detects non-persistent attributes and makes sure that they go
	 * into {@link EntityIO#npAttributeMap}.
	 */
	protected AttributeMetaData initAttributeMetaData(
			AttributeMetaData amd, RAnnotatedElement attribute, EClass eClass) {

		EntityMetaData emd = entityIO.getEntityMetaData();
		
		boolean isNewAttribute = amd == null;
		if (isNewAttribute) {
			GenericEntityModelManager mm = entityIO.getModelManager();
			amd = (AttributeMetaData)mm.getFactory().create(eClass);			
			amd.setName(Utils.getAttributeName(attribute));
			amd.registerAdapterFactory(this);			
			amd.eAdapters().add(entityIO);						
			if (attribute.isField()) {
				amd.setGetterMethodName(null);			
			} else {
				amd.setGetterMethodName(attribute.getName());
			}
		}
		
		// figure out if our attribute is persistent or not
		boolean persistent;
		if (attribute.isAnnotationPresent(Transient.class)) {
			persistent = false;
		} else if (attribute.isField()) {
			persistent = emd.getAccessType() 
				== EntityMetaData.ACCESS_TYPE_FIELD;
		} else {
			persistent = emd.getAccessType() 
				== EntityMetaData.ACCESS_TYPE_PROPERTY;
		}
		setPersistent(amd, persistent, isNewAttribute);

		TypeMetaData tmd = entityIO.findTypeByClassName(attribute.getTypeName());
		if (tmd == null) {
			if (attribute.isTypeEnum()) {
				tmd = entityIO.findTypeByClassName("java.lang.Enum");
			} else if (attribute.isTypeSerializable()) {
				tmd = entityIO.findTypeByClassName("java.io.Serializable");
			}
		}
		amd.setJavaType(tmd);
		return amd;
	}	
	
	/**
	 * Make the attribute persistent or not. This is a NOP if nothing needs
	 * to change. 
	 */
	protected void setPersistent(boolean persistent) {
		setPersistent(getAttributeMetaData(), persistent, false);
	}

	/**
	 * Make the attribute persistent or not. This is a NOP if nothing needs
	 * to change unless isNewAttribute is true (newly created attributes
	 * appear non-persistent as they have no owning EntityMetaData). This 
	 * version is used when {@link #getAttributeMetaData()} might return null. 
	 */
	protected void setPersistent(AttributeMetaData amd, boolean persistent,
			boolean isNewAttribute) {
		if (persistent) {
			if (isNewAttribute || amd.isNonPersistent()) {
				entityIO.removeNonPersistentAttribute(amd);
				entityIO.getEntityMetaData().getAttributeList().add(amd);
			}
		} else {
			if (isNewAttribute || !amd.isNonPersistent()) {
				amd.makeNonPersistent();
				entityIO.addNonPersistentAttribute(amd);
			}
		}		
	}
	
	/**
	 * Get comment info to attach to OrmColumn's and so on that belong to us.
	 */
	protected String getComment(AttributeMetaData amd) {
		return amd.getEntityMetaData().getSchemaName() + "." + amd.getName();
	}

	public EntityIO getEntityIO() {
		return entityIO;
	}
	
	/**
	 * Make sure that the attribute has a Transient annotation if it needs
	 * one (i.e. it would be persistent otherwise).
	 */
	public void ensureTransient(RAnnotatedElement attribute) {
		if (entityIO.isPersistentByDefault(attribute)) {
			AnnotationEx t = (AnnotationEx)attribute.getAnnotation(
					Transient.class, true);
			t.setMarker(true);
		}
	}
	
	/**
	 * Does target have a table and at leasy one primary key column? This
	 * will return false if target is null.
	 */
	protected boolean hasTableAndPrimaryKey(EntityMetaData target) {
		if (target == null || target.getTable() == null 
				|| EntityIO.get(target).getModelUpdateStatus() 
					< EntityIO.STATUS_STARTED_ATTRIBUTES) {
			return false;
		}
		return !target.getTable().getPrimaryKeyList().isEmpty();
	}
	
	/**
	 * Return the mappedBy attribute or null if it does not exist. This
	 * handles renamed attributes correctly.
	 */
	protected AttributeMetaData getMappedByAttribute(AnnotationEx main, 
			String mappedBy, EntityMetaData target) {
		EntityIO targetEntityIO = EntityIO.get(target);
		String newName = targetEntityIO.getNewAttributeName(mappedBy);
		if (newName != null) {
			// our meta data needs to be written out once the model
			// has been updated to reflect the new name of our 
			// mappedBy attribute
			entityIO.getModelManager().registerForMetaDataUpdate(entityIO);
			mappedBy = newName;
		}
		AttributeMetaData ans = target.findAttributeMetaData(mappedBy);
		if (ans == null) {
			entityIO.addProblem(
					"Attribute '" + mappedBy + "' not found on " + 
					target.getClassName(),
					main.getLocation("mappedBy"));					
		} else if (ans.getMappedBy() != null) {
			entityIO.addProblem(
					"Attribute '" + mappedBy + "' also has mappedBy set",
					main.getLocation("mappedBy"));	
			ans = null;
		}
		return ans;
	}	
		
	protected int getCascadeBits(CascadeType[] cascade) {
		int bits = 0;
		for (int i = 0; i < cascade.length; i++) {
			bits |= getCascadeBits(cascade[i]);
		}
		return bits;
	}
	
	protected int getCascadeBits(CascadeType ct) {
		switch (ct) {
		case ALL:		return AttributeMetaData.CASCADE_TYPE_ALL;
		case MERGE:		return AttributeMetaData.CASCADE_TYPE_MERGE;
		case PERSIST:	return AttributeMetaData.CASCADE_TYPE_PERSIST;
		case REFRESH:	return AttributeMetaData.CASCADE_TYPE_REFRESH;
		case REMOVE:	return AttributeMetaData.CASCADE_TYPE_REMOVE;
		}
		return 0;
	}	
	
	/**
	 * Convert bits (see {@link AttributeMetaData#CASCADE_TYPE_PERSIST} etc.)
	 * to {@link CascadeType} array or null if the array would be empty. 
	 */
	protected CascadeType[] getCascadeTypes(int bits) {
		if (bits == 0) {
			return null;
		}
		if ((bits & AttributeMetaData.CASCADE_TYPE_ALL) 
				== AttributeMetaData.CASCADE_TYPE_ALL) {
			return CASCADE_TYPE_ALL; 
		}
		ArrayList ans = new ArrayList(3);
		if ((bits & AttributeMetaData.CASCADE_TYPE_PERSIST) != 0) {
			ans.add(CascadeType.PERSIST);
		}
		if ((bits & AttributeMetaData.CASCADE_TYPE_MERGE) != 0) {
			ans.add(CascadeType.MERGE);
		}
		if ((bits & AttributeMetaData.CASCADE_TYPE_REMOVE) != 0) {
			ans.add(CascadeType.REMOVE);
		}
		if ((bits & AttributeMetaData.CASCADE_TYPE_REFRESH) != 0) {
			ans.add(CascadeType.REFRESH);
		}
		CascadeType[] a = new CascadeType[ans.size()];
		ans.toArray(a);
		return a;
	}

	/**
	 * Get the list of attributes that can be used for mappedBy for this
	 * attribute.
	 */
	public List getValidMappedByAttributes() {
		return Collections.EMPTY_LIST;
	}
	
	/**
	 * Get the base {@link OrmColumn#getRelativePositionInTable()} value for
	 * columns belonging to this attribute. This is based on the position of 
	 * the attribute in its owner and on the position of this class in the 
	 * heirachy.
	 */
	public int getRelativePosition(RAnnotatedElement attribute) {
		return entityIO.getRelativePosition() + 
			attribute.getRelativeIndexInOwner() * RELATIVE_POSITION_FACTOR;
	}
	
	/**
	 * Get a short string representing the type of target for use in error
	 * messages and so on. 
	 */
	public String getTypeName(EntityMetaData target) {
		switch (target.getEntityType()) {
		default:
		case EntityMetaData.TYPE_ENTITY:
			return "Entity";
		case EntityMetaData.TYPE_EMBEDDABLE:
			return "Embeddable";
		case EntityMetaData.TYPE_EMBEDDABLE_SUPERCLASS:
			return "EmbeddableSuperclass";
		}
	}
	
	/**
	 * Add a problem for a null java type. Typically this means that the
	 * type is not persistent. 
	 */
	protected void addNullJavaTypeProblem(RAnnotatedElement attribute) {
		String typeName = attribute.getTypeName();
		if (typeName != null) {
			entityIO.addProblem(typeName + " is not persistent", 
					attribute.getLocation());
		}
		// if typeName is null then Eclipse has most likely added a
		// problem for the missing type already
	}	
	
}
