/*
 * Copyright (c) 2005 Versant Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Versant Corporation - initial API and implementation
 */

package org.eclipse.jsr220orm.generic;

import java.io.IOException;
import java.sql.Types;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.xmlbeans.XmlException;
import org.eclipse.emf.common.util.EList;
import org.eclipse.jsr220Orm.generic.xml.VendorDefinitionDocument;
import org.eclipse.jsr220Orm.generic.xml.VendorDefinitionDocument.VendorDefinition.CollectionType;
import org.eclipse.jsr220Orm.generic.xml.VendorDefinitionDocument.VendorDefinition.JdbcTypeFallback;
import org.eclipse.jsr220Orm.generic.xml.VendorDefinitionDocument.VendorDefinition.JdbcTypeMap;
import org.eclipse.jsr220Orm.generic.xml.VendorDefinitionDocument.VendorDefinition.SimpleType;
import org.eclipse.jsr220Orm.generic.xml.VendorDefinitionDocument.VendorDefinition.JdbcTypeMap.JdbcType;
import org.eclipse.jsr220orm.core.util.JdbcUtils;
import org.eclipse.jsr220orm.generic.io.AnnotationRegistry;
import org.eclipse.wst.rdb.internal.core.definition.DatabaseDefinition;
import org.eclipse.wst.rdb.internal.models.dbdefinition.PredefinedDataTypeDefinition;

/**
 * Information about the persistence vendor e.g. JDBC type mappings. 
 * This is read from an XML file (e.g. vendor-generic.xml).
 */
public class VendorDef {

	protected static final String VENDOR_DEFINITION_XML_NAMESPACE = 
		"http://jsr220orm.eclipse.org/generic/xml";
	
	protected String name;
    protected Map<Integer, JdbcTypeInfo> jdbcTypeMap;

	protected VendorDefinitionDocument.VendorDefinition vd;

    /**
     * Call {@link #init} to initialize.
     */
    public VendorDef() {
    }
    
    /**
     * Create new from XML resource on classpath. 
     */
    public void init(String vendorDefResouce, GenericEntityModelManager mm) 
    		throws XmlException, IOException {
        String namespace = mm.getNature().getActiveOrmProject().getProduct().getNamespace();
        VendorDefinitionDocument doc = Utils.loadVendorDefinition(
        		namespace, vendorDefResouce);
	    vd = doc.getVendorDefinition();
		name = vd.getName();
	    AnnotationRegistry reg = mm.getAnnotationRegistry();
	    DatabaseDefinition dbdef = mm.getDatabaseDefinition();
	    
	    // find the best type table for our database
		String dbProduct = dbdef.getProduct();
		String dbVersion = dbdef.getVersion();
	    JdbcTypeMap match = null;
	    for (int i = 0; i < vd.sizeOfJdbcTypeMapArray(); i++) {
	    	JdbcTypeMap map = vd.getJdbcTypeMapArray(i);
	    	String database = map.getDatabase();
			if (database.equals(dbProduct)) {
	    		if (match == null) {
	    			match = map;
	    		}
	    		if (dbVersion.equals(map.getVersion())) {
	    			match = map;
	    			break;
	    		}
	    	}
	    }
	    if (match == null) {
	    	match = vd.getJdbcTypeMapArray(0);
	    }
	    
	    // populate the map resolving SQL type names to matching RDB stuff
	    jdbcTypeMap = new HashMap();
		for (int i =  0; i < match.sizeOfJdbcTypeArray(); i++) {
	    	JdbcType t = match.getJdbcTypeArray(i);
	    	int jdbcType = JdbcUtils.getJdbcTypeValue(t.getJdbc());
	    	JdbcTypeInfo info = findPredefinedDataTypeDefinition(dbdef, 
	    			jdbcType, t.getSql());
	    	if (info != null) {
		    	jdbcTypeMap.put(jdbcType, info);
	    	} else {
	    		// TODO output warning about unmatched RDB type
	    	}
	    }

	    // complete the type table using the jdbc-type-fallback to fill
	    // in for missing types
	    Map<Integer, Integer> fallbackMap = 
	    	createJdbcTypeFallbackMap(vd, reg);
	    List failed = new ArrayList();
	    Set<Integer> ignore = createJdbcTypeIgnoreSet();
	    int[] allTypes = mm.getAllJdbcTypeInts();
	    for (int i = allTypes.length - 1; i >= 0; i--) {
	    	int jdbcType = allTypes[i];
	    	if (jdbcTypeMap.containsKey(jdbcType)) {
	    		continue;
	    	}
	    	JdbcTypeInfo info = null;
	    	for (int t = jdbcType; fallbackMap.containsKey(t); ) {
	    		t = fallbackMap.get(t);
	    		info = jdbcTypeMap.get(t);
	    		if (info != null) {
	    			jdbcTypeMap.put(jdbcType, info);
	    			break;
	    		}
	    	}
	    	if (info == null && !ignore.contains(jdbcType)) {
	    		failed.add(JdbcUtils.getJdbcTypeName(jdbcType));
	    	}
	    }
	    
	    // issue a warning about unmapped types that we care about
	    if (!failed.isEmpty()) {
	    	String msg = "JDBC types for vendor " + name + " on " + dbProduct + 
	    		" " + dbVersion + " are unmapped: " + failed;
	    	System.out.println(msg);
	    	// TODO issue the warning to the error log
	    }
    }
    
    /**
     * Get a Set of JDBC types that we will not warn about if they are
     * unmapped.
     */
    protected Set<Integer> createJdbcTypeIgnoreSet() {
	    int[] a = new int[]{
	    	Types.ARRAY,
	    	Types.DATALINK,
	    	Types.DISTINCT,
	    	Types.JAVA_OBJECT,
	    	Types.NULL,
	    	Types.OTHER,
	    	Types.REF,
	    	Types.STRUCT,
	    };
	    Set<Integer> ignore = new HashSet();
	    for (int i = a.length - 1; i >= 0; i--) {
	    	ignore.add(a[i]);
	    }
	    return ignore;
    }
    
    /**
     * Construct a fallback map of JDBC types to handle types not provided
     * by the RBD database definition.
     */
    protected Map<Integer, Integer> createJdbcTypeFallbackMap(
    		VendorDefinitionDocument.VendorDefinition vd, AnnotationRegistry reg) {
    	Map<Integer, Integer> ans = new HashMap();
    	int n = vd.sizeOfJdbcTypeFallbackArray();
    	for (int i = 0; i < n; i++) {
    		JdbcTypeFallback f = vd.getJdbcTypeFallbackArray(i);
    		int jdbcType = JdbcUtils.getJdbcTypeValue(f.getJdbc());
    		int alt = JdbcUtils.getJdbcTypeValue(f.getAlt());
    		ans.put(jdbcType, alt);
    	}
    	return ans;
    }
    
    /**
     * Return the datatype that is the best match for the jdbcType (from
     * java.sql.Types) and sqlType (actual database type name) or null if
     * none found.
     */
    protected JdbcTypeInfo findPredefinedDataTypeDefinition(
    		DatabaseDefinition dbdef, int jdbcType, String sqlType) {
		List tl = dbdef.getPredefinedDataTypeDefinitionsByJDBCEnumType(
				jdbcType);
		if (tl.isEmpty()) {
			return null;
		}
		JdbcTypeInfo info = new JdbcTypeInfo();
		info.jdbcType = jdbcType;
		info.dataTypeDef = (PredefinedDataTypeDefinition)tl.get(0);
		info.dataTypeName = (String)info.dataTypeDef.getName().get(0);
    	for (Iterator j = tl.iterator(); j.hasNext(); ) {
    		PredefinedDataTypeDefinition dt = 
    			(PredefinedDataTypeDefinition)j.next();
    		EList nl = dt.getName();
    		for (Iterator k = nl.iterator(); k.hasNext(); ) {
    			String n = (String)k.next();
	    		if (sqlType.equals(n)) {
	    			info.dataTypeDef = dt;
	    			info.dataTypeName = n;
	    			return info;
	    		}
	    		if (n.startsWith(sqlType)) {
	    			info.dataTypeDef = dt;
	    			info.dataTypeName = n;
	    			break;
	    		}
    		}
    	}
    	return info;
    }

    /**
     * Get the JdbcTypeInfo for the jdbcType or null if none.
     */
    public JdbcTypeInfo getJdbcTypeInfo(int jdbcType) {
    	return jdbcTypeMap.get(jdbcType);
    }
    
    /**
     * Is the jdbcType a numeric type i.e. one that uses precision and not
     * length?
     */
    public boolean isNumericType(int jdbcType) {
    	switch (jdbcType) {
    	case Types.DECIMAL:
    	case Types.NUMERIC:
    		return true;
    	}
    	return false;
    }
    
	/**
	 * Get the definitions for the simple types.
	 */
    public SimpleType[] getSimpleTypes() {
		return vd.getSimpleTypeArray();
	}

    /**
     * Get the definitions for the collection types. 
     */
    public CollectionType[] getCollectionTypes() {
    	return vd.getCollectionTypeArray();
    }
    
    /**
     * Discard information not needed after the model manager has completed
     * initialization.
     */
    public void initCleanup() {
    	vd = null;
    }
           
    /**
     * The selected RDB data type mapping for a JDBC type from java.sql.Types. 
     */
    public static class JdbcTypeInfo {

    	public int jdbcType;
    	public PredefinedDataTypeDefinition dataTypeDef;
    	public String dataTypeName;
    	
    	public String toString() {
    		return "JdbcTypeInfo " + jdbcType + " " + dataTypeName;
    	}
    	
    	public boolean isLengthSupported() {
    		if (dataTypeDef != null) {
    			return dataTypeDef.isLengthSupported() 
    				|| dataTypeDef.isPrecisionSupported();
    		}
        	return true;    		
    	}
    	
    	public boolean isScaleSupported() {
    		if (dataTypeDef != null) {
    			return dataTypeDef.isScaleSupported();
    		}
        	switch (jdbcType) {
        	case Types.DECIMAL:
        	case Types.NUMERIC:
        		return true;
        	}
        	return false;
    	}
    	
    	public boolean isNullableSupported() {
    		if (dataTypeDef != null) {
    			return dataTypeDef.isNullableSupported(); 
    		}
        	return true;    		    		
    	}
    	
    }

}
