#
# Copyright (c) 2024 Boeing
#
# This program and the accompanying materials are made
# available under the terms of the Eclipse Public License 2.0
# which is available at https://www.eclipse.org/legal/epl-2.0/
#
# SPDX-License-Identifier: EPL-2.0
#
# Contributors:
#     Boeing - initial API and implementation
# 
load("//bat/private:toolchains_repo.bzl", "PLATFORMS", "toolchains_repo")
load("//bat/private:exists.bzl", "assert_bat_exists_for_host", "bat_exists_for_os")
load("//bat/private:bat_versions.bzl", "BAT_VERSIONS")
load("//bat/private:host_alias.bzl","bat_repo_host_os_alias")
load("@aspect_bazel_lib//lib:repo_utils.bzl","repo_utils")
#highly derivative of rules_nodes/nodejs/repositories.bzl

DEFAULT_BAT_REPOSITORY="bat"
DEFAULT_BAT_VERSION = "1.0.3"
BUILT_IN_BAT_PLATFORMS = PLATFORMS.keys()
BAT_EXTRACT_DIR = "bin/bat"

GET_SCRIPT_DIR = """
# From stackoverflow.com
SOURCE="${BASH_SOURCE[0]}"
# Resolve $SOURCE until the file is no longer a symlink
while [ -h "$SOURCE" ]; do
  DIR="$(cd -P "$(dirname "$SOURCE" )" >/dev/null && pwd)"
  SOURCE="$(readlink "$SOURCE")"
  # if $SOURCE was a relative symlink, we need to resolve it relative to the
  # path where the symlink file was located.
  [[ $SOURCE != /* ]] && SOURCE="$DIR/$SOURCE"
done
SCRIPT_DIR="$(cd -P "$( dirname "$SOURCE" )" >/dev/null && pwd)"
"""

_ATTRS = {
    "bat_repositories": attr.string_list_dict(),
    "bat_urls":attr.string_list(
        default = [
            "https://download.eclipse.org/technology/osee/downloads/bat/{os}/{version}/bat{ext}"
        ]
    ),
    "bat_version":attr.string(
        default=DEFAULT_BAT_VERSION
    ),
    "platform": attr.string(
        doc = "Internal use only. Which platform to install as a toolchain. If unset, we assume the repository is named bat_[platform]",
        values = BUILT_IN_BAT_PLATFORMS,
    ),
}
def _download_bat(repository_ctx):
    host_os = repository_ctx.attr.platform or repository_ctx.name.split("bat_", 1)[1]
    bat_version = repository_ctx.attr.bat_version
    bat_repositories = repository_ctx.attr.bat_repositories
    if not bat_repositories.items():
        bat_repositories = BAT_VERSIONS
    if not bat_exists_for_os(bat_version, host_os, bat_repositories):
        return
    bat_urls = repository_ctx.attr.bat_urls
    version_host_os = "%s-%s" % (bat_version, host_os)
    if not version_host_os in bat_repositories:
        fail("Unknown BAT version-host %s" % version_host_os)
    os_prefix, version, ext, sha256 = bat_repositories[version_host_os]

    urls = [url.format(os=os_prefix, version=version, ext=ext) for url in bat_urls]
    repository_ctx.download(
        url = urls,
        output=BAT_EXTRACT_DIR,
        sha256=sha256,
        executable = True
    )

def _setup_bat(repository_ctx):
    bat_path = BAT_EXTRACT_DIR
    # bat_bin =("%s/bin/bat" % bat_path) if not repo_utils.is_windows(repository_ctx) else ("%s/bat.exe" % bat_path)
    bat_bin =("%s" % bat_path) if not repo_utils.is_windows(repository_ctx) else ("%s/bat.exe" % bat_path)
    ext = ".cmd" if repo_utils.is_windows(repository_ctx) else ".sh"
    bat_entry = "bin/bat%s" % ext
    relative_bat_bin = _strip_bin(bat_bin)
    relative_bat_entry = _strip_bin(bat_entry)
    
    if not repo_utils.is_windows(repository_ctx):
        repository_ctx.file("bin/bat.sh", content = """#!/usr/bin/env bash
# Generated by bat_repositories.bzl
# Immediately exit if any command fails.
set -e
{get_script_dir}
export PATH="$SCRIPT_DIR":$PATH
exec "$SCRIPT_DIR/{bat}" "$@"
""".format(
    get_script_dir= GET_SCRIPT_DIR,
    bat = relative_bat_bin
    ))
    else:
        repository_ctx.file("bin/bat.cmd", content = """
@echo off
SET SCRIPT_DIR=%~dp0
SET PATH=%SCRIPT_DIR%;%PATH%
CALL "%SCRIPT_DIR%\\{bat}" %*
""".format(bat = relative_bat_bin))
    build_content = """# Generated by bat_repositories.bzl
package(default_visibility = ["//visibility:public"])
exports_files(["{bat_entry}"])
alias(name="bat_bin",actual="{bat_bin}")
alias(name="bat",actual="{bat_entry}")
filegroup(
    name="bat_files",
    srcs=[":bat",":bat_bin"]
)
    """.format(
       bat_bin_export = "\n  \"%s\"," % bat_bin,
       bat_bin=bat_bin,
       bat_entry=bat_entry 
    )
    if repository_ctx.attr.platform:
        build_content +="""
load("@rules_osee//bat:toolchain.bzl","bat_toolchain")
bat_toolchain(
    name="bat_toolchain",
    target_tool=":bat_bin"
)
        """
        repository_ctx.file("BUILD.bazel",content=build_content)

def bat_register_toolchains(name = DEFAULT_BAT_REPOSITORY, register= True, ** kwargs):
    for platform in BUILT_IN_BAT_PLATFORMS:
        bat_repositories(
            name = name + "_" + platform,
            platform = platform,
            **kwargs
        )
        if register:
            native.register_toolchains(
                "@%s//:%s_toolchain_target" % (name, platform),
                "@%s//:%s_toolchain" % (name, platform),
            )
    toolchains_repo(
        name = name,
        user_bat_repository_name = name,
    )

def _strip_bin(path):
    if not path.startswith("bin/"):
        fail("Expected path to start with 'bin/' but was %s" % path)

    return path[len("bin/"):]
def _bat_repo_impl(repository_ctx):
    assert_bat_exists_for_host(repository_ctx)
    _download_bat(repository_ctx)
    _setup_bat(repository_ctx)
    #todo: add support for downloading the plconfig here if setup

bat_repositories = repository_rule(
    _bat_repo_impl,
    attrs=_ATTRS,
)