/*********************************************************************
 * Copyright (c) 2022 Boeing
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Boeing - initial API and implementation
 **********************************************************************/

package org.eclipse.osee.synchronization.rest;

import java.util.Objects;
import org.eclipse.osee.synchronization.rest.IdentifierType.Identifier;
import org.eclipse.osee.synchronization.util.IndentedString;

/**
 * A specialization of {@link AbstractHierarchyTreeGrove} for the storage of {@link SpecObjectGroveThing} objects needed for a
 * Synchronization Artifact.
 *
 * @author Loren K. Ashley
 */

public class SpecObjectGrove extends AbstractHierarchyTreeGrove {

   /**
    * Creates a new empty {@link SpecObjetGrove}.
    */

   SpecObjectGrove() {
      super(IdentifierType.SPEC_OBJECT);
   }

   @Override
   public GroveThing get(Identifier groveThingKey) {
      return null;
   }

   public SpecificationGroveThing getSpecification(Identifier specificationIdentifier) {
      var hierarchyTree = this.hierarchyTrees.get(specificationIdentifier);
      assert Objects.nonNull(hierarchyTree);
      var specification = hierarchyTree.get(specificationIdentifier);
      assert Objects.nonNull(specification);

      return (SpecificationGroveThing) specification;
   }

   public SpecObjectGroveThing getSpecObject(Identifier specificationIdentifier, Identifier specObjectIdentifier) {
      var hierarchyTree = this.hierarchyTrees.get(specificationIdentifier);
      assert Objects.nonNull(hierarchyTree);
      var specObject = hierarchyTree.get(specObjectIdentifier);
      assert Objects.nonNull(specObject);

      return (SpecObjectGroveThing) specObject;

   }

   /**
    * Recursively adds the hierarchical children of a SpecObjectGroveThing to the message.
    *
    * @param indent the current message indent level
    * @param message the {@link StringBuilder} to append to
    * @param rootGroveThingKey key of the root item on the hierarchy tree. Used to access the hierarchy for the desired
    * specification.
    * @param parentGroveThingKey the key for the SpecObjectGroveThing to generate messages for each child.
    * @return the appended to {@link StringBuilder}.
    */

   private StringBuilder toMessageChildrenKeys(int indent, StringBuilder message, Identifier rootGroveThingKey, Identifier parentGroveThingKey) {
      String indent0 = IndentedString.indentString(indent + 0);

      this.streamGroveThingChildKeys(rootGroveThingKey, parentGroveThingKey).forEach(childGroveThingKey -> {
         //@formatter:off
         message
            .append( indent0 ).append( "Spec Object: ").append( childGroveThingKey ).append( "\n" )
            ;
         //@formatter:on

         this.toMessageChildrenKeys(indent + 1, message, rootGroveThingKey, childGroveThingKey);
      });

      return message;
   }

   /**
    * Generates a debugging message with a hierarchical report of the {@link SpecObjectGroveThing} objects for each
    * {@link SpecificationGroveThing}.
    */

   @Override
   public StringBuilder toMessage(int indent, StringBuilder message) {
      var outMessage = (message != null) ? message : new StringBuilder(1 * 1024);
      var indent0 = IndentedString.indentString(indent + 0);
      var indent1 = IndentedString.indentString(indent + 1);

      //@formatter:off
      outMessage
         .append( indent0 ).append( "SpecObjectGrove:" ).append( "\n" );
      //@formatter:on

      this.streamRootGroveThingKeys().forEach(rootGroveThingKey -> {
         //@formatter:off
         outMessage
            .append( indent1 ).append( "Hierarcy Tree For: " ).append( rootGroveThingKey ).append( "\n" );
         //@formatter:on

         this.toMessageChildrenKeys(indent + 2, message, rootGroveThingKey, rootGroveThingKey);

      });

      return outMessage;
   }

   /**
    * Returns the message generated by the method {@link #toMessage} with an indent level of zero. {@inheritDoc}
    */

   @Override
   public String toString() {
      return this.toMessage(0, null).toString();
   }
}

/* EOF */
