/*********************************************************************
 * Copyright (c) 2021 Boeing
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Boeing - initial API and implementation
 **********************************************************************/
package org.eclipse.osee.mim.internal;

import java.lang.reflect.InvocationTargetException;
import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.stream.Collectors;
import org.eclipse.osee.framework.core.data.ArtifactId;
import org.eclipse.osee.framework.core.data.AttributeTypeId;
import org.eclipse.osee.framework.core.data.BranchId;
import org.eclipse.osee.framework.core.data.RelationTypeSide;
import org.eclipse.osee.framework.core.enums.CoreAttributeTypes;
import org.eclipse.osee.framework.core.enums.CoreRelationTypes;
import org.eclipse.osee.mim.ArtifactAccessor;
import org.eclipse.osee.mim.InterfaceElementApi;
import org.eclipse.osee.mim.InterfaceElementArrayApi;
import org.eclipse.osee.mim.InterfacePlatformTypeApi;
import org.eclipse.osee.mim.InterfaceStructureApi;
import org.eclipse.osee.mim.types.InterfaceStructureElementToken;
import org.eclipse.osee.mim.types.InterfaceStructureToken;
import org.eclipse.osee.mim.types.MimAttributeQuery;
import org.eclipse.osee.mim.types.PlatformTypeToken;
import org.eclipse.osee.orcs.OrcsApi;
import org.eclipse.osee.orcs.data.ArtifactReadable;

/**
 * @author Luciano T. Vaglienti
 */
public class InterfaceStructureApiImpl implements InterfaceStructureApi {

   private ArtifactAccessor<InterfaceStructureToken> accessor;
   private final InterfacePlatformTypeApi interfacePlatformTypeApi;
   private final InterfaceElementApi interfaceElementApi;
   private final InterfaceElementArrayApi interfaceElementArrayApi;
   private final List<AttributeTypeId> structureAttributeList;
   private final List<AttributeTypeId> elementAttributeList;
   private final List<RelationTypeSide> relations;

   InterfaceStructureApiImpl(OrcsApi orcsApi, InterfacePlatformTypeApi interfacePlatformTypeApi, InterfaceElementApi interfaceElementApi, InterfaceElementArrayApi interfaceElementArrayApi) {
      this.setAccessor(new InterfaceStructureAccessor(orcsApi));
      this.interfacePlatformTypeApi = interfacePlatformTypeApi;
      this.interfaceElementApi = interfaceElementApi;
      this.interfaceElementArrayApi = interfaceElementArrayApi;
      this.structureAttributeList = this.createStructureAttributeList();
      this.elementAttributeList = this.createElementAttributeList();
      this.relations = this.createRelationTypeSideList();
   }

   private ArtifactAccessor<InterfaceStructureToken> getAccessor() {
      return this.accessor;
   }

   /**
    * @param accessor the accessor to set
    */
   public void setAccessor(ArtifactAccessor<InterfaceStructureToken> accessor) {
      this.accessor = accessor;
   }

   private List<RelationTypeSide> createRelationTypeSideList() {
      List<RelationTypeSide> relations = new LinkedList<RelationTypeSide>();
      relations.add(CoreRelationTypes.InterfaceStructureContent_DataElement);
      relations.add(CoreRelationTypes.InterfaceElementPlatformType_PlatformType);
      return relations;
   }

   private List<AttributeTypeId> createStructureAttributeList() {
      List<AttributeTypeId> attributes = new LinkedList<AttributeTypeId>();
      attributes.add(CoreAttributeTypes.Name);
      attributes.add(CoreAttributeTypes.Description);
      attributes.add(CoreAttributeTypes.InterfaceStructureCategory);
      attributes.add(CoreAttributeTypes.InterfaceMinSimultaneity);
      attributes.add(CoreAttributeTypes.InterfaceMaxSimultaneity);
      attributes.add(CoreAttributeTypes.InterfaceTaskFileType);
      return attributes;
   }

   private List<AttributeTypeId> createElementAttributeList() {
      List<AttributeTypeId> attributes = new LinkedList<AttributeTypeId>();
      attributes.add(CoreAttributeTypes.Name);
      attributes.add(CoreAttributeTypes.Description);
      attributes.add(CoreAttributeTypes.Notes);
      attributes.add(CoreAttributeTypes.InterfaceElementAlterable);
      attributes.add(CoreAttributeTypes.InterfaceElementIndexEnd);
      attributes.add(CoreAttributeTypes.InterfaceElementIndexStart);
      return attributes;
   }

   private InterfaceStructureElementToken defaultSetUpElement(InterfaceStructureElementToken element, InterfaceStructureElementToken previousElement) {
      if (previousElement.isInvalid() && !previousElement.isAutogenerated()) {
         element.setBeginByte((double) 0);
         element.setBeginWord((double) 0);
      } else {
         element.setBeginByte((previousElement.getEndByte() + 1) % 4);
         element.setBeginWord(Math.floor(((previousElement.getEndWord() * 4) + previousElement.getEndByte() + 1) / 4));
      }
      return element;
   }

   private InterfaceStructureToken parseStructure(BranchId branch, InterfaceStructureToken structure) {
      return this.parseStructure(branch, structure, new LinkedList<InterfaceStructureElementToken>());
   }

   private InterfaceStructureToken parseStructure(BranchId branch, InterfaceStructureToken structure, String elementFilter) {
      return this.parseStructure(branch, structure,
         this.interfaceElementApi.getAllRelatedAndFilter(branch, ArtifactId.valueOf(structure.getId()), elementFilter));
   }

   private InterfaceStructureToken parseStructure(BranchId branch, InterfaceStructureToken structure, List<InterfaceStructureElementToken> defaultElements) {
      try {
         Collection<InterfaceStructureElementToken> elements = new LinkedList<>();
         elements.addAll(defaultElements.size() > 0 ? defaultElements : interfaceElementApi.getAllRelated(branch,
            ArtifactId.valueOf(structure.getId())));
         Collection<InterfaceStructureElementToken> tempElements = new LinkedList<>();
         if (elements.size() >= 2) {
            Iterator<InterfaceStructureElementToken> elementIterator = elements.iterator();
            InterfaceStructureElementToken previousElement = elementIterator.next();

            InterfaceStructureElementToken currentElement = elementIterator.next();
            previousElement.setBeginByte((double) 0);
            previousElement.setBeginWord((double) 0);
            tempElements.add(previousElement);
            if (!elementIterator.hasNext()) {
               /**
                * If currentElement = last, set it up so that it may be added/serialized
                */
               currentElement = this.defaultSetUpElement(currentElement, previousElement);
            }
            while (elementIterator.hasNext()) {

               InterfaceStructureElementToken nextElement = elementIterator.next();
               currentElement = this.defaultSetUpElement(currentElement, previousElement);
               if (currentElement.getInterfacePlatformTypeByteSize() >= 4) {
                  if (previousElement.getEndByte() != 3) {
                     /**
                      * Make sure elements of size word or greater start on 0
                      */
                     previousElement = new InterfaceStructureElementToken("Insert Spare",
                        "byte align spare for aligning to word start", (previousElement.getEndByte() + 1) % 4,
                        Math.floor(((previousElement.getEndWord() * 4) + previousElement.getEndByte() + 1) / 4),
                        (int) Math.floor(3 - (previousElement.getEndByte())), true);
                     tempElements.add(previousElement);
                  }
                  if (currentElement.getInterfacePlatformTypeWordSize() > 1 && (previousElement.getEndWord() + 1) % currentElement.getInterfacePlatformTypeWordSize() != 0) {
                     /**
                      * Make sure elements of size larger than 2 words start on m*n indexed words
                      */
                     previousElement = new InterfaceStructureElementToken("Insert Spare",
                        "byte align spare for byte alignment", (previousElement.getEndByte() + 1) % 4,
                        Math.floor(((previousElement.getEndWord() * 4) + previousElement.getEndByte() + 1) / 4),
                        (int) (Math.floor(
                           (currentElement.getInterfacePlatformTypeWordSize() - ((previousElement.getEndWord() + 1) % currentElement.getInterfacePlatformTypeWordSize()))) * 4) - 1);
                     tempElements.add(previousElement);
                     //make a spare to fill remaining area until beginWord % WordSize=1
                  }
                  //re-set up current Element based on spare
                  currentElement = this.defaultSetUpElement(currentElement, previousElement);
               }
               tempElements.add(currentElement);
               previousElement = currentElement;
               currentElement = nextElement;
            }
            /**
             * Handle last element outside of while loop
             */
            currentElement = this.defaultSetUpElement(currentElement, previousElement);
            if (currentElement.getInterfacePlatformTypeByteSize() >= 4) {
               if (previousElement.getEndByte() != 3) {
                  /**
                   * Make sure elements of size word or greater start on 0
                   */
                  previousElement = new InterfaceStructureElementToken("Insert Spare",
                     "byte align spare for aligning to word start", (previousElement.getEndByte() + 1) % 4,
                     Math.floor(((previousElement.getEndWord() * 4) + previousElement.getEndByte() + 1) / 4),
                     (int) Math.floor(3 - (previousElement.getEndByte())), true);
                  tempElements.add(previousElement);
               }
               if (currentElement.getInterfacePlatformTypeWordSize() > 1 && (previousElement.getEndWord() + 1) % currentElement.getInterfacePlatformTypeWordSize() != 0) {
                  /**
                   * Make sure elements of size larger than 2 words start on m*n indexed words
                   */
                  previousElement = new InterfaceStructureElementToken("Insert Spare",
                     "byte align spare for byte alignment", (previousElement.getEndByte() + 1) % 4,
                     Math.floor(((previousElement.getEndWord() * 4) + previousElement.getEndByte() + 1) / 4),
                     (int) (Math.floor(
                        (currentElement.getInterfacePlatformTypeWordSize() - ((previousElement.getEndWord() + 1) % currentElement.getInterfacePlatformTypeWordSize()))) * 4) - 1);
                  tempElements.add(previousElement);
                  //make a spare to fill remaining area until beginWord % WordSize=1
               }
               //re-set up current Element based on spare
               currentElement = this.defaultSetUpElement(currentElement, previousElement);
            }
            tempElements.add(currentElement);
            if (currentElement.getEndByte() != 3) {
               /**
                * Rule for making sure last element ends on last byte of word(no partials)
                */
               tempElements.add(
                  new InterfaceStructureElementToken("Insert Spare", "byte align spare for aligning to word start",
                     ((currentElement.getEndWord() * 4) + currentElement.getEndByte() + 1) % 4,
                     Math.floor(((currentElement.getEndWord() * 4) + currentElement.getEndByte() + 1) / 4),
                     (int) Math.floor(3 - (currentElement.getEndByte())), true));
            }
            if (currentElement.getEndWord() % 2 != 1) {
               /**
                * Rule for making sure next element on next structure sent is on boundary of 2n
                */
               currentElement =
                  new InterfaceStructureElementToken("Insert Spare", "byte align spare for byte alignment",
                     ((currentElement.getEndWord() * 4) + currentElement.getEndByte() + 1) % 4,
                     Math.floor(((currentElement.getEndWord() * 4) + currentElement.getEndByte() + 1) / 4), 4);
               tempElements.add(currentElement);
            }
            structure.setElements(tempElements);
         } else {
            /**
             * Condition for when less than 2 elements
             */
            InterfaceStructureElementToken lastElement = new InterfaceStructureElementToken("Insert Spare",
               "byte align spare for aligning to word start", 0.0, 0.0, 0);
            for (InterfaceStructureElementToken element : elements) {
               element.setBeginByte(0.0);
               element.setBeginWord(0.0);
               PlatformTypeToken currentPlatformType;
               currentPlatformType = this.interfacePlatformTypeApi.getAccessor().getByRelationWithoutId(branch,
                  CoreRelationTypes.InterfaceElementPlatformType_Element, ArtifactId.valueOf(element.getId()),
                  PlatformTypeToken.class);
               element.setPlatformTypeId(currentPlatformType.getId());
               element.setPlatformTypeName(currentPlatformType.getName());
               element.setInterfacePlatformTypeBitSize(currentPlatformType.getInterfacePlatformTypeBitSize());
               element.setLogicalType(
                  currentPlatformType.getInterfaceLogicalType() != null ? currentPlatformType.getInterfaceLogicalType() : "");
               element.setInterfacePlatformTypeMinval(
                  currentPlatformType.getInterfacePlatformTypeMinval() != null ? currentPlatformType.getInterfacePlatformTypeMinval() : "");
               element.setInterfacePlatformTypeMaxval(
                  currentPlatformType.getInterfacePlatformTypeMaxval() != null ? currentPlatformType.getInterfacePlatformTypeMaxval() : "");
               element.setInterfacePlatformTypeDefaultValue(
                  currentPlatformType.getInterfacePlatformTypeDefaultValue() != null ? currentPlatformType.getInterfacePlatformTypeDefaultValue() : "");
               element.setUnits(
                  currentPlatformType.getInterfacePlatformTypeUnits() != null ? currentPlatformType.getInterfacePlatformTypeUnits() : "");
               lastElement = element;
            }
            tempElements.addAll(elements);
            if (lastElement.getEndByte() != 3) {
               /**
                * Rule for making sure last element ends on last byte of word(no partials)
                */
               tempElements.add(
                  new InterfaceStructureElementToken("Insert Spare", "byte align spare for aligning to word start",
                     ((lastElement.getEndWord() * 4) + lastElement.getEndByte() + 1) % 4,
                     Math.floor(((lastElement.getEndWord() * 4) + lastElement.getEndByte() + 1) / 4),
                     (int) Math.floor(3 - (lastElement.getEndByte())), true));
            }
            if (lastElement.getEndWord() % 2 != 1) {
               /**
                * Rule for making sure next element on next structure sent is on boundary of 2n
                */
               lastElement = new InterfaceStructureElementToken("Insert Spare", "byte align spare for byte alignment",
                  ((lastElement.getEndWord() * 4) + lastElement.getEndByte() + 1) % 4,
                  Math.floor(((lastElement.getEndWord() * 4) + lastElement.getEndByte() + 1) / 4), 4);
               tempElements.add(lastElement);
            }
            structure.setElements(tempElements);

         }
      } catch (Exception ex) {
         //do nothing
      }
      return structure;
   }

   @Override
   public List<InterfaceStructureToken> getAllRelated(BranchId branch, ArtifactId subMessageId) {
      List<InterfaceStructureToken> structureList = new LinkedList<InterfaceStructureToken>();
      try {
         structureList = (List<InterfaceStructureToken>) this.getAccessor().getAllByRelation(branch,
            CoreRelationTypes.InterfaceSubMessageContent_SubMessage, subMessageId, this.getFollowRelationDetails(),
            InterfaceStructureToken.class);
         for (InterfaceStructureToken structure : structureList) {
            structure = this.parseStructure(branch, structure, structure.getElements());
         }

         return structureList;
      } catch (Exception ex) {
         System.out.println(ex);
         return structureList;
      }
   }

   @Override
   public List<InterfaceStructureToken> getAll(BranchId branch) {
      List<InterfaceStructureToken> structureList = new LinkedList<InterfaceStructureToken>();
      try {
         structureList = (List<InterfaceStructureToken>) this.getAccessor().getAll(branch,
            this.getFollowRelationDetails(), InterfaceStructureToken.class);
         for (InterfaceStructureToken structure : structureList) {
            structure = this.parseStructure(branch, structure);
         }

         return structureList;
      } catch (Exception ex) {
         System.out.println(ex);
         return structureList;
      }
   }

   @Override
   public List<InterfaceStructureToken> getAllWithoutRelations(BranchId branch) {
      List<InterfaceStructureToken> structureList = new LinkedList<InterfaceStructureToken>();
      try {
         structureList = (List<InterfaceStructureToken>) this.getAccessor().getAll(branch,
            this.getFollowRelationDetails(), InterfaceStructureToken.class);
         return structureList;
      } catch (Exception ex) {
         System.out.println(ex);
         return structureList;
      }
   }

   @Override
   public List<InterfaceStructureToken> getFiltered(BranchId branch, String filter) {
      List<InterfaceStructureToken> structureList = new LinkedList<InterfaceStructureToken>();
      try {
         structureList = (List<InterfaceStructureToken>) this.getAccessor().getAllByFilter(branch, filter,
            this.structureAttributeList, this.getFollowRelationDetails(), InterfaceStructureToken.class);
         for (InterfaceStructureToken structure : structureList) {
            structure = this.parseStructure(branch, structure);
         }

         return structureList;
      } catch (Exception ex) {
         System.out.println(ex);
         return structureList;
      }
   }

   @Override
   public List<InterfaceStructureToken> getFilteredWithoutRelations(BranchId branch, String filter) {
      List<InterfaceStructureToken> structureList = new LinkedList<InterfaceStructureToken>();
      try {
         structureList = (List<InterfaceStructureToken>) this.getAccessor().getAllByFilter(branch, filter,
            this.structureAttributeList, this.getFollowRelationDetails(), InterfaceStructureToken.class);
         return structureList;
      } catch (Exception ex) {
         System.out.println(ex);
         return structureList;
      }
   }

   @Override
   public InterfaceStructureToken get(BranchId branch, ArtifactId artId) {
      InterfaceStructureToken structure;
      try {
         structure =
            this.getAccessor().get(branch, artId, this.getFollowRelationDetails(), InterfaceStructureToken.class);
      } catch (InstantiationException | IllegalAccessException | IllegalArgumentException | InvocationTargetException
         | NoSuchMethodException | SecurityException ex) {
         System.out.println(ex);
         structure = InterfaceStructureToken.SENTINEL;
      }
      return structure;
   }

   @Override
   public InterfaceStructureToken getRelated(BranchId branch, ArtifactId subMessageId, ArtifactId structureId) {
      InterfaceStructureToken structure;
      try {
         structure = this.getAccessor().getByRelation(branch, structureId,
            CoreRelationTypes.InterfaceSubMessageContent_SubMessage, subMessageId, this.getFollowRelationDetails(),
            InterfaceStructureToken.class);
         structure = this.parseStructure(branch, structure);

         return structure;
      } catch (Exception ex) {
         System.out.println(ex);
         structure = new InterfaceStructureToken();
         return structure;
      }
   }

   @Override
   public List<InterfaceStructureToken> getAllRelatedAndFilter(BranchId branch, ArtifactId subMessageId, String filter) {
      List<InterfaceStructureToken> totalStructureList = new LinkedList<InterfaceStructureToken>();
      try {
         /**
          * Gets total list of all related structures for lookup later
          */
         totalStructureList = this.getAllRelated(branch, subMessageId);
         for (InterfaceStructureToken structure : totalStructureList) {
            structure = this.parseStructure(branch, structure);
         }
         /**
          * Gets all structures that match filter conditions
          */
         List<InterfaceStructureToken> structureList =
            (List<InterfaceStructureToken>) this.getAccessor().getAllByRelationAndFilter(branch,
               CoreRelationTypes.InterfaceSubMessageContent_SubMessage, subMessageId, filter,
               this.structureAttributeList, this.getFollowRelationDetails(), InterfaceStructureToken.class);
         for (InterfaceStructureToken structure : structureList) {
            structure = this.parseStructure(branch, structure);
         }
         /**
          * Gets all elements that match filter conditions, then find their related structures and attach
          */
         List<InterfaceStructureElementToken> elements = this.interfaceElementApi.getFiltered(branch, filter);
         for (InterfaceStructureElementToken element : elements) {
            List<InterfaceStructureToken> subStructureList =
               (List<InterfaceStructureToken>) this.getAccessor().getAllByRelation(branch,
                  CoreRelationTypes.InterfaceStructureContent_DataElement, ArtifactId.valueOf(element.getId()),
                  this.getFollowRelationDetails(), InterfaceStructureToken.class);
            for (InterfaceStructureToken alternateStructure : subStructureList) {
               if (totalStructureList.indexOf(alternateStructure) != -1 && totalStructureList.get(
                  totalStructureList.indexOf(alternateStructure)).getElements().indexOf(element) != -1) {
                  totalStructureList.set(totalStructureList.indexOf(alternateStructure),
                     this.parseStructure(branch, totalStructureList.get(totalStructureList.indexOf(alternateStructure)),
                        totalStructureList.get(totalStructureList.indexOf(alternateStructure)).getElements()));
               }
               List<InterfaceStructureElementToken> elementList = new LinkedList<InterfaceStructureElementToken>();
               elementList.add(element);
               alternateStructure.setElements(elementList);
               if (totalStructureList.indexOf(alternateStructure) != -1) {
                  InterfaceStructureToken tempStructure =
                     totalStructureList.get(totalStructureList.indexOf(alternateStructure));
                  if (!structureList.contains(alternateStructure)) {
                     structureList.add(alternateStructure);
                  } else {
                     InterfaceStructureToken tempStructure2 =
                        structureList.get(structureList.indexOf(alternateStructure));
                     structureList.remove(alternateStructure);
                     tempStructure2.getElements().add(element);
                     structureList.add(tempStructure2);
                  }
               }
            }
         }
         return structureList;
      } catch (Exception ex) {
         System.out.println(ex);
         return totalStructureList;
      }
   }

   @Override
   public InterfaceStructureToken getRelatedAndFilter(BranchId branch, ArtifactId subMessageId, ArtifactId structureId, String filter) {
      InterfaceStructureToken structure;
      try {
         structure = this.getAccessor().getByRelation(branch, structureId,
            CoreRelationTypes.InterfaceSubMessageContent_SubMessage, subMessageId, this.getFollowRelationDetails(),
            InterfaceStructureToken.class);
         structure = this.parseStructure(branch, structure, filter);

         return structure;
      } catch (Exception ex) {
         System.out.println(ex);
         structure = new InterfaceStructureToken();
         return structure;
      }
   }

   @Override
   public Collection<InterfaceStructureToken> query(BranchId branch, MimAttributeQuery query) {
      try {
         List<InterfaceStructureToken> structureList = new LinkedList<InterfaceStructureToken>();
         structureList = (List<InterfaceStructureToken>) this.getAccessor().getAllByQuery(branch, query,
            this.getFollowRelationDetails(), InterfaceStructureToken.class);
         for (InterfaceStructureToken structure : structureList) {
            structure = this.parseStructure(branch, structure);
         }
         return structureList;
      } catch (InstantiationException | IllegalAccessException | IllegalArgumentException | InvocationTargetException
         | NoSuchMethodException | SecurityException ex) {
         System.out.println(ex);
      }
      return new LinkedList<InterfaceStructureToken>();
   }

   @Override
   public List<RelationTypeSide> getFollowRelationDetails() {
      return this.relations;
   }

   @Override
   public List<InterfaceStructureToken> getAllRelatedFromElement(InterfaceStructureElementToken element) {
      ArtifactReadable elementReadable = element.getArtifactReadable();
      if (elementReadable != null && elementReadable.isValid()) {
         return elementReadable.getRelated(
            CoreRelationTypes.InterfaceStructureContent_Structure).getList().stream().map(
               a -> new InterfaceStructureToken(a)).collect(Collectors.toList());
      }
      return new LinkedList<>();
   }

   @Override
   public InterfaceStructureToken getWithAllParentRelations(BranchId branch, ArtifactId structureId) {
      try {
         List<RelationTypeSide> parentRelations = Arrays.asList(CoreRelationTypes.InterfaceSubMessageContent_SubMessage,
            CoreRelationTypes.InterfaceMessageSubMessageContent_Message,
            CoreRelationTypes.InterfaceConnectionContent_Connection);
         return this.getAccessor().get(branch, structureId, parentRelations, InterfaceStructureToken.class);
      } catch (InstantiationException | IllegalAccessException | IllegalArgumentException | InvocationTargetException
         | NoSuchMethodException | SecurityException ex) {
         System.out.println(ex);
      }
      return InterfaceStructureToken.SENTINEL;
   }

}
