# This file tests the tclfCmd.c file.
#
# This file contains a collection of tests for one or more of the Tcl
# built-in commands.  Sourcing this file into Tcl runs the tests and
# generates output for errors.  No output means no errors were found.
#
# Copyright (c) 1997 Sun Microsystems, Inc.
# Copyright (c) 1998-1999 by Scriptics Corporation.
#
# See the file "license.terms" for information on usage and redistribution
# of this file, and for a DISCLAIMER OF ALL WARRANTIES.
#
# RCS: @(#) $Id: macFCmd.tcl,v 1.1 2006/12/29 09:02:53 fourdman Exp $
#

if {[lsearch [namespace children] ::tcltest] == -1} {
    package require tcltest
    namespace import -force ::tcltest::*
}

# These tests really need to be run from a writable directory, which
# it is assumed [temporaryDirectory] is.
set oldcwd [pwd]
cd [temporaryDirectory]

catch {file delete -force foo.dir}
file mkdir foo.dir
if {[catch {file attributes foo.dir -readonly 1}]} {
    set ::tcltest::testConstraints(fileSharing) 0
    set ::tcltest::testConstraints(notFileSharing) 1
} else {
    set ::tcltest::testConstraints(fileSharing) 1
    set ::tcltest::testConstraints(notFileSharing) 0
}
file delete -force foo.dir

test macFCmd-1.1 {GetFileFinderAttributes - no file} {macOnly} {
    catch {file delete -force foo.file}
    list [catch {file attributes foo.file -creator} msg] $msg
} {1 {could not read "foo.file": no such file or directory}}
test macFCmd-1.2 {GetFileFinderAttributes - creator} {macOnly} {
    catch {file delete -force foo.file}
    catch {close [open foo.file w]}
    list [catch {file attributes foo.file -creator} msg] \
	    [regexp {MPW |CWIE} $msg] [file delete -force foo.file]
} {0 1 {}}
test macFCmd-1.3 {GetFileFinderAttributes - type} {macOnly} {
    catch {file delete -force foo.file}
    catch {close [open foo.file w]}
    list [catch {file attributes foo.file -type} msg] $msg \
	    [file delete -force foo.file]
} {0 TEXT {}}
test macFCmd-1.4 {GetFileFinderAttributes - not hidden} {macOnly} {
    catch {file delete -force foo.file}
    catch {close [open foo.file w]}
    list [catch {file attributes foo.file -hidden} msg] $msg \
	    [file delete -force foo.file]
} {0 0 {}}
test macFCmd-1.5 {GetFileFinderAttributes - hidden} {macOnly} {
    catch {file delete -force foo.file}
    catch {close [open foo.file w]}
    file attributes foo.file -hidden 1
    list [catch {file attributes foo.file -hidden} msg] $msg \
	    [file delete -force foo.file]
} {0 1 {}}
test macFCmd-1.6 {GetFileFinderAttributes - folder creator} {macOnly} {
    catch {file delete -force foo.dir}
    file mkdir foo.dir
    list [catch {file attributes foo.dir -creator} msg] $msg \
	    [file delete -force foo.dir]
} {0 Fldr {}}
test macFCmd-1.7 {GetFileFinderAttributes - folder type} {macOnly} {
    catch {file delete -force foo.dir}
    file mkdir foo.dir
    list [catch {file attributes foo.dir -type} msg] $msg \
	    [file delete -force foo.dir]
} {0 Fldr {}}
test macFCmd-1.8 {GetFileFinderAttributes - folder hidden} {macOnly} {
    catch {file delete -force foo.dir}
    file mkdir foo.dir
    list [catch {file attributes foo.dir -hidden} msg] $msg \
	    [file delete -force foo.dir]
} {0 0 {}}

test macFCmd-2.1 {GetFileReadOnly - bad file} {macOnly} {
    catch {file delete -force foo.file}
    list [catch {file attributes foo.file -readonly} msg] $msg
} {1 {could not read "foo.file": no such file or directory}}
test macFCmd-2.2 {GetFileReadOnly - file not read only} {macOnly} {
    catch {file delete -force foo.file}
    close [open foo.file w]
    list [catch {file attributes foo.file -readonly} msg] $msg \
	    [file delete -force foo.file]
} {0 0 {}}
test macFCmd-2.3 {GetFileReadOnly - file read only} {macOnly} {
    catch {file delete -force foo.file}
    close [open foo.file w]
    file attributes foo.file -readonly 1
    list [catch {file attributes foo.file -readonly} msg] $msg \
	    [file delete -force foo.file]
} {0 1 {}}
test macFCmd-2.4 {GetFileReadOnly - directory not read only} {macOnly} {
    catch {file delete -force foo.dir}
    file mkdir foo.dir
    list [catch {file attributes foo.dir -readonly} msg] $msg \
	    [file delete -force foo.dir]
} {0 0 {}}
test macFCmd-2.5 {GetFileReadOnly - directory read only} {macOnly fileSharing} {
    catch {file delete -force foo.dir}
    file mkdir foo.dir
    file attributes foo.dir -readonly 1
    list [catch {file attributes foo.dir -readonly} msg] $msg \
	    [file delete -force foo.dir]
} {0 1 {}}

test macFCmd-3.1 {SetFileFinderAttributes - bad file} {macOnly} {
    catch {file delete -force foo.file}
    list [catch {file attributes foo.file -creator FOOO} msg] $msg
} {1 {could not read "foo.file": no such file or directory}}
test macFCmd-3.2 {SetFileFinderAttributes - creator} {macOnly} {
    catch {file delete -force foo.file}
    close [open foo.file w]
    list [catch {file attributes foo.file -creator FOOO} msg] $msg \
	    [file attributes foo.file -creator] [file delete -force foo.file]
} {0 {} FOOO {}}
test macFCmd-3.3 {SetFileFinderAttributes - bad creator} {macOnly} {
    catch {file delete -force foo.file}
    close [open foo.file w]
    list [catch {file attributes foo.file -creator 0} msg] $msg \
	    [file delete -force foo.file]
} {1 {expected Macintosh OS type but got "0"} {}}
test macFCmd-3.4 {SetFileFinderAttributes - hidden} {macOnly} {
    catch {file delete -force foo.file}
    close [open foo.file w]
    list [catch {file attributes foo.file -hidden 1} msg] $msg \
	    [file attributes foo.file -hidden] [file delete -force foo.file]
} {0 {} 1 {}}
test macFCmd-3.5 {SetFileFinderAttributes - type} {macOnly} {
    catch {file delete -force foo.file}
    close [open foo.file w]
    list [catch {file attributes foo.file -type FOOO} msg] $msg \
	    [file attributes foo.file -type] [file delete -force foo.file]
} {0 {} FOOO {}}
test macFCmd-3.6 {SetFileFinderAttributes - bad type} {macOnly} {
    catch {file delete -force foo.file}
    close [open foo.file w]
    list [catch {file attributes foo.file -type 0} msg] $msg \
	    [file delete -force foo.file]
} {1 {expected Macintosh OS type but got "0"} {}}
test macFCmd-3.7 {SetFileFinderAttributes - directory} {macOnly} {
    catch {file delete -force foo.dir}
    file mkdir foo.dir
    list [catch {file attributes foo.dir -creator FOOO} msg] \
	    $msg [file delete -force foo.dir]
} {1 {cannot set -creator: "foo.dir" is a directory} {}}

test macFCmd-4.1 {SetFileReadOnly - bad file} {macOnly} {
    catch {file delete -force foo.file}
    list [catch {file attributes foo.file -readonly 1} msg] $msg
} {1 {could not read "foo.file": no such file or directory}}
test macFCmd-4.2 {SetFileReadOnly - file not readonly} {macOnly} {
    catch {file delete -force foo.file}
    close [open foo.file w]
    list [catch {file attributes foo.file -readonly 0} msg] \
	    $msg [file attributes foo.file -readonly] [file delete -force foo.file]
} {0 {} 0 {}}
test macFCmd-4.3 {SetFileReadOnly - file readonly} {macOnly} {
    catch {file delete -force foo.file}
    close [open foo.file w]
    list [catch {file attributes foo.file -readonly 1} msg] \
	    $msg [file attributes foo.file -readonly] [file delete -force foo.file]
} {0 {} 1 {}}
test macFCmd-4.4 {SetFileReadOnly - directory not readonly} \
	{macOnly fileSharing} {
    catch {file delete -force foo.dir}
    file mkdir foo.dir
    list [catch {file attributes foo.dir -readonly 0} msg] \
	    $msg [file attributes foo.dir -readonly] [file delete -force foo.dir]
} {0 {} 0 {}}
test macFCmd-4.5 {SetFileReadOnly - directory not readonly} \
	{macOnly notFileSharing} {
    catch {file delete -force foo.dir}
    file mkdir foo.dir
    list [catch {file attributes foo.dir -readonly 0} msg] $msg \
	    [file delete -force foo.dir]
} {1 {cannot set a directory to read-only when File Sharing is turned off} {}}
test macFCmd-4.6 {SetFileReadOnly - directory readonly} {macOnly fileSharing} {
    catch {file delete -force foo.dir}
    file mkdir foo.dir
    list [catch {file attributes foo.dir -readonly 1} msg] $msg \
	    [file attributes foo.dir -readonly] [file delete -force foo.dir]
} {0 {} 1 {}}
test macFCmd-4.7 {SetFileReadOnly - directory readonly} {macOnly notFileSharing} {
    catch {file delete -force foo.dir}
    file mkdir foo.dir
    list [catch {file attributes foo.dir -readonly 1} msg] $msg \
	    [file delete -force foo.dir]
} {1 {cannot set a directory to read-only when File Sharing is turned off} {}}

# cleanup
cd $oldcwd
::tcltest::cleanupTests
return
