/**
 * <copyright>
 * 
 * Copyright (c) 2014-2016 itemis and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     itemis - Initial API and implementation
 *     itemis - [503063] Provide launching support for Sphinx Workflows
 * 
 * </copyright>
 */
package org.eclipse.sphinx.jdt.util;

import com.google.common.base.Objects;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.regex.Pattern;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.launching.IVMInstall;
import org.eclipse.jdt.launching.IVMInstall2;
import org.eclipse.jdt.launching.JavaRuntime;
import org.eclipse.osgi.util.NLS;
import org.eclipse.sphinx.jdt.internal.Activator;
import org.eclipse.sphinx.jdt.internal.messages.Messages;
import org.eclipse.sphinx.platform.util.ExtendedPlatform;
import org.eclipse.sphinx.platform.util.PlatformLogUtil;
import org.eclipse.sphinx.platform.util.StatusUtil;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;

@SuppressWarnings("all")
public class JavaExtensions {
  public static final String VERSION_1_8 = "1.8";
  
  public static final String DEFAULT_OUTPUT_FOLDER_NAME = "bin";
  
  public static final Pattern CLASS_NAME_PATTERN = Pattern.compile("^([a-z][a-z_0-9]*\\.)*[A-Z_]($[A-Z_]|[\\w_])*$");
  
  private static final String PLUGIN_ID_VERSION_SEPARATOR = "_";
  
  public static File getFile(final IClasspathEntry entry) {
    File _xifexpression = null;
    boolean _exists = entry.getPath().toFile().exists();
    if (_exists) {
      _xifexpression = entry.getPath().toFile();
    } else {
      _xifexpression = ResourcesPlugin.getWorkspace().getRoot().getLocation().append(entry.getPath()).toFile();
    }
    return _xifexpression;
  }
  
  public static IJavaProject getJavaProject(final String projectName) {
    return JavaCore.create(ResourcesPlugin.getWorkspace().getRoot().getProject(projectName));
  }
  
  /**
   * Returns the absolute path in the local file system corresponding to given workspace-relative path.
   * 
   * @param workspacePath the workspace-relative path to some resource in the workspace
   * 
   * @return the absolute path in the local file system corresponding to given <code>workspacePath</code>, or null if no path can be determined
   */
  public static IPath getLocation(final IPath workspacePath) {
    IResource _findMember = ResourcesPlugin.getWorkspace().getRoot().findMember(workspacePath);
    IPath _location = null;
    if (_findMember!=null) {
      _location=_findMember.getLocation();
    }
    return _location;
  }
  
  /**
   * Returns the plug-in installation location. It is composed of the Platform's installation location
   * followed by the name of the "plugins" folder inside it.
   * 
   * @return The plug-in installation location.
   */
  public static IPath getPluginInstallLocation() {
    String _path = Platform.getInstallLocation().getURL().getPath();
    return new Path(_path).append("plugins");
  }
  
  /**
   * Returns the common postfix of the classpath root locations of "dev mode" plug-ins. It is equal to the name
   * of the default output folder ("bin") in Java projects.
   * 
   * @return The common classpath root location postfix of "dev" mode plug-ins.
   */
  public static String getDevModePluginClasspathRootLocationPostfix() {
    return JavaExtensions.DEFAULT_OUTPUT_FOLDER_NAME;
  }
  
  public static boolean isInstalledPluginClasspathRootLocationOf(final String pluginId, final IPath classpathLocation) {
    boolean _xblockexpression = false;
    {
      Assert.isNotNull(classpathLocation);
      _xblockexpression = (JavaExtensions.getPluginInstallLocation().isPrefixOf(classpathLocation) && classpathLocation.lastSegment().split(JavaExtensions.PLUGIN_ID_VERSION_SEPARATOR)[0].equals(pluginId));
    }
    return _xblockexpression;
  }
  
  public static boolean isDevModePluginClasspathLocationOf(final String pluginId, final IPath classpathLocation) {
    boolean _xblockexpression = false;
    {
      Assert.isNotNull(classpathLocation);
      _xblockexpression = classpathLocation.toString().contains(new Path(pluginId).append(JavaExtensions.getDevModePluginClasspathRootLocationPostfix()).toString());
    }
    return _xblockexpression;
  }
  
  public static Bundle getBundle(final URL pluginClasspathRootLocationURL) {
    try {
      Object _xblockexpression = null;
      {
        Assert.isNotNull(pluginClasspathRootLocationURL);
        String _path = pluginClasspathRootLocationURL.getPath();
        IPath pluginClasspathRootLocation = new Path(_path);
        boolean _isPrefixOf = JavaExtensions.getPluginInstallLocation().isPrefixOf(pluginClasspathRootLocation);
        if (_isPrefixOf) {
          return Activator.getContext().getBundle(("reference:" + pluginClasspathRootLocationURL));
        }
        boolean _equals = pluginClasspathRootLocation.lastSegment().equals(JavaExtensions.getDevModePluginClasspathRootLocationPostfix());
        if (_equals) {
          BundleContext _context = Activator.getContext();
          URL _uRL = pluginClasspathRootLocation.removeLastSegments(1).toFile().toURI().toURL();
          String _plus = ("reference:" + _uRL);
          return _context.getBundle(_plus);
        }
        _xblockexpression = null;
      }
      return ((Bundle)_xblockexpression);
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public static IPath getPluginClasspathRootLocation(final String pluginId) {
    final Bundle bundle = ExtendedPlatform.loadBundle(pluginId);
    boolean _notEquals = (!Objects.equal(bundle, null));
    if (_notEquals) {
      try {
        final URL classpathRootURL = bundle.getResource("/");
        final URL resolvedClasspathRootURL = FileLocator.resolve(classpathRootURL);
        String _path = resolvedClasspathRootURL.getPath();
        return new Path(_path).removeTrailingSeparator();
      } catch (final Throwable _t) {
        if (_t instanceof IOException) {
          final IOException ex = (IOException)_t;
          final IStatus status = StatusUtil.createErrorStatus(Activator.getDefault(), ex);
          Activator.getDefault().getLog().log(status);
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
    }
    return null;
  }
  
  public static void validateCompilerCompliance(final String compliance) {
    final IVMInstall install = JavaRuntime.getDefaultVMInstall();
    if ((install instanceof IVMInstall2)) {
      final String compilerCompliance = JavaExtensions.getCompilerCompliance(((IVMInstall2)install), compliance);
      boolean _equals = compilerCompliance.equals(compliance);
      boolean _not = (!_equals);
      if (_not) {
        try {
          final float complianceToSet = Float.parseFloat(compliance);
          final float complianceFromVM = Float.parseFloat(compilerCompliance);
          Assert.isLegal((complianceToSet <= complianceFromVM), 
            NLS.bind(Messages.error_JRECompliance_NotCompatible, compliance, compilerCompliance));
        } catch (final Throwable _t) {
          if (_t instanceof NumberFormatException) {
            final NumberFormatException ex = (NumberFormatException)_t;
            PlatformLogUtil.logAsWarning(Activator.getDefault(), ex);
          } else {
            throw Exceptions.sneakyThrow(_t);
          }
        }
      }
    }
  }
  
  private static String getCompilerCompliance(final IVMInstall2 vmInstall, final String defaultCompliance) {
    Assert.isNotNull(vmInstall);
    final String version = vmInstall.getJavaVersion();
    boolean _equals = Objects.equal(version, null);
    if (_equals) {
      return defaultCompliance;
    } else {
      boolean _startsWith = version.startsWith(JavaExtensions.VERSION_1_8);
      if (_startsWith) {
        return JavaExtensions.VERSION_1_8;
      } else {
        boolean _startsWith_1 = version.startsWith(JavaCore.VERSION_1_7);
        if (_startsWith_1) {
          return JavaCore.VERSION_1_7;
        } else {
          boolean _startsWith_2 = version.startsWith(JavaCore.VERSION_1_6);
          if (_startsWith_2) {
            return JavaCore.VERSION_1_6;
          } else {
            boolean _startsWith_3 = version.startsWith(JavaCore.VERSION_1_5);
            if (_startsWith_3) {
              return JavaCore.VERSION_1_5;
            } else {
              boolean _startsWith_4 = version.startsWith(JavaCore.VERSION_1_4);
              if (_startsWith_4) {
                return JavaCore.VERSION_1_4;
              } else {
                boolean _startsWith_5 = version.startsWith(JavaCore.VERSION_1_3);
                if (_startsWith_5) {
                  return JavaCore.VERSION_1_3;
                } else {
                  boolean _startsWith_6 = version.startsWith(JavaCore.VERSION_1_2);
                  if (_startsWith_6) {
                    return JavaCore.VERSION_1_3;
                  } else {
                    boolean _startsWith_7 = version.startsWith(JavaCore.VERSION_1_1);
                    if (_startsWith_7) {
                      return JavaCore.VERSION_1_3;
                    }
                  }
                }
              }
            }
          }
        }
      }
    }
    return defaultCompliance;
  }
}
