/**
 * Copyright (c) 2017 itemis AG and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Tamas Miklossy (itemis AG) - initial API and implementation
 */
package org.eclipse.gef.dot.internal.language;

import com.google.common.base.Objects;
import java.util.List;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.gef.dot.internal.DotAttributes;
import org.eclipse.gef.dot.internal.language.dot.AttrList;
import org.eclipse.gef.dot.internal.language.dot.AttrStmt;
import org.eclipse.gef.dot.internal.language.dot.Attribute;
import org.eclipse.gef.dot.internal.language.dot.AttributeType;
import org.eclipse.gef.dot.internal.language.dot.DotGraph;
import org.eclipse.gef.dot.internal.language.dot.EdgeStmtNode;
import org.eclipse.gef.dot.internal.language.dot.NodeStmt;
import org.eclipse.gef.dot.internal.language.dot.Stmt;
import org.eclipse.gef.dot.internal.language.dot.Subgraph;
import org.eclipse.gef.dot.internal.language.terminals.ID;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * This class provides helper methods for walking the DOT abstract syntax tree.
 */
@SuppressWarnings("all")
public class DotAstHelper {
  /**
   * Returns the color scheme attribute value that is set for the given
   * attribute.
   * 
   * @param attribute
   *            The attribute to determine the color scheme attribute value
   *            for.
   * @return The color scheme value that is set for the given attribute, or
   *         null if it cannot be determined.
   */
  public static String getColorSchemeAttributeValue(final Attribute attribute) {
    EdgeStmtNode edgeStmtNode = EcoreUtil2.<EdgeStmtNode>getContainerOfType(attribute, EdgeStmtNode.class);
    boolean _tripleNotEquals = (edgeStmtNode != null);
    if (_tripleNotEquals) {
      EList<AttrList> _attrLists = edgeStmtNode.getAttrLists();
      ID colorScheme = DotAstHelper.getAttributeValue(_attrLists, 
        DotAttributes.COLORSCHEME__GCNE);
      boolean _tripleNotEquals_1 = (colorScheme != null);
      if (_tripleNotEquals_1) {
        return colorScheme.toValue();
      }
      ID _globalColorSchemeValue = DotAstHelper.getGlobalColorSchemeValue(edgeStmtNode, AttributeType.EDGE);
      colorScheme = _globalColorSchemeValue;
      boolean _tripleNotEquals_2 = (colorScheme != null);
      if (_tripleNotEquals_2) {
        return colorScheme.toValue();
      }
    }
    final NodeStmt nodeStmt = EcoreUtil2.<NodeStmt>getContainerOfType(attribute, NodeStmt.class);
    boolean _tripleNotEquals_3 = (nodeStmt != null);
    if (_tripleNotEquals_3) {
      EList<AttrList> _attrLists_1 = nodeStmt.getAttrLists();
      ID colorScheme_1 = DotAstHelper.getAttributeValue(_attrLists_1, 
        DotAttributes.COLORSCHEME__GCNE);
      boolean _tripleNotEquals_4 = (colorScheme_1 != null);
      if (_tripleNotEquals_4) {
        return colorScheme_1.toValue();
      }
      ID _globalColorSchemeValue_1 = DotAstHelper.getGlobalColorSchemeValue(nodeStmt, AttributeType.NODE);
      colorScheme_1 = _globalColorSchemeValue_1;
      boolean _tripleNotEquals_5 = (colorScheme_1 != null);
      if (_tripleNotEquals_5) {
        return colorScheme_1.toValue();
      }
    }
    final AttrStmt attrStmt = EcoreUtil2.<AttrStmt>getContainerOfType(attribute, AttrStmt.class);
    boolean _tripleNotEquals_6 = (attrStmt != null);
    if (_tripleNotEquals_6) {
      EList<AttrList> _attrLists_2 = attrStmt.getAttrLists();
      ID colorScheme_2 = DotAstHelper.getAttributeValue(_attrLists_2, 
        DotAttributes.COLORSCHEME__GCNE);
      boolean _tripleNotEquals_7 = (colorScheme_2 != null);
      if (_tripleNotEquals_7) {
        return colorScheme_2.toValue();
      }
    }
    final DotGraph dotGraph = EcoreUtil2.<DotGraph>getContainerOfType(attribute, DotGraph.class);
    boolean _tripleNotEquals_8 = (dotGraph != null);
    if (_tripleNotEquals_8) {
      ID colorScheme_3 = DotAstHelper.getAttributeValueAll(dotGraph, DotAttributes.COLORSCHEME__GCNE);
      boolean _tripleNotEquals_9 = (colorScheme_3 != null);
      if (_tripleNotEquals_9) {
        return colorScheme_3.toValue();
      }
      ID _globalColorSchemeValue_2 = DotAstHelper.getGlobalColorSchemeValue(dotGraph, 
        AttributeType.GRAPH);
      colorScheme_3 = _globalColorSchemeValue_2;
      boolean _tripleNotEquals_10 = (colorScheme_3 != null);
      if (_tripleNotEquals_10) {
        return colorScheme_3.toValue();
      }
    }
    return null;
  }
  
  private static ID getGlobalColorSchemeValue(final EObject eObject, final AttributeType attributeType) {
    Object _xblockexpression = null;
    {
      EObject container = EcoreUtil2.<Subgraph>getContainerOfType(eObject, Subgraph.class);
      boolean _tripleNotEquals = (container != null);
      if (_tripleNotEquals) {
        EList<Stmt> _stmts = ((Subgraph) container).getStmts();
        final ID value = DotAstHelper.getAttributeValue(_stmts, attributeType, DotAttributes.COLORSCHEME__GCNE);
        boolean _tripleNotEquals_1 = (value != null);
        if (_tripleNotEquals_1) {
          return value;
        }
      }
      DotGraph _containerOfType = EcoreUtil2.<DotGraph>getContainerOfType(eObject, DotGraph.class);
      container = _containerOfType;
      boolean _tripleNotEquals_2 = (container != null);
      if (_tripleNotEquals_2) {
        EList<Stmt> _stmts_1 = ((DotGraph) container).getStmts();
        final ID value_1 = DotAstHelper.getAttributeValue(_stmts_1, attributeType, DotAttributes.COLORSCHEME__GCNE);
        boolean _tripleNotEquals_3 = (value_1 != null);
        if (_tripleNotEquals_3) {
          return value_1;
        }
      }
      _xblockexpression = null;
    }
    return ((ID)_xblockexpression);
  }
  
  private static ID getAttributeValue(final EList<Stmt> stmts, final AttributeType attributeType, final String attributeName) {
    Object _xblockexpression = null;
    {
      for (final Stmt stmt : stmts) {
        if ((stmt instanceof AttrStmt)) {
          final AttrStmt attrStmt = ((AttrStmt) stmt);
          AttributeType _type = attrStmt.getType();
          boolean _equals = Objects.equal(_type, attributeType);
          if (_equals) {
            EList<AttrList> _attrLists = attrStmt.getAttrLists();
            return DotAstHelper.getAttributeValue(_attrLists, attributeName);
          }
        }
      }
      _xblockexpression = null;
    }
    return ((ID)_xblockexpression);
  }
  
  public static ID getAttributeValue(final DotGraph graph, final String name) {
    Object _xblockexpression = null;
    {
      EList<Stmt> _stmts = graph.getStmts();
      for (final Stmt stmt : _stmts) {
        {
          ID _switchResult = null;
          boolean _matched = false;
          if (!_matched) {
            if (stmt instanceof Attribute) {
              _matched=true;
              _switchResult = DotAstHelper.getAttributeValue(((Attribute)stmt), name);
            }
          }
          ID value = _switchResult;
          boolean _tripleNotEquals = (value != null);
          if (_tripleNotEquals) {
            return value;
          }
        }
      }
      _xblockexpression = null;
    }
    return ((ID)_xblockexpression);
  }
  
  public static ID getAttributeValueAll(final DotGraph graph, final String name) {
    Object _xblockexpression = null;
    {
      EList<Stmt> _stmts = graph.getStmts();
      for (final Stmt stmt : _stmts) {
        {
          ID _switchResult = null;
          boolean _matched = false;
          if (!_matched) {
            if (stmt instanceof AttrStmt) {
              _matched=true;
              EList<AttrList> _attrLists = ((AttrStmt)stmt).getAttrLists();
              _switchResult = DotAstHelper.getAttributeValue(_attrLists, name);
            }
          }
          if (!_matched) {
            if (stmt instanceof Attribute) {
              _matched=true;
              _switchResult = DotAstHelper.getAttributeValue(((Attribute)stmt), name);
            }
          }
          ID value = _switchResult;
          boolean _tripleNotEquals = (value != null);
          if (_tripleNotEquals) {
            return value;
          }
        }
      }
      _xblockexpression = null;
    }
    return ((ID)_xblockexpression);
  }
  
  public static ID getAttributeValue(final Subgraph subgraph, final String name) {
    Object _xblockexpression = null;
    {
      EList<Stmt> _stmts = subgraph.getStmts();
      for (final Stmt stmt : _stmts) {
        {
          ID _switchResult = null;
          boolean _matched = false;
          if (!_matched) {
            if (stmt instanceof Attribute) {
              _matched=true;
              _switchResult = DotAstHelper.getAttributeValue(((Attribute)stmt), name);
            }
          }
          ID value = _switchResult;
          boolean _tripleNotEquals = (value != null);
          if (_tripleNotEquals) {
            return value;
          }
        }
      }
      _xblockexpression = null;
    }
    return ((ID)_xblockexpression);
  }
  
  /**
   * Returns the value of the first attribute with the give name or
   * <code>null</code> if no attribute could be found.
   * 
   * @param attrLists
   *            The {@link AttrList}s to search.
   * @param name
   *            The name of the attribute whose value is to be retrieved.
   * @return The attribute value or <code>null</code> in case the attribute
   *         could not be found.
   */
  public static ID getAttributeValue(final List<AttrList> attrLists, final String name) {
    Object _xblockexpression = null;
    {
      for (final AttrList attrList : attrLists) {
        {
          final ID value = DotAstHelper.getAttributeValue(attrList, name);
          boolean _tripleNotEquals = (value != null);
          if (_tripleNotEquals) {
            return value;
          }
        }
      }
      _xblockexpression = null;
    }
    return ((ID)_xblockexpression);
  }
  
  private static ID getAttributeValue(final AttrList attrList, final String name) {
    EList<Attribute> _attributes = attrList.getAttributes();
    final Function1<Attribute, Boolean> _function = new Function1<Attribute, Boolean>() {
      public Boolean apply(final Attribute it) {
        ID _name = it.getName();
        String _value = _name.toValue();
        return Boolean.valueOf(Objects.equal(_value, name));
      }
    };
    Attribute _findFirst = IterableExtensions.<Attribute>findFirst(_attributes, _function);
    ID _value = null;
    if (_findFirst!=null) {
      _value=_findFirst.getValue();
    }
    return _value;
  }
  
  private static ID getAttributeValue(final Attribute attribute, final String name) {
    Object _xblockexpression = null;
    {
      ID _name = attribute.getName();
      String _value = _name.toValue();
      boolean _equals = _value.equals(name);
      if (_equals) {
        return attribute.getValue();
      }
      _xblockexpression = null;
    }
    return ((ID)_xblockexpression);
  }
}
