/**
 * Copyright (c) 2015 Codetrails GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.epp.internal.logging.aeri.ide.dialogs;

import static org.eclipse.core.databinding.UpdateValueStrategy.POLICY_ON_REQUEST;
import static org.eclipse.emf.databinding.EMFProperties.value;
import static org.eclipse.epp.internal.logging.aeri.ide.dialogs.UI.createCheckButton;
import static org.eclipse.epp.internal.logging.aeri.ide.dialogs.UI.createLabelWithText;
import static org.eclipse.epp.internal.logging.aeri.ide.dialogs.UI.gd;
import static org.eclipse.epp.internal.logging.aeri.ide.dialogs.UI.gdGrabH;
import static org.eclipse.epp.internal.logging.aeri.ide.dialogs.UI.gl;
import static org.eclipse.epp.logging.aeri.core.SystemControl.executeHandler;
import static org.eclipse.jface.databinding.swt.WidgetProperties.selection;
import static org.eclipse.jface.databinding.swt.WidgetProperties.text;

import org.eclipse.core.databinding.DataBindingContext;
import org.eclipse.core.databinding.UpdateValueStrategy;
import org.eclipse.core.databinding.observable.value.IObservableValue;
import org.eclipse.emf.databinding.EMFProperties;
import org.eclipse.epp.internal.logging.aeri.ide.handlers.OpenConfigureServersDialogHandler;
import org.eclipse.epp.internal.logging.aeri.ide.l10n.Messages;
import org.eclipse.epp.logging.aeri.core.IModelPackage;
import org.eclipse.epp.logging.aeri.core.ISystemSettings;
import org.eclipse.epp.logging.aeri.core.SendMode;
import org.eclipse.epp.logging.aeri.core.SystemControl;
import org.eclipse.jface.databinding.swt.ISWTObservableValue;
import org.eclipse.jface.databinding.swt.WidgetProperties;
import org.eclipse.jface.databinding.viewers.IViewerObservableValue;
import org.eclipse.jface.databinding.viewers.ViewersObservables;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.ComboViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.RowLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Link;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPreferencePage;
import org.eclipse.ui.dialogs.PreferencesUtil;

public class PreferencePage extends org.eclipse.jface.preference.PreferencePage
        implements IWorkbenchPreferencePage, IModelPackage.Literals {

    private DataBindingContext context;
    private ISystemSettings systemSettings;

    public PreferencePage() {
        context = new DataBindingContext();
        systemSettings = SystemControl.getSystemSettings();
    }

    @Override
    public void init(IWorkbench workbench) {
        setDescription(Messages.PREFERENCE_PAGE_DESCRIPTION);
    }

    @Override
    protected Control createContents(Composite parent) {
        Composite container = new Composite(parent, SWT.NONE);
        {
            Composite personalGroup = createPersonalGroup(container);
            Group anonymizeDefaults = createAnonymizeControl(container);

            gdGrabH().indent(0, 10).applyTo(personalGroup);
            gdGrabH().indent(0, 10).applyTo(anonymizeDefaults);
        }

        Composite sendModeDefaults = createSendModeControl(container);
        addConfigureButton(container);
        {
            Button debug = createCheckButton(container, Messages.FIELD_LABEL_ENABLE_DEBUG, Messages.TOOLTIP_ENABLE_DEBUG);
            // gdGrabH().align(SWT.END, SWT.CENTER).applyTo(debug);

            IObservableValue swt = selection().observe(debug);
            IObservableValue emf = value(SYSTEM_SETTINGS__DEBUG_ENABLED).observe(systemSettings);
            context.bindValue(swt, emf, new UpdateValueStrategy(POLICY_ON_REQUEST), null);
        }
        addLinks(container);

        gl().margins(5, 5).applyTo(container);
        gd().applyTo(container);

        gd().applyTo(sendModeDefaults);
        applyDialogFont(container);
        return container;
    }

    private Composite createSendModeControl(Composite parent) {
        Composite container = new Composite(parent, SWT.NONE);

        Label label = new Label(container, SWT.NONE);
        label.setText(Messages.LABEL_TEXT_SEND_MODE);
        label.setToolTipText(Messages.TOOLTIP_SEND_MODE);

        ComboViewer sendMode = new ComboViewer(container);
        sendMode.getControl().setToolTipText(Messages.TOOLTIP_SEND_MODE);
        sendMode.setContentProvider(new ArrayContentProvider());
        sendMode.setInput(SendMode.VALUES);
        sendMode.setLabelProvider(new SendModeLabelProvider());

        IViewerObservableValue jface = ViewersObservables.observeSinglePostSelection(sendMode);
        IObservableValue emf = EMFProperties.value(SYSTEM_SETTINGS__SEND_MODE).observe(systemSettings);
        context.bindValue(jface, emf, new UpdateValueStrategy(POLICY_ON_REQUEST), null);

        gl().numColumns(2).applyTo(container);
        gd().align(SWT.BEGINNING, SWT.CENTER).applyTo(label);
        gdGrabH().align(SWT.END, SWT.CENTER).applyTo(sendMode.getCombo());
        return container;
    }

    // TODO Move to UI class. Similar methods in SystemSettingsPage and ConfigureServerDialog
    private Composite createPersonalGroup(Composite parent) {
        Group personalGroup = new Group(parent, SWT.NONE);
        personalGroup.setText(Messages.GROUP_TEXT_CONTACT_INFORMATION);
        gl().margins(5, 5).numColumns(2).applyTo(personalGroup);
        {
            Text name = createLabelWithText(personalGroup, Messages.FIELD_LABEL_NAME, Messages.FIELD_MESSAGE_NAME,
                    Messages.FIELD_MESSAGE_NAME);
            ISWTObservableValue swt = text(SWT.FocusOut).observe(name);
            IObservableValue emf = value(USER_SETTINGS__REPORTER_NAME).observe(systemSettings);
            context.bindValue(swt, emf, new UpdateValueStrategy(POLICY_ON_REQUEST), null);
        }
        {
            String tooltip = Messages.FIELD_MESSAGE_EMAIL + '\n' + Messages.TOOLTIP_SETTINGS_EMAIL;
            Text email = createLabelWithText(personalGroup, Messages.FIELD_LABEL_EMAIL, Messages.FIELD_MESSAGE_EMAIL, tooltip);
            ISWTObservableValue swt = WidgetProperties.text(SWT.Modify).observe(email);
            IObservableValue emf = EMFProperties.value(USER_SETTINGS__REPORTER_EMAIL).observe(systemSettings);
            context.bindValue(swt, emf, new UpdateValueStrategy(POLICY_ON_REQUEST), null);
        }
        return personalGroup;
    }

    private Group createAnonymizeControl(Composite parent) {
        Group container = new Group(parent, SWT.SHADOW_ETCHED_IN | SWT.SHADOW_ETCHED_OUT | SWT.SHADOW_IN | SWT.SHADOW_OUT);
        container.setLayout(new RowLayout(SWT.VERTICAL));
        container.setText(Messages.GROUP_TEXT_ANONYMIZATION);
        {
            Button stackTraces = createCheckButton(container, Messages.FIELD_LABEL_ANONYMIZE_STACKTRACES,
                    Messages.TOOLTIP_SETTINGS_MAKE_STACKTRACE_ANONYMOUS);
            stackTraces.setFocus();
            IObservableValue swt = selection().observe(stackTraces);
            IObservableValue emf = value(USER_SETTINGS__ANONYMIZE_STACK_TRACES).observe(systemSettings);
            context.bindValue(swt, emf, new UpdateValueStrategy(POLICY_ON_REQUEST), null);
        }
        {
            Button messages = createCheckButton(container, Messages.FIELD_LABEL_ANONYMIZE_MESSAGES,
                    Messages.TOOLTIP_SETTINGS_MAKE_MESSAGES_ANONYMOUS);
            IObservableValue swt = selection().observe(messages);
            IObservableValue emf = value(USER_SETTINGS__ANONYMIZE_MESSAGES).observe(systemSettings);
            context.bindValue(swt, emf, new UpdateValueStrategy(POLICY_ON_REQUEST), null);
        }
        {
            Button disableWiringAnalysis = createCheckButton(container, Messages.FIELD_LABEL_DISABLE_AUTOMATIC_WIRING_ANALYSIS,
                    Messages.TOOLTIP_DISABLE_AUTOMATIC_WIRING_ANALYSIS);
            IObservableValue swt = selection().observe(disableWiringAnalysis);
            IObservableValue emf = value(IModelPackage.Literals.USER_SETTINGS__DISABLE_AUTOMATIC_WIRING_ANALYSIS).observe(systemSettings);
            context.bindValue(swt, emf, new UpdateValueStrategy(POLICY_ON_REQUEST), null);
        }
        return container;
    }

    private void addConfigureButton(Composite parent) {
        Button button = new Button(parent, SWT.PUSH);
        button.setText(Messages.BUTTON_TEXT_CONFIGURE_SERVERS);
        button.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                executeHandler(OpenConfigureServersDialogHandler.class);
            }
        });
        gd().align(SWT.END, SWT.CENTER).applyTo(button);
    }

    private void addLinks(Composite parent) {
        Composite links = new Composite(parent, SWT.NONE);
        links.setLayout(new RowLayout(SWT.VERTICAL));
        GridDataFactory.fillDefaults().span(2, 1).grab(true, false).applyTo(links);

        createNotificationsLabelAndLink(links);
        //
        // // placeholder
        // new Label(links, SWT.NONE);
        //
        // // TODO use links from configuration of current server
        // createAndConfigureLink(links, Messages.LINK_LEARN_MORE, HELP_URL);
        // createAndConfigureLink(links, Messages.LINK_PROVIDE_FEEDBACK, FEEDBACK_FORM_URL);
    }

    private void createNotificationsLabelAndLink(Composite links) {
        Link link = new Link(links, SWT.NONE);
        link.setText(Messages.LINK_TEXT_NOTIFICATION_PREFERENCE_PAGE);
        link.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                PreferencesUtil.createPreferenceDialogOn(getShell(),
                        "org.eclipse.mylyn.commons.notifications.preferencePages.Notifications", null, null); //$NON-NLS-1$
            }
        });

    }

    private static String[][] createModeLabelAndValues() {
        SendMode[] modes = SendMode.values();
        String[][] labelAndValues = new String[modes.length][2];
        for (int i = 0; i < modes.length; i++) {
            SendMode mode = modes[i];
            labelAndValues[i][0] = descriptionForMode(mode);
            labelAndValues[i][1] = mode.name();
        }
        return labelAndValues;
    }

    private static String descriptionForMode(SendMode mode) {
        return new SendModeLabelProvider().getText(mode);
    }

    @Override
    public boolean performOk() {
        context.updateModels();
        return super.performOk();
    }

}
