/**
 * Copyright (c) 2015 Codetrails GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.epp.internal.logging.aeri.ide.dialogs;

import static org.apache.commons.lang3.StringUtils.defaultString;
import static org.eclipse.e4.ui.services.IServiceConstants.ACTIVE_SHELL;
import static org.eclipse.emf.databinding.EMFProperties.value;
import static org.eclipse.epp.internal.logging.aeri.ide.di.ImageRegistryCreationFunction.IMG_WIZBAN;
import static org.eclipse.epp.internal.logging.aeri.ide.dialogs.UI.*;
import static org.eclipse.epp.internal.logging.aeri.ide.utils.Servers.copySystemDefaultsIfUnset;
import static org.eclipse.epp.logging.aeri.core.IModelPackage.Literals.*;
import static org.eclipse.epp.logging.aeri.core.util.Links.*;
import static org.eclipse.jface.databinding.swt.WidgetProperties.*;

import javax.inject.Inject;
import javax.inject.Named;

import org.eclipse.core.databinding.DataBindingContext;
import org.eclipse.core.databinding.observable.value.IObservableValue;
import org.eclipse.e4.core.di.annotations.Optional;
import org.eclipse.emf.databinding.EMFProperties;
import org.eclipse.epp.internal.logging.aeri.ide.IServerDescriptor;
import org.eclipse.epp.internal.logging.aeri.ide.l10n.Messages;
import org.eclipse.epp.internal.logging.aeri.ide.utils.Browsers;
import org.eclipse.epp.internal.logging.aeri.ide.utils.Formats;
import org.eclipse.epp.logging.aeri.core.ILink;
import org.eclipse.epp.logging.aeri.core.ISystemSettings;
import org.eclipse.epp.logging.aeri.core.util.Links;
import org.eclipse.jface.databinding.swt.ISWTObservableValue;
import org.eclipse.jface.databinding.swt.WidgetProperties;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.TitleAreaDialog;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.resource.ImageRegistry;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.window.DefaultToolTip;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.RowLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Link;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

public class ConfigureServerDialog extends TitleAreaDialog {

    /**
     * Return code to indicate a cancel using the esc-button.
     */
    public static final int ESC_CANCEL = 42 + 42;

    private static final Point TOOLTIP_DISPLACEMENT = new Point(5, 20);
    private static int TOOLTIP_MS_HIDE_DELAY = 20000;

    private IServerDescriptor server;
    private DataBindingContext context;

    @Inject
    public ConfigureServerDialog(IServerDescriptor server, ISystemSettings settings, @Named(ACTIVE_SHELL) @Optional Shell parentShell,
            ImageRegistry registry) {
        super(parentShell);
        this.server = server;
        setTitleImage(registry.get(IMG_WIZBAN));
        setHelpAvailable(false);
        setShellStyle(SWT.SHELL_TRIM | SWT.RESIZE | SWT.SHEET);
        context = new DataBindingContext();
        copySystemDefaultsIfUnset(settings, server);
    }

    @Override
    protected void createButtonsForButtonBar(Composite parent) {
        createButton(parent, IDialogConstants.OK_ID, Messages.BUTTON_TEXT_ENABLE, true);
        createButton(parent, IDialogConstants.CANCEL_ID, Messages.BUTTON_TEXT_DISABLE, false);
    }

    @Override
    public void create() {
        super.create();
        setTitle(Formats.format(Messages.DIALOG_TITLE_CONFIGURE_SERVER, server.getName()));
        setMessage(Messages.DIALOG_MESSAGE_CONFIGURE_SERVER);
    }

    @Override
    protected Control createDialogArea(Composite parent) {
        Composite container = new Composite(parent, SWT.NONE);
        gl().extendedMargins(5, 5, 5, 0).applyTo(container);
        gdGrabHV().applyTo(container);

        Composite personalGroup = createPersonalGroup(container);
        GridDataFactory.fillDefaults().indent(0, 10).grab(true, false).applyTo(personalGroup);

        Group makeAnonymousGroup = createAnonymizeGroup(container);
        gd().indent(0, 5).applyTo(makeAnonymousGroup);

        ILink termsOfUse = Links.Link(server, REL_TERMS_OF_USE);
        ILink privacyPolicy = Links.Link(server, REL_PRIVACY_POLICY);

        String legalText = null;
        if (termsOfUse != null && privacyPolicy != null) {
            legalText = Formats.format(Messages.LINK_TEXT_ENABLE_ADMONITION_WITH_TERMS_OF_USE_AND_PRIVACY_POLICY,
                    defaultString(termsOfUse.getTitle(), Messages.LINK_TEXT_TERMS_OF_USE), termsOfUse.getHref(),
                    defaultString(privacyPolicy.getTitle(), Messages.LINK_TEXT_PRIVACY_POLICY), privacyPolicy.getHref());
        } else if (termsOfUse != null && privacyPolicy == null) {
            legalText = Formats.format(Messages.LINK_TEXT_ENABLE_ADMONITION_WITH_TERMS_OF_USE,
                    defaultString(termsOfUse.getTitle(), Messages.LINK_TEXT_TERMS_OF_USE), termsOfUse.getHref());
        } else if (termsOfUse == null && privacyPolicy != null) {
            legalText = Formats.format(Messages.LINK_TEXT_ENABLE_ADMONITION_WITH_PRIVACY_POLICY,
                    defaultString(privacyPolicy.getTitle(), Messages.LINK_TEXT_PRIVACY_POLICY), privacyPolicy.getHref());
        }
        if (legalText != null) {
            Link termsAndConditions = new Link(container, SWT.NONE);
            termsAndConditions.setText(legalText);
            termsAndConditions.addSelectionListener(new SelectionAdapter() {
                @Override
                public void widgetSelected(SelectionEvent e) {
                    Browsers.openInExternalBrowser(e.text);
                }
            });
            gd().indent(10, 10).hint(300, SWT.DEFAULT).align(SWT.FILL, SWT.CENTER).applyTo(termsAndConditions);
        }

        Dialog.applyDialogFont(container);
        return container;
    }

    // TODO Move to UI class. Similar methods in PreferencePage and SystemSettingsPage
    private Composite createPersonalGroup(Composite parent) {
        Group personalGroup = new Group(parent, SWT.NONE);
        personalGroup.setText(Messages.GROUP_TEXT_CONTACT_INFORMATION);
        gl().margins(5, 5).numColumns(2).applyTo(personalGroup);
        {
            Text name = createLabelWithText(personalGroup, Messages.FIELD_LABEL_NAME, Messages.FIELD_MESSAGE_NAME,
                    Messages.FIELD_MESSAGE_NAME);
            ISWTObservableValue swt = text(SWT.Modify).observe(name);
            IObservableValue emf = value(USER_SETTINGS__REPORTER_NAME).observe(server);
            context.bindValue(swt, emf);
        }
        {
            String tooltip = Messages.FIELD_MESSAGE_EMAIL + '\n' + Messages.TOOLTIP_SETTINGS_EMAIL;
            Text email = createLabelWithText(personalGroup, Messages.FIELD_LABEL_EMAIL, Messages.FIELD_MESSAGE_EMAIL, tooltip);
            ISWTObservableValue swt = WidgetProperties.text(SWT.Modify).observe(email);
            IObservableValue emf = EMFProperties.value(USER_SETTINGS__REPORTER_EMAIL).observe(server);
            context.bindValue(swt, emf);
        }
        return personalGroup;
    }

    private Group createAnonymizeGroup(Composite container) {
        Group makeAnonymousGroup = new Group(container, SWT.SHADOW_ETCHED_IN | SWT.SHADOW_ETCHED_OUT | SWT.SHADOW_IN | SWT.SHADOW_OUT);
        makeAnonymousGroup.setLayout(new RowLayout(SWT.VERTICAL));
        makeAnonymousGroup.setText(Messages.GROUP_TEXT_ANONYMIZATION);
        {
            Button anonStackTraces = createGroupCheckButton(makeAnonymousGroup, Messages.FIELD_LABEL_ANONYMIZE_STACKTRACES,
                    Messages.TOOLTIP_SETTINGS_MAKE_STACKTRACE_ANONYMOUS);
            anonStackTraces.setFocus();
            IObservableValue swt = selection().observe(anonStackTraces);
            IObservableValue emf = value(USER_SETTINGS__ANONYMIZE_STACK_TRACES).observe(server);
            context.bindValue(swt, emf);
        }
        {
            Button anonMessages = createGroupCheckButton(makeAnonymousGroup, Messages.FIELD_LABEL_ANONYMIZE_MESSAGES,
                    Messages.TOOLTIP_SETTINGS_MAKE_MESSAGES_ANONYMOUS);
            IObservableValue swt = selection().observe(anonMessages);
            IObservableValue emf = value(USER_SETTINGS__ANONYMIZE_MESSAGES).observe(server);
            context.bindValue(swt, emf);
        }
        return makeAnonymousGroup;
    }

    private Button createGroupCheckButton(Group group, String buttonText, String toolTipText) {
        Button button = new Button(group, SWT.CHECK);
        button.setText(buttonText);
        calibrateTooltip(new DefaultToolTip(button), toolTipText);
        return button;
    }

    private void calibrateTooltip(DefaultToolTip toolTip, String toolTipText) {
        toolTip.setText(toolTipText);
        toolTip.setFont(JFaceResources.getDialogFont());
        toolTip.setShift(TOOLTIP_DISPLACEMENT);
        toolTip.setHideDelay(TOOLTIP_MS_HIDE_DELAY);
    }

    @Override
    protected void okPressed() {
        server.setEnabled(true);
        super.okPressed();
    }

    @Override
    protected void cancelPressed() {
        server.setEnabled(false);
        super.cancelPressed();
    }
}
