/*****************************************************************************
 * Copyright (c) 2016 CEA LIST.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   CEA LIST - Initial API and implementation
 *   Nicolas FAUVERGUE (ALL4TEC) nicolas.fauvergue@all4tec.net - Bug 496905
 *
 *****************************************************************************/
package org.eclipse.papyrus.uml.diagram.common.part;

import org.eclipse.core.commands.operations.IUndoContext;
import org.eclipse.emf.common.notify.Adapter;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.Notifier;
import org.eclipse.gef.KeyHandler;
import org.eclipse.gmf.runtime.notation.Diagram;
import org.eclipse.gmf.runtime.notation.NotationPackage;
import org.eclipse.papyrus.infra.core.services.ServiceException;
import org.eclipse.papyrus.infra.core.services.ServicesRegistry;
import org.eclipse.papyrus.infra.gmfdiag.common.SynchronizableGmfDiagramEditor;
import org.eclipse.papyrus.infra.gmfdiag.common.helper.DiagramHelper;
import org.eclipse.papyrus.infra.internationalization.InternationalizationPackage;
import org.eclipse.papyrus.infra.internationalization.common.editor.IInternationalizationEditor;
import org.eclipse.papyrus.infra.internationalization.utils.utils.LabelInternationalization;
import org.eclipse.papyrus.infra.internationalization.utils.utils.LabelInternationalizationUtils;
import org.eclipse.papyrus.infra.ui.lifecycleevents.ISaveAndDirtyService;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.contexts.IContextService;

/**
 * Common ancestor of GMF based editors for UML. This class allows to declare
 * stuff commons to all this kind of editors.
 *
 * @author cedric dumoulin
 *
 */
public class UmlGmfDiagramEditor extends SynchronizableGmfDiagramEditor implements IInternationalizationEditor {

	/**
	 * The associated Diagram.
	 */
	private Diagram diagram;

	/**
	 * Object used to synchronize the name of the editor with the name of the
	 * diagram.
	 */
	private PartNameSynchronizer partNameSynchronizer;

	/**
	 * service registry of the backbone
	 */
	private ServicesRegistry servicesRegistry;

	/**
	 * Constructor.
	 *
	 * @param servicesRegistry
	 * @param diagram
	 * @throws ServiceException
	 */
	public UmlGmfDiagramEditor(ServicesRegistry servicesRegistry, Diagram diagram) throws ServiceException {
		super(true);
		this.diagram = diagram;
		this.servicesRegistry = servicesRegistry;
		// Install synchronizer
		partNameSynchronizer = new PartNameSynchronizer(diagram);

		// Need to manage the part label synchronizer for the table labels
		LabelInternationalizationUtils.managePartLabelSynchronizer(diagram, this);

		// Register this part to the ISaveAndDirtyService.
		// This will allows to be notified of saveAs events, and the isDirty
		// flag will be taken into
		// account.
		ISaveAndDirtyService saveAndDirtyService = servicesRegistry.getService(ISaveAndDirtyService.class);
		saveAndDirtyService.registerIsaveablePart(this);
	}

	/**
	 * Dispose services used in this part.
	 *
	 * @see org.eclipse.gmf.runtime.diagram.ui.resources.editor.parts.DiagramDocumentEditor#dispose()
	 *
	 */
	@Override
	public void dispose() {
		this.setUndoContext(new IUndoContext() {

			@Override
			public String getLabel() {
				return "Disposed undo context";
			}

			@Override
			public boolean matches(IUndoContext context) {
				return false;
			}

		}); // Avoid disposing the shared UndoContext when this nestedEditor is dispose
		// Super.dispose() will try to dispose the IUndoContext

		super.dispose();

		ISaveAndDirtyService saveAndDirtyService;
		try {
			saveAndDirtyService = servicesRegistry.getService(ISaveAndDirtyService.class);
			saveAndDirtyService.removeIsaveablePart(this);

		} catch (ServiceException e) {
			// the service can't be found. Maybe it is already disposed.
			// Do nothing
		}

		partNameSynchronizer = null;
		diagram = null;
		servicesRegistry = null;
	}

	@Override
	public Object getAdapter(Class type) {
		if (type == ServicesRegistry.class) {
			return servicesRegistry;
		}
		return super.getAdapter(type);
	}

	@Override
	protected void stopListening() {
		super.stopListening();
	}

	/**
	 *
	 * @return the backbone service registry. it cannot return null.
	 */
	public ServicesRegistry getServicesRegistry() {
		return servicesRegistry;
	}

	/**
	 * Set the associated Diagram.
	 */
	public void setDiagram(Diagram diagram) {
		this.diagram = diagram;
		partNameSynchronizer.setDiagram(diagram);
	}

	/**
	 * Get the associated Diagram
	 */
	@Override
	public Diagram getDiagram() {
		return diagram;
	}

	/**
	 *
	 * @see org.eclipse.gmf.runtime.diagram.ui.parts.DiagramEditor#getKeyHandler()
	 *
	 * @return
	 */
	@Override
	protected KeyHandler getKeyHandler() {
		// we remove all keybinding provided by GMF
		KeyHandler keyHandler = new KeyHandler();
		return keyHandler;
	}

	/**
	 * A class taking in charge the synchronization of the partName and the
	 * diagram name. When diagram name change, the other is automatically
	 * updated.
	 *
	 * @author cedric dumoulin
	 *
	 */
	public class PartNameSynchronizer {

		Diagram diagram;

		/**
		 * Listener on diagram name change.
		 */
		private Adapter diagramNameListener = new Adapter() {

			@Override
			public void notifyChanged(Notification notification) {
				if (notification.getFeatureID(Diagram.class) == NotationPackage.DIAGRAM__NAME && notification.getNotifier() == diagram) {
					setPartName(LabelInternationalization.getInstance().getDiagramLabel(diagram));
				} else if (notification.getFeature() == InternationalizationPackage.eINSTANCE.getInternationalizationEntry_Value() && notification.getNotifier() == diagram) {
					setPartName(LabelInternationalization.getInstance().getDiagramLabel(diagram));
				}

			}

			@Override
			public Notifier getTarget() {
				return null;
			}

			@Override
			public void setTarget(Notifier newTarget) {
			}

			@Override
			public boolean isAdapterForType(Object type) {
				return false;
			}

		};

		/**
		 *
		 * Constructor.
		 *
		 * @param diagram
		 */
		PartNameSynchronizer(Diagram diagram) {
			setDiagram(diagram);
		}

		/**
		 * Change the associated diagram.
		 *
		 * @param diagram
		 */
		public void setDiagram(Diagram diagram) {
			// Remove from old diagram, if any
			if (this.diagram != null) {
				diagram.eAdapters().remove(diagramNameListener);
			}

			// Set new Diagram
			this.diagram = diagram;
			// Set editor name
			setPartName(LabelInternationalization.getInstance().getDiagramLabel(diagram));
			// Listen to name change
			diagram.eAdapters().add(diagramNameListener);
		}
	}

	@Override
	public void createPartControl(Composite parent) {
		IContextService contextService = (IContextService) getSite().getService(IContextService.class);
		// FIXME : before Eclipse Juno, this line was not necessary
		// see bug 367816 and bug 382218
		contextService.activateContext("org.eclipse.gmf.runtime.diagram.ui.diagramContext"); //$NON-NLS-1$
		super.createPartControl(parent);

	}

	/**
	 * @see org.eclipse.papyrus.infra.internationalization.common.editor.IInternationalizationEditor#modifyPartName(java.lang.String)
	 *
	 * @param name
	 */
	@Override
	public void modifyPartName(final String name) {
		setPartName(name);
	}
	
	/**
	 * {@inheritDoc}
	 * 
	 * @see org.eclipse.papyrus.infra.internationalization.common.editor.IInternationalizationEditor#refreshEditorPart()
	 */
	@Override
	public void refreshEditorPart(){
		DiagramHelper.forceRefresh(getDiagramEditPart());
	}
}
