/******************************************************************************
 * Copyright (c) 2002, 2010 IBM Corporation and others.
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *    IBM Corporation - initial API and implementation
 ****************************************************************************/

package org.eclipse.gmf.runtime.common.ui.internal.resources;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.IPath;


/**
 * Utility class that describes a file change event.
 * 
 * @author Anthony Hunter <a
 *         href="mailto:ahunter@rational.com">ahunter@rational.com </a>
 */
public class FileChangeEvent {

	/**
	 * the event type.
	 */
	private FileChangeEventType eventType;

	/**
	 * the original file of a rename or move event.
	 */
	private IFile oldFile;

	/**
	 * the event file.
	 */
	private IFile file;
	
	/**
	 * The original file's absolute path.
	 */
	private IPath oldFilePath;
	
	/**
	 * The event file's absolute path.
	 */
	private IPath filePath;

	/**
	 * Constructor for a file change event.
	 * 
	 * @param anEventType
	 *            the event type, either changed or deleted.
	 * @param aFile
	 *            the changed or deleted file.
	 */
	public FileChangeEvent(FileChangeEventType anEventType, IFile aFile) {
		assert (anEventType == FileChangeEventType.CHANGED
			|| anEventType == FileChangeEventType.DELETED);
		setEventType(anEventType);
		setFile(aFile);
		setOldFile(null);
	}

	/**
	 * Constructor for a file change event.
	 * 
	 * @param anEventType
	 *            the event type, either moved or renamed.
	 * @param anOldFile
	 *            the original file before the event.
	 * @param newFile
	 *            the new file after the event.
	 */
	public FileChangeEvent(FileChangeEventType anEventType, IFile anOldFile,
			IFile newFile) {
		assert (anEventType == FileChangeEventType.RENAMED
			|| anEventType == FileChangeEventType.MOVED);
		setEventType(anEventType);
		setFile(newFile);
		setOldFile(anOldFile);
	}

	/**
	 * Get the event type.
	 * 
	 * @return the event type.
	 */
	public FileChangeEventType getEventType() {
		return eventType;
	}

	/**
	 * Get the file for the event.
	 * 
	 * @return the file for the event.
	 */
	public IFile getFile() {
		return file;
	}

	/**
	 * Get the original file before a rename or move the event.
	 * 
	 * @return the original file before a rename or move the event.
	 */
	public IFile getOldFile() {
		return oldFile;
	}

	/**
	 * Set the event type.
	 * 
	 * @param type
	 *            the event type.
	 */
	private void setEventType(FileChangeEventType type) {
		this.eventType = type;
	}

	/**
	 * Set the file for the event.
	 * 
	 * @param aFile
	 *            the file for the event.
	 */
	private void setFile(IFile aFile) {
		this.file = aFile;
	}

	/**
	 * Set the original file before a rename or move the event.
	 * 
	 * @param aFile
	 *            the original file before a rename or move the event.
	 */
	private void setOldFile(IFile aFile) {
		this.oldFile = aFile;
	}

	/**
	 * Gets the cached original file's absolute path.
	 * 
	 * @return
	 */
	public IPath getOldFilePath() {
		if (oldFilePath == null) {
			oldFilePath = oldFile.getFullPath();
		}
		
		return oldFilePath;
	}

	/**
	 * Gets the cached event file's absolute path.
	 * 
	 * @return
	 */
	public IPath getFilePath() {
		if (filePath == null) {
			filePath = file.getFullPath();
		}
		
		return filePath;
	}	

}