/**
 * Copyright (c) 2020 CEA LIST.
 * 
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * Contributors:
 *  Ansgar Radermacher  ansgar.radermacher@cea.fr
 */
package org.eclipse.papyrus.robotics.ros2.codegen.python.component;

import java.util.List;
import java.util.Objects;
import org.eclipse.emf.common.util.EList;
import org.eclipse.papyrus.designer.infra.base.StringUtils;
import org.eclipse.papyrus.robotics.core.utils.ParameterUtils;
import org.eclipse.papyrus.robotics.profile.robotics.parameters.ParameterEntry;
import org.eclipse.papyrus.robotics.ros2.codegen.python.utils.RosPythonTypes;
import org.eclipse.uml2.uml.Behavior;
import org.eclipse.uml2.uml.LiteralBoolean;
import org.eclipse.uml2.uml.OpaqueBehavior;
import org.eclipse.uml2.uml.Operation;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.Type;
import org.eclipse.uml2.uml.UMLPackage;
import org.eclipse.uml2.uml.ValueSpecification;
import org.eclipse.uml2.uml.util.UMLUtil;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.StringExtensions;

@SuppressWarnings("all")
public class ParameterTransformations {
  /**
   * Declare ROS 2 parameters, taking default values and descriptions into account
   */
  public static boolean declareParameters(final org.eclipse.uml2.uml.Class component) {
    boolean _xblockexpression = false;
    {
      final Operation declareParamsOp = component.createOwnedOperation("declareParameters", null, null);
      Behavior _createOwnedBehavior = component.createOwnedBehavior(component.getName(), 
        UMLPackage.eINSTANCE.getOpaqueBehavior());
      final OpaqueBehavior declareParamsOb = ((OpaqueBehavior) _createOwnedBehavior);
      declareParamsOp.getMethods().add(declareParamsOb);
      declareParamsOb.getLanguages().add("Python");
      EList<String> _bodies = declareParamsOb.getBodies();
      StringConcatenation _builder = new StringConcatenation();
      {
        List<Property> _allParameters = ParameterUtils.getAllParameters(component);
        for(final Property parameter : _allParameters) {
          final String description = ParameterTransformations.getDescription(parameter);
          _builder.newLineIfNotEmpty();
          String descriptorVar = null;
          _builder.newLineIfNotEmpty();
          {
            if ((description != null)) {
            }
          }
          _builder.append("self.declare_parameter(\'");
          String _name = parameter.getName();
          _builder.append(_name);
          _builder.append("\', ");
          String _strDefaultValue = ParameterTransformations.strDefaultValue(parameter, (descriptorVar != null));
          _builder.append(_strDefaultValue);
          _builder.append(")");
          _builder.newLineIfNotEmpty();
        }
      }
      _xblockexpression = _bodies.add(_builder.toString());
    }
    return _xblockexpression;
  }

  public static boolean initParameters(final org.eclipse.uml2.uml.Class component) {
    boolean _xblockexpression = false;
    {
      final Operation initParamsOp = component.createOwnedOperation("initParameterVars", null, null);
      Behavior _createOwnedBehavior = component.createOwnedBehavior(component.getName(), 
        UMLPackage.eINSTANCE.getOpaqueBehavior());
      final OpaqueBehavior initParamsOb = ((OpaqueBehavior) _createOwnedBehavior);
      initParamsOp.getMethods().add(initParamsOb);
      initParamsOb.getLanguages().add("Python");
      EList<String> _bodies = initParamsOb.getBodies();
      StringConcatenation _builder = new StringConcatenation();
      {
        List<Property> _allParameters = ParameterUtils.getAllParameters(component);
        for(final Property parameter : _allParameters) {
          _builder.append("self.");
          String _varName = StringUtils.varName(parameter.getName());
          _builder.append(_varName);
          _builder.append("_ = self.get_parameter(\"");
          String _name = parameter.getName();
          _builder.append(_name);
          _builder.append("\")");
          _builder.newLineIfNotEmpty();
        }
      }
      _xblockexpression = _bodies.add(_builder.toString());
    }
    return _xblockexpression;
  }

  /**
   * Move parameters into the main components, rename it eventually to
   * ensure that it is a valid identifier and remove default values (as
   * the have been taken into account for parameter declaration already
   * At the end, it also destroys the nested class holding the parameters
   */
  public static void moveParameters(final org.eclipse.uml2.uml.Class component) {
    final org.eclipse.uml2.uml.Class paramSet = ParameterUtils.getParameterClass(component);
    if ((paramSet != null)) {
      Property[] _clone = ((Property[])Conversions.unwrapArray(paramSet.getOwnedAttributes(), Property.class)).clone();
      for (final Property parameter : _clone) {
        {
          component.getOwnedAttributes().add(parameter);
          String _varName = StringUtils.varName(parameter.getName());
          String _plus = (_varName + "_");
          parameter.setName(_plus);
          ValueSpecification _defaultValue = parameter.getDefaultValue();
          boolean _tripleNotEquals = (_defaultValue != null);
          if (_tripleNotEquals) {
            parameter.getDefaultValue().destroy();
          }
        }
      }
      paramSet.destroy();
    }
  }

  /**
   * Produce a string representation of the default value
   * @param nonNull if true, return emptyParameterValue instead of null. This is
   *   useful, since a non-empty description requires the presence of a default value
   */
  public static String strDefaultValue(final Property parameter, final boolean nonNull) {
    ValueSpecification _defaultValue = parameter.getDefaultValue();
    boolean _tripleNotEquals = (_defaultValue != null);
    if (_tripleNotEquals) {
      String value = parameter.getDefaultValue().stringValue();
      Type _type = parameter.getType();
      boolean _tripleNotEquals_1 = (_type != null);
      if (_tripleNotEquals_1) {
        final String typeName = parameter.getType().getName().toLowerCase().replace("std::", "");
        boolean _equals = Objects.equals(typeName, "string");
        if (_equals) {
          value = String.format("std::string(\"%s\")", value);
          parameter.setType(RosPythonTypes.getType(parameter, "ros2Library::stdlib::string"));
        } else {
          ValueSpecification _defaultValue_1 = parameter.getDefaultValue();
          if ((_defaultValue_1 instanceof LiteralBoolean)) {
            value = StringExtensions.toFirstUpper(value);
          }
        }
      }
      return String.format("%s", value);
    }
    if (nonNull) {
      return "rclcpp::ParameterValue()";
    }
    return null;
  }

  public static String getDescription(final Property umlParameter) {
    final ParameterEntry parameter = UMLUtil.<ParameterEntry>getStereotypeApplication(umlParameter, ParameterEntry.class);
    if ((parameter != null)) {
      final String description = parameter.getDescription();
      if ((description != null)) {
        return StringUtils.quote(description.trim());
      }
    }
    return null;
  }
}
