/**
 * Copyright (c) 2014, 2015, 2018, 2020, 2023 Christian W. Damus and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 * Christian W. Damus - Initial API and implementation
 * Ansgar Radermacher - Bug 526155, enable re-generation from profile: copy existing advices
 * Ansgar Radermacher - Bug 526156, reference semantic base element type, bug 582492, move to com.google.inject
 * Camille Letavernier - Bug 569354: remove StereotypeAdvice; use StereotypeMatcherAdvice instead
 */
package org.eclipse.papyrus.uml.profile.types.generator;

import com.google.inject.Inject;
import com.google.inject.Singleton;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Objects;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.gmf.runtime.emf.type.core.ElementTypeRegistry;
import org.eclipse.gmf.runtime.emf.type.core.IElementType;
import org.eclipse.papyrus.infra.types.AbstractAdviceBindingConfiguration;
import org.eclipse.papyrus.infra.types.AbstractEditHelperAdviceConfiguration;
import org.eclipse.papyrus.infra.types.AbstractMatcherConfiguration;
import org.eclipse.papyrus.infra.types.ElementTypeConfiguration;
import org.eclipse.papyrus.infra.types.ElementTypesConfigurationsFactory;
import org.eclipse.papyrus.infra.types.IconEntry;
import org.eclipse.papyrus.infra.types.SpecializationTypeConfiguration;
import org.eclipse.papyrus.infra.types.core.impl.ConfiguredHintedSpecializationElementType;
import org.eclipse.papyrus.uml.profile.types.generator.strategy.ElementTypeConfigHelper;
import org.eclipse.papyrus.uml.types.core.advices.applystereotype.ApplyStereotypeAdviceConfiguration;
import org.eclipse.papyrus.uml.types.core.advices.applystereotype.StereotypeToApply;
import org.eclipse.papyrus.uml.types.core.matchers.stereotype.StereotypeApplicationMatcherConfiguration;
import org.eclipse.papyrus.uml.types.core.matchers.stereotype.StereotypeApplicationMatcherFactory;
import org.eclipse.papyrus.uml.types.core.matchers.stereotype.StereotypeMatcherAdviceConfiguration;
import org.eclipse.uml2.uml.Image;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.Stereotype;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.eclipse.xtext.xbase.lib.StringExtensions;

/**
 * Transformation rule for generating a {@link SpecializationTypeConfiguration} from a UML metaclass {@link Extension}.
 */
@Singleton
@SuppressWarnings("all")
public class ElementTypeRule {
  @Extension
  private static ElementTypesConfigurationsFactory elementtypesconfigurationsFactory = ElementTypesConfigurationsFactory.eINSTANCE;

  @Extension
  private static StereotypeApplicationMatcherFactory stereotypeApplicationMatcherConfigurationFactory = StereotypeApplicationMatcherFactory.eINSTANCE;

  @Inject
  @Extension
  private UMLElementTypes _uMLElementTypes;

  @Inject
  @Extension
  private Identifiers _identifiers;

  @Inject
  @Extension
  private ElementTypeConfigHelper _elementTypeConfigHelper;

  public SpecializationTypeConfiguration toElementType(final ImpliedExtension umlExtension, final ElementTypeConfiguration supertype) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(umlExtension, supertype);
    final SpecializationTypeConfiguration _result;
    synchronized (_createCache_toElementType) {
      if (_createCache_toElementType.containsKey(_cacheKey)) {
        return _createCache_toElementType.get(_cacheKey);
      }
      SpecializationTypeConfiguration _createSpecializationTypeConfiguration = ElementTypeRule.elementtypesconfigurationsFactory.createSpecializationTypeConfiguration();
      _result = _createSpecializationTypeConfiguration;
      _createCache_toElementType.put(_cacheKey, _result);
    }
    _init_toElementType(_result, umlExtension, supertype);
    return _result;
  }

  private final HashMap<ArrayList<?>, SpecializationTypeConfiguration> _createCache_toElementType = CollectionLiterals.newHashMap();

  private void _init_toElementType(final SpecializationTypeConfiguration it, final ImpliedExtension umlExtension, final ElementTypeConfiguration supertype) {
    it.setIdentifier(this._identifiers.toElementTypeID(umlExtension, supertype));
    boolean _hasSemanticSupertype = this._uMLElementTypes.hasSemanticSupertype(supertype);
    if (_hasSemanticSupertype) {
      final String baseTypeId = this._identifiers.toSemanticElementTypeID(umlExtension, this._uMLElementTypes.getElementTypeConfiguration(umlExtension.getMetaclass()));
      final IElementType baseTypeFromRegistry = ElementTypeRegistry.getInstance().getType(baseTypeId);
      if ((baseTypeFromRegistry instanceof ConfiguredHintedSpecializationElementType)) {
        final ElementTypeConfiguration baseType = ((ConfiguredHintedSpecializationElementType) baseTypeFromRegistry).getConfiguration();
        it.getSpecializedTypes().add(baseType);
      } else {
        final SpecializationTypeConfiguration baseType_1 = ElementTypeRule.elementtypesconfigurationsFactory.createSpecializationTypeConfiguration();
        baseType_1.setIdentifier(this._identifiers.toElementTypeID(umlExtension, this._uMLElementTypes.getElementTypeConfiguration(umlExtension.getMetaclass())));
        baseType_1.getSpecializedTypes().add(this._uMLElementTypes.getElementTypeConfiguration(umlExtension.getMetaclass()));
        baseType_1.setHint(this._uMLElementTypes.getElementTypeConfiguration(umlExtension.getMetaclass()).getHint());
        baseType_1.setName(this._identifiers.toElementTypeName(umlExtension, this._uMLElementTypes.getElementTypeConfiguration(umlExtension.getMetaclass())));
        IconEntry icon = this.getIconEntry(umlExtension.getStereotype());
        IconEntry _xifexpression = null;
        boolean _notEquals = (!Objects.equals(icon, null));
        if (_notEquals) {
          _xifexpression = icon;
        } else {
          _xifexpression = this._uMLElementTypes.getIconEntry(umlExtension.getMetaclass());
        }
        baseType_1.setIconEntry(_xifexpression);
        final ElementTypeConfiguration addedBaseType = ConfigurationSetRule.addElementType(baseType_1);
        it.getSpecializedTypes().add(addedBaseType);
      }
    }
    it.getSpecializedTypes().add(supertype);
    it.setHint(supertype.getHint());
    it.setName(this._identifiers.toElementTypeName(umlExtension, supertype));
    this._elementTypeConfigHelper.setSource(it, EcoreUtil.getURI(umlExtension.getStereotype()).toString());
    final IElementType elemTypeFromRegistry = ElementTypeRegistry.getInstance().getType(it.getIdentifier());
    if ((elemTypeFromRegistry instanceof ConfiguredHintedSpecializationElementType)) {
      final ElementTypeConfiguration elemTypeConfigFromRegistry = ((ConfiguredHintedSpecializationElementType)elemTypeFromRegistry).getConfiguration();
      if ((elemTypeConfigFromRegistry instanceof SpecializationTypeConfiguration)) {
        final AbstractEditHelperAdviceConfiguration helperAdviceFromRegistry = ((SpecializationTypeConfiguration) elemTypeConfigFromRegistry).getEditHelperAdviceConfiguration();
        boolean _notEquals_1 = (!Objects.equals(helperAdviceFromRegistry, null));
        if (_notEquals_1) {
          it.setEditHelperAdviceConfiguration(helperAdviceFromRegistry);
        }
      }
    }
    IconEntry icon_1 = this.getIconEntry(umlExtension.getStereotype());
    IconEntry _xifexpression_1 = null;
    boolean _notEquals_2 = (!Objects.equals(icon_1, null));
    if (_notEquals_2) {
      _xifexpression_1 = icon_1;
    } else {
      _xifexpression_1 = this._uMLElementTypes.getIconEntry(umlExtension.getMetaclass());
    }
    it.setIconEntry(_xifexpression_1);
    boolean _hasSemanticSupertype_1 = this._uMLElementTypes.hasSemanticSupertype(supertype);
    boolean _not = (!_hasSemanticSupertype_1);
    if (_not) {
      it.setMatcherConfiguration(this.toMatcherConfiguration(umlExtension, supertype));
    }
  }

  /**
   * Change the stereotype qualified name for this type configuration (i.e. modify
   * the qualified name of the ApplyStereotypeAdvice and StereotypeMatcher)
   * 
   * @param typeConfig
   * 		The element type to edit
   * @param stereotype
   * 		The current stereotype, that was renamed
   */
  public void setStereotypeName(final ElementTypeConfiguration typeConfig, final Stereotype stereotype) {
    final String newName = stereotype.getQualifiedName();
    final ImpliedExtension ext = this._elementTypeConfigHelper.getExtension(typeConfig, stereotype);
    if ((ext == null)) {
      return;
    }
    final List<AbstractAdviceBindingConfiguration> advices = new ElementTypeConfigHelper().getRelatedAdvices(typeConfig);
    int _size = advices.size();
    boolean _notEquals = (_size != 1);
    if (_notEquals) {
      return;
    }
    final AbstractAdviceBindingConfiguration advice = advices.get(0);
    if ((advice instanceof ApplyStereotypeAdviceConfiguration)) {
      int _size_1 = ((ApplyStereotypeAdviceConfiguration)advice).getStereotypesToApply().size();
      boolean _equals = (_size_1 == 1);
      if (_equals) {
        final StereotypeToApply stereotypeToApply = ((ApplyStereotypeAdviceConfiguration)advice).getStereotypesToApply().get(0);
        stereotypeToApply.setStereotypeQualifiedName(newName);
        int _size_2 = stereotypeToApply.getRequiredProfiles().size();
        boolean _equals_1 = (_size_2 == 1);
        if (_equals_1) {
          final String profileName = this.getProfileName(newName);
          boolean _notEquals_1 = (!Objects.equals(profileName, null));
          if (_notEquals_1) {
            stereotypeToApply.getRequiredProfiles().set(0, profileName);
          }
        }
      }
    } else {
      if ((advice instanceof StereotypeMatcherAdviceConfiguration)) {
        int _size_3 = ((StereotypeMatcherAdviceConfiguration)advice).getStereotypesQualifiedNames().size();
        boolean _equals_2 = (_size_3 == 1);
        if (_equals_2) {
          ((StereotypeMatcherAdviceConfiguration)advice).getStereotypesQualifiedNames().set(0, newName);
        }
      }
    }
    if ((typeConfig instanceof SpecializationTypeConfiguration)) {
      final AbstractMatcherConfiguration matcher = ((SpecializationTypeConfiguration)typeConfig).getMatcherConfiguration();
      if ((matcher instanceof StereotypeApplicationMatcherConfiguration)) {
        int _size_4 = ((StereotypeApplicationMatcherConfiguration)matcher).getStereotypesQualifiedNames().size();
        boolean _equals_3 = (_size_4 == 1);
        if (_equals_3) {
          ((StereotypeApplicationMatcherConfiguration)matcher).getStereotypesQualifiedNames().set(0, newName);
        }
      }
    }
    if ((typeConfig instanceof SpecializationTypeConfiguration)) {
      final ElementTypeConfiguration supertype = ((SpecializationTypeConfiguration)typeConfig).getSpecializedTypes().get(0);
      ((SpecializationTypeConfiguration)typeConfig).setName(this._identifiers.toElementTypeName(ext, supertype));
      ((SpecializationTypeConfiguration)typeConfig).setIdentifier(this._identifiers.toElementTypeID(ext, supertype));
    }
  }

  private String getProfileName(final String stereoName) {
    final int sep = stereoName.lastIndexOf(NamedElement.SEPARATOR);
    if ((sep > 0)) {
      final String profileName = stereoName.substring(0, sep);
      return profileName;
    }
    return null;
  }

  private StereotypeMatcherAdviceConfiguration toMatcherConfiguration(final ImpliedExtension umlExtension, final ElementTypeConfiguration supertype) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(umlExtension, supertype);
    final StereotypeMatcherAdviceConfiguration _result;
    synchronized (_createCache_toMatcherConfiguration) {
      if (_createCache_toMatcherConfiguration.containsKey(_cacheKey)) {
        return _createCache_toMatcherConfiguration.get(_cacheKey);
      }
      StereotypeMatcherAdviceConfiguration _createStereotypeMatcherAdviceConfiguration = ElementTypeRule.stereotypeApplicationMatcherConfigurationFactory.createStereotypeMatcherAdviceConfiguration();
      _result = _createStereotypeMatcherAdviceConfiguration;
      _createCache_toMatcherConfiguration.put(_cacheKey, _result);
    }
    _init_toMatcherConfiguration(_result, umlExtension, supertype);
    return _result;
  }

  private final HashMap<ArrayList<?>, StereotypeMatcherAdviceConfiguration> _createCache_toMatcherConfiguration = CollectionLiterals.newHashMap();

  private void _init_toMatcherConfiguration(final StereotypeMatcherAdviceConfiguration it, final ImpliedExtension umlExtension, final ElementTypeConfiguration supertype) {
    final Stereotype umlStereotype = umlExtension.getStereotype();
    String _qualified = this._identifiers.getQualified(StringExtensions.toFirstLower(umlStereotype.getName()));
    String _hintSuffix = this._identifiers.hintSuffix(supertype);
    String _plus = (_qualified + _hintSuffix);
    it.setIdentifier(_plus);
    it.getStereotypesQualifiedNames().add(umlStereotype.getQualifiedName());
    String _name = umlStereotype.getName();
    String _plus_1 = ("Apply Stereotype " + _name);
    it.setDescription(_plus_1);
  }

  private IconEntry getIconEntry(final Stereotype stereotype) {
    IconEntry _xblockexpression = null;
    {
      final Function1<Image, Boolean> _function = (Image it) -> {
        boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(it.getLocation());
        return Boolean.valueOf((!_isNullOrEmpty));
      };
      final Image image = IterableExtensions.<Image>findFirst(stereotype.getIcons(), _function);
      IconEntry _xifexpression = null;
      boolean _notEquals = (!Objects.equals(image, null));
      if (_notEquals) {
        IconEntry _xblockexpression_1 = null;
        {
          final URI uri = URI.createURI(image.getLocation(), true);
          IconEntry _xifexpression_1 = null;
          boolean _notEquals_1 = (!Objects.equals(uri, null));
          if (_notEquals_1) {
            IconEntry _createIconEntry = ElementTypeRule.elementtypesconfigurationsFactory.createIconEntry();
            final Procedure1<IconEntry> _function_1 = (IconEntry it) -> {
              boolean _isPlatform = uri.isPlatform();
              if (_isPlatform) {
                it.setBundleId(uri.segment(1));
                final Function1<String, CharSequence> _function_2 = (String it_1) -> {
                  return URI.decode(it_1);
                };
                String _join = IterableExtensions.<String>join(IterableExtensions.<String>drop(uri.segmentsList(), 2), "/", _function_2);
                String _plus = ("/" + _join);
                it.setIconPath(_plus);
              } else {
                boolean _isRelative = uri.isRelative();
                if (_isRelative) {
                  it.setBundleId(this.containingProject(stereotype).getName());
                  String _decode = URI.decode(uri.toString());
                  String _plus_1 = ("/" + _decode);
                  it.setIconPath(_plus_1);
                } else {
                  it.setIconPath(uri.toString());
                }
              }
            };
            _xifexpression_1 = ObjectExtensions.<IconEntry>operator_doubleArrow(_createIconEntry, _function_1);
          }
          _xblockexpression_1 = _xifexpression_1;
        }
        _xifexpression = _xblockexpression_1;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }

  private IProject containingProject(final EObject object) {
    return ResourcesPlugin.getWorkspace().getRoot().getProject(object.eResource().getURI().segment(2));
  }
}
