/**
 * Copyright (c) 2006-2013, 2021 Borland Software Corporation, CEA LIST, Artal and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 * Dmitry Stadnik (Borland) - initial API and implementation
 * Michael Golubev (Montages) - #386838 - migrate to Xtend2
 * Etienne Allogo (ARTAL) - etienne.allogo@artal.fr - Bug 569174 : 1.4 Merge papyrus extension templates into codegen.xtend
 * Etienne Allogo (ARTAL) - etienne.allogo@artal.fr - Bug 569174 : L1.2 clean up + nonNLS(String) helper
 */
package xpt;

import com.google.inject.Singleton;
import java.util.regex.Pattern;
import org.eclipse.papyrus.gmf.codegen.gmfgen.GenCommonBase;
import org.eclipse.papyrus.gmf.codegen.gmfgen.GenDiagram;
import org.eclipse.papyrus.gmf.codegen.gmfgen.GenEditorGenerator;
import org.eclipse.papyrus.gmf.internal.common.codegen.Conversions;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.IntegerRange;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import xpt.editor.VisualIDRegistry;

/**
 * XXX: [MG] I don't like dependency from Common -> VisualIdRegistry
 */
@Singleton
@SuppressWarnings("all")
public class Common {
  public CharSequence copyright(final GenEditorGenerator it) {
    CharSequence _xifexpression = null;
    String _copyrightText = it.getCopyrightText();
    boolean _tripleNotEquals = (_copyrightText != null);
    if (_tripleNotEquals) {
      CharSequence _xblockexpression = null;
      {
        final String[] split = it.getCopyrightText().split("\n");
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("/**");
        {
          for(final String element : split) {
            _builder.append("\n *");
            {
              boolean _isBlank = element.isBlank();
              boolean _not = (!_isBlank);
              if (_not) {
                _builder.append(" ");
                _builder.append(element);
              }
            }
          }
        }
        _builder.newLineIfNotEmpty();
        _builder.append(" ");
        _builder.append("*/");
        _builder.newLine();
        _xblockexpression = _builder;
      }
      _xifexpression = _xblockexpression;
    }
    return _xifexpression;
  }
  
  public CharSequence xcopyright(final GenEditorGenerator it) {
    StringConcatenation _builder = new StringConcatenation();
    {
      String _copyrightText = it.getCopyrightText();
      boolean _tripleNotEquals = (_copyrightText != null);
      if (_tripleNotEquals) {
        _builder.append("<!--");
        _builder.newLine();
        String _escapeXML = this.escapeXML(it.getCopyrightText());
        _builder.append(_escapeXML);
        _builder.newLineIfNotEmpty();
        _builder.append("-->");
        _builder.newLine();
      }
    }
    return _builder;
  }
  
  public String escapeXML(final CharSequence forXML) {
    return Conversions.escapeXML(forXML.toString());
  }
  
  public CharSequence generatedClassComment() {
    return this.generatedClassComment("");
  }
  
  public CharSequence generatedClassComment(final String comment) {
    return this.doGeneratedComment(comment, "");
  }
  
  public CharSequence generatedMemberComment() {
    return this.doGeneratedComment("", "");
  }
  
  public CharSequence generatedMemberComment(final String comment) {
    return this.doGeneratedComment(comment, "");
  }
  
  public CharSequence generatedMemberComment(final String comment, final String comment2) {
    return this.doGeneratedComment(comment, comment2);
  }
  
  /**
   * XXX: FIXME: merge all generatedXXXcomment to go here
   */
  public CharSequence doGeneratedComment(final String comment, final String comment2) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("/**");
    {
      int _length = comment.length();
      boolean _greaterThan = (_length > 0);
      if (_greaterThan) {
        _builder.append("\n");
        _builder.append(" * ");
        String _replaceAll = comment.replaceAll("\n", "\n * ").replaceAll("\\* \n", "*\n");
        _builder.append(_replaceAll);
        _builder.newLineIfNotEmpty();
        _builder.append(" ");
        _builder.append("*");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append(" ");
    _builder.append("* @generated");
    {
      int _length_1 = comment2.length();
      boolean _greaterThan_1 = (_length_1 > 0);
      if (_greaterThan_1) {
        _builder.append("\n", " ");
        _builder.append("* ");
        String _replaceAll_1 = comment2.replaceAll("\n", 
          "\n* ").replaceAll("\\* \n", "*\n ");
        _builder.append(_replaceAll_1, " ");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append(" ");
    _builder.append("*/");
    _builder.newLine();
    return _builder;
  }
  
  public CharSequence xmlGeneratedTag() {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("<?gmfgen generated=\"true\"?>");
    return _builder;
  }
  
  public CharSequence nonNLS_All(final Iterable<?> list) {
    StringConcatenation _builder = new StringConcatenation();
    {
      boolean _isEmpty = IterableExtensions.isEmpty(list);
      boolean _not = (!_isEmpty);
      if (_not) {
        {
          int _size = IterableExtensions.size(list);
          IntegerRange _upTo = new IntegerRange(1, _size);
          boolean _hasElements = false;
          for(final Integer i : _upTo) {
            if (!_hasElements) {
              _hasElements = true;
            } else {
              _builder.appendImmediate(" ", "");
            }
            CharSequence _nonNLS = this.nonNLS((i).intValue());
            _builder.append(_nonNLS);
          }
        }
      }
    }
    return _builder;
  }
  
  public CharSequence nonNLS() {
    StringConcatenation _builder = new StringConcatenation();
    CharSequence _nonNLS = this.nonNLS(1);
    _builder.append(_nonNLS);
    return _builder;
  }
  
  public CharSequence nonNLS(final Object xptSelf, final int i) {
    StringConcatenation _builder = new StringConcatenation();
    CharSequence _nonNLS = this.nonNLS(i);
    _builder.append(_nonNLS);
    return _builder;
  }
  
  public CharSequence nonNLS(final int xptSelf) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("//$NON-NLS-");
    _builder.append(xptSelf);
    _builder.append("$");
    return _builder;
  }
  
  /**
   * Generates nonNLS(n) a line of java code by counting the quoted strings.
   */
  public CharSequence nonNLS(final String toExternalize) {
    StringConcatenation _builder = new StringConcatenation();
    int count = this.countQuotedStrings(toExternalize);
    {
      if ((count > 0)) {
        {
          IntegerRange _upTo = new IntegerRange(1, count);
          boolean _hasElements = false;
          for(final Integer i : _upTo) {
            if (!_hasElements) {
              _hasElements = true;
            } else {
              _builder.appendImmediate(" ", "");
            }
            CharSequence _nonNLS = this.nonNLS((i).intValue());
            _builder.append(_nonNLS);
          }
        }
      }
    }
    return _builder;
  }
  
  /**
   * Counts quoted strings.
   */
  public int countQuotedStrings(final String toCount) {
    return Long.valueOf(Pattern.compile("\"([^\"]*)\"").matcher(toCount).results().count()).intValue();
  }
  
  /**
   * Provides handy single point to override generation of assert statements
   * TODO refactor this Common.xpt into different flavours - like CommonCodeStyle (nls, assert), CommonSnippets(pkgStmt, setCharset, getSaveOptions) and so on
   * TODO condition.xpandToCharList()->count('"') / 2 gives better guess about number of nonNLS to generate
   */
  public CharSequence _assert(final String condition) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("assert ");
    _builder.append(condition);
    _builder.append(";");
    {
      int _indexOf = condition.indexOf("\"");
      boolean _greaterThan = (_indexOf > 0);
      if (_greaterThan) {
        _builder.append(" ");
        CharSequence _nonNLS = this.nonNLS();
        _builder.append(_nonNLS);
      }
    }
    return _builder;
  }
  
  public CharSequence addShortcutAnnotation(final GenDiagram it, final String viewVar) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("org.eclipse.emf.ecore.EAnnotation shortcutAnnotation = org.eclipse.emf.ecore.EcoreFactory.eINSTANCE.createEAnnotation();");
    _builder.newLine();
    _builder.append("shortcutAnnotation.setSource(\"Shortcut\"); ");
    CharSequence _nonNLS = this.nonNLS();
    _builder.append(_nonNLS);
    _builder.newLineIfNotEmpty();
    _builder.append("shortcutAnnotation.getDetails().put(\"modelID\", ");
    CharSequence _modelID = VisualIDRegistry.modelID(it);
    _builder.append(_modelID);
    _builder.append("); ");
    CharSequence _nonNLS_1 = this.nonNLS();
    _builder.append(_nonNLS_1);
    _builder.newLineIfNotEmpty();
    _builder.append(viewVar);
    _builder.append(".getEAnnotations().add(shortcutAnnotation);");
    _builder.newLineIfNotEmpty();
    return _builder;
  }
  
  public String tripleSpace(final int amount) {
    StringBuilder b = new StringBuilder();
    int counter = 0;
    while ((counter < amount)) {
      {
        b.append("   ");
        counter = (counter + 1);
      }
    }
    return b.toString();
  }
  
  public String stringVisualID(final GenCommonBase it) {
    String _xifexpression = null;
    String _visualIDOverride = it.getVisualIDOverride();
    boolean _tripleNotEquals = (_visualIDOverride != null);
    if (_tripleNotEquals) {
      _xifexpression = it.getVisualIDOverride();
    } else {
      _xifexpression = Integer.valueOf(it.getVisualID()).toString();
    }
    return _xifexpression;
  }
  
  public String stringUniqueIdentifier(final GenCommonBase it) {
    String _xifexpression = null;
    String _visualIDOverride = it.getVisualIDOverride();
    boolean _tripleNotEquals = (_visualIDOverride != null);
    if (_tripleNotEquals) {
      _xifexpression = it.getVisualIDOverride();
    } else {
      _xifexpression = it.getUniqueIdentifier();
    }
    return _xifexpression;
  }
}
