/**
 * Copyright (c) 2014, 2018 Christian W. Damus and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 * Christian W. Damus - Initial API and implementation
 * Ansgar Radermacher - Bug 526156, reference semantic base element type
 */
package org.eclipse.papyrus.uml.profile.types.generator;

import java.util.ArrayList;
import java.util.Arrays;
import javax.inject.Inject;
import javax.inject.Singleton;
import org.eclipse.emf.common.notify.Adapter;
import org.eclipse.emf.common.notify.AdapterFactory;
import org.eclipse.emf.common.util.ResourceLocator;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.edit.provider.IItemLabelProvider;
import org.eclipse.papyrus.infra.types.ElementTypeConfiguration;
import org.eclipse.papyrus.infra.types.ElementTypeSetConfiguration;
import org.eclipse.papyrus.infra.types.SpecializationTypeConfiguration;
import org.eclipse.uml2.common.util.UML2Util;
import org.eclipse.uml2.uml.Stereotype;
import org.eclipse.xtend.lib.annotations.Accessors;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.eclipse.xtext.xbase.lib.Pure;
import org.eclipse.xtext.xbase.lib.StringExtensions;

/**
 * Utility extensions for working with and generating unique identifiers in the element types model.
 */
@Singleton
@SuppressWarnings("all")
public class Identifiers {
  @Inject
  @Extension
  private UMLElementTypes _uMLElementTypes;
  
  @Accessors
  private String prefix;
  
  @Accessors
  private final String umlElementTypesSet = "org.eclipse.papyrus.uml.service.types.UMLElementTypeSet";
  
  @Accessors
  private final String contextId = "org.eclipse.papyrus.infra.services.edit.TypeContext";
  
  @Accessors
  private String baseElementTypesSet = this.umlElementTypesSet;
  
  @Accessors
  private ElementTypeSetConfiguration baseElementTypesSetConfiguration;
  
  @Accessors
  private boolean suppressSemanticSuperElementTypes;
  
  @Accessors
  private AdapterFactory adapterFactory;
  
  private String identifierBase;
  
  private boolean useDiPostfix;
  
  /**
   * Constant for postfix that is appended in case of DI element types
   * @since 2.1
   */
  public static String diPostfix() {
    return ".di";
  }
  
  public void setUseDiPostfix(final boolean useDiPostfix) {
    this.useDiPostfix = useDiPostfix;
  }
  
  public boolean useDiPostfix() {
    return this.useDiPostfix;
  }
  
  public String setIdentifierBase(final org.eclipse.uml2.uml.Package umlPackage) {
    return this.identifierBase = this.prefix;
  }
  
  public String getQualified(final String id) {
    return ((this.identifierBase + ".") + id);
  }
  
  public String toElementTypeID(final ImpliedExtension umlExtension, final ElementTypeConfiguration supertype) {
    String _xblockexpression = null;
    {
      final Stereotype stereo = umlExtension.getStereotype();
      String _xifexpression = null;
      int _size = stereo.getAllExtendedMetaclasses().size();
      boolean _lessEqualsThan = (_size <= 1);
      if (_lessEqualsThan) {
        _xifexpression = stereo.getName();
      } else {
        String _name = stereo.getName();
        String _plus = (_name + "_");
        String _name_1 = umlExtension.getMetaclass().getName();
        _xifexpression = (_plus + _name_1);
      }
      String name = _xifexpression;
      String _qualified = this.getQualified(UML2Util.getValidJavaIdentifier(name));
      String _hintSuffix = this.hintSuffix(supertype);
      _xblockexpression = (_qualified + _hintSuffix);
    }
    return _xblockexpression;
  }
  
  /**
   * Return ID of eventually already existing semantic element types. Removes ".di" postfix from
   * the base identifier. This assures that the user can use different IDs for DI and semantic
   * element types - if the naming convention to use a ".di" postfix compared to the semantic element
   * is followed.
   * @since 2.1
   */
  public String toSemanticElementTypeID(final ImpliedExtension umlExtension, final ElementTypeConfiguration supertype) {
    String _xblockexpression = null;
    {
      final Stereotype stereo = umlExtension.getStereotype();
      String _xifexpression = null;
      int _size = stereo.getAllExtendedMetaclasses().size();
      boolean _lessEqualsThan = (_size <= 1);
      if (_lessEqualsThan) {
        _xifexpression = stereo.getName();
      } else {
        String _name = stereo.getName();
        String _plus = (_name + "_");
        String _name_1 = umlExtension.getMetaclass().getName();
        _xifexpression = (_plus + _name_1);
      }
      String name = _xifexpression;
      String baseId = this.identifierBase;
      String _xifexpression_1 = null;
      boolean _endsWith = baseId.endsWith(Identifiers.diPostfix());
      if (_endsWith) {
        int _length = baseId.length();
        int _minus = (_length - 2);
        String _substring = baseId.substring(0, _minus);
        String _validJavaIdentifier = UML2Util.getValidJavaIdentifier(name);
        String _plus_1 = (_substring + _validJavaIdentifier);
        String _hintSuffix = this.hintSuffix(supertype);
        _xifexpression_1 = (_plus_1 + _hintSuffix);
      } else {
        String _qualified = this.getQualified(UML2Util.getValidJavaIdentifier(name));
        String _hintSuffix_1 = this.hintSuffix(supertype);
        _xifexpression_1 = (_qualified + _hintSuffix_1);
      }
      _xblockexpression = _xifexpression_1;
    }
    return _xblockexpression;
  }
  
  public String toElementTypeName(final ImpliedExtension umlExtension, final ElementTypeConfiguration supertype) {
    String _xblockexpression = null;
    {
      final Stereotype stereo = umlExtension.getStereotype();
      ArrayList<Object> _newArrayList = CollectionLiterals.<Object>newArrayList();
      final Procedure1<ArrayList<Object>> _function = new Procedure1<ArrayList<Object>>() {
        @Override
        public void apply(final ArrayList<Object> it) {
          int _size = stereo.getExtensions().size();
          boolean _greaterThan = (_size > 1);
          if (_greaterThan) {
            it.add(umlExtension.getMetaclass().getName());
          }
          if (((!StringExtensions.isNullOrEmpty(supertype.getHint())) && (IterableExtensions.size(Identifiers.this._uMLElementTypes.getDiagramSpecificElementTypes(umlExtension.getMetaclass())) > 1))) {
            it.add(supertype.getHint());
          }
        }
      };
      final ArrayList<Object> discriminators = ObjectExtensions.<ArrayList<Object>>operator_doubleArrow(_newArrayList, _function);
      String _xifexpression = null;
      boolean _isNullOrEmpty = IterableExtensions.isNullOrEmpty(discriminators);
      if (_isNullOrEmpty) {
        String _xifexpression_1 = null;
        int _size = stereo.getAllExtendedMetaclasses().size();
        boolean _lessEqualsThan = (_size <= 1);
        if (_lessEqualsThan) {
          _xifexpression_1 = stereo.getName();
        } else {
          String _name = stereo.getName();
          String _plus = (_name + " ");
          String _name_1 = umlExtension.getMetaclass().getName();
          _xifexpression_1 = (_plus + _name_1);
        }
        _xifexpression = _xifexpression_1;
      } else {
        String _name_2 = stereo.getName();
        final Function1<Object, CharSequence> _function_1 = new Function1<Object, CharSequence>() {
          @Override
          public CharSequence apply(final Object it) {
            return it.toString();
          }
        };
        String _join = IterableExtensions.<Object>join(discriminators, " (", ", ", ")", _function_1);
        _xifexpression = (_name_2 + _join);
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  protected String _hintSuffix(final ElementTypeConfiguration elementType) {
    return "";
  }
  
  protected String _hintSuffix(final SpecializationTypeConfiguration elementType) {
    String _xifexpression = null;
    boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(elementType.getHint());
    if (_isNullOrEmpty) {
      _xifexpression = "";
    } else {
      String _hint = elementType.getHint();
      _xifexpression = ("_" + _hint);
    }
    return _xifexpression;
  }
  
  protected String _getLabel(final EObject object) {
    String _xblockexpression = null;
    {
      Adapter _adapt = null;
      if (this.adapterFactory!=null) {
        _adapt=this.adapterFactory.adapt(object, IItemLabelProvider.class);
      }
      final IItemLabelProvider labels = ((IItemLabelProvider) _adapt);
      String _text = null;
      if (labels!=null) {
        _text=labels.getText(object);
      }
      _xblockexpression = _text;
    }
    return _xblockexpression;
  }
  
  protected String _getLabel(final EClassifier eClassifier) {
    String _xtrycatchfinallyexpression = null;
    try {
      ResourceLocator _resourceLocator = this.getResourceLocator(eClassifier);
      String _string = null;
      if (_resourceLocator!=null) {
        String _name = eClassifier.getName();
        String _plus = ("_UI_" + _name);
        String _plus_1 = (_plus + "_type");
        _string=_resourceLocator.getString(_plus_1);
      }
      _xtrycatchfinallyexpression = _string;
    } catch (final Throwable _t) {
      if (_t instanceof Exception) {
        _xtrycatchfinallyexpression = eClassifier.getName();
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
    return _xtrycatchfinallyexpression;
  }
  
  private ResourceLocator getResourceLocator(final EObject object) {
    Object _switchResult = null;
    Adapter _adapt = null;
    if (this.adapterFactory!=null) {
      _adapt=this.adapterFactory.adapt(object, IItemLabelProvider.class);
    }
    final Adapter adapter = _adapt;
    boolean _matched = false;
    if (adapter instanceof ResourceLocator) {
      _matched=true;
      _switchResult = ((Object)adapter);
    }
    return ((ResourceLocator)_switchResult);
  }
  
  public String hintSuffix(final ElementTypeConfiguration elementType) {
    if (elementType instanceof SpecializationTypeConfiguration) {
      return _hintSuffix((SpecializationTypeConfiguration)elementType);
    } else if (elementType != null) {
      return _hintSuffix(elementType);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(elementType).toString());
    }
  }
  
  public String getLabel(final EObject eClassifier) {
    if (eClassifier instanceof EClassifier) {
      return _getLabel((EClassifier)eClassifier);
    } else if (eClassifier != null) {
      return _getLabel(eClassifier);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(eClassifier).toString());
    }
  }
  
  @Pure
  public String getPrefix() {
    return this.prefix;
  }
  
  public void setPrefix(final String prefix) {
    this.prefix = prefix;
  }
  
  @Pure
  public String getUmlElementTypesSet() {
    return this.umlElementTypesSet;
  }
  
  @Pure
  public String getContextId() {
    return this.contextId;
  }
  
  @Pure
  public String getBaseElementTypesSet() {
    return this.baseElementTypesSet;
  }
  
  public void setBaseElementTypesSet(final String baseElementTypesSet) {
    this.baseElementTypesSet = baseElementTypesSet;
  }
  
  @Pure
  public ElementTypeSetConfiguration getBaseElementTypesSetConfiguration() {
    return this.baseElementTypesSetConfiguration;
  }
  
  public void setBaseElementTypesSetConfiguration(final ElementTypeSetConfiguration baseElementTypesSetConfiguration) {
    this.baseElementTypesSetConfiguration = baseElementTypesSetConfiguration;
  }
  
  @Pure
  public boolean isSuppressSemanticSuperElementTypes() {
    return this.suppressSemanticSuperElementTypes;
  }
  
  public void setSuppressSemanticSuperElementTypes(final boolean suppressSemanticSuperElementTypes) {
    this.suppressSemanticSuperElementTypes = suppressSemanticSuperElementTypes;
  }
  
  @Pure
  public AdapterFactory getAdapterFactory() {
    return this.adapterFactory;
  }
  
  public void setAdapterFactory(final AdapterFactory adapterFactory) {
    this.adapterFactory = adapterFactory;
  }
}
