/*******************************************************************************
 * Copyright (c) 2008 Ecliptical Software Inc. and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Ecliptical Software Inc. - initial API and implementation
 *******************************************************************************/
package org.eclipse.emf.mint.util;

import java.util.Collections;
import java.util.List;

import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.Notifier;
import org.eclipse.emf.common.notify.impl.AdapterImpl;
import org.eclipse.emf.common.util.BasicEList;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;

/**
 * Adapter for tracking changes to a group of related objects. Each time a
 * related object is added to the group (typically through a reference), changes
 * to its features are tracked and notifications forwarded to registered
 * listeners. This is particularly useful when an object's derived feature
 * depends on a feature that belongs to one of its referenced objects.
 * 
 * <p>
 * Clients may extend this class.
 * </p>
 */
public abstract class GroupAdapterImpl extends AdapterImpl {

	/**
	 * Simple list for tracking notifiers.
	 */
	@SuppressWarnings("serial")
	protected static class NotifierEList extends
			BasicEList.FastCompare<Notifier> {
		@Override
		protected boolean canContainNull() {
			return false;
		}
	}

	/**
	 * Tracked targets (the group).
	 */
	protected NotifierEList targets;

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.emf.common.notify.impl.AdapterImpl#setTarget(org.eclipse.
	 * emf.common.notify.Notifier)
	 */
	@Override
	public void setTarget(Notifier target) {
		if (this.target != null) {
			if (targets == null)
				targets = new NotifierEList();

			targets.add(this.target);
		}

		super.setTarget(target);

		if (target instanceof EObject)
			addGroupTargets(getGroupTargets((EObject) target));
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.emf.common.notify.impl.AdapterImpl#unsetTarget(org.eclipse
	 * .emf.common.notify.Notifier)
	 */
	@Override
	public void unsetTarget(Notifier target) {
		if (target instanceof EObject)
			removeGroupTargets(getGroupTargets((EObject) target));

		if (target == this.target) {
			if (targets == null) {
				super.setTarget(null);
			} else {
				super.setTarget(targets.remove(targets.size() - 1));
				if (targets.isEmpty())
					targets = null;
			}
		} else if (targets != null) {
			if (targets.remove(target) && targets.isEmpty())
				targets = null;
		}
	}

	/**
	 * Disposes the adapter by removing itself from all its targets.
	 */
	public void dispose() {
		Notifier oldTarget = target;
		target = null;

		List<Notifier> oldTargets = targets;
		targets = null;

		if (oldTarget != null)
			oldTarget.eAdapters().remove(this);

		if (oldTargets != null) {
			for (Notifier otherTarget : oldTargets)
				otherTarget.eAdapters().remove(this);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.emf.common.notify.impl.AdapterImpl#notifyChanged(org.eclipse
	 * .emf.common.notify.Notification)
	 */
	public void notifyChanged(Notification msg) {
		Object notifier = msg.getNotifier();
		Object feature = msg.getFeature();
		if (notifier instanceof EObject && feature instanceof EReference) {
			EObject object = (EObject) notifier;
			EReference ref = (EReference) feature;
			switch (msg.getEventType()) {
			case Notification.ADD:
				if (isGroupReference(object, ref)) {
					List<EObject> values = Collections
							.singletonList((EObject) msg.getNewValue());
					addGroupTargets(values);
				}

				break;
			case Notification.ADD_MANY:
				if (isGroupReference(object, ref)) {
					@SuppressWarnings("unchecked")
					List<EObject> values = (List<EObject>) msg.getNewValue();
					addGroupTargets(values);
				}

				break;
			case Notification.REMOVE:
				if (isGroupReference(object, ref)) {
					List<EObject> values = Collections
							.singletonList((EObject) msg.getOldValue());
					removeGroupTargets(values);
				}

				break;
			case Notification.REMOVE_MANY:
				if (isGroupReference(object, ref)) {
					@SuppressWarnings("unchecked")
					List<EObject> values = (List<EObject>) msg.getOldValue();
					removeGroupTargets(values);
				}

				break;
			case Notification.SET:
				if (isGroupReference(object, ref)) {
					if (msg.getOldValue() != null) {
						List<EObject> values = Collections
								.singletonList((EObject) msg.getOldValue());
						removeGroupTargets(values);
					}

					if (msg.getNewValue() != null) {
						List<EObject> values = Collections
								.singletonList((EObject) msg.getNewValue());
						addGroupTargets(values);
					}
				}

				break;
			case Notification.UNSET:
				if (isGroupReference(object, ref) && msg.getOldValue() != null) {
					List<EObject> values = Collections
							.singletonList((EObject) msg.getOldValue());
					removeGroupTargets(values);
				}

				break;
			}
		}
	}

	/**
	 * Adds the given targets to the group of tracked targets.
	 * 
	 * @param groupTargets
	 *            new targets to track
	 */
	protected void addGroupTargets(List<? extends Notifier> groupTargets) {
		for (Notifier groupTarget : groupTargets) {
			if (!groupTarget.eAdapters().contains(this))
				groupTarget.eAdapters().add(this);
		}
	}

	/**
	 * Removes the targets from the group of tracked targets.
	 * 
	 * @param groupTargets
	 *            targets to remove from the group
	 */
	protected void removeGroupTargets(List<? extends Notifier> groupTargets) {
		for (Notifier groupTarget : groupTargets)
			groupTarget.eAdapters().remove(this);
	}

	/**
	 * Returns the object's direct relations.
	 * 
	 * @param object
	 *            object whose direct relations to return
	 * @return object's direct relations
	 */
	protected List<EObject> getGroupTargets(EObject object) {
		BasicEList.FastCompare<EObject> list = new BasicEList.FastCompare<EObject>();
		List<EReference> refs = getGroupReferences(object);
		for (EReference ref : refs) {
			Object value = object.eGet(ref);
			if (ref.isMany()) {
				@SuppressWarnings("unchecked")
				EList<EObject> values = (EList<EObject>) value;
				list.addAll(values);
			} else if (value != null) {
				list.add((EObject) value);
			}
		}

		return list;
	}

	/**
	 * Determines whether the object's reference participates in defining its
	 * relations.
	 * 
	 * @param object
	 *            object
	 * @param ref
	 *            object's reference
	 * @return {@code true} if the given reference contributes its direct
	 *         relations
	 */
	protected boolean isGroupReference(EObject object, EReference ref) {
		List<EReference> list = getGroupReferences(object);
		if (list != null)
			return list.contains(ref);

		return false;
	}

	/**
	 * Returns the object's references that define its direct relations.
	 * 
	 * @param object
	 *            object whose references to return
	 * @return object's references that define its direct relations
	 */
	protected abstract List<EReference> getGroupReferences(EObject object);

	/**
	 * Returns the group's root object.
	 * 
	 * @return the group's root object
	 */
	protected Notifier getRootTarget() {
		return targets == null ? getTarget() : targets.get(0);
	}
}
