/**
 * Copyright (c) 2016, 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package org.eclipse.gemoc.trace.gemoc.traceaddon;

import com.google.common.collect.BiMap;
import com.google.common.collect.HashBiMap;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.transaction.RecordingCommand;
import org.eclipse.emf.transaction.TransactionalEditingDomain;
import org.eclipse.emf.transaction.util.TransactionUtil;
import org.eclipse.gemoc.commons.eclipse.emf.EMFResource;
import org.eclipse.gemoc.executionframework.engine.core.CommandExecution;
import org.eclipse.gemoc.trace.commons.model.launchconfiguration.LaunchConfiguration;
import org.eclipse.gemoc.trace.commons.model.trace.Dimension;
import org.eclipse.gemoc.trace.commons.model.trace.State;
import org.eclipse.gemoc.trace.commons.model.trace.Step;
import org.eclipse.gemoc.trace.commons.model.trace.Trace;
import org.eclipse.gemoc.trace.commons.model.trace.TracedObject;
import org.eclipse.gemoc.trace.commons.model.trace.Value;
import org.eclipse.gemoc.trace.gemoc.api.IMultiDimensionalTraceAddon;
import org.eclipse.gemoc.trace.gemoc.api.IStateManager;
import org.eclipse.gemoc.trace.gemoc.api.ITraceConstructor;
import org.eclipse.gemoc.trace.gemoc.api.ITraceExplorer;
import org.eclipse.gemoc.trace.gemoc.api.ITraceExtractor;
import org.eclipse.gemoc.trace.gemoc.api.ITraceNotifier;
import org.eclipse.gemoc.xdsmlframework.api.core.IExecutionContext;
import org.eclipse.gemoc.xdsmlframework.api.core.IExecutionEngine;
import org.eclipse.gemoc.xdsmlframework.api.engine_addon.IEngineAddon;
import org.eclipse.gemoc.xdsmlframework.api.engine_addon.modelchangelistener.BatchModelChangeListener;
import org.eclipse.gemoc.xdsmlframework.api.extensions.engine_addon.EngineAddonSpecificationExtensionPoint;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.StringExtensions;

@SuppressWarnings("all")
public abstract class AbstractTraceAddon implements IEngineAddon, IMultiDimensionalTraceAddon<Step<?>, State<?, ?>, TracedObject<?>, Dimension<?>, Value<?>> {
  private IExecutionContext<?, ?, ?> _executionContext;
  
  private ITraceExplorer<Step<?>, State<?, ?>, TracedObject<?>, Dimension<?>, Value<?>> traceExplorer;
  
  private ITraceExtractor<Step<?>, State<?, ?>, TracedObject<?>, Dimension<?>, Value<?>> traceExtractor;
  
  private ITraceConstructor traceConstructor;
  
  private ITraceNotifier traceNotifier;
  
  private BatchModelChangeListener traceListener;
  
  private boolean needTransaction = true;
  
  private BatchModelChangeListener listenerAddon;
  
  private Trace<Step<?>, TracedObject<?>, State<?, ?>> trace;
  
  protected boolean activateUpdateEquivalenceClasses = true;
  
  protected abstract ITraceConstructor constructTraceConstructor(final Resource modelResource, final Resource traceResource, final Map<EObject, TracedObject<?>> exeToTraced);
  
  protected abstract IStateManager<State<?, ?>> constructStateManager(final Resource modelResource, final Map<TracedObject<?>, EObject> tracedToExe);
  
  @Override
  public ITraceExplorer<Step<?>, State<?, ?>, TracedObject<?>, Dimension<?>, Value<?>> getTraceExplorer() {
    return this.traceExplorer;
  }
  
  @Override
  public ITraceConstructor getTraceConstructor() {
    return this.traceConstructor;
  }
  
  @Override
  public ITraceExtractor<Step<?>, State<?, ?>, TracedObject<?>, Dimension<?>, Value<?>> getTraceExtractor() {
    return this.traceExtractor;
  }
  
  @Override
  public ITraceNotifier getTraceNotifier() {
    return this.traceNotifier;
  }
  
  @Override
  public void load(final Resource traceResource) {
    final EObject root = IterableExtensions.<EObject>head(traceResource.getContents());
    if ((root instanceof Trace<?, ?, ?>)) {
      this.trace = ((Trace<Step<?>, TracedObject<?>, State<?, ?>>) root);
      GenericTraceExplorer _genericTraceExplorer = new GenericTraceExplorer(this.trace);
      this.traceExplorer = _genericTraceExplorer;
      GenericTraceExtractor _genericTraceExtractor = new GenericTraceExtractor(this.trace, this.activateUpdateEquivalenceClasses);
      this.traceExtractor = _genericTraceExtractor;
    } else {
      this.traceExplorer = null;
      this.traceExtractor = null;
    }
  }
  
  private static String getEPackageFQN(final EPackage p, final String separator) {
    final EPackage superP = p.getESuperPackage();
    if ((superP != null)) {
      String _ePackageFQN = AbstractTraceAddon.getEPackageFQN(superP, separator);
      String _plus = (_ePackageFQN + separator);
      String _name = p.getName();
      return (_plus + _name);
    } else {
      return StringExtensions.toFirstUpper(p.getName());
    }
  }
  
  @Override
  public void aboutToExecuteStep(final IExecutionEngine<?> executionEngine, final Step<?> step) {
    this.manageStep(step, true);
  }
  
  @Override
  public void stepExecuted(final IExecutionEngine<?> engine, final Step<?> step) {
    this.manageStep(step, false);
  }
  
  private void manageStep(final Step<?> step, final boolean add) {
    if ((step != null)) {
      final Runnable _function = () -> {
        this.traceConstructor.addState(this.listenerAddon.getChanges(this));
        if (add) {
          this.traceConstructor.addStep(step);
        } else {
          this.traceConstructor.endStep(step);
        }
        this.traceNotifier.notifyListener(this.traceExtractor);
        this.traceNotifier.notifyListener(this.traceExplorer);
        this.traceNotifier.notifyListeners();
        this.traceExplorer.updateCallStack(step);
      };
      this.modifyTrace(_function);
    }
  }
  
  /**
   * To construct the trace manager
   */
  @Override
  public void engineAboutToStart(final IExecutionEngine<?> engine) {
    if ((this._executionContext == null)) {
      this._executionContext = engine.getExecutionContext();
      this.activateUpdateEquivalenceClasses = (this._executionContext.getRunConfiguration().getAttribute("org.eclipse.gemoc.trace.gemoc.addon_booleanOption", Boolean.valueOf(false))).booleanValue();
      final Resource modelResource = this._executionContext.getResourceModel();
      final ResourceSet rs = new ResourceSetImpl();
      final TransactionalEditingDomain ed = TransactionUtil.getEditingDomain(rs);
      this.needTransaction = (ed != null);
      String _string = this._executionContext.getWorkspace().getExecutionPath().toString();
      String _plus = (_string + "/execution.trace");
      final URI traceModelURI = URI.createPlatformResourceURI(_plus, false);
      final Resource traceResource = rs.createResource(traceModelURI);
      Set<Resource> _relatedResources = EMFResource.getRelatedResources(engine.getExecutionContext().getResourceModel());
      BatchModelChangeListener _batchModelChangeListener = new BatchModelChangeListener(_relatedResources);
      this.listenerAddon = _batchModelChangeListener;
      this.listenerAddon.registerObserver(this);
      final LaunchConfiguration launchConfiguration = engine.extractLaunchConfiguration();
      final BiMap<EObject, TracedObject<?>> exeToTraced = HashBiMap.<EObject, TracedObject<?>>create();
      this.traceConstructor = this.constructTraceConstructor(modelResource, traceResource, exeToTraced);
      final Runnable _function = () -> {
        this.traceConstructor.initTrace(launchConfiguration);
      };
      this.modifyTrace(_function);
      final EObject root = IterableExtensions.<EObject>head(traceResource.getContents());
      if ((root instanceof Trace<?, ?, ?>)) {
        this.trace = ((Trace<Step<?>, TracedObject<?>, State<?, ?>>) root);
        final IStateManager<State<?, ?>> stateManager = this.constructStateManager(modelResource, exeToTraced.inverse());
        GenericTraceExplorer _genericTraceExplorer = new GenericTraceExplorer(this.trace, stateManager);
        this.traceExplorer = _genericTraceExplorer;
        GenericTraceExtractor _genericTraceExtractor = new GenericTraceExtractor(this.trace, this.activateUpdateEquivalenceClasses);
        this.traceExtractor = _genericTraceExtractor;
        Set<Resource> _relatedResources_1 = EMFResource.getRelatedResources(traceResource);
        BatchModelChangeListener _batchModelChangeListener_1 = new BatchModelChangeListener(_relatedResources_1);
        this.traceListener = _batchModelChangeListener_1;
        GenericTraceNotifier _genericTraceNotifier = new GenericTraceNotifier(this.traceListener);
        this.traceNotifier = _genericTraceNotifier;
        this.traceNotifier.addListener(this.traceExtractor);
        this.traceNotifier.addListener(this.traceExplorer);
      }
    }
  }
  
  /**
   * Wrapper using lambda to always use a RecordingCommand when modifying the trace
   */
  private void modifyTrace(final Runnable r, final String message) {
    try {
      if (this.needTransaction) {
        final TransactionalEditingDomain ed = TransactionUtil.getEditingDomain(this._executionContext.getResourceModel());
        final Set<Throwable> catchedException = new HashSet<Throwable>();
        RecordingCommand command = new RecordingCommand(ed, message) {
          @Override
          protected void doExecute() {
            try {
              r.run();
            } catch (final Throwable _t) {
              if (_t instanceof Throwable) {
                final Throwable t = (Throwable)_t;
                catchedException.add(t);
              } else {
                throw Exceptions.sneakyThrow(_t);
              }
            }
          }
        };
        CommandExecution.execute(ed, command);
        boolean _isEmpty = catchedException.isEmpty();
        boolean _not = (!_isEmpty);
        if (_not) {
          throw IterableExtensions.<Throwable>head(catchedException);
        }
      } else {
        r.run();
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  /**
   * Same as above, but without message.
   */
  private void modifyTrace(final Runnable r) {
    this.modifyTrace(r, "");
  }
  
  @Override
  public List<String> validate(final List<IEngineAddon> otherAddons) {
    final ArrayList<String> errors = new ArrayList<String>();
    boolean found = false;
    String addonName = "";
    for (final IEngineAddon iEngineAddon : otherAddons) {
      if (((iEngineAddon instanceof AbstractTraceAddon) && (iEngineAddon != this))) {
        found = true;
        addonName = EngineAddonSpecificationExtensionPoint.getName(iEngineAddon);
      }
    }
    if (found) {
      final String thisName = EngineAddonSpecificationExtensionPoint.getName(this);
      errors.add(((thisName + " can\'t run with ") + addonName));
    }
    return errors;
  }
  
  @Override
  public Trace<?, ?, ?> getTrace() {
    return this.trace;
  }
}
