/**
 * Copyright (c) 2016, 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package org.eclipse.gemoc.trace.gemoc.generator.util;

import com.google.common.base.Objects;
import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import org.eclipse.core.resources.ICommand;
import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.xtext.xbase.lib.Exceptions;

/**
 * Adapted from http://architecturware.cvs.sourceforge.net/viewvc/architecturware/oaw_v4/core.plugin/plugin.oaw4/main/src/org/openarchitectureware/wizards/EclipseHelper.java?revision=1.13&content-type=text%2Fplain
 */
@SuppressWarnings("all")
public class PluginProjectHelper {
  public static final String ISO_8859_1 = "iso-8859-1";
  
  public static IProject createPluginProject(final String projectName, final List<String> srcFolders, final List<IProject> referencedProjects, final Set<String> requiredBundles, final List<String> exportedPackages, final IProgressMonitor progressMonitor) {
    try {
      IProject project = null;
      progressMonitor.beginTask("", 10);
      progressMonitor.subTask(("Creating project " + projectName));
      final IWorkspace workspace = ResourcesPlugin.getWorkspace();
      project = workspace.getRoot().getProject(projectName);
      final IJavaProject javaProject = JavaCore.create(project);
      final IProjectDescription projectDescription = ResourcesPlugin.getWorkspace().newProjectDescription(projectName);
      projectDescription.setLocation(null);
      boolean _exists = project.exists();
      boolean _not = (!_exists);
      if (_not) {
        SubProgressMonitor _subProgressMonitor = new SubProgressMonitor(progressMonitor, 1);
        project.create(projectDescription, _subProgressMonitor);
      }
      final List<IClasspathEntry> classpathEntries = new ArrayList<IClasspathEntry>();
      int _size = referencedProjects.size();
      boolean _notEquals = (_size != 0);
      if (_notEquals) {
        final IProject[] referencedProjectsArray = new IProject[referencedProjects.size()];
        projectDescription.setReferencedProjects(referencedProjects.<IProject>toArray(referencedProjectsArray));
        for (final IProject referencedProject : referencedProjects) {
          {
            final IClasspathEntry referencedProjectClasspathEntry = JavaCore.newProjectEntry(
              referencedProject.getFullPath());
            classpathEntries.add(referencedProjectClasspathEntry);
          }
        }
      }
      final String[] natureIdsArray = { JavaCore.NATURE_ID, "org.eclipse.pde.PluginNature" };
      projectDescription.setNatureIds(natureIdsArray);
      final ICommand java = projectDescription.newCommand();
      java.setBuilderName(JavaCore.BUILDER_ID);
      final ICommand manifest = projectDescription.newCommand();
      manifest.setBuilderName("org.eclipse.pde.ManifestBuilder");
      final ICommand schema = projectDescription.newCommand();
      schema.setBuilderName("org.eclipse.pde.SchemaBuilder");
      final ICommand[] commandArray = { java, manifest, schema };
      projectDescription.setBuildSpec(commandArray);
      SubProgressMonitor _subProgressMonitor_1 = new SubProgressMonitor(progressMonitor, 1);
      project.open(_subProgressMonitor_1);
      SubProgressMonitor _subProgressMonitor_2 = new SubProgressMonitor(progressMonitor, 1);
      project.setDescription(projectDescription, _subProgressMonitor_2);
      Collections.reverse(srcFolders);
      for (final String src : srcFolders) {
        {
          final IFolder srcContainer = project.getFolder(src);
          boolean _exists_1 = srcContainer.exists();
          boolean _not_1 = (!_exists_1);
          if (_not_1) {
            SubProgressMonitor _subProgressMonitor_3 = new SubProgressMonitor(progressMonitor, 1);
            srcContainer.create(false, true, _subProgressMonitor_3);
          }
          final IClasspathEntry srcClasspathEntry = JavaCore.newSourceEntry(srcContainer.getFullPath());
          classpathEntries.add(0, srcClasspathEntry);
        }
      }
      Path _path = new Path(
        "org.eclipse.jdt.launching.JRE_CONTAINER/org.eclipse.jdt.internal.debug.ui.launcher.StandardVMType/J2SE-1.5");
      classpathEntries.add(JavaCore.newContainerEntry(_path));
      Path _path_1 = new Path("org.eclipse.pde.core.requiredPlugins");
      classpathEntries.add(JavaCore.newContainerEntry(_path_1));
      final IClasspathEntry[] classPathEntriesArray = new IClasspathEntry[classpathEntries.size()];
      IClasspathEntry[] _array = classpathEntries.<IClasspathEntry>toArray(classPathEntriesArray);
      SubProgressMonitor _subProgressMonitor_3 = new SubProgressMonitor(progressMonitor, 1);
      javaProject.setRawClasspath(_array, _subProgressMonitor_3);
      Path _path_2 = new Path((("/" + projectName) + "/bin"));
      SubProgressMonitor _subProgressMonitor_4 = new SubProgressMonitor(progressMonitor, 1);
      javaProject.setOutputLocation(_path_2, _subProgressMonitor_4);
      PluginProjectHelper.createManifest(projectName, requiredBundles, exportedPackages, progressMonitor, project);
      PluginProjectHelper.createBuildProps(progressMonitor, project, srcFolders);
      return project;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public static IFile createFile(final String name, final IContainer container, final String content, final IProgressMonitor progressMonitor) {
    try {
      Path _path = new Path(name);
      final IFile file = container.getFile(_path);
      PluginProjectHelper.assertExist(file.getParent());
      byte[] _bytes = content.getBytes(file.getCharset());
      final InputStream stream = new ByteArrayInputStream(_bytes);
      try {
        boolean _exists = file.exists();
        if (_exists) {
          file.setContents(stream, true, true, progressMonitor);
        } else {
          file.create(stream, true, progressMonitor);
        }
        progressMonitor.worked(1);
        return file;
      } finally {
        stream.close();
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private static void createBuildProps(final IProgressMonitor progressMonitor, final IProject project, final List<String> srcFolders) {
    final StringBuilder bpContent = new StringBuilder("source.. = ");
    for (final Iterator<String> iterator = srcFolders.iterator(); iterator.hasNext();) {
      {
        bpContent.append(iterator.next()).append("/");
        boolean _hasNext = iterator.hasNext();
        if (_hasNext) {
          bpContent.append(",");
        }
      }
    }
    bpContent.append("\n");
    bpContent.append("bin.includes = META-INF/,.\n");
    PluginProjectHelper.createFile("build.properties", project, bpContent.toString(), progressMonitor);
  }
  
  private static void createManifest(final String projectName, final Set<String> requiredBundles, final List<String> exportedPackages, final IProgressMonitor progressMonitor, final IProject project) throws CoreException {
    final StringBuilder maniContent = new StringBuilder("Manifest-Version: 1.0\n");
    maniContent.append("Bundle-ManifestVersion: 2\n");
    maniContent.append((("Bundle-Name: " + projectName) + "\n"));
    maniContent.append((("Bundle-SymbolicName: " + projectName) + "; singleton:=true\n"));
    maniContent.append("Bundle-Version: 1.0.0\n");
    boolean _isEmpty = requiredBundles.isEmpty();
    boolean _not = (!_isEmpty);
    if (_not) {
      maniContent.append("Require-Bundle: ");
    }
    for (final String entry : requiredBundles) {
      maniContent.append(((" " + entry) + ",\n"));
    }
    if (((!Objects.equal(exportedPackages, null)) && (!exportedPackages.isEmpty()))) {
      String _get = exportedPackages.get(0);
      String _plus = ("Require-Bundle: " + _get);
      maniContent.append(_plus);
      final int x = exportedPackages.size();
      for (int i = 1; (i < x); i++) {
        String _get_1 = exportedPackages.get(i);
        String _plus_1 = (",\n " + _get_1);
        maniContent.append(_plus_1);
      }
      maniContent.append("\n");
    }
    maniContent.append("Bundle-RequiredExecutionEnvironment: J2SE-1.5\r\n");
    final IFolder metaInf = project.getFolder("META-INF");
    SubProgressMonitor _subProgressMonitor = new SubProgressMonitor(progressMonitor, 1);
    metaInf.create(false, true, _subProgressMonitor);
    PluginProjectHelper.createFile("MANIFEST.MF", metaInf, maniContent.toString(), progressMonitor);
  }
  
  private static void assertExist(final IContainer c) {
    try {
      boolean _exists = c.exists();
      boolean _not = (!_exists);
      if (_not) {
        boolean _exists_1 = c.getParent().exists();
        boolean _not_1 = (!_exists_1);
        if (_not_1) {
          PluginProjectHelper.assertExist(c.getParent());
        }
        if ((c instanceof IFolder)) {
          NullProgressMonitor _nullProgressMonitor = new NullProgressMonitor();
          ((IFolder) c).create(false, true, _nullProgressMonitor);
        }
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
}
